/*
 * Copyright (c) 2023 European Commission
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eu.europa.ec.eudi.openid4vci

import com.nimbusds.jose.JOSEObjectType
import com.nimbusds.jose.JWSAlgorithm
import com.nimbusds.jose.JWSObject
import com.nimbusds.jose.crypto.MACSigner
import com.nimbusds.jose.jwk.JWK
import com.nimbusds.jose.util.JSONObjectUtils
import com.nimbusds.jwt.SignedJWT
import eu.europa.ec.eudi.openid4vci.internal.*
import kotlinx.serialization.Required
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject
import java.net.URL
import java.time.Clock
import java.time.Instant

/**
 * These JWTs are transmitted via HTTP headers in an HTTP request from a Client Instance
 * to an Authorization Server or Resource Server. The primary purpose of these headers is
 * to authenticate the Client Instance
 */
typealias ClientAttestation = Pair<ClientAttestationJWT, ClientAttestationPoPJWT>

/**
 * Qualification of a JWT that adheres to client attestation JWT
 * as described in Attestation-Based Client authentication
 *
 * @param jwt A JSON Web Token (JWT) generated by the client backend
 * which is bound to a key managed by a Client Instance which can then
 * be used by the instance for client authentication
 */
@JvmInline
value class ClientAttestationJWT(val jwt: SignedJWT) {
    init {
        jwt.ensureType(JOSEObjectType(AttestationBasedClientAuthenticationSpec.ATTESTATION_JWT_TYPE))
        requireNotNull(jwt.jwtClaimsSet.issuer) { "Invalid Attestation JWT. Misses `iss` claim" }
        requireNotNull(jwt.jwtClaimsSet.subject) { "Invalid Attestation JWT. Misses `sub` claim" }
        requireNotNull(jwt.jwtClaimsSet.expirationTime) { "Invalid Attestation JWT. Misses `exp` claim" }
        val cnf = requireNotNull(jwt.jwtClaimsSet.cnf()) { "Invalid Attestation JWT. Misses `cnf` claim" }
        requireNotNull(cnf.cnfJwk()) { "Invalid Attestation JWT. Misses `jwk` claim from `cnf`" }
        jwt.ensureSignedOrVerified()
    }

    val clientId: ClientId get() = checkNotNull(jwt.jwtClaimsSet.subject)
    val cnf: JsonObject get() = checkNotNull(jwt.jwtClaimsSet.cnf())
    val publicKey: JWK get() = checkNotNull(cnf.cnfJwk())
}

/**
 * Qualification of a JWT that adheres to client attestation PoP JWT
 * as described in Attestation-Based Client authentication
 *
 * @param jwt A Proof of Possession generated by the Client Instance
 * using the key that the Client Attestation JWT is bound to.
 */
@JvmInline
value class ClientAttestationPoPJWT(val jwt: SignedJWT) {
    init {
        jwt.ensureType(JOSEObjectType(AttestationBasedClientAuthenticationSpec.ATTESTATION_POP_JWT_TYPE))
        requireNotNull(jwt.jwtClaimsSet.issuer) { "Invalid PoP JWT. Misses `iss` claim" }
        val audience = requireNotNull(jwt.jwtClaimsSet.audience) { "Invalid PoP JWT. Misses `aud` claim" }
        require(1 == audience.size) { "Invalid PoP JWT. Has more than one values in `aud` claim" }
        requireNotNull(jwt.jwtClaimsSet.jwtid) { "Invalid PoP JWT. Misses `jti` claim" }
        requireNotNull(jwt.jwtClaimsSet.issueTime) { "Invalid PoP JWT. Misses `iat` claim" }
        jwt.ensureSignedNotMAC()
    }

    val clientId: ClientId get() = checkNotNull(jwt.jwtClaimsSet.issuer)
    val claims: ClientAttestationPOPClaims
        get() =
            JsonSupport.decodeFromString<ClientAttestationPOPClaims>(
                JSONObjectUtils.toJSONString(jwt.jwtClaimsSet.toJSONObject()),
            )
}

@Serializable
data class ClientAttestationPOPClaims(
    @SerialName(RFC7519.ISSUER) @Required val issuer: ClientId,
    @SerialName(RFC7519.AUDIENCE) @Required @Serializable(with = URLSerializer::class) val audience: URL,
    @SerialName(RFC7519.JWT_ID) @Required val jwtId: JwtId,
    @SerialName(RFC7519.ISSUED_AT) @Required @Serializable(with = NumericInstantSerializer::class) val issuedAt: Instant,
    @SerialName(AttestationBasedClientAuthenticationSpec.CHALLENGE_CLAIM) val challenge: Nonce? = null,
    @SerialName(RFC7519.NOT_BEFORE) @Serializable(with = NumericInstantSerializer::class) val notBefore: Instant? = null,
)

//
// Creation of ClientAttestationPoPJWT
//

data class ClientAttestationPoPJWTSpec(
    val signer: Signer<JWK>,
) {
    init {
        requireIsNotMAC(signer.javaAlgorithm.toJoseAlg())
    }
}

/**
 * A function for building a [ClientAttestationPoPJWT]
 * in the context of a [ClientAuthentication.AttestationBased] client
 */
fun interface ClientAttestationPoPBuilder {

    /**
     * Builds a PoP JWT
     *
     * @param clock wallet's clock
     * @param authorizationServerId the issuer claim of the OAuth 2.0 authorization server to which
     * the attestation will be presented for authentication.
     * @receiver the client for which to create the PoP
     *
     * @return the PoP JWT
     */
    suspend fun ClientAuthentication.AttestationBased.attestationPoPJWT(
        clock: Clock,
        authorizationServerId: URL,
        challenge: Nonce?,
    ): ClientAttestationPoPJWT

    companion object {
        val Default: ClientAttestationPoPBuilder = DefaultClientAttestationPoPBuilder
    }
}

internal fun SignedJWT.ensureSignedOrVerified() {
    require(state == JWSObject.State.SIGNED || state == JWSObject.State.VERIFIED) {
        "Provided JWT is not signed"
    }
}

internal fun SignedJWT.ensureSignedNotMAC() {
    ensureSignedOrVerified()
    val alg = requireNotNull(header.algorithm) { "Invalid JWT misses header alg" }
    requireIsNotMAC(alg)
}

internal fun requireIsNotMAC(alg: JWSAlgorithm) =
    require(!alg.isMACSigning()) { "MAC signing algorithm not allowed" }

internal fun JWSAlgorithm.isMACSigning(): Boolean = this in MACSigner.SUPPORTED_ALGORITHMS

private fun SignedJWT.ensureType(expectedType: JOSEObjectType) {
    require(expectedType == header.type) {
        "Expected SignedJWT `typ` to be '${expectedType.type}', but found '${header.type?.type}' instead"
    }
}
