/*
 * Copyright (c) 2023 European Commission
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eu.europa.ec.eudi.openid4vci

import com.nimbusds.jose.jwk.JWK
import com.nimbusds.jwt.SignedJWT
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject

@Serializable
@JvmInline
value class KeyAttestationJWT(val value: String) {

    val attestedKeys: List<JWK>
        get() = validateJwt(value).getOrThrow()

    init {
        validateJwt(value).getOrThrow()
    }

    companion object {

        fun validateJwt(value: String): Result<List<JWK>> = runCatching {
            val jwt = SignedJWT.parse(value)
            jwt.ensureSignedNotMAC()

            require(jwt.header?.type?.type == OpenId4VCISpec.KEY_ATTESTATION_JWT_TYPE) {
                "Invalid Key Attestation JWT. Type must be set to `$OpenId4VCISpec.KEY_ATTESTATION_JWT_TYPE`"
            }

            val claims = Json.parseToJsonElement(jwt.jwtClaimsSet.toString()).jsonObject
            requireNotNull(claims["iat"]) { "Invalid Key Attestation JWT. Misses `iat` claim" }

            val attestedKeysClaimEntries = claims[OpenId4VCISpec.KEY_ATTESTATION_ATTESTED_KEYS]?.jsonArray
            requireNotNull(attestedKeysClaimEntries) { "Invalid Key Attestation JWT. Misses `attested_keys` claim" }
            require(attestedKeysClaimEntries.isNotEmpty()) {
                "Invalid Key Attestation JWT. `attested_keys` claim must not be empty"
            }
            attestedKeysClaimEntries.mapIndexed { index, keyElement ->
                require(keyElement is JsonObject) {
                    "Invalid Key Attestation JWT. Item at index $index in `attested_keys` is not a JSON object."
                }
                runCatching {
                    JWK.parse(keyElement.toString())
                }.getOrElse { e ->
                    throw IllegalArgumentException(
                        "Invalid Key Attestation JWT. Item at index $index in `attested_keys` is not a valid JWK: ${e.message}",
                        e,
                    )
                }.also {
                    require(!it.isPrivate) {
                        "Invalid Key Attestation JWT. Item at index $index in `attested_keys` must be a public key."
                    }
                }
            }
        }
    }
}
