/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.transport;

import io.ably.lib.http.HttpUtils;
import io.ably.lib.transport.ConnectionManager;
import io.ably.lib.transport.ITransport;
import io.ably.lib.transport.SafeSSLSocketFactory;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.Param;
import io.ably.lib.types.ProtocolMessage;
import io.ably.lib.types.ProtocolSerializer;
import io.ably.lib.util.Log;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Timer;
import java.util.TimerTask;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.SSLSession;
import org.java_websocket.WebSocket;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.exceptions.WebsocketNotConnectedException;
import org.java_websocket.framing.Framedata;
import org.java_websocket.handshake.ServerHandshake;

public class WebSocketTransport
implements ITransport {
    private static final String TAG = WebSocketTransport.class.getName();
    private final ITransport.TransportParams params;
    private final ConnectionManager connectionManager;
    private final boolean channelBinaryMode;
    private String wsUri;
    private ITransport.ConnectListener connectListener;
    private WsClient wsConnection;
    private static final int NEVER_CONNECTED = -1;
    private static final int BUGGYCLOSE = -2;
    private static final int CLOSE_NORMAL = 1000;
    private static final int GOING_AWAY = 1001;
    private static final int CLOSE_PROTOCOL_ERROR = 1002;
    private static final int REFUSE = 1003;
    private static final int ABNORMAL_CLOSE = 1006;
    private static final int NO_UTF8 = 1007;
    private static final int POLICY_VALIDATION = 1008;
    private static final int TOOBIG = 1009;
    private static final int EXTENSION = 1010;
    private static final int UNEXPECTED_CONDITION = 1011;
    private static final int TLS_ERROR = 1015;

    protected WebSocketTransport(ITransport.TransportParams params, ConnectionManager connectionManager) {
        this.params = params;
        this.connectionManager = connectionManager;
        this.channelBinaryMode = params.options.useBinaryProtocol;
        params.heartbeats = false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void connect(ITransport.ConnectListener connectListener) {
        this.connectListener = connectListener;
        try {
            boolean isTls = this.params.options.tls;
            String wsScheme = isTls ? "wss://" : "ws://";
            this.wsUri = wsScheme + this.params.host + ':' + this.params.port + "/";
            Param[] authParams = this.connectionManager.ably.auth.getAuthParams();
            Param[] connectParams = this.params.getConnectParams(authParams);
            if (connectParams.length > 0) {
                this.wsUri = HttpUtils.encodeParams(this.wsUri, connectParams);
            }
            Log.d(TAG, "connect(); wsUri = " + this.wsUri);
            WebSocketTransport webSocketTransport = this;
            synchronized (webSocketTransport) {
                this.wsConnection = new WsClient(URI.create(this.wsUri));
                if (isTls) {
                    SSLContext sslContext = SSLContext.getInstance("TLS");
                    sslContext.init(null, null, null);
                    SafeSSLSocketFactory factory = new SafeSSLSocketFactory(sslContext.getSocketFactory());
                    this.wsConnection.setSocketFactory(factory);
                }
            }
            this.wsConnection.connect();
        }
        catch (AblyException e) {
            Log.e(TAG, "Unexpected exception attempting connection; wsUri = " + this.wsUri, e);
            connectListener.onTransportUnavailable(this, e.errorInfo);
        }
        catch (Throwable t) {
            Log.e(TAG, "Unexpected exception attempting connection; wsUri = " + this.wsUri, t);
            connectListener.onTransportUnavailable(this, AblyException.fromThrowable((Throwable)t).errorInfo);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Log.d(TAG, "close()");
        WebSocketTransport webSocketTransport = this;
        synchronized (webSocketTransport) {
            if (this.wsConnection != null) {
                this.wsConnection.close();
                this.wsConnection = null;
            }
        }
    }

    @Override
    public void send(ProtocolMessage msg) throws AblyException {
        Log.d(TAG, "send(); action = " + (Object)((Object)msg.action));
        try {
            if (this.channelBinaryMode) {
                byte[] encodedMsg = ProtocolSerializer.writeMsgpack(msg);
                if (Log.level <= 2) {
                    ProtocolMessage decodedMsg = ProtocolSerializer.readMsgpack(encodedMsg);
                    Log.v(TAG, "send(): " + (Object)((Object)decodedMsg.action) + ": " + new String(ProtocolSerializer.writeJSON(decodedMsg)));
                }
                this.wsConnection.send(encodedMsg);
            } else {
                if (Log.level <= 2) {
                    Log.v(TAG, "send(): " + new String(ProtocolSerializer.writeJSON(msg)));
                }
                this.wsConnection.send(ProtocolSerializer.writeJSON(msg));
            }
        }
        catch (WebsocketNotConnectedException e) {
            if (this.connectListener != null) {
                this.connectListener.onTransportUnavailable(this, AblyException.fromThrowable((Throwable)e).errorInfo);
            }
            throw AblyException.fromThrowable(e);
        }
        catch (Exception e) {
            throw AblyException.fromThrowable(e);
        }
    }

    @Override
    public String getHost() {
        return this.params.host;
    }

    protected void preProcessReceivedMessage(ProtocolMessage message) {
    }

    public String toString() {
        return WebSocketTransport.class.getName() + " {" + this.getURL() + "}";
    }

    @Override
    public String getURL() {
        return this.wsUri;
    }

    class WsClient
    extends WebSocketClient {
        private Timer timer;
        private TimerTask activityTimerTask;
        private long lastActivityTime;
        private boolean shouldExplicitlyVerifyHostname;

        WsClient(URI serverUri) {
            super(serverUri);
            this.timer = new Timer();
            this.activityTimerTask = null;
            this.shouldExplicitlyVerifyHostname = true;
        }

        public void onOpen(ServerHandshake handshakedata) {
            Log.d(TAG, "onOpen()");
            if (((WebSocketTransport)WebSocketTransport.this).params.options.tls && this.shouldExplicitlyVerifyHostname && !this.isHostnameVerified(((WebSocketTransport)WebSocketTransport.this).params.host)) {
                this.close();
            } else {
                WebSocketTransport.this.connectListener.onTransportAvailable(WebSocketTransport.this);
                this.flagActivity();
            }
        }

        private boolean isHostnameVerified(String hostname) {
            SSLSession session = this.getSSLSession();
            if (HttpsURLConnection.getDefaultHostnameVerifier().verify(hostname, session)) {
                Log.v(TAG, "Successfully verified hostname");
                return true;
            }
            Log.e(TAG, "Hostname verification failed, expected " + hostname + ", found " + session.getPeerHost());
            return false;
        }

        public void onMessage(ByteBuffer blob) {
            try {
                ProtocolMessage msg = ProtocolSerializer.readMsgpack(blob.array());
                Log.d(TAG, "onMessage(): msg (binary) = " + msg);
                WebSocketTransport.this.preProcessReceivedMessage(msg);
                WebSocketTransport.this.connectionManager.onMessage(WebSocketTransport.this, msg);
            }
            catch (AblyException e) {
                String msg = "Unexpected exception processing received binary message";
                Log.e(TAG, msg, e);
            }
            this.flagActivity();
        }

        public void onMessage(String string) {
            try {
                ProtocolMessage msg = ProtocolSerializer.fromJSON(string);
                Log.d(TAG, "onMessage(): msg (text) = " + msg);
                WebSocketTransport.this.preProcessReceivedMessage(msg);
                WebSocketTransport.this.connectionManager.onMessage(WebSocketTransport.this, msg);
            }
            catch (AblyException e) {
                String msg = "Unexpected exception processing received text message";
                Log.e(TAG, msg, e);
            }
            this.flagActivity();
        }

        public void onWebsocketPing(WebSocket conn, Framedata f) {
            Log.d(TAG, "onWebsocketPing()");
            super.onWebsocketPing(conn, f);
            this.flagActivity();
        }

        public void onClose(int wsCode, String wsReason, boolean remote) {
            ErrorInfo reason;
            Log.d(TAG, "onClose(): wsCode = " + wsCode + "; wsReason = " + wsReason + "; remote = " + remote);
            switch (wsCode) {
                case -2: 
                case -1: 
                case 1000: 
                case 1001: 
                case 1006: {
                    reason = ConnectionManager.REASON_DISCONNECTED;
                    break;
                }
                case 1003: 
                case 1008: {
                    reason = ConnectionManager.REASON_REFUSED;
                    break;
                }
                case 1009: {
                    reason = ConnectionManager.REASON_TOO_BIG;
                    break;
                }
                default: {
                    reason = ConnectionManager.REASON_FAILED;
                }
            }
            WebSocketTransport.this.connectListener.onTransportUnavailable(WebSocketTransport.this, reason);
            this.dispose();
        }

        public void onError(Exception e) {
            Log.e(TAG, "Connection error ", e);
            WebSocketTransport.this.connectListener.onTransportUnavailable(WebSocketTransport.this, new ErrorInfo(e.getMessage(), 503, 80000));
        }

        protected void onSetSSLParameters(SSLParameters sslParameters) {
            try {
                super.onSetSSLParameters(sslParameters);
                this.shouldExplicitlyVerifyHostname = false;
            }
            catch (NoSuchMethodError exception) {
                Log.w(TAG, "Error when trying to set SSL parameters, most likely due to an old Java API version", exception);
                this.shouldExplicitlyVerifyHostname = true;
            }
        }

        private synchronized void dispose() {
            try {
                this.timer.cancel();
                this.timer = null;
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
        }

        private synchronized void flagActivity() {
            this.lastActivityTime = System.currentTimeMillis();
            WebSocketTransport.this.connectionManager.setLastActivity(this.lastActivityTime);
            if (this.activityTimerTask == null && ((WebSocketTransport)WebSocketTransport.this).connectionManager.maxIdleInterval != 0L) {
                this.checkActivity();
            }
        }

        private synchronized void checkActivity() {
            long next;
            long timeout = ((WebSocketTransport)WebSocketTransport.this).connectionManager.maxIdleInterval;
            if (timeout == 0L) {
                Log.v(TAG, "checkActivity: infinite timeout");
                return;
            }
            if (this.activityTimerTask != null) {
                return;
            }
            long now = System.currentTimeMillis();
            if (now < (next = this.lastActivityTime + (timeout += ((WebSocketTransport)WebSocketTransport.this).connectionManager.ably.options.realtimeRequestTimeout))) {
                Log.v(TAG, "checkActivity: ok");
                this.activityTimerTask = new TimerTask(){

                    @Override
                    public void run() {
                        try {
                            WsClient.this.checkActivity();
                        }
                        catch (Throwable t) {
                            Log.e(TAG, "Unexpected exception in activity timer handler", t);
                        }
                    }
                };
                this.schedule(this.activityTimerTask, next - now);
            } else {
                Log.e(TAG, "No activity for " + timeout + "ms, closing connection");
                this.closeConnection(1006, "timed out");
            }
        }

        private synchronized void schedule(TimerTask task, long delay) {
            if (this.timer != null) {
                try {
                    this.timer.schedule(task, delay);
                }
                catch (IllegalStateException ise) {
                    Log.e(TAG, "Unexpected exception scheduling activity timer", ise);
                }
            }
        }
    }

    public static class Factory
    implements ITransport.Factory {
        @Override
        public WebSocketTransport getTransport(ITransport.TransportParams params, ConnectionManager connectionManager) {
            return new WebSocketTransport(params, connectionManager);
        }
    }
}

