/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant;

import com.google.common.collect.ImmutableSet;
import io.confluent.kafka.multitenant.LogicalClusterMetadata;
import io.confluent.kafka.multitenant.PhysicalClusterMetadata;
import io.confluent.kafka.multitenant.SslCertificateManager;
import io.confluent.kafka.multitenant.TenantLifecycleManager;
import io.confluent.kafka.multitenant.Utils;
import io.confluent.kafka.multitenant.quota.QuotaConfig;
import io.confluent.kafka.multitenant.quota.TenantQuotaCallback;
import io.confluent.kafka.multitenant.quota.TestCluster;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.AlterConfigsOptions;
import org.apache.kafka.clients.admin.MockAdminClient;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.quota.ClientQuotaType;
import org.apache.kafka.test.TestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;

public class PhysicalClusterMetadataTest {
    private static final Long TEST_CACHE_RELOAD_DELAY_MS = TimeUnit.SECONDS.toMillis(5L);
    private static final long TEST_MAX_WAIT_MS = TimeUnit.SECONDS.toMillis(60L);
    private static final String SSL_CERTS_DIR = "mnt/sslcerts/";
    private static final String BROKER_ID = "0";
    private static final String BROKER_UUID = "test-uuid-3";
    private static final URL TEST_SSL_CERTS_AUG = PhysicalClusterMetadataTest.class.getResource("/cert_exp_aug");
    private static final URL TEST_SSL_CERTS_MAY = PhysicalClusterMetadataTest.class.getResource("/cert_exp_may");
    private AdminClient mockAdminClient;
    private MockTime time;
    private Metrics metrics;
    private PhysicalClusterMetadata lcCache;
    private String sslCertsPath;
    private String endpoint;
    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();
    private String fullchain1 = "-----BEGIN CERTIFICATE-----\nMIIFfDCCBGSgAwIBAgISBFmAQQIS/v9qbmrDF5BxHGeWMA0GCSqGSIb3DQEBCwUA\nMEoxCzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1MZXQncyBFbmNyeXB0MSMwIQYDVQQD\nExpMZXQncyBFbmNyeXB0IEF1dGhvcml0eSBYMzAeFw0xOTAzMDIwNDMwNDhaFw0x\nOTA1MzEwNDMwNDhaMC0xKzApBgNVBAMMIioudXMtY2VudHJhbDEuZ2NwLnByaXYu\nY3BkZXYuY2xvdWQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQD7fcdS\nHYnMBb7zL2Nez+Mfy0g2FEBAT+duDDPGlP9AVyw3D7mZxIhaIy/kz3oWguXrtvZA\nbvmLoTvnYm4suUD/iWpfr5OBOunnzSb772TbCTMjVrrVjjaKR+hHMscGMeV7O8yv\nG3urz8KA7ms66COIXWFE1cKTIfhTqoIj0sgQ0J+WuLHvqrn5V0F5P0oZ2rRBMcUS\nmO2/FGUgGMs/Tnh+6DdE1GuBeQxztLRMWTlKWKY6zKd5H305ef1uvGhMgDkBnR97\n0nFi6rjvbL+OXfQuYXf/rhR1MtjHTNprmGBVhqHci3oZdE1vFvnxUQ98nY/ibOgZ\nWrEXuXc9QN5MgZeJAgMBAAGjggJ3MIICczAOBgNVHQ8BAf8EBAMCBaAwHQYDVR0l\nBBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYE\nFL+UaLPv4/rIx38v2b2kw1YvJIOwMB8GA1UdIwQYMBaAFKhKamMEfd265tE5t6ZF\nZe/zqOyhMG8GCCsGAQUFBwEBBGMwYTAuBggrBgEFBQcwAYYiaHR0cDovL29jc3Au\naW50LXgzLmxldHNlbmNyeXB0Lm9yZzAvBggrBgEFBQcwAoYjaHR0cDovL2NlcnQu\naW50LXgzLmxldHNlbmNyeXB0Lm9yZy8wLQYDVR0RBCYwJIIiKi51cy1jZW50cmFs\nMS5nY3AucHJpdi5jcGRldi5jbG91ZDBMBgNVHSAERTBDMAgGBmeBDAECATA3Bgsr\nBgEEAYLfEwEBATAoMCYGCCsGAQUFBwIBFhpodHRwOi8vY3BzLmxldHNlbmNyeXB0\nLm9yZzCCAQQGCisGAQQB1nkCBAIEgfUEgfIA8AB2AHR+2oMxrTMQkSGcziVPQnDC\nv/1eQiAIxjc1eeYQe8xWAAABaTziKRUAAAQDAEcwRQIhAJ8fxSwNiqLbtz/pCCPY\ntA+yyTJ5jhwbxIxeDg7x1q6FAiA3ClZpy5EVG/ng997wyZjWW8n6dH9Owee8wjVV\nOXK1VAB2ACk8UZZUyDlluqpQ/FgH1Ldvv1h6KXLcpMMM9OVFR/R4AAABaTziKSoA\nAAQDAEcwRQIhAMPvT2P/PY6xux8Jf7T8ZhiluaSa8PUQDXqJNrd3I5rMAiBiaaAX\nJsWP2RZrUXGLECy8s5L8kUPD5ZpBmFCaxUnl6zANBgkqhkiG9w0BAQsFAAOCAQEA\nMChPeoJ15MwxFuVc2FuHFGyBWuKrOSMcQ+1l8T86IOV5xh8grnZO55hlO2jbfXkF\nsWQpSMyfi4QRyX3U5o9R4HzsnC2zmFoPZ6SkYM/SUJX6qY0asW5+WmQk920EXuuZ\naP1lGAIbZXOI4OmJDBeVYzSFQOh8o5Mbsa9geGxUSLQoRf1KAjGIbsFlGwe+/7gN\nzpmPfciV+hwM9QGO3BXOV3MWdi8UvHppr9YpjuPbCNnVlm6Cqq8KxoKSa/DS1MrK\n1YMkFSGZJoDJFi5AAhBgUW1i5KEi1bfwleDQfJtoGcXH/0CpdnVfMWdYejHEyn2b\nIjD2mMLVhq9uEr1O6l9C5g==\n-----END CERTIFICATE-----\n\n-----BEGIN CERTIFICATE-----\nMIIEkjCCA3qgAwIBAgIQCgFBQgAAAVOFc2oLheynCDANBgkqhkiG9w0BAQsFADA/\nMSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT\nDkRTVCBSb290IENBIFgzMB4XDTE2MDMxNzE2NDA0NloXDTIxMDMxNzE2NDA0Nlow\nSjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUxldCdzIEVuY3J5cHQxIzAhBgNVBAMT\nGkxldCdzIEVuY3J5cHQgQXV0aG9yaXR5IFgzMIIBIjANBgkqhkiG9w0BAQEFAAOC\nAQ8AMIIBCgKCAQEAnNMM8FrlLke3cl03g7NoYzDq1zUmGSXhvb418XCSL7e4S0EF\nq6meNQhY7LEqxGiHC6PjdeTm86dicbp5gWAf15Gan/PQeGdxyGkOlZHP/uaZ6WA8\nSMx+yk13EiSdRxta67nsHjcAHJyse6cF6s5K671B5TaYucv9bTyWaN8jKkKQDIZ0\nZ8h/pZq4UmEUEz9l6YKHy9v6Dlb2honzhT+Xhq+w3Brvaw2VFn3EK6BlspkENnWA\na6xK8xuQSXgvopZPKiAlKQTGdMDQMc2PMTiVFrqoM7hD8bEfwzB/onkxEz0tNvjj\n/PIzark5McWvxI0NHWQWM6r6hCm21AvA2H3DkwIDAQABo4IBfTCCAXkwEgYDVR0T\nAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAYYwfwYIKwYBBQUHAQEEczBxMDIG\nCCsGAQUFBzABhiZodHRwOi8vaXNyZy50cnVzdGlkLm9jc3AuaWRlbnRydXN0LmNv\nbTA7BggrBgEFBQcwAoYvaHR0cDovL2FwcHMuaWRlbnRydXN0LmNvbS9yb290cy9k\nc3Ryb290Y2F4My5wN2MwHwYDVR0jBBgwFoAUxKexpHsscfrb4UuQdf/EFWCFiRAw\nVAYDVR0gBE0wSzAIBgZngQwBAgEwPwYLKwYBBAGC3xMBAQEwMDAuBggrBgEFBQcC\nARYiaHR0cDovL2Nwcy5yb290LXgxLmxldHNlbmNyeXB0Lm9yZzA8BgNVHR8ENTAz\nMDGgL6AthitodHRwOi8vY3JsLmlkZW50cnVzdC5jb20vRFNUUk9PVENBWDNDUkwu\nY3JsMB0GA1UdDgQWBBSoSmpjBH3duubRObemRWXv86jsoTANBgkqhkiG9w0BAQsF\nAAOCAQEA3TPXEfNjWDjdGBX7CVW+dla5cEilaUcne8IkCJLxWh9KEik3JHRRHGJo\nuM2VcGfl96S8TihRzZvoroed6ti6WqEBmtzw3Wodatg+VyOeph4EYpr/1wXKtx8/\nwApIvJSwtmVi4MFU5aMqrSDE6ea73Mj2tcMyo5jMd6jmeWUHK8so/joWUoHOUgwu\nX4Po1QYz+3dszkDqMp4fklxBwXRsW10KXzPMTZ+sOPAveyxindmjkW8lGy+QsRlG\nPfZ+G6Z6h7mjem0Y+iWlkYcV4PIWL1iwBi8saCbGS5jN2p8M+X+Q7UNKEkROb3N6\nKOqkqm57TH2H3eDJAkSnh6/DNFu0Qg==\n-----END CERTIFICATE-----\n";
    private String privkey1 = "-----BEGIN RSA PRIVATE KEY-----\nMIIEowIBAAKCAQEA+33HUh2JzAW+8y9jXs/jH8tINhRAQE/nbgwzxpT/QFcsNw+5\nmcSIWiMv5M96FoLl67b2QG75i6E752JuLLlA/4lqX6+TgTrp580m++9k2wkzI1a6\n1Y42ikfoRzLHBjHlezvMrxt7q8/CgO5rOugjiF1hRNXCkyH4U6qCI9LIENCflrix\n76q5+VdBeT9KGdq0QTHFEpjtvxRlIBjLP054fug3RNRrgXkMc7S0TFk5SlimOsyn\neR99OXn9brxoTIA5AZ0fe9JxYuq472y/jl30LmF3/64UdTLYx0zaa5hgVYah3It6\nGXRNbxb58VEPfJ2P4mzoGVqxF7l3PUDeTIGXiQIDAQABAoIBAH8DkEY1quGCyWSy\nu0IoRjJJjafaZHTWpjCbMw8JMz0AidEpPPifHKpBeS/bZXK3G34HwqjaI2hUvxdm\nS/SEf4JPmYzH9Pxgj7/FifnVdx90rwIbDHNMxtjh5jsHNyM20gqCMicB/1zPqhFJ\n2JhAo6l8V+LW/tUmY++FfwKusuJiKtq8gTAhaf+qr9ARnNwRQ0hq+BwEr6XD9ful\nrfLEIl3ecvaKXjj6b1inrG2yVN3coibNuDkLXY2EgJG64n1vTu6aBMj0P6HeiXiH\nm04OrTP43nQrKV/+7IzrBlqEMzHtj09OgWvXkV6NhIEWswkkY+HkzLj9hgoDdV4Z\nFmmmbV0CgYEA/O6cPK4K+ElrGvyJRlkRuD3WzZByaqJyeR6Jdm3qNCQ/9bI5PH/X\ngpPqHaFAA0RVlUiV4dZ5BksxW6Lvokbv/i1xlAs8TFobdr5+qcO518e5C8V+CCut\ny01LOj0I9ZvFdAHJP514QgHLHhG8/Ua5/kgFWtnZWi4mhak6T04NOk8CgYEA/oqx\nzzHHhpaWS4tfQEL0Ibxx1VKT8M4R3QkOnngLJonA1lfqSujIeodLZWm6uzcy30u7\nikCuoKCxK0vE8dOltopwTtlTOuYI2JfdvmFaanOG/857VoFIQZmI8bakaQPr4PXe\nffDZ22k4n4B7k05fZ1MqaCHnZPI1niamfGkJEqcCgYEA0G35BfAOTiiCQIzWusfv\nWDptZpygDMutNa46bQOKukkdA+VIUViwSYSGqsAUthx7wjc8fAx3Uv5nwDH2820t\nm/Hq5KqVl/2xIBs+2brWzMBi9xZaE3WbFCuv0GA3n94ryrsmEmw7i3la3n6TlMvR\nvX+wGfvnpu7dA8w+pteVAvUCgYAO//FWemJ9peYZcY8dZFSqoEY9Ae7B5ALdeako\n4X4WuUtp1ihyXaFixxJEWaStX6VZz0av8PvZb17BZGeosIY1aZcQrnHfKKsgyGJC\n083WNBSignJ2OIwfgYK2a8LohVijGxoPZeAQs/SoQZQGrDmnBxmapVTTeAp81V4+\nOppURQKBgFBqIIXH06SH4FVxTbo+cq6gOHuRIXyLR++AyzJyn8wVfHVJ01r6zhGd\nazTCm+fyfw/6jsJfK3b8mLjiz39GAM7vyvkVV5FKYC2wQldxPyow/MhmKBjdqp2p\n5gq8MSFaXoZOtRgMWdT4JPLTi7xp8429yRM8JK8E6A9+5aTkUM7H\n-----END RSA PRIVATE KEY-----\n";
    private String fullchain2 = "-----BEGIN CERTIFICATE-----\nMIIFfDCCBGSgAwIBAgISA+SIWPXUZozkvzhi5xKWfne+MA0GCSqGSIb3DQEBCwUA\nMEoxCzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1MZXQncyBFbmNyeXB0MSMwIQYDVQQD\nExpMZXQncyBFbmNyeXB0IEF1dGhvcml0eSBYMzAeFw0xOTA1MTcwMzU5MTdaFw0x\nOTA4MTUwMzU5MTdaMC0xKzApBgNVBAMMIioudXMtY2VudHJhbDEuZ2NwLnByaXYu\nY3BkZXYuY2xvdWQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQD7fcdS\nHYnMBb7zL2Nez+Mfy0g2FEBAT+duDDPGlP9AVyw3D7mZxIhaIy/kz3oWguXrtvZA\nbvmLoTvnYm4suUD/iWpfr5OBOunnzSb772TbCTMjVrrVjjaKR+hHMscGMeV7O8yv\nG3urz8KA7ms66COIXWFE1cKTIfhTqoIj0sgQ0J+WuLHvqrn5V0F5P0oZ2rRBMcUS\nmO2/FGUgGMs/Tnh+6DdE1GuBeQxztLRMWTlKWKY6zKd5H305ef1uvGhMgDkBnR97\n0nFi6rjvbL+OXfQuYXf/rhR1MtjHTNprmGBVhqHci3oZdE1vFvnxUQ98nY/ibOgZ\nWrEXuXc9QN5MgZeJAgMBAAGjggJ3MIICczAOBgNVHQ8BAf8EBAMCBaAwHQYDVR0l\nBBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYE\nFL+UaLPv4/rIx38v2b2kw1YvJIOwMB8GA1UdIwQYMBaAFKhKamMEfd265tE5t6ZF\nZe/zqOyhMG8GCCsGAQUFBwEBBGMwYTAuBggrBgEFBQcwAYYiaHR0cDovL29jc3Au\naW50LXgzLmxldHNlbmNyeXB0Lm9yZzAvBggrBgEFBQcwAoYjaHR0cDovL2NlcnQu\naW50LXgzLmxldHNlbmNyeXB0Lm9yZy8wLQYDVR0RBCYwJIIiKi51cy1jZW50cmFs\nMS5nY3AucHJpdi5jcGRldi5jbG91ZDBMBgNVHSAERTBDMAgGBmeBDAECATA3Bgsr\nBgEEAYLfEwEBATAoMCYGCCsGAQUFBwIBFhpodHRwOi8vY3BzLmxldHNlbmNyeXB0\nLm9yZzCCAQQGCisGAQQB1nkCBAIEgfUEgfIA8AB3AHR+2oMxrTMQkSGcziVPQnDC\nv/1eQiAIxjc1eeYQe8xWAAABasQonMoAAAQDAEgwRgIhAJCoN9WGHlqLCDD6cBg2\nQLlwETH+I8J0n/k8NYnrBpGTAiEAvjwyafxyws7p6Ay2NosLJ9elSvqMepPfuGIO\nG7wis9gAdQBj8tvN6DvMLM8LcoQnV2szpI1hd4+9daY4scdoVEvYjQAAAWrEKJzr\nAAAEAwBGMEQCIE397fFPAFeDRaI7ByAE/hqwhQKeTf8sPc4nKB6f98QLAiAon1kz\nO99aKvICl8N7z63Y5rIAqV1jEde2Ie58KPzVgDANBgkqhkiG9w0BAQsFAAOCAQEA\nKQdMKH9f2twvcVcYX+nkyzwhc11sf8n5dt100DHnnU0sD3R6LvaYpGqy6F/52rMl\nDFC/Lj98Xp+aATEGv31CYfZBdd8yxJ1XKs3xm0avjhPW+amWnNz5T9MENCvTss6x\nhZKc18Xwj8ZQH8zw9+xTp5wi1x6kYyIfL6s2L76ogf3FgrqksVh8mHGkJbYs6hdj\nd4NbXhTwyVrimVLaFRX1ijULG2YX/E9uQXLqMorl0P+Sy+XywR5X+Y9I/You0wko\nYLMT+6MCieNjV2wR97Q+J2G8Hfg4gavUgd/SGiPvBtrmx51SANNzjA6GJzODfbjm\nQ6UBvzOlbr3oBx5+/yjtkg==\n-----END CERTIFICATE-----\n\n-----BEGIN CERTIFICATE-----\nMIIEkjCCA3qgAwIBAgIQCgFBQgAAAVOFc2oLheynCDANBgkqhkiG9w0BAQsFADA/\nMSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT\nDkRTVCBSb290IENBIFgzMB4XDTE2MDMxNzE2NDA0NloXDTIxMDMxNzE2NDA0Nlow\nSjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUxldCdzIEVuY3J5cHQxIzAhBgNVBAMT\nGkxldCdzIEVuY3J5cHQgQXV0aG9yaXR5IFgzMIIBIjANBgkqhkiG9w0BAQEFAAOC\nAQ8AMIIBCgKCAQEAnNMM8FrlLke3cl03g7NoYzDq1zUmGSXhvb418XCSL7e4S0EF\nq6meNQhY7LEqxGiHC6PjdeTm86dicbp5gWAf15Gan/PQeGdxyGkOlZHP/uaZ6WA8\nSMx+yk13EiSdRxta67nsHjcAHJyse6cF6s5K671B5TaYucv9bTyWaN8jKkKQDIZ0\nZ8h/pZq4UmEUEz9l6YKHy9v6Dlb2honzhT+Xhq+w3Brvaw2VFn3EK6BlspkENnWA\na6xK8xuQSXgvopZPKiAlKQTGdMDQMc2PMTiVFrqoM7hD8bEfwzB/onkxEz0tNvjj\n/PIzark5McWvxI0NHWQWM6r6hCm21AvA2H3DkwIDAQABo4IBfTCCAXkwEgYDVR0T\nAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAYYwfwYIKwYBBQUHAQEEczBxMDIG\nCCsGAQUFBzABhiZodHRwOi8vaXNyZy50cnVzdGlkLm9jc3AuaWRlbnRydXN0LmNv\nbTA7BggrBgEFBQcwAoYvaHR0cDovL2FwcHMuaWRlbnRydXN0LmNvbS9yb290cy9k\nc3Ryb290Y2F4My5wN2MwHwYDVR0jBBgwFoAUxKexpHsscfrb4UuQdf/EFWCFiRAw\nVAYDVR0gBE0wSzAIBgZngQwBAgEwPwYLKwYBBAGC3xMBAQEwMDAuBggrBgEFBQcC\nARYiaHR0cDovL2Nwcy5yb290LXgxLmxldHNlbmNyeXB0Lm9yZzA8BgNVHR8ENTAz\nMDGgL6AthitodHRwOi8vY3JsLmlkZW50cnVzdC5jb20vRFNUUk9PVENBWDNDUkwu\nY3JsMB0GA1UdDgQWBBSoSmpjBH3duubRObemRWXv86jsoTANBgkqhkiG9w0BAQsF\nAAOCAQEA3TPXEfNjWDjdGBX7CVW+dla5cEilaUcne8IkCJLxWh9KEik3JHRRHGJo\nuM2VcGfl96S8TihRzZvoroed6ti6WqEBmtzw3Wodatg+VyOeph4EYpr/1wXKtx8/\nwApIvJSwtmVi4MFU5aMqrSDE6ea73Mj2tcMyo5jMd6jmeWUHK8so/joWUoHOUgwu\nX4Po1QYz+3dszkDqMp4fklxBwXRsW10KXzPMTZ+sOPAveyxindmjkW8lGy+QsRlG\nPfZ+G6Z6h7mjem0Y+iWlkYcV4PIWL1iwBi8saCbGS5jN2p8M+X+Q7UNKEkROb3N6\nKOqkqm57TH2H3eDJAkSnh6/DNFu0Qg==\n-----END CERTIFICATE-----";
    private String privkey2 = "-----BEGIN RSA PRIVATE KEY-----\nMIIEowIBAAKCAQEA+33HUh2JzAW+8y9jXs/jH8tINhRAQE/nbgwzxpT/QFcsNw+5\nmcSIWiMv5M96FoLl67b2QG75i6E752JuLLlA/4lqX6+TgTrp580m++9k2wkzI1a6\n1Y42ikfoRzLHBjHlezvMrxt7q8/CgO5rOugjiF1hRNXCkyH4U6qCI9LIENCflrix\n76q5+VdBeT9KGdq0QTHFEpjtvxRlIBjLP054fug3RNRrgXkMc7S0TFk5SlimOsyn\neR99OXn9brxoTIA5AZ0fe9JxYuq472y/jl30LmF3/64UdTLYx0zaa5hgVYah3It6\nGXRNbxb58VEPfJ2P4mzoGVqxF7l3PUDeTIGXiQIDAQABAoIBAH8DkEY1quGCyWSy\nu0IoRjJJjafaZHTWpjCbMw8JMz0AidEpPPifHKpBeS/bZXK3G34HwqjaI2hUvxdm\nS/SEf4JPmYzH9Pxgj7/FifnVdx90rwIbDHNMxtjh5jsHNyM20gqCMicB/1zPqhFJ\n2JhAo6l8V+LW/tUmY++FfwKusuJiKtq8gTAhaf+qr9ARnNwRQ0hq+BwEr6XD9ful\nrfLEIl3ecvaKXjj6b1inrG2yVN3coibNuDkLXY2EgJG64n1vTu6aBMj0P6HeiXiH\nm04OrTP43nQrKV/+7IzrBlqEMzHtj09OgWvXkV6NhIEWswkkY+HkzLj9hgoDdV4Z\nFmmmbV0CgYEA/O6cPK4K+ElrGvyJRlkRuD3WzZByaqJyeR6Jdm3qNCQ/9bI5PH/X\ngpPqHaFAA0RVlUiV4dZ5BksxW6Lvokbv/i1xlAs8TFobdr5+qcO518e5C8V+CCut\ny01LOj0I9ZvFdAHJP514QgHLHhG8/Ua5/kgFWtnZWi4mhak6T04NOk8CgYEA/oqx\nzzHHhpaWS4tfQEL0Ibxx1VKT8M4R3QkOnngLJonA1lfqSujIeodLZWm6uzcy30u7\nikCuoKCxK0vE8dOltopwTtlTOuYI2JfdvmFaanOG/857VoFIQZmI8bakaQPr4PXe\nffDZ22k4n4B7k05fZ1MqaCHnZPI1niamfGkJEqcCgYEA0G35BfAOTiiCQIzWusfv\nWDptZpygDMutNa46bQOKukkdA+VIUViwSYSGqsAUthx7wjc8fAx3Uv5nwDH2820t\nm/Hq5KqVl/2xIBs+2brWzMBi9xZaE3WbFCuv0GA3n94ryrsmEmw7i3la3n6TlMvR\nvX+wGfvnpu7dA8w+pteVAvUCgYAO//FWemJ9peYZcY8dZFSqoEY9Ae7B5ALdeako\n4X4WuUtp1ihyXaFixxJEWaStX6VZz0av8PvZb17BZGeosIY1aZcQrnHfKKsgyGJC\n083WNBSignJ2OIwfgYK2a8LohVijGxoPZeAQs/SoQZQGrDmnBxmapVTTeAp81V4+\nOppURQKBgFBqIIXH06SH4FVxTbo+cq6gOHuRIXyLR++AyzJyn8wVfHVJ01r6zhGd\nazTCm+fyfw/6jsJfK3b8mLjiz39GAM7vyvkVV5FKYC2wQldxPyow/MhmKBjdqp2p\n5gq8MSFaXoZOtRgMWdT4JPLTi7xp8429yRM8JK8E6A9+5aTkUM7H\n-----END RSA PRIVATE KEY-----\n";
    private String fullchain3 = "-----BEGIN CERTIFICATE-----\nMIIFfDCCBGSgAwIBAgISA+SIWPXUZozkvzhi5xKWfne+MA0GCSqGSIb3DQEBCwUA\nMEoxCzAJBgNVBAYTAlVTMRYwFAYDVQQKEw1MZXQncyBFbmNyeXB0MSMwIQYDVQQD\nExpMZXQncyBFbmNyeXB0IEF1dGhvcml0eSBYMzAeFw0xOTA1MTcwMzU5MTdaFw0x\nOTA4MTUwMzU5MTdaMC0xKzApBgNVBAMMIioudXMtY2VudHJhbDEuZ2NwLnByaXYu\nY3BkZXYuY2xvdWQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQD7fcdS\nHYnMBb7zL2Nez+Mfy0g2FEBAT+duDDPGlP9AVyw3D7mZxIhaIy/kz3oWguXrtvZA\nbvmLoTvnYm4suUD/iWpfr5OBOunnzSb772TbCTMjVrrVjjaKR+hHMscGMeV7O8yv\nG3urz8KA7ms66COIXWFE1cKTIfhTqoIj0sgQ0J+WuLHvqrn5V0F5P0oZ2rRBMcUS\nmO2/FGUgGMs/Tnh+6DdE1GuBeQxztLRMWTlKWKY6zKd5H305ef1uvGhMgDkBnR97\n0nFi6rjvbL+OXfQuYXf/rhR1MtjHTNprmGBVhqHci3oZdE1vFvnxUQ98nY/ibOgZ\nWrEXuXc9QN5MgZeJAgMBAAGjggJ3MIICczAOBgNVHQ8BAf8EBAMCBaAwHQYDVR0l\nBBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYE\nFL+UaLPv4/rIx38v2b2kw1YvJIOwMB8GA1UdIwQYMBaAFKhKamMEfd265tE5t6ZF\nZe/zqOyhMG8GCCsGAQUFBwEBBGMwYTAuBggrBgEFBQcwAYYiaHR0cDovL29jc3Au\naW50LXgzLmxldHNlbmNyeXB0Lm9yZzAvBggrBgEFBQcwAoYjaHR0cDovL2NlcnQu\naW50LXgzLmxldHNlbmNyeXB0Lm9yZy8wLQYDVR0RBCYwJIIiKi51cy1jZW50cmFs\nMS5nY3AucHJpdi5jcGRldi5jbG91ZDBMBgNVHSAERTBDMAgGBmeBDAECATA3Bgsr\nBgEEAYLfEwEBATAoMCYGCCsGAQUFBwIBFhpodHRwOi8vY3BzLmxldHNlbmNyeXB0\nLm9yZzCCAQQGCisGAQQB1nkCBAIEgfUEgfIA8AB3AHR+2oMxrTMQkSGcziVPQnDC\nv/1eQiAIxjc1eeYQe8xWAAABasQonMoAAAQDAEgwRgIhAJCoN9WGHlqLCDD6cBg2\nQLlwETH+I8J0n/k8NYnrBpGTAiEAvjwyafxyws7p6Ay2NosLJ9elSvqMepPfuGIO\nG7wis9gAdQBj8tvN6DvMLM8LcoQnV2szpI1hd4+9daY4scdoVEvYjQAAAWrEKJzr\nAAAEAwBGMEQCIE397fFPAFeDRaI7ByAE/hqwhQKeTf8sPc4nKB6f98QLAiAon1kz\nO99aKvICl8N7z63Y5rIAqV1jEde2Ie58KPzVgDANBgkqhkiG9w0BAQsFAAOCAQEA\nKQdMKH9f2twvcVcYX+nkyzwhc11sf8n5dt100DHnnU0sD3R6LvaYpGqy6F/52rMl\nDFC/Lj98Xp+aATEGv31CYfZBdd8yxJ1XKs3xm0avjhPW+amWnNz5T9MENCvTss6x\nhZKc18Xwj8ZQH8zw9+xTp5wi1x6kYyIfL6s2L76ogf3FgrqksVh8mHGkJbYs6hdj\nd4NbXhTwyVrimVLaFRX1ijULG2YX/E9uQXLqMorl0P+Sy+XywR5X+Y9I/You0wko\nYLMT+6MCieNjV2wR97Q+J2G8Hfg4gavUgd/SGiPvBtrmx51SANNzjA6GJzODfbjm\nQ6UBvzOlbr3oBx5+/yjtkg==\n-----END CERTIFICATE-----\n\n-----BEGIN CERTIFICATE-----\nMIIEkjCCA3qgAwIBAgIQCgFBQgAAAVOFc2oLheynCDANBgkqhkiG9w0BAQsFADA/\nMSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT\nDkRTVCBSb290IENBIFgzMB4XDTE2MDMxNzE2NDA0NloXDTIxMDMxNzE2NDA0Nlow\nSjELMAkGA1UEBhMCVVMxFjAUBgNVBAoTDUxldCdzIEVuY3J5cHQxIzAhBgNVBAMT\nGkxldCdzIEVuY3J5cHQgQXV0aG9yaXR5IFgzMIIBIjANBgkqhkiG9w0BAQEFAAOC\nAQ8AMIIBCgKCAQEAnNMM8FrlLke3cl03g7NoYzDq1zUmGSXhvb418XCSL7e4S0EF\nq6meNQhY7LEqxGiHC6PjdeTm86dicbp5gWAf15Gan/PQeGdxyGkOlZHP/uaZ6WA8\nSMx+yk13EiSdRxta67nsHjcAHJyse6cF6s5K671B5TaYucv9bTyWaN8jKkKQDIZ0\nZ8h/pZq4UmEUEz9l6YKHy9v6Dlb2honzhT+Xhq+w3Brvaw2VFn3EK6BlspkENnWA\na6xK8xuQSXgvopZPKiAlKQTGdMDQMc2PMTiVFrqoM7hD8bEfwzB/onkxEz0tNvjj\n/PIzark5McWvxI0NHWQWM6r6hCm21AvA2H3DkwIDAQABo4IBfTCCAXkwEgYDVR0T\nAQH/BAgwBgEB/wIBADAOBgNVHQ8BAf8EBAMCAYYwfwYIKwYBBQUHAQEEczBxMDIG\nCCsGAQUFBzABhiZodHRwOi8vaXNyZy50cnVzdGlkLm9jc3AuaWRlbnRydXN0LmNv\nbTA7BggrBgEFBQcwAoYvaHR0cDovL2FwcHMuaWRlbnRydXN0LmNvbS9yb290cy9k\nc3Ryb290Y2F4My5wN2MwHwYDVR0jBBgwFoAUxKexpHsscfrb4UuQdf/EFWCFiRAw\nVAYDVR0gBE0wSzAIBgZngQwBAgEwPwYLKwYBBAGC3xMBAQEwMDAuBggrBgEFBQcC\nARYiaHR0cDovL2Nwcy5yb290LXgxLmxldHNlbmNyeXB0Lm9yZzA8BgNVHR8ENTAz\nMDGgL6AthitodHRwOi8vY3JsLmlkZW50cnVzdC5jb20vRFNUUk9PVENBWDNDUkwu\nY3JsMB0GA1UdDgQWBBSoSmpjBH3duubRObemRWXv86jsoTANBgkqhkiG9w0BAQsF\nAAOCAQEA3TPXEfNjWDjdGBX7CVW+dla5cEilaUcne8IkCJLxWh9KEik3JHRRHGJo\nuM2VcGfl96S8TihRzZvoroed6ti6WqEBmtzw3Wodatg+VyOeph4EYpr/1wXKtx8/\nwApIvJSwtmVi4MFU5aMqrSDE6ea73Mj2tcMyo5jMd6jmeWUHK8so/joWUoHOUgwu\nX4Po1QYz+3dszkDqMp4fklxBwXRsW10KXzPMTZ+sOPAveyxindmjkW8lGy+QsRlG\nPfZ+G6Z6h7mjem0Y+iWlkYcV4PIWL1iwBi8saCbGS5jN2p8M+X+Q7UNKEkROb3N6\nKOqkqm57TH2H3eDJAkSnh6/DNFu0Qg==\n-----END CERTIFICATE-----";
    private String privkey3 = "-----BEGIN RSA PRIVATE KEY-----\nMIIEowIBAAKCAQEA+33HUh2JzAW+8y9jXs/jH8tINhRAQE/nbgwzxpT/QFcsNw+5\nmcSIWiMv5M96FoLl67b2QG75i6E752JuLLlA/4lqX6+TgTrp580m++9k2wkzI1a6\n1Y42ikfoRzLHBjHlezvMrxt7q8/CgO5rOugjiF1hRNXCkyH4U6qCI9LIENCflrix\n76q5+VdBeT9KGdq0QTHFEpjtvxRlIBjLP054fug3RNRrgXkMc7S0TFk5SlimOsyn\neR99OXn9brxoTIA5AZ0fe9JxYuq472y/jl30LmF3/64UdTLYx0zaa5hgVYah3It6\nGXRNbxb58VEPfJ2P4mzoGVqxF7l3PUDeTIGXiQIDAQABAoIBAH8DkEY1quGCyWSy\nu0IoRjJJjafaZHTWpjCbMw8JMz0AidEpPPifHKpBeS/bZXK3G34HwqjaI2hUvxdm\nS/SEf4JPmYzH9Pxgj7/FifnVdx90rwIbDHNMxtjh5jsHNyM20gqCMicB/1zPqhFJ\n2JhAo6l8V+LW/tUmY++FfwKusuJiKtq8gTAhaf+qr9ARnNwRQ0hq+BwEr6XD9ful\nrfLEIl3ecvaKXjj6b1inrG2yVN3coibNuDkLXY2EgJG64n1vTu6aBMj0P6HeiXiH\nm04OrTP43nQrKV/+7IzrBlqEMzHtj09OgWvXkV6NhIEWswkkY+HkzLj9hgoDdV4Z\nFmmmbV0CgYEA/O6cPK4K+ElrGvyJRlkRuD3WzZByaqJyeR6Jdm3qNCQ/9bI5PH/X\ngpPqHaFAA0RVlUiV4dZ5BksxW6Lvokbv/i1xlAs8TFobdr5+qcO518e5C8V+CCut\ny01LOj0I9ZvFdAHJP514QgHLHhG8/Ua5/kgFWtnZWi4mhak6T04NOk8CgYEA/oqx\nzzHHhpaWS4tfQEL0Ibxx1VKT8M4R3QkOnngLJonA1lfqSujIeodLZWm6uzcy30u7\nikCuoKCxK0vE8dOltopwTtlTOuYI2JfdvmFaanOG/857VoFIQZmI8bakaQPr4PXe\nffDZ22k4n4B7k05fZ1MqaCHnZPI1niamfGkJEqcCgYEA0G35BfAOTiiCQIzWusfv\nWDptZpygDMutNa46bQOKukkdA+VIUViwSYSGqsAUthx7wjc8fAx3Uv5nwDH2820t\nm/Hq5KqVl/2xIBs+2brWzMBi9xZaE3WbFCuv0GA3n94ryrsmEmw7i3la3n6TlMvR\nvX+wGfvnpu7dA8w+pteVAvUCgYAO//FWemJ9peYZcY8dZFSqoEY9Ae7B5ALdeako\n4X4WuUtp1ihyXaFixxJEWaStX6VZz0av8PvZb17BZGeosIY1aZcQrnHfKKsgyGJC\n083WNBSignJ2OIwfgYK2a8LohVijGxoPZeAQs/SoQZQGrDmnBxmapVTTeAp81V4+\nOppURQKBgFBqIIXH06SH4FVxTbo+cq6gOHuRIXyLR++AyzJyn8wVfHVJ01r6zhGd\nazTCm+fyfw/6jsJfK3b8mLjiz39GAM7vyvkVV5FKYC2wQldxPyow/MhmKBjdqp2p\n5gq8MSFaXoZOtRgMWdT4JPLTi7xp8429yRM8JK8E6A9+5aTkUM7H\n-----END RSA PRIVATE KEY-----\n";

    @Before
    public void setUp() throws Exception {
        this.metrics = new Metrics();
        this.time = new MockTime();
        this.lcCache = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        Node node = new Node(0, "localhost", 9092);
        this.endpoint = node.host() + ":" + node.port();
        this.mockAdminClient = (AdminClient)Mockito.spy((Object)new MockAdminClient.Builder().brokers(Collections.singletonList(node)).controller(0).build());
        this.sslCertsPath = this.tempFolder.getRoot().getCanonicalPath() + "/" + SSL_CERTS_DIR + "spec.json";
        String logicalClustersDir = this.tempFolder.getRoot().getCanonicalPath();
        this.lcCache.configure(logicalClustersDir, TEST_CACHE_RELOAD_DELAY_MS.longValue(), this.mockAdminClient, BROKER_ID, this.sslCertsPath);
    }

    @After
    public void tearDown() {
        this.lcCache.shutdown();
    }

    @Test
    public void testLoadFromFileSystemUpdateMetric() throws Exception {
        long now = 1000L + 1000L * (System.currentTimeMillis() / 1000L);
        this.time.setCurrentTimeMs(now);
        long delay1 = 5000L;
        long delay2 = 8000L;
        long lastMod1 = now - delay1;
        Utils.createOrUpdateLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder, lastMod1);
        long lastMod2 = now - delay2;
        Utils.createOrUpdateLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder, lastMod2);
        this.lcCache.start();
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)this.lcCache.isUp());
        Assert.assertEquals((double)delay1, (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-min"), (double)0.0);
        Assert.assertEquals((double)delay2, (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-max"), (double)0.0);
        Assert.assertEquals((double)(0.5 * (double)(delay1 + delay2)), (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-avg"), (double)0.0);
    }

    @Test
    public void testEndToEndLoadTimeMetric() throws Exception {
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        long delay1 = this.time.milliseconds() - Utils.CREATION_DATE_1.getTime();
        long delay2 = this.time.milliseconds() - Utils.CREATION_DATE_2.getTime();
        this.lcCache.start();
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)this.lcCache.isUp());
        Assert.assertEquals((double)delay1, (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-end-to-end-load-time-min"), (double)0.0);
        Assert.assertEquals((double)delay2, (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-end-to-end-load-time-max"), (double)0.0);
        Assert.assertEquals((double)(0.5 * (double)(delay1 + delay2)), (double)TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-end-to-end-load-time-avg"), (double)0.0);
    }

    @Test
    public void testNumTenantsMetric() throws Exception {
        Assert.assertEquals((long)0L, (long)TestUtils.getIntMetricValue((Metrics)this.metrics, (String)"number-of-tenants"));
        this.lcCache.start();
        List<LogicalClusterMetadata> metadatas = Arrays.asList(Utils.LC_META_ABC, Utils.LC_META_XYZ, Utils.LC_META_HEALTHCHECK);
        int count = 0;
        for (LogicalClusterMetadata metadata : metadatas) {
            Utils.createLogicalClusterFile(metadata, this.tempFolder);
            this.lcCache.reloadCache();
            Assert.assertNotNull((Object)this.lcCache.metadata(metadata.logicalClusterId()));
            Assert.assertEquals((long)(++count), (long)TestUtils.getIntMetricValue((Metrics)this.metrics, (String)"number-of-tenants"));
        }
        for (LogicalClusterMetadata metadata : metadatas) {
            Utils.deleteLogicalClusterFile(metadata, this.tempFolder);
            this.lcCache.reloadCache();
            Assert.assertNull((Object)this.lcCache.metadata(metadata.logicalClusterId()));
            Assert.assertEquals((long)(--count), (long)TestUtils.getIntMetricValue((Metrics)this.metrics, (String)"number-of-tenants"));
        }
    }

    @Test
    public void testCreateAndRemoveInstance() throws Exception {
        String brokerUUID = "test-uuid";
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.session.uuid", String.valueOf("test-uuid"));
        configs.put("broker.id", BROKER_ID);
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath() + "/subdir/anotherdir/");
        Assert.assertNull((Object)PhysicalClusterMetadata.getInstance((String)"test-uuid"));
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)metadata.isUp());
        Assert.assertEquals((Object)metadata, (Object)PhysicalClusterMetadata.getInstance((String)"test-uuid"));
        metadata.close("test-uuid");
        Assert.assertNull((Object)PhysicalClusterMetadata.getInstance((String)"test-uuid"));
    }

    @Test(expected=ConfigException.class)
    public void testConfigureInstanceWithoutDirConfigThrowsException() {
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.session.uuid", "test-uuid-1");
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
    }

    @Test
    public void testConfigureInstanceWithSameBrokerUuid() throws IOException {
        String brokerUUID = "test-uuid-2";
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.session.uuid", "test-uuid-2");
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath());
        PhysicalClusterMetadata meta1 = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        meta1.configure(configs);
        Assert.assertEquals((Object)meta1, (Object)PhysicalClusterMetadata.getInstance((String)"test-uuid-2"));
        meta1.configure(configs);
        Assert.assertEquals((Object)meta1, (Object)PhysicalClusterMetadata.getInstance((String)"test-uuid-2"));
        PhysicalClusterMetadata meta2 = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        try {
            meta2.configure(configs);
            Assert.fail((String)"Exception not thrown when configuring another instance with the same broker UUID");
        }
        catch (UnsupportedOperationException e) {
            Assert.assertEquals((Object)meta1, (Object)PhysicalClusterMetadata.getInstance((String)"test-uuid-2"));
        }
        meta2.close("test-uuid-2");
        Assert.assertEquals((Object)meta1, (Object)PhysicalClusterMetadata.getInstance((String)"test-uuid-2"));
        meta1.close("test-uuid-2");
        Assert.assertNull((Object)PhysicalClusterMetadata.getInstance((String)"test-uuid-2"));
    }

    @Test
    public void testStartWithInaccessibleDirShouldThrowException() throws IOException {
        Assert.assertTrue((boolean)this.tempFolder.getRoot().setReadable(false));
        try {
            this.lcCache.start();
            Assert.fail((String)"IOException not thrown when starting with inaccessible directory.");
        }
        catch (IOException ioe) {
            Assert.assertFalse((boolean)this.lcCache.isUp());
            Assert.assertFalse((boolean)this.lcCache.dirWatcher.isRegistered());
        }
        Assert.assertTrue((boolean)this.tempFolder.getRoot().setReadable(true));
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
    }

    @Test
    public void testExistingFilesLoaded() throws IOException, InterruptedException {
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        this.lcCache.start();
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)this.lcCache.isUp());
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
    }

    @Test
    public void testHandlesFileEvents() throws IOException, InterruptedException {
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        String lcXyzId = Utils.LC_META_XYZ.logicalClusterId();
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(lcXyzId) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        Assert.assertEquals((Object)Utils.LC_META_XYZ, (Object)this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()));
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
        Assert.assertEquals((Object)this.lcCache.logicalClusterIds(), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        LogicalClusterMetadata updatedLcMeta = new LogicalClusterMetadata(Utils.LC_META_XYZ.logicalClusterId(), Utils.LC_META_XYZ.physicalClusterId(), "new-name", "new-account", Utils.LC_META_XYZ.k8sClusterId(), Utils.LC_META_XYZ.logicalClusterType(), Utils.LC_META_XYZ.storageBytes(), Utils.LC_META_XYZ.producerByteRate(), Utils.LC_META_XYZ.consumerByteRate(), null, null, Long.valueOf(Utils.LC_META_XYZ.brokerRequestPercentage().longValue()), Utils.LC_META_XYZ.networkQuotaOverhead(), null, null);
        Utils.updateLogicalClusterFile(updatedLcMeta, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(lcXyzId).logicalClusterName().equals("new-name"), (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be updated");
        Utils.deleteLogicalClusterFile(updatedLcMeta, this.tempFolder);
        Utils.deleteLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(lcXyzId) == null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be removed from the cache");
    }

    @Test
    public void testEventsForJsonFileWithInvalidContentDoNotImpactValidLogicalClusters() throws IOException, InterruptedException {
        Utils.createInvalidLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        this.lcCache.start();
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        Utils.updateInvalidLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        LogicalClusterMetadata anotherMeta = new LogicalClusterMetadata("lkc-123", "pkc-123", "123", "my-account", "k8s-123", "kafka", Long.valueOf(0xA00000L), Long.valueOf(102400L), Long.valueOf(204800L), null, null, Long.valueOf(LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER.longValue()), LogicalClusterMetadata.DEFAULT_NETWORK_QUOTA_OVERHEAD_PERCENTAGE, null, null);
        Utils.createLogicalClusterFile(anotherMeta, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(anotherMeta.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)anotherMeta.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Utils.deleteLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.logicalClusterIds().size() == this.lcCache.logicalClusterIdsIncludingStale().size(), (long)TEST_MAX_WAIT_MS, (String)"Deleting file with bad content should remove corresponding logical cluster from stale list.");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)anotherMeta.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)anotherMeta.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)anotherMeta.logicalClusterId(), (Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
    }

    @Test
    public void testWatcherIsClosedAfterShutdown() throws IOException, InterruptedException {
        Assert.assertFalse((boolean)this.lcCache.dirWatcher.isRegistered());
        this.lcCache.start();
        TestUtils.waitForCondition(() -> ((PhysicalClusterMetadata.MetadataChangeListener)this.lcCache.dirWatcher).isRegistered(), (String)"Timed out waiting for WatchService to start.");
        this.lcCache.shutdown();
        Assert.assertFalse((boolean)this.lcCache.dirWatcher.isRegistered());
    }

    @Test(expected=IllegalStateException.class)
    public void testStartAfterShutdownShouldThrowException() throws IOException {
        this.lcCache.shutdown();
        this.lcCache.start();
    }

    @Test(expected=IllegalStateException.class)
    public void testLogicalClusterIdsAfterShutdownShouldThrowException() throws IOException {
        this.lcCache.start();
        this.lcCache.shutdown();
        this.lcCache.logicalClusterIds();
    }

    @Test(expected=IllegalStateException.class)
    public void testLogicalClusterIdsBeforeStartShouldThrowException() {
        this.lcCache.logicalClusterIds();
    }

    @Test(expected=IllegalStateException.class)
    public void testGetMetadataAfterShutdownShouldThrowException() throws IOException {
        this.lcCache.start();
        this.lcCache.shutdown();
        this.lcCache.metadata("some-cluster-id");
    }

    @Test(expected=IllegalStateException.class)
    public void testGetMetadataBeforeStartShouldThrowException() {
        Assert.assertFalse((boolean)this.lcCache.isUp());
        this.lcCache.metadata("some-cluster-id");
    }

    @Test
    public void testShouldSilentlySkipSubdirectoryEvents() throws IOException, InterruptedException {
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
        File subdir = this.tempFolder.newFolder("lkc-hjf");
        Assert.assertTrue((subdir.exists() && subdir.isDirectory() ? 1 : 0) != 0);
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.logicalClusterIds().size() >= 2, (long)TEST_MAX_WAIT_MS, (String)"Expected two new logical clusters to be added to the cache.");
        Assert.assertEquals((Object)Utils.LC_META_XYZ, (Object)this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()));
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId(), (Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId(), (Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
    }

    @Test
    public void testShouldRetryOnFailureToReadFile() throws IOException, InterruptedException {
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.setPosixFilePermissions(Utils.LC_META_ABC, "-wx-wx-wx", this.tempFolder);
        this.lcCache.start();
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of 'lkc-xyz' logical cluster to be present in metadata cache");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Utils.setPosixFilePermissions(Utils.LC_META_ABC, "rwxrwxrwx", this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.logicalClusterIds().size() == this.lcCache.logicalClusterIdsIncludingStale().size(), (long)TEST_MAX_WAIT_MS, (String)"Expected cache to recover");
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
    }

    @Test
    public void testListenerThreadShouldNotDieOnUnsetQuotas() throws IOException, InterruptedException {
        LogicalClusterMetadata tenantWithUnsetQuota = new LogicalClusterMetadata("lkc-inv", "pkc-a7sjfe", "my-poc-cluster", "my-account", "k8s-abc", "kafka", Long.valueOf(5242880000L), Long.valueOf(0x6400000L), null, null, null, Long.valueOf(0L), LogicalClusterMetadata.DEFAULT_NETWORK_QUOTA_OVERHEAD_PERCENTAGE, new LogicalClusterMetadata.LifecycleMetadata("my-poc-cluster", "pkc-a7sjfe", null, null), null);
        int numBrokers = 8;
        TenantQuotaCallback quotaCallback = this.setupCallbackAndTenant(Utils.LC_META_ABC.logicalClusterId(), 8);
        this.lcCache.configure(this.tempFolder.getRoot().getCanonicalPath(), TimeUnit.MINUTES.toMillis(60L), this.mockAdminClient, BROKER_ID, this.sslCertsPath);
        this.lcCache.start();
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
        Utils.createLogicalClusterFile(tenantWithUnsetQuota, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(tenantWithUnsetQuota.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        Map<String, String> tags1 = Collections.singletonMap("tenant", tenantWithUnsetQuota.logicalClusterId());
        TestUtils.waitForCondition(() -> quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags1).longValue() == 0xA00000L, (long)TEST_MAX_WAIT_MS, (String)"Expected min producer quota for tenant with no topics and limited quota");
        Assert.assertEquals((double)QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.FETCH), (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags1), (double)0.001);
        Assert.assertEquals((double)LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags1), (double)0.001);
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of 'lkc-abc' logical cluster to be present in metadata cache");
        quotaCallback.updateClusterMetadata(quotaCallback.cluster());
        Map<String, String> tags = Collections.singletonMap("tenant", Utils.LC_META_ABC.logicalClusterId());
        TestUtils.waitForCondition(() -> quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags) > 1.048576E7, (long)TEST_MAX_WAIT_MS, (String)"Tenant quota not configured within timeout, expected to be greater than min quota");
        TestUtils.waitForCondition(() -> quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags) < QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.PRODUCE), (long)TEST_MAX_WAIT_MS, (String)"Tenant quota not configured within timeout, expected to be lower than unlimited quota");
        Assert.assertEquals((double)(Utils.LC_META_ABC.producerByteRate() / 8L), (double)quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags), (double)0.001);
        Assert.assertEquals((double)(Utils.LC_META_ABC.consumerByteRate() / 8L), (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags), (double)0.001);
        Assert.assertEquals((double)LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags), (double)0.001);
    }

    @Test
    public void testListenerThreadShouldNotDieOnException() throws IOException, InterruptedException {
        TenantLifecycleManager lifecycleManager = (TenantLifecycleManager)Mockito.spy((Object)new TenantLifecycleManager(0L, null));
        SslCertificateManager sslCertificateManager = new SslCertificateManager((Object)BROKER_ID, (Object)this.sslCertsPath, this.mockAdminClient);
        this.lcCache.configure(this.tempFolder.getRoot().getCanonicalPath(), TimeUnit.MINUTES.toMillis(60L), lifecycleManager, sslCertificateManager);
        this.lcCache.start();
        ((TenantLifecycleManager)Mockito.doThrow((Throwable[])new Throwable[]{new RuntimeException()}).when((Object)lifecycleManager)).deleteTenants();
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of 'lkc-abc' logical cluster to be present in metadata cache");
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of 'lkc-xyz' logical cluster to be present in metadata cache");
    }

    @Test
    public void testShouldSkipInvalidJsonButUpdateCacheWhenJsonGetsFixed() throws IOException, InterruptedException {
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
        Utils.createInvalidLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of 'lkc-xyz' logical cluster to be present in metadata cache");
        TestUtils.waitForCondition(() -> this.lcCache.logicalClusterIds().size() < this.lcCache.logicalClusterIdsIncludingStale().size(), (long)TEST_MAX_WAIT_MS, (String)"Expected inaccessible logical cluster file to cause stale cluster in cache.");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        Utils.updateLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be updated");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId(), (Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId(), (Object)Utils.LC_META_XYZ.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
    }

    @Test
    public void testShouldSkipInvalidMetadataButUpdateCacheWhenFixed() throws IOException, InterruptedException {
        LogicalClusterMetadata lcMeta = new LogicalClusterMetadata("lkc-qwr", "pkc-qwr", "xyz", "my-account", "k8s-abc", "kafka", Long.valueOf(0x6400000L), Long.valueOf(1024L), null, null, null, Long.valueOf(LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER.longValue()), LogicalClusterMetadata.DEFAULT_NETWORK_QUOTA_OVERHEAD_PERCENTAGE, null, null);
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
        Utils.createLogicalClusterFile(lcMeta, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.logicalClusterIdsIncludingStale().size() > 0, (long)TEST_MAX_WAIT_MS, (String)"Expected logical cluster with invalid metadata to be in the stale list");
        LogicalClusterMetadata lcValidMeta = new LogicalClusterMetadata("lkc-qwr", "pkc-qwr", "xyz", "my-account", "k8s-abc", "kafka", Long.valueOf(0x6400000L), Long.valueOf(1024L), Long.valueOf(2048L), null, null, Long.valueOf(LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER.longValue()), LogicalClusterMetadata.DEFAULT_NETWORK_QUOTA_OVERHEAD_PERCENTAGE, null, null);
        Utils.updateLogicalClusterFile(lcValidMeta, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(lcValidMeta.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be updated");
        Assert.assertEquals((Object)ImmutableSet.of((Object)lcValidMeta.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((String)"Expect no stale logical clusters anymore", (Object)ImmutableSet.of((Object)lcValidMeta.logicalClusterId()), (Object)this.lcCache.logicalClusterIdsIncludingStale());
    }

    @Test
    public void testLoadAndRemoveMetadataCallsQuotaCallback() throws IOException, InterruptedException {
        TenantQuotaCallback quotaCallback = new TenantQuotaCallback();
        quotaCallback.configure(Collections.singletonMap("broker.id", "1"));
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        this.lcCache.start();
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)this.lcCache.isUp());
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
        Map<String, String> tags = Collections.singletonMap("tenant", Utils.LC_META_ABC.logicalClusterId());
        Assert.assertEquals((double)1.048576E7, (double)quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags), (double)0.001);
        Assert.assertEquals((double)1.048576E7, (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags), (double)0.001);
        Assert.assertEquals((double)LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags), (double)0.001);
        Thread.sleep(1000L);
        Utils.deleteLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) == null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be removed from the cache");
        TestUtils.waitForCondition(() -> quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags).doubleValue() == QuotaConfig.UNLIMITED_QUOTA.quota(ClientQuotaType.PRODUCE), (long)TEST_MAX_WAIT_MS, (String)"Expected unlimited quota for tenant with no quota");
    }

    @Test
    public void testHealthCheckAndKafkaTenant() throws IOException, InterruptedException {
        TenantQuotaCallback quotaCallback = new TenantQuotaCallback();
        quotaCallback.configure(Collections.singletonMap("broker.id", "1"));
        this.lcCache.start();
        Assert.assertTrue((boolean)this.lcCache.isUp());
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
        Utils.createLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_HEALTHCHECK, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_LINK_HEALTHCHECK, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_XYZ.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata of new logical cluster to be present in metadata cache");
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_HEALTHCHECK.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected healthcheck tenant metadata to be present in metadata cache");
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_LINK_HEALTHCHECK.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected link-healthcheck tenant metadata to be present in metadata cache");
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_XYZ.logicalClusterId(), (Object)Utils.LC_META_HEALTHCHECK.logicalClusterId(), (Object)Utils.LC_META_LINK_HEALTHCHECK.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        for (LogicalClusterMetadata healthcheckMetadata : ImmutableSet.of((Object)Utils.LC_META_HEALTHCHECK, (Object)Utils.LC_META_LINK_HEALTHCHECK)) {
            Map<String, String> tags = Collections.singletonMap("tenant", healthcheckMetadata.logicalClusterId());
            Assert.assertEquals((double)1.048576E7, (double)quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags), (double)0.001);
            Assert.assertEquals((double)1.048576E7, (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags), (double)0.001);
            Assert.assertEquals((double)LogicalClusterMetadata.DEFAULT_REQUEST_PERCENTAGE_PER_BROKER, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags), (double)0.001);
        }
    }

    @Test
    public void testShouldFailToLoadApiKeysAndHealthcheckFiles() throws IOException, InterruptedException {
        String apikeysJson = "{\"keys\": {\"key1\": {\"user_id\": \"user1\",\"logical_cluster_id\": \"myCluster\",\"sasl_mechanism\": \"PLAIN\",\"hashed_secret\": \"no hash\",\"hash_function\": \"none\"}}}";
        Utils.updateJsonFile("apikeys.json", "{\"keys\": {\"key1\": {\"user_id\": \"user1\",\"logical_cluster_id\": \"myCluster\",\"sasl_mechanism\": \"PLAIN\",\"hashed_secret\": \"no hash\",\"hash_function\": \"none\"}}}", false, this.tempFolder);
        String hcJson = "{\"kafka_key\":\"Q4L43O\",\"kafka_secret\":\"J\",\"kafka_link_key\":\"R5M541\",\"kafka_link_secret\":\"K\",\"dd_api_key\":\"\"}";
        Utils.updateJsonFile("kafka-healthcheck-external.json", "{\"kafka_key\":\"Q4L43O\",\"kafka_secret\":\"J\",\"kafka_link_key\":\"R5M541\",\"kafka_link_secret\":\"K\",\"dd_api_key\":\"\"}", false, this.tempFolder);
        this.lcCache.start();
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
        Assert.assertEquals((Object)ImmutableSet.of((Object)"apikeys", (Object)"kafka-healthcheck-external"), (Object)this.lcCache.logicalClusterIdsIncludingStale());
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null, (long)TEST_MAX_WAIT_MS, (String)"Expected new logical cluster to be added to the cache.");
        Assert.assertEquals((Object)Utils.LC_META_ABC, (Object)this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()));
        Assert.assertEquals((Object)ImmutableSet.of((Object)Utils.LC_META_ABC.logicalClusterId()), (Object)this.lcCache.logicalClusterIds());
        Utils.updateJsonFile("apikeys.json", "{\"keys\": {\"key1\": {\"user_id\": \"user1\",\"logical_cluster_id\": \"myCluster\",\"sasl_mechanism\": \"PLAIN\",\"hashed_secret\": \"no hash\",\"hash_function\": \"none\"}}}", false, this.tempFolder);
        Utils.deleteLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) == null, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be removed from the cache");
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)this.lcCache.logicalClusterIds());
    }

    @Test
    public void testShouldNotReturnDeletedLogicalClusters() throws IOException, InterruptedException {
        this.lcCache.start();
        Utils.createLogicalClusterFile(Utils.LC_META_ABC, this.tempFolder);
        Utils.createLogicalClusterFile(Utils.LC_META_DED, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(Utils.LC_META_ABC.logicalClusterId()) != null && !this.lcCache.logicalClusterIds().contains(Utils.LC_META_DED.logicalClusterId()), (long)TEST_MAX_WAIT_MS, (String)"Expected new logical cluster to be added to the cache and the deleted to not be.");
        TestUtils.waitForCondition(() -> !this.lcCache.logicalClusterIdsIncludingStale().contains(Utils.LC_META_DED.logicalClusterId()), (long)TEST_MAX_WAIT_MS, (String)"We expect that deactivated clusters will not be in cache, even as stale.");
        Assert.assertFalse((String)"We expect that the deactivated cluster will be in process of getting deleted", (boolean)this.lcCache.tenantLifecycleManager.deletedClusters().contains(Utils.LC_META_DED));
    }

    @Test
    public void testLogicalClustersExpiring() throws IOException, InterruptedException {
        long now = 1000L + 1000L * (System.currentTimeMillis() / 1000L);
        long delay1 = 1000L;
        long delay2 = 2000L;
        long delay3 = 3000L;
        long lastMod1 = now - delay1;
        long lastMod2 = now - delay2;
        long lastMod3 = now - delay3;
        this.time.setCurrentTimeMs(now);
        this.lcCache.start();
        Utils.createOrUpdateLogicalClusterFile(Utils.LC_META_XYZ, this.tempFolder, lastMod1);
        TestUtils.waitForCondition(() -> TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-avg") == (double)delay1, (long)TEST_MAX_WAIT_MS, (String)"Expected sensor to update with a load of an active cluster");
        LogicalClusterMetadata updatedLcMeta = new LogicalClusterMetadata(Utils.LC_META_XYZ.logicalClusterId(), Utils.LC_META_XYZ.physicalClusterId(), Utils.LC_META_XYZ.logicalClusterName(), Utils.LC_META_XYZ.accountId(), Utils.LC_META_XYZ.k8sClusterId(), Utils.LC_META_XYZ.logicalClusterType(), Utils.LC_META_XYZ.storageBytes(), Utils.LC_META_XYZ.producerByteRate(), Utils.LC_META_XYZ.consumerByteRate(), null, null, Long.valueOf(Utils.LC_META_XYZ.brokerRequestPercentage().longValue()), Utils.LC_META_XYZ.networkQuotaOverhead(), new LogicalClusterMetadata.LifecycleMetadata(Utils.LC_META_XYZ.logicalClusterId(), Utils.LC_META_XYZ.physicalClusterId(), Utils.LC_META_XYZ.lifecycleMetadata().creationDate(), new Date()), null);
        Utils.createOrUpdateLogicalClusterFile(updatedLcMeta, this.tempFolder, lastMod2);
        TestUtils.waitForCondition(() -> TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-avg") == (double)(delay1 + delay2) * 0.5, (long)TEST_MAX_WAIT_MS, (String)"Expected sensor to only be updated with the initial load of inactive cluster.");
        Utils.createOrUpdateLogicalClusterFile(updatedLcMeta, this.tempFolder, lastMod3);
        TestUtils.waitForCondition(() -> TestUtils.getMetricValue((Metrics)this.metrics, (String)"lkc-metadata-load-time-from-fs-update-avg") == (double)(delay1 + delay2) * 0.5, (long)TEST_MAX_WAIT_MS, (String)"Expected sensor to never be updated after the initial load of inactive cluster.");
    }

    @Test
    public void testTenantQuotaUpdateUpdatesQuotaLimitAndUpdatesQuotaResetRequired() throws IOException, InterruptedException {
        LogicalClusterMetadata lowIngressEgressTenant = new LogicalClusterMetadata("lkc-leg", "pkc-a7sjfe", "my-poc-cluster", "my-account", "k8s-abc", "kafka", Long.valueOf(5242880000L), Long.valueOf(0x500000L), Long.valueOf(0x500000L), null, null, Long.valueOf(50L), LogicalClusterMetadata.DEFAULT_NETWORK_QUOTA_OVERHEAD_PERCENTAGE, new LogicalClusterMetadata.LifecycleMetadata("my-poc-cluster", "pkc-a7sjfe", null, null), null);
        LogicalClusterMetadata upgradedTenant = Utils.updateQuotas(lowIngressEgressTenant, 0x6400000L, 0x6400000L, 250L);
        Utils.createLogicalClusterFile(lowIngressEgressTenant, this.tempFolder);
        int numBrokers = 8;
        TenantQuotaCallback quotaCallback = this.setupCallbackAndTenant(lowIngressEgressTenant.logicalClusterId(), 8);
        Assert.assertFalse((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assert.assertFalse((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assert.assertFalse((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        this.lcCache.start();
        Assert.assertTrue((String)"Expected cache to be initialized", (boolean)this.lcCache.isUp());
        Map<String, String> tags = Collections.singletonMap("tenant", lowIngressEgressTenant.logicalClusterId());
        Assert.assertEquals((double)655360.0, (double)quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags), (double)0.001);
        Assert.assertEquals((double)655360.0, (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags), (double)0.001);
        Assert.assertEquals((double)50.0, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags), (double)0.001);
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
        Utils.updateLogicalClusterFile(upgradedTenant, this.tempFolder);
        TestUtils.waitForCondition(() -> this.lcCache.metadata(upgradedTenant.logicalClusterId()).producerByteRate() == 0x6400000L, (long)TEST_MAX_WAIT_MS, (String)"Expected metadata to be updated");
        Assert.assertEquals((double)1.31072E7, (double)quotaCallback.quotaLimit(ClientQuotaType.PRODUCE, tags), (double)0.001);
        Assert.assertEquals((double)1.31072E7, (double)quotaCallback.quotaLimit(ClientQuotaType.FETCH, tags), (double)0.001);
        Assert.assertEquals((double)250.0, (double)quotaCallback.quotaLimit(ClientQuotaType.REQUEST, tags), (double)0.001);
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.PRODUCE));
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.FETCH));
        Assert.assertTrue((boolean)quotaCallback.quotaResetRequired(ClientQuotaType.REQUEST));
    }

    private TenantQuotaCallback setupCallbackAndTenant(String lcId, int numBrokers) {
        boolean brokerId = true;
        TenantQuotaCallback quotaCallback = new TenantQuotaCallback();
        quotaCallback.configure(Collections.singletonMap("broker.id", String.valueOf(1)));
        TestCluster testCluster = new TestCluster();
        for (int i = 1; i <= numBrokers; ++i) {
            testCluster.addNode(i, "rack0");
        }
        Cluster cluster = testCluster.cluster();
        quotaCallback.updateClusterMetadata(cluster);
        String topic = lcId + "_topic1";
        for (int i = 1; i <= numBrokers; ++i) {
            testCluster.setPartitionLeaders(topic, i, 1, i);
        }
        quotaCallback.updateClusterMetadata(testCluster.cluster());
        return quotaCallback;
    }

    @Test
    public void testByteComparisonForPEMCertificateFiles() {
        byte[] fullchain1Bytes = this.fullchain1.getBytes();
        byte[] privkey1Bytes = this.privkey1.getBytes();
        byte[] fullchain2Bytes = this.fullchain2.getBytes();
        byte[] privkey2Bytes = this.privkey2.getBytes();
        Assert.assertFalse((Arrays.equals(fullchain1Bytes, fullchain2Bytes) && Arrays.equals(privkey1Bytes, privkey2Bytes) ? 1 : 0) != 0);
        byte[] fullchain3Bytes = this.fullchain3.getBytes();
        byte[] privkey3Bytes = this.privkey3.getBytes();
        Assert.assertTrue((Arrays.equals(fullchain2Bytes, fullchain3Bytes) && Arrays.equals(privkey3Bytes, privkey2Bytes) ? 1 : 0) != 0);
    }

    @Test
    public void testAdminClientCreatedWithRequiredConfigs() throws Exception {
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.id", BROKER_ID);
        configs.put("broker.session.uuid", BROKER_UUID);
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath());
        configs.put("multitenant.metadata.ssl.certs.path", this.tempFolder.getRoot().getCanonicalPath() + "/mnt/sslcerts/");
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
        metadata.handleSocketServerInitialized(this.endpoint);
        Assert.assertNotNull((Object)metadata.sslCertificateManager.getAdminClient());
        metadata.close(BROKER_UUID);
    }

    @Test
    public void testAdminClientNotCreatedWithoutBrokerId() throws Exception {
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.session.uuid", BROKER_UUID);
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath());
        configs.put("multitenant.metadata.ssl.certs.path", this.tempFolder.getRoot().getCanonicalPath() + "/mnt/sslcerts/");
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
        metadata.handleSocketServerInitialized(this.endpoint);
        Assert.assertNull((Object)metadata.sslCertificateManager.getAdminClient());
        metadata.close(BROKER_UUID);
    }

    @Test
    public void testAdminClientNotCreatedWithoutSpecConfig() throws Exception {
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.id", BROKER_ID);
        configs.put("broker.session.uuid", BROKER_UUID);
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath());
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
        metadata.handleSocketServerInitialized(this.endpoint);
        Assert.assertNull((Object)metadata.sslCertificateManager.getAdminClient());
        metadata.close(BROKER_UUID);
    }

    @Test
    public void testAdminClientNotCreatedWithMalformedSpecConfig() throws Exception {
        HashMap<String, String> configs = new HashMap<String, String>();
        configs.put("broker.id", BROKER_ID);
        configs.put("broker.session.uuid", BROKER_UUID);
        configs.put("multitenant.metadata.dir", this.tempFolder.getRoot().getCanonicalPath());
        configs.put("multitenant.metadata.ssl.certs.path", "tempfolderpathmntsslcerts");
        PhysicalClusterMetadata metadata = new PhysicalClusterMetadata(this.metrics, (Time)this.time);
        metadata.configure(configs);
        metadata.handleSocketServerInitialized(this.endpoint);
        Assert.assertNull((Object)metadata.sslCertificateManager.getAdminClient());
        metadata.close(BROKER_UUID);
    }

    @Test
    public void testAdminClientInvocationOnIdenticalSslCertsSync() throws Exception {
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.lcCache.start();
        this.lcCache.sslCertificateManager.loadSslCertFiles();
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.times((int)1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempFolder, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }

    @Test
    public void testAdminClientInvocationOnDifferentSslCertsSync() throws Exception {
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.lcCache.start();
        this.lcCache.sslCertificateManager.loadSslCertFiles();
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempFolder, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(2))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Utils.deleteFiles(this.tempFolder, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(3))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }

    @Test
    public void testWatchServiceDoesNotTerminateOnDirectoryDeletion() throws Exception {
        String logicalClustersDir = this.tempFolder.getRoot().getCanonicalPath() + "logical_clusters";
        this.lcCache.configure(logicalClustersDir, TEST_CACHE_RELOAD_DELAY_MS.longValue(), this.mockAdminClient, BROKER_ID, this.sslCertsPath);
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_MAY, SSL_CERTS_DIR);
        this.lcCache.start();
        this.lcCache.sslCertificateManager.loadSslCertFiles();
        Assert.assertTrue((boolean)Files.exists(Paths.get(logicalClustersDir, new String[0]), new LinkOption[0]));
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(1))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
        Files.delete(Paths.get(logicalClustersDir, new String[0]));
        Utils.deleteFiles(this.tempFolder, SSL_CERTS_DIR);
        Utils.syncCerts(this.tempFolder, TEST_SSL_CERTS_AUG, SSL_CERTS_DIR);
        Assert.assertFalse((boolean)Files.exists(Paths.get(logicalClustersDir, new String[0]), new LinkOption[0]));
        ((AdminClient)Mockito.verify((Object)this.mockAdminClient, (VerificationMode)Mockito.timeout((long)TEST_MAX_WAIT_MS).times(2))).incrementalAlterConfigs((Map)ArgumentMatchers.any(), (AlterConfigsOptions)ArgumentMatchers.any());
    }
}

