/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.server.plugins.auth;

import io.confluent.kafka.server.plugins.auth.MultiTenantSaslConfigEntry;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslSecrets;
import io.confluent.kafka.server.plugins.auth.MultiTenantSaslSecretsLoader;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.attribute.FileTime;
import java.util.Collections;
import java.util.HashSet;
import kafka.server.BrokerSession;
import kafka.server.KafkaConfig;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.network.PublicCredential;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class MultiTenantSaslSecretsLoaderTest {
    static final String JSON1 = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key1\": {", "      \"user_id\": \"user1\",", "      \"logical_cluster_id\": \"myCluster\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none\"", "    },", "    \"key2\": {", "      \"user_id\": \"user2\",", "      \"logical_cluster_id\": \"myCluster2\",", "      \"sasl_mechanism\": \"SSL\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none\"", "    }", "  }", "}");
    static final String JSON2 = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key3\": {", "      \"user_id\": \"user3\",", "      \"logical_cluster_id\": \"myCluster3\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none3\"", "    }", "  }", "}");
    static final String JSON3 = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key3\": {", "      \"user_id\": \"user4\",", "      \"logical_cluster_id\": \"myCluster4\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none3\"", "    }", "  }", "}");
    static final String MALFORMED_JSON = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    key3: {", "      \"user_id\": \"user3\",", "      \"logical_cluster_id\": \"myCluster3\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none3\"", "    }", "  }", "}");
    static final String JSON_WITH_NULL = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key3\": null", "  }", "}");
    static final String JSON_WITH_MISSING_FIELDS = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    key3: {", "      \"logical_cluster_id\": \"myCluster3\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none3\"", "    }", "  }", "}");
    static final String JSON_WITH_EXTRA_FIELDS = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key1\": {", "      \"user_id\": \"user1\",", "      \"logical_cluster_id\": \"myCluster\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none\",", "      \"a_new_field\": \"some value\"", "    }", "  }", "}");
    static final String JSON_WITH_USER_RESOURCE_ID_1 = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key1\": {", "      \"user_id\": \"user1\",", "      \"logical_cluster_id\": \"myCluster\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none\",", "      \"user_resource_id\": \"u-xyz123\"", "    }", "  }", "}");
    static final String JSON_WITH_USER_RESOURCE_ID_2 = String.join((CharSequence)"\n", "{", "  \"keys\": {", "    \"key2\": {", "      \"user_id\": \"user1\",", "      \"logical_cluster_id\": \"myCluster\",", "      \"sasl_mechanism\": \"PLAIN\",", "      \"hashed_secret\": \"no hash\",", "      \"hash_function\": \"none\",", "      \"user_resource_id\": \"u-xyz123\"", "    }", "  }", "}");

    @Test
    public void testLoad() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(1);
        try (TempFile file = new TempFile(JSON1);){
            MultiTenantSaslSecrets secrets = loader.load(file.path(), 100L, 10L);
            MultiTenantSaslConfigEntry entry = (MultiTenantSaslConfigEntry)secrets.entries().get("key1");
            Assertions.assertNotNull((Object)entry);
            Assertions.assertEquals((Object)"user1", (Object)entry.userId());
            Assertions.assertEquals((Object)"myCluster", (Object)entry.logicalClusterId());
            Assertions.assertEquals((Object)"PLAIN", (Object)entry.saslMechanism());
            Assertions.assertEquals((Object)"no hash", (Object)entry.hashedSecret());
            Assertions.assertEquals((Object)"none", (Object)entry.hashFunction());
            MultiTenantSaslConfigEntry entry2 = (MultiTenantSaslConfigEntry)secrets.entries().get("key2");
            Assertions.assertNotNull((Object)entry2);
            Assertions.assertEquals((Object)"user2", (Object)entry2.userId());
            Assertions.assertEquals((Object)"myCluster2", (Object)entry2.logicalClusterId());
            Assertions.assertEquals((Object)"SSL", (Object)entry2.saslMechanism());
            Assertions.assertEquals((Object)"no hash", (Object)entry2.hashedSecret());
            Assertions.assertEquals((Object)"none", (Object)entry2.hashFunction());
        }
    }

    @Test
    public void testReloadTimeout() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(1);
        try (TempFile file = new TempFile(JSON1);){
            MultiTenantSaslSecrets secrets = loader.load(file.path(), 100L, 1000000000L);
            Assertions.assertEquals((Object)"user2", (Object)((MultiTenantSaslConfigEntry)secrets.entries().get("key2")).userId());
            file.delete();
            MultiTenantSaslSecrets secrets2 = loader.load(file.path(), 100L, 1000000000L);
            Assertions.assertEquals((Object)"user2", (Object)((MultiTenantSaslConfigEntry)secrets2.entries().get("key2")).userId());
            Assertions.assertThrows(ConfigException.class, () -> loader.load(file.path(), 200L, 10L));
            file.replace(JSON2);
            MultiTenantSaslSecrets secrets3 = loader.load(file.path(), 300L, 10L);
            Assertions.assertEquals((Object)"user3", (Object)((MultiTenantSaslConfigEntry)secrets3.entries().get("key3")).userId());
        }
    }

    @Test
    public void testMultipleCacheEntries() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON1);
             TempFile file2 = new TempFile(JSON2);
             TempFile file3 = new TempFile(JSON1);){
            loader.load(file.path(), 100L, 10L);
            loader.load(file2.path(), 100L, 10L);
            Assertions.assertTrue((boolean)loader.contains(file.path()));
            Assertions.assertTrue((boolean)loader.contains(file2.path()));
            loader.load(file3.path(), 100L, 10L);
            Assertions.assertFalse((boolean)loader.contains(file.path()));
            Assertions.assertTrue((boolean)loader.contains(file2.path()));
            Assertions.assertTrue((boolean)loader.contains(file3.path()));
        }
    }

    @Test
    public void testReloadOnModTimeChange() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON1);
             TempFile file2 = new TempFile(JSON2);){
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            loader.load(file2.path(), 100L, 0L);
            Assertions.assertEquals((Object)file2.path(), (Object)loader.mostRecentlyLoaded());
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file2.path(), (Object)loader.mostRecentlyLoaded());
            file.changeModTime(0L);
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
        }
    }

    @Test
    public void testReloadOnContentsChange() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON1);
             TempFile file2 = new TempFile(JSON2);){
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            loader.load(file2.path(), 100L, 0L);
            Assertions.assertEquals((Object)file2.path(), (Object)loader.mostRecentlyLoaded());
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file2.path(), (Object)loader.mostRecentlyLoaded());
            long oldModTime = file.getModTime();
            file.replace(JSON2);
            file.changeModTime(oldModTime);
            MultiTenantSaslSecrets secrets = loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            Assertions.assertEquals((Object)"user3", (Object)((MultiTenantSaslConfigEntry)secrets.entries().get("key3")).userId());
        }
    }

    @Test
    public void testShouldThrowConfigExceptionOnParseError() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(MALFORMED_JSON);){
            Assertions.assertThrows(ConfigException.class, () -> loader.load(file.path(), 100L, 0L));
        }
    }

    @Test
    public void testShouldThrowConfigExceptionOnMissingValue() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON_WITH_NULL);){
            Assertions.assertThrows(ConfigException.class, () -> loader.load(file.path(), 100L, 0L));
        }
    }

    @Test
    public void testShouldThrowConfigExceptionOnMissingField() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON_WITH_MISSING_FIELDS);){
            Assertions.assertThrows(ConfigException.class, () -> loader.load(file.path(), 100L, 0L));
        }
    }

    @Test
    public void testApiKeyDeleteWithConnectionTerminationEnabled() throws Exception {
        this.verifyApiKeyDelete(true);
    }

    @Test
    public void testApiKeyDeleteWithConnectionTerminationDisabled() throws Exception {
        this.verifyApiKeyDelete(false);
    }

    @Test
    public void testIgnoreNewFields() throws Exception {
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(1);
        try (TempFile file = new TempFile(JSON_WITH_EXTRA_FIELDS);){
            MultiTenantSaslSecrets secrets = loader.load(file.path(), 100L, 10L);
            MultiTenantSaslConfigEntry entry = (MultiTenantSaslConfigEntry)secrets.entries().get("key1");
            Assertions.assertNotNull((Object)entry);
            Assertions.assertEquals((Object)"user1", (Object)entry.userId());
            Assertions.assertEquals((Object)"myCluster", (Object)entry.logicalClusterId());
            Assertions.assertEquals((Object)"PLAIN", (Object)entry.saslMechanism());
            Assertions.assertEquals((Object)"no hash", (Object)entry.hashedSecret());
            Assertions.assertEquals((Object)"none", (Object)entry.hashFunction());
        }
    }

    private void verifyApiKeyDelete(boolean closeConnectionsOnCredentialDelete) throws Exception {
        HashSet deletedCredentials = new HashSet();
        KafkaConfig config = (KafkaConfig)Mockito.mock(KafkaConfig.class);
        Mockito.when((Object)config.brokerSessionUuid()).thenReturn((Object)"1");
        Mockito.when((Object)config.closeConnectionsOnCredentialDelete()).thenReturn((Object)closeConnectionsOnCredentialDelete);
        BrokerSession.addSession((KafkaConfig)config, deletedCredentials::add);
        MultiTenantSaslSecretsLoader loader = new MultiTenantSaslSecretsLoader(2);
        try (TempFile file = new TempFile(JSON1);){
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            Assertions.assertEquals(Collections.emptySet(), deletedCredentials);
            TestUtils.writeToFile((File)file.tempFile, (String)JSON2);
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            if (closeConnectionsOnCredentialDelete) {
                PublicCredential key1 = PublicCredential.saslCredential((String)"key1", (String)"PLAIN");
                PublicCredential key2 = PublicCredential.saslCredential((String)"key2", (String)"PLAIN");
                Assertions.assertEquals((Object)Utils.mkSet((Object[])new PublicCredential[]{key1, key2}), deletedCredentials);
                deletedCredentials.clear();
            } else {
                Assertions.assertEquals(Collections.emptySet(), deletedCredentials);
            }
            TestUtils.writeToFile((File)file.tempFile, (String)JSON3);
            loader.load(file.path(), 100L, 0L);
            Assertions.assertEquals((Object)file.path(), (Object)loader.mostRecentlyLoaded());
            Assertions.assertEquals(Collections.emptySet(), deletedCredentials);
        }
    }

    private static class TempFile
    implements Closeable {
        final File tempFile;

        TempFile(String value) throws IOException {
            this.tempFile = TestUtils.tempFile((String)value);
        }

        String path() {
            return this.tempFile.toString();
        }

        void delete() {
            this.tempFile.delete();
        }

        void replace(String value) throws IOException {
            File newFile = TestUtils.tempFile((String)value);
            try {
                newFile.renameTo(this.tempFile);
            }
            catch (Throwable t) {
                newFile.delete();
                throw t;
            }
        }

        long getModTime() throws IOException {
            FileTime fileTime = (FileTime)Files.getAttribute(this.tempFile.toPath(), "lastModifiedTime", new LinkOption[0]);
            return fileTime.toMillis();
        }

        void changeModTime(long value) throws IOException {
            Files.setAttribute(this.tempFile.toPath(), "lastModifiedTime", FileTime.fromMillis(value), new LinkOption[0]);
        }

        @Override
        public void close() {
            this.delete();
        }
    }
}

