/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license;

import com.google.common.base.Preconditions;
import io.confluent.license.ExpiredLicenseException;
import io.confluent.license.InvalidLicenseException;
import io.confluent.license.License;
import io.confluent.license.LicenseChanged;
import io.confluent.license.LicenseManagers;
import io.confluent.license.LicenseStore;
import io.confluent.license.util.StringUtils;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.common.errors.ClusterAuthorizationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.utils.Time;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.InvalidJwtSignatureException;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseManager {
    private static final Logger log = LoggerFactory.getLogger(LicenseManager.class);
    protected static final long WARN_THRESHOLD_DAYS = 10L;
    private final String invalidLicenseErrorMessage = "Supplied license is invalid.";
    private final LicenseStore licenseStore;
    private final Time time;
    private final ClusterClient primaryClusterClient;
    private final Map<String, ClusterClient> clusterClients = new ConcurrentHashMap<String, ClusterClient>();
    private final CopyOnWriteArrayList<Consumer<LicenseChanged>> listeners = new CopyOnWriteArrayList();
    private final License configuredLicense;
    private boolean isConfiguredLicenseInvalid = false;
    private boolean allowInvalidLicenseToRun = false;
    private long timeUntilLicenseExpirationMs = -1L;
    static final long DEFAULT_INITIAL_DELAY = 1L;
    static final long DEFAULT_PERIOD = 1L;
    static final TimeUnit DEFAULT_TIMEUNIT = TimeUnit.DAYS;

    public LicenseManager(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig) {
        this(new BasicClusterClient(topicConfig), new LicenseStore(topic, producerConfig, consumerConfig, topicConfig), Time.SYSTEM);
    }

    public LicenseManager(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, Duration topicCreateTimeout, Duration retryBackoffMinMs, Duration retryBackoffMaxMs) {
        this(new BasicClusterClient(topicConfig), new LicenseStore(topic, producerConfig, consumerConfig, topicConfig, topicCreateTimeout, retryBackoffMinMs, retryBackoffMaxMs, Time.SYSTEM), Time.SYSTEM);
    }

    public LicenseManager(Map<String, Object> topicConfig, LicenseStore licenseStore, String configuredLicense, boolean startStore) {
        this(new BasicClusterClient(topicConfig), licenseStore, Time.SYSTEM, configuredLicense, startStore, false);
    }

    public LicenseManager(Map<String, Object> topicConfig, LicenseStore licenseStore, String configuredLicense, boolean startStore, boolean allowInvalidLicenseToRun) {
        this(new BasicClusterClient(topicConfig), licenseStore, Time.SYSTEM, configuredLicense, startStore, allowInvalidLicenseToRun);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time) {
        this(primaryClusterClient, licenseStore, time, null, true, false);
    }

    protected LicenseManager(ClusterClient primaryClusterClient, LicenseStore licenseStore, Time time, String configuredLicenseStr, boolean startStore, boolean allowInvalidLicenseToRun) {
        this.licenseStore = licenseStore;
        this.time = time;
        this.configuredLicense = this.readAndValidateConfiguredLicense(configuredLicenseStr);
        this.primaryClusterClient = primaryClusterClient;
        this.allowInvalidLicenseToRun = allowInvalidLicenseToRun;
        if (startStore) {
            licenseStore.start();
        }
    }

    public void addCluster(String key, Map<String, Object> adminConfig) {
        if (adminConfig != null && !adminConfig.isEmpty()) {
            this.addCluster(key, new BasicClusterClient(adminConfig));
        }
    }

    public void addCluster(String key, ClusterClient client) {
        Objects.nonNull(key);
        Objects.nonNull(client);
        this.clusterClients.put(key, client);
    }

    public boolean removeCluster(String key) {
        return key != null && this.clusterClients.remove(key) != null;
    }

    public boolean addListener(Consumer<LicenseChanged> listener) {
        return listener != null && this.listeners.addIfAbsent(listener);
    }

    public boolean removeListener(Consumer<LicenseChanged> listener) {
        return listener != null && this.listeners.remove(listener);
    }

    public void start() {
        this.start(1L, 1L, DEFAULT_TIMEUNIT);
    }

    public void start(long initialDelay, long period, TimeUnit timeUnit) {
        Preconditions.checkArgument((initialDelay >= 0L ? 1 : 0) != 0, (Object)"The initial delay should be non-negative.");
        Preconditions.checkArgument((period > 0L ? 1 : 0) != 0, (Object)"The period should be positive.");
        LicenseManagers.INSTANCE.start(this, initialDelay, period, timeUnit);
    }

    public void stop() {
        if (!LicenseManagers.INSTANCE.stop(this)) {
            this.doStop();
        }
    }

    protected void doStart() {
        this.licenseStore.start();
    }

    protected void doStop() {
        this.licenseStore.stop();
    }

    public License configuredLicense() {
        return this.configuredLicense;
    }

    public License registerOrValidateLicense(String license) throws InvalidLicenseException {
        License latestLicense;
        PublicKey publicKey = LicenseManager.loadPublicKey();
        License newLicense = this.configuredLicense;
        if (StringUtils.isNotBlank(license)) {
            try {
                newLicense = this.readLicense(publicKey, license, false);
                log.debug("Found valid new license: {}", (Object)newLicense);
            }
            catch (InvalidLicenseException e) {
                this.notifyLicense(null, null, "Supplied license is invalid. " + e.getMessage());
                throw new InvalidLicenseException("Supplied license is invalid. " + e.getMessage(), (Throwable)((Object)e));
            }
        }
        long now = this.time.milliseconds();
        String reason = "";
        String storedLicenseStr = this.licenseStore.licenseScan();
        License storedLicense = null;
        if (StringUtils.isNotBlank(storedLicenseStr)) {
            try {
                storedLicense = this.readLicense(publicKey, storedLicenseStr, true);
                License.expiration(storedLicense.jwtClaims());
            }
            catch (Throwable t) {
                if (newLicense == null) {
                    throw new InvalidLicenseException("Stored license is invalid", t);
                }
                reason = "New license replaces invalid stored license (" + t.getMessage() + ").";
                storedLicense = null;
            }
        }
        if (newLicense != null) {
            try {
                License.expiration(newLicense.jwtClaims());
            }
            catch (Throwable t) {
                if (storedLicense != null) {
                    newLicense = storedLicense;
                    reason = "Using stored license because new license has an invalid expiration.";
                    log.warn(reason, t);
                }
                throw new InvalidLicenseException("No stored license, and error extracting expiration date from valid license", t);
            }
        }
        if (storedLicense == null) {
            if (newLicense == null) {
                if (this.isConfiguredLicenseInvalid && this.allowInvalidLicenseToRun) {
                    this.notifyLicense(null, null, "Supplied license is invalid.");
                    throw new InvalidLicenseException("Supplied license is invalid.");
                }
                newLicense = this.generateLicense(publicKey, now);
            }
            if (!newLicense.isFreeTier()) {
                log.debug("Storing license: {}", (Object)newLicense);
                this.licenseStore.registerLicense(newLicense.serializedForm());
            }
            latestLicense = newLicense;
        } else {
            latestLicense = storedLicense;
            if (storedLicense.expiresBefore(newLicense) && !newLicense.isFreeTier()) {
                log.debug("Storing updated license with later expiration: {}", (Object)newLicense);
                this.licenseStore.registerLicense(newLicense.serializedForm());
                latestLicense = newLicense;
            }
        }
        assert (latestLicense != null);
        License finalLicense = new License(latestLicense.jwtClaims(), this.time, latestLicense.serializedForm(), this.primaryClusterClient.clusterId());
        this.checkLicense(finalLicense, storedLicense, now, reason);
        return finalLicense;
    }

    protected License readLicense(PublicKey publicKey, String licenseStr, boolean stored) throws InvalidLicenseException {
        try {
            JwtClaims givenJwtClaims = stored ? License.verifyStored(publicKey, licenseStr) : License.verify(publicKey, licenseStr);
            License result = new License(givenJwtClaims, this.time, licenseStr);
            return result;
        }
        catch (InvalidJwtSignatureException e) {
            throw new InvalidLicenseException("Invalid signature", e);
        }
        catch (InvalidJwtException e) {
            throw new InvalidLicenseException("License does not match expected form.", e);
        }
        catch (Throwable e) {
            throw new InvalidLicenseException("Invalid license with invalid expiration.", e);
        }
    }

    private License readAndValidateConfiguredLicense(String licenseStr) {
        if (StringUtils.isNotBlank(licenseStr)) {
            try {
                License license = this.readLicense(LicenseManager.loadPublicKey(), licenseStr, false);
                try {
                    License.expiration(license.jwtClaims());
                }
                catch (Throwable t) {
                    throw new InvalidLicenseException("Configured license has invalid expiration", t);
                }
                this.checkLicense(license, license, this.time.milliseconds(), "Configured license");
                log.debug("Found valid configured license: {}", (Object)license);
                return license;
            }
            catch (InvalidLicenseException e) {
                log.warn("Supplied license is invalid. Will attempt to use stored license.", (Throwable)((Object)e));
                this.isConfiguredLicenseInvalid = true;
                return null;
            }
        }
        return null;
    }

    protected License generateLicense(PublicKey publicKey, long now) throws InvalidLicenseException {
        if (this.hasOnlySingleNodeClusters()) {
            JwtClaims jwtClaims = License.baseClaims("free tier", Long.MAX_VALUE, true);
            jwtClaims.setClaim("licenseType", (Object)"free");
            String generatedLicense = LicenseManager.generateFreeLicense(jwtClaims);
            License license = new License(jwtClaims, this.time, generatedLicense);
            log.debug("All single-node cluster checks satisfied; using {}", (Object)license);
            return license;
        }
        JwtClaims trialClaims = License.baseClaims("trial", now + TimeUnit.DAYS.toMillis(30L) + 1000L, true);
        trialClaims.setClaim("licenseType", (Object)"trial");
        String generatedLicense = License.generateTrialLicense(trialClaims);
        License license = new License(trialClaims, this.time, generatedLicense);
        log.debug("Creating new {}", (Object)license);
        return license;
    }

    private void checkLicense(License newLicense, License oldLicense, long now, String reason) throws InvalidLicenseException {
        if (newLicense.isFreeTier()) {
            reason = "License for single cluster, single node";
            log.info(reason);
            this.notifyLicense(newLicense, oldLicense, reason);
        } else {
            if (now > newLicense.expirationMillis()) {
                String msg = newLicense.toString();
                if (StringUtils.isBlank(reason)) {
                    reason = msg;
                }
                this.notifyLicense(newLicense, null, reason);
                throw new ExpiredLicenseException(newLicense, msg);
            }
            if (newLicense.expirationMillis() < Long.MAX_VALUE) {
                String msg = newLicense.toString();
                if (newLicense.timeRemaining(TimeUnit.DAYS) < 10L) {
                    log.warn(msg);
                } else {
                    log.info(msg);
                }
                reason = StringUtils.isBlank(reason) ? msg : reason + " " + msg;
                this.notifyLicense(newLicense, oldLicense, reason);
            }
        }
    }

    protected boolean hasOnlySingleNodeClusters() {
        return this.primaryClusterClient != null && this.primaryClusterClient.brokerCount() == 1 && this.clusterClients.values().stream().allMatch(client -> client.brokerCount() == 1);
    }

    protected void notifyLicense(License newLicense, License oldLicense, String reason) {
        if (newLicense == null && oldLicense == null) {
            LicenseChangedEvent event = new LicenseChangedEvent(null, LicenseChanged.Type.INVALID, reason);
            this.notifyEventToListeners(event);
            return;
        }
        if (newLicense.isEquivalentTo(oldLicense)) {
            log.debug("Skipping notifying {} listeners of unchanged license: {}", (Object)this.listeners.size(), (Object)newLicense);
            return;
        }
        LicenseChanged.Type changeType = LicenseChanged.Type.UPDATED;
        if (newLicense.isExpired()) {
            changeType = LicenseChanged.Type.EXPIRED;
        } else if (newLicense.isRenewalOf(oldLicense)) {
            changeType = LicenseChanged.Type.RENEWAL;
        }
        LicenseChangedEvent event = new LicenseChangedEvent(newLicense, changeType, reason);
        this.notifyEventToListeners(event);
    }

    private void notifyEventToListeners(LicenseChangedEvent event) {
        log.debug("Notifying {} listeners of license change: {}", (Object)this.listeners.size(), (Object)event);
        for (Consumer<LicenseChanged> listener : this.listeners) {
            try {
                listener.accept(event);
            }
            catch (Throwable t) {
                log.error("Unexpected error when calling license manager listener:", t);
            }
        }
    }

    public static PublicKey loadPublicKey() {
        try {
            return License.loadPublicKey();
        }
        catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new IllegalStateException("Internal license validation error", e);
        }
    }

    private static String generateFreeLicense(JwtClaims claims) throws InvalidLicenseException {
        JsonWebSignature jws = new JsonWebSignature();
        jws.setAlgorithmConstraints(AlgorithmConstraints.NO_CONSTRAINTS);
        jws.setAlgorithmHeaderValue("none");
        jws.setPayload(claims.toJson());
        try {
            return jws.getCompactSerialization();
        }
        catch (JoseException e) {
            log.error("Error while attempting to start free tier: ", (Throwable)e);
            throw new InvalidLicenseException("Error creating license for trial version: ", e);
        }
    }

    public void timeUntilLicenseExpirationMs(long timeMs) {
        this.timeUntilLicenseExpirationMs = timeMs;
    }

    public long timeUntilLicenseExpirationMs() {
        return this.timeUntilLicenseExpirationMs;
    }

    protected static class BasicClusterClient
    implements ClusterClient {
        private final Map<String, Object> adminConfig;

        BasicClusterClient(Map<String, Object> adminConfig) {
            this.adminConfig = new HashMap<String, Object>(adminConfig);
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public int brokerCount() {
            try (AdminClient admin = AdminClient.create(this.adminConfig);){
                int count = ((Collection)admin.describeCluster().nodes().get()).size();
                log.debug("Found {} brokers in Kafka cluster at {}", (Object)count, (Object)this);
                int n = count;
                return n;
            }
            catch (ExecutionException e) {
                Throwable cause = e.getCause();
                if (cause instanceof UnsupportedVersionException) {
                    log.debug("Unable to use admin client to connect to older Kafka cluster at {}", (Object)this);
                    return -2;
                }
                if (cause instanceof ClusterAuthorizationException) {
                    log.debug("Not authorized to use admin client to connect to Kafka cluster at {}", (Object)this);
                    return -3;
                }
                if (!(cause instanceof TimeoutException)) return -1;
                log.debug("Timed out waiting to connect to Kafka cluster at {}", (Object)this);
                return -4;
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                return -5;
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public String clusterId() {
            try (AdminClient admin = AdminClient.create(this.adminConfig);){
                String string = (String)admin.describeCluster().clusterId().get();
                return string;
            }
            catch (InterruptedException | ExecutionException e) {
                log.debug("Failed to fetch cLusterId of Kafka cluster at {}", (Object)this);
                return null;
            }
        }

        public String toString() {
            Object servers = this.adminConfig.get("bootstrap.servers");
            return servers == null ? "<unknown>" : servers.toString();
        }
    }

    protected static interface ClusterClient {
        public int brokerCount();

        public String clusterId();
    }

    protected static class LicenseChangedEvent
    implements LicenseChanged {
        private final License license;
        private final LicenseChanged.Type type;
        private final String description;

        LicenseChangedEvent(License license, LicenseChanged.Type type, String description) {
            Objects.nonNull((Object)type);
            Objects.nonNull(description);
            this.license = license;
            this.type = type;
            this.description = description;
        }

        @Override
        public License license() {
            return this.license;
        }

        @Override
        public LicenseChanged.Type type() {
            return this.type;
        }

        @Override
        public String description() {
            return this.description;
        }

        public int hashCode() {
            return this.type().hashCode();
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (obj instanceof LicenseChangedEvent) {
                LicenseChangedEvent that = (LicenseChangedEvent)obj;
                return this.type() == that.type() && Objects.equals(this.license(), that.license());
            }
            return false;
        }

        public String toString() {
            return (Object)((Object)this.type) + " " + this.license + " (" + this.description + ")";
        }
    }
}

