/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.client.oauth;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import javax.security.auth.login.AppConfigurationEntry;
import org.apache.kafka.common.security.oauthbearer.JwtRetriever;
import org.apache.kafka.common.security.oauthbearer.JwtRetrieverException;
import org.apache.kafka.common.security.oauthbearer.internals.secured.ConfigurationUtils;
import org.apache.kafka.common.utils.ExponentialBackoff;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UamiJwtRetriever
implements JwtRetriever {
    private static final Logger log = LoggerFactory.getLogger(UamiJwtRetriever.class);
    private static final String JSON_PATH = "/access_token";
    private static final int MAX_RESPONSE_BODY_LENGTH = 1000;
    private static final int HTTP_CLIENT_CONNECT_TIMEOUT = 5;
    private static final int HTTP_REQUEST_TIMEOUT = 5;
    private final Time time;
    private long loginRetryBackoffMs;
    private long loginRetryBackoffMaxMs;
    private String uamiTokenUrl;
    private final HttpClient httpClient;
    private ExponentialBackoff exponentialBackoff;

    public UamiJwtRetriever() {
        this.time = Time.SYSTEM;
        this.httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(5L)).build();
    }

    public UamiJwtRetriever(Time time, HttpClient httpClient) {
        this.time = time;
        this.httpClient = httpClient;
    }

    public void configure(Map<String, ?> configs, String saslMechanism, List<AppConfigurationEntry> jaasConfigEntries) {
        ConfigurationUtils cu = new ConfigurationUtils(configs, saslMechanism);
        this.uamiTokenUrl = cu.validateUrl("sasl.oauthbearer.token.endpoint.url").toString();
        this.loginRetryBackoffMs = cu.validateLong("sasl.login.retry.backoff.ms");
        this.loginRetryBackoffMaxMs = cu.validateLong("sasl.login.retry.backoff.max.ms");
        this.exponentialBackoff = new ExponentialBackoff(this.loginRetryBackoffMs, 2, this.loginRetryBackoffMaxMs, 0.2);
    }

    public String retrieve() throws JwtRetrieverException {
        HttpRequest request = HttpRequest.newBuilder().uri(URI.create(this.uamiTokenUrl)).timeout(Duration.ofSeconds(5L)).header("Metadata", "true").GET().build();
        long endMs = this.time.milliseconds() + this.loginRetryBackoffMaxMs;
        int currAttempt = 0;
        IOException ioException = null;
        while (this.time.milliseconds() < endMs) {
            ++currAttempt;
            try {
                log.debug("Sending request to UAMI attempt {}", (Object)currAttempt);
                HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                int statusCode = response.statusCode();
                if (statusCode == 200) {
                    return this.parseJwt(response.body());
                }
                if (UamiJwtRetriever.isRetryable(statusCode)) {
                    log.warn("Error retrieving UAMI JWT during attempt {}. failed with status code {}.", (Object)currAttempt, (Object)response.statusCode());
                    this.time.sleep(this.exponentialBackoff.backoff((long)currAttempt));
                    continue;
                }
                throw new JwtRetrieverException("UAMI token request failed with non-retryable status " + statusCode + " on attempt " + currAttempt);
            }
            catch (IOException e) {
                log.warn("Error retrieving JWT from UAMI during attempt {}", (Object)currAttempt, (Object)e);
                ioException = e;
                this.time.sleep(this.exponentialBackoff.backoff((long)currAttempt));
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.error("Error while retrieving JWT from UAMI: {}", (Object)e.getMessage(), (Object)e);
                throw new JwtRetrieverException("Error while retrieving JWT from UAMI", (Throwable)e);
            }
        }
        throw new JwtRetrieverException("Error while retrieving JWT from UAMI after " + currAttempt + " attempts.", ioException);
    }

    private String parseJwt(String responseBody) throws JwtRetrieverException {
        String value;
        JsonNode rootNode;
        ObjectMapper mapper = new ObjectMapper();
        try {
            rootNode = mapper.readTree(responseBody);
        }
        catch (IOException e) {
            throw new JwtRetrieverException((Throwable)e);
        }
        JsonNode node = rootNode.at(JSON_PATH);
        if (node != null && !node.isMissingNode() && !Utils.isBlank((String)(value = node.textValue()))) {
            return value.trim();
        }
        String snippet = responseBody;
        if (snippet.length() > 1000) {
            int actualLength = responseBody.length();
            String s = responseBody.substring(0, 1000);
            snippet = String.format("%s (trimmed to first %d characters out of %d total)", s, 1000, actualLength);
        }
        throw new JwtRetrieverException(String.format("The token endpoint response did not contain a valid JWT. Response: (%s)", snippet));
    }

    private static boolean isRetryable(int status) {
        return status == 408 || status == 429 || status >= 500 && status < 600;
    }
}

