/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.source;

import io.confluent.connect.jdbc.dialect.DatabaseDialect;
import io.confluent.connect.jdbc.util.ConnectionProvider;
import io.confluent.connect.jdbc.util.QuoteMethod;
import io.confluent.connect.jdbc.util.TableId;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.connect.connector.ConnectorContext;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TableMonitorThread
extends Thread {
    private static final Logger log = LoggerFactory.getLogger(TableMonitorThread.class);
    private final DatabaseDialect dialect;
    private final ConnectionProvider connectionProvider;
    private final ConnectorContext context;
    private final CountDownLatch shutdownLatch;
    private final long pollMs;
    private Set<String> whitelist;
    private Set<String> blacklist;
    private List<TableId> tables;
    private Map<String, List<TableId>> duplicates;

    public TableMonitorThread(DatabaseDialect dialect, ConnectionProvider connectionProvider, ConnectorContext context, long pollMs, Set<String> whitelist, Set<String> blacklist) {
        this.dialect = dialect;
        this.connectionProvider = connectionProvider;
        this.context = context;
        this.shutdownLatch = new CountDownLatch(1);
        this.pollMs = pollMs;
        this.whitelist = whitelist;
        this.blacklist = blacklist;
        this.tables = null;
    }

    @Override
    public void run() {
        log.info("Starting thread to monitor tables.");
        while (this.shutdownLatch.getCount() > 0L) {
            try {
                if (this.updateTables()) {
                    this.context.requestTaskReconfiguration();
                }
            }
            catch (Exception e) {
                this.context.raiseError(e);
                throw e;
            }
            try {
                log.debug("Waiting {} ms to check for changed.", (Object)this.pollMs);
                boolean shuttingDown = this.shutdownLatch.await(this.pollMs, TimeUnit.MILLISECONDS);
                if (!shuttingDown) continue;
                return;
            }
            catch (InterruptedException e) {
                log.error("Unexpected InterruptedException, ignoring: ", (Throwable)e);
            }
        }
    }

    public synchronized List<TableId> tables() {
        long started;
        long timeout = 10000L;
        long now = started = System.currentTimeMillis();
        while (this.tables == null && now - started < 10000L) {
            try {
                this.wait(10000L - (now - started));
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            now = System.currentTimeMillis();
        }
        if (this.tables == null) {
            throw new ConnectException("Tables could not be updated quickly enough.");
        }
        if (!this.duplicates.isEmpty()) {
            String configText = this.whitelist != null ? "'table.whitelist'" : (this.blacklist != null ? "'table.blacklist'" : "'table.whitelist' or 'table.blacklist'");
            String msg = "The connector uses the unqualified table name as the topic name and has detected duplicate unqualified table names. This could lead to mixed data types in the topic and downstream processing errors. To prevent such processing errors, the JDBC Source connector fails to start when it detects duplicate table name configurations. Update the connector's " + configText + " config to include exactly one table in each of the tables listed below.\n\t";
            throw new ConnectException(msg + this.duplicates.values());
        }
        return this.tables;
    }

    public void shutdown() {
        log.info("Shutting down thread monitoring tables.");
        this.shutdownLatch.countDown();
    }

    private synchronized boolean updateTables() {
        List<TableId> tables;
        try {
            tables = this.dialect.tableIds(this.connectionProvider.getConnection());
            log.debug("Got the following tables: " + Arrays.toString(tables.toArray()));
        }
        catch (SQLException e) {
            log.error("Error while trying to get updated table list, ignoring and waiting for next table poll interval", (Throwable)e);
            this.connectionProvider.close();
            return false;
        }
        ArrayList<TableId> filteredTables = new ArrayList<TableId>(tables.size());
        if (this.whitelist != null) {
            for (TableId table : tables) {
                String fqn1 = this.dialect.expressionBuilder().append((Object)table, QuoteMethod.NEVER).toString();
                String fqn2 = this.dialect.expressionBuilder().append((Object)table, QuoteMethod.ALWAYS).toString();
                if (!this.whitelist.contains(fqn1) && !this.whitelist.contains(fqn2) && !this.whitelist.contains(table.tableName())) continue;
                filteredTables.add(table);
            }
        } else if (this.blacklist != null) {
            for (TableId table : tables) {
                String fqn1 = this.dialect.expressionBuilder().append((Object)table, QuoteMethod.NEVER).toString();
                String fqn2 = this.dialect.expressionBuilder().append((Object)table, QuoteMethod.ALWAYS).toString();
                if (this.blacklist.contains(fqn1) || this.blacklist.contains(fqn2) || this.blacklist.contains(table.tableName())) continue;
                filteredTables.add(table);
            }
        } else {
            filteredTables.addAll(tables);
        }
        if (!filteredTables.equals(this.tables)) {
            log.info("After filtering the tables are: {}", (Object)this.dialect.expressionBuilder().appendList().delimitedBy(",").of(filteredTables));
            Map<String, List> duplicates = filteredTables.stream().collect(Collectors.groupingBy(TableId::tableName)).entrySet().stream().filter(entry -> ((List)entry.getValue()).size() > 1).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            this.duplicates = duplicates;
            List<TableId> previousTables = this.tables;
            this.tables = filteredTables;
            this.notifyAll();
            return previousTables != null;
        }
        return false;
    }
}

