/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.dialect;

import io.confluent.connect.jdbc.dialect.DatabaseDialect;
import io.confluent.connect.jdbc.dialect.DatabaseDialectProvider;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ServiceLoader;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.kafka.common.config.AbstractConfig;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseDialects {
    private static final Pattern PROTOCOL_PATTERN = Pattern.compile("jdbc:([^:]+):(.*)");
    private static final Logger LOG = LoggerFactory.getLogger(DatabaseDialects.class);
    private static final ConcurrentMap<String, DatabaseDialectProvider> REGISTRY = new ConcurrentSkipListMap<String, DatabaseDialectProvider>();

    private static void loadAllDialects() {
        LOG.debug("Searching for and loading all JDBC source dialects on the classpath");
        final AtomicInteger count = new AtomicInteger();
        AccessController.doPrivileged(new PrivilegedAction<Void>(){

            @Override
            public Void run() {
                ServiceLoader<DatabaseDialectProvider> loadedDialects = ServiceLoader.load(DatabaseDialectProvider.class);
                Iterator<DatabaseDialectProvider> dialectIterator = loadedDialects.iterator();
                try {
                    while (dialectIterator.hasNext()) {
                        try {
                            DatabaseDialectProvider provider = dialectIterator.next();
                            REGISTRY.put(provider.getClass().getName(), provider);
                            count.incrementAndGet();
                            LOG.debug("Found '{}' provider {}", (Object)provider, provider.getClass());
                        }
                        catch (Throwable t) {
                            LOG.debug("Skipping dialect provider after error while loading", t);
                        }
                    }
                }
                catch (Throwable t) {
                    LOG.debug("Error loading dialect providers", t);
                }
                return null;
            }
        });
        LOG.debug("Registered {} source dialects", (Object)count.get());
    }

    public static DatabaseDialect findBestFor(String jdbcUrl, AbstractConfig config) throws ConnectException {
        DatabaseDialectProvider.JdbcUrlInfo info = DatabaseDialects.extractJdbcUrlInfo(jdbcUrl);
        LOG.debug("Finding best dialect for {}", (Object)info);
        int bestScore = 0;
        DatabaseDialectProvider bestMatch = null;
        for (DatabaseDialectProvider provider : REGISTRY.values()) {
            int score = provider.score(info);
            LOG.debug("Dialect {} scored {} against {}", new Object[]{provider, score, info});
            if (score <= bestScore) continue;
            bestMatch = provider;
            bestScore = score;
        }
        LOG.debug("Using dialect {} with score {} against {}", new Object[]{bestMatch, bestScore, info});
        return bestMatch.create(config);
    }

    public static DatabaseDialect create(String dialectName, AbstractConfig config) throws ConnectException {
        LOG.debug("Looking for named dialect '{}'", (Object)dialectName);
        HashSet<String> dialectNames = new HashSet<String>();
        for (DatabaseDialectProvider provider : REGISTRY.values()) {
            dialectNames.add(provider.dialectName());
            if (!provider.dialectName().equals(dialectName)) continue;
            return provider.create(config);
        }
        for (DatabaseDialectProvider provider : REGISTRY.values()) {
            if (!provider.dialectName().equalsIgnoreCase(dialectName)) continue;
            return provider.create(config);
        }
        throw new ConnectException("Unable to find dialect with name '" + dialectName + "' in the available dialects: " + dialectNames);
    }

    static DatabaseDialectProvider.JdbcUrlInfo extractJdbcUrlInfo(String url) {
        Matcher matcher = PROTOCOL_PATTERN.matcher(url);
        if (matcher.matches()) {
            return new JdbcUrlDetails(matcher.group(1), matcher.group(2), url);
        }
        throw new ConnectException("Not a valid JDBC URL: " + url);
    }

    public static Collection<DatabaseDialectProvider> registeredDialectProviders() {
        return new HashSet<DatabaseDialectProvider>(REGISTRY.values());
    }

    public static Collection<String> registeredDialectNames() {
        return REGISTRY.values().stream().map(DatabaseDialectProvider::dialectName).collect(Collectors.toSet());
    }

    private DatabaseDialects() {
    }

    static {
        DatabaseDialects.loadAllDialects();
    }

    static class JdbcUrlDetails
    implements DatabaseDialectProvider.JdbcUrlInfo {
        final String subprotocol;
        final String subname;
        final String url;

        public JdbcUrlDetails(String subprotocol, String subname, String url) {
            this.subprotocol = subprotocol;
            this.subname = subname;
            this.url = url;
        }

        @Override
        public String subprotocol() {
            return this.subprotocol;
        }

        @Override
        public String subname() {
            return this.subname;
        }

        @Override
        public String url() {
            return this.url;
        }

        public String toString() {
            return "JDBC subprotocol '" + this.subprotocol + "' and source '" + this.url + "'";
        }
    }
}

