/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.rbac;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.confluent.security.authorizer.utils.JsonMapper;
import io.confluent.security.rbac.AccessPolicy;
import io.confluent.security.rbac.BindingScopes;
import io.confluent.security.rbac.InvalidRoleDefinitionException;
import io.confluent.security.rbac.Role;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class RbacRoles {
    private static final String DEFAULT_POLICY_FILE = "default_rbac_roles.json";
    private static final String CLOUD_CONTROL_PLANE_POLICY_FILE = "cloud_rbac_roles.json";
    private static final String CLOUD_KAFKA_POLICY_FILE = "cloud_kafka_rbac_roles.json";
    private static final String CLOUD_SDS_SR_POLICY_FILE = "cloud_sds_schema_registry_rbac_roles.json";
    private static final String CLOUD_SDS_KSQL_POLICY_FILE = "cloud_sds_ksql_rbac_roles.json";
    private BindingScopes bindingScopes;
    private final Map<String, Role> roles;

    @JsonCreator
    public RbacRoles(@JsonProperty(value="roles") List<Role> roles, @JsonProperty(value="bindingScopes") LinkedHashMap<String, Object> bindingScopes) {
        this.bindingScopes = new BindingScopes(bindingScopes);
        this.roles = new HashMap<String, Role>();
        roles.forEach(this::addRole);
    }

    public RbacRoles(List<Role> roles, BindingScopes bindingScopes) {
        this.bindingScopes = bindingScopes;
        this.roles = new HashMap<String, Role>();
        roles.forEach(this::addRole);
    }

    public Role role(String roleName, String namespace) {
        Role role = this.roles.get(roleName);
        if (role != null && role.isInNamespace(namespace)) {
            return role;
        }
        return null;
    }

    public Role role(String roleName) {
        return this.roles.get(roleName);
    }

    public Collection<Role> roles(String namespace) {
        return this.roles.values().stream().filter(role -> role.isInNamespace(namespace)).collect(Collectors.toList());
    }

    public Collection<Role> roles() {
        return new ArrayList<Role>(this.roles.values());
    }

    public Map<String, Role> rolesMap() {
        return this.roles;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RbacRoles)) {
            return false;
        }
        RbacRoles that = (RbacRoles)o;
        return Objects.equals(this.roles, that.roles);
    }

    public int hashCode() {
        return Objects.hash(this.roles);
    }

    void addRole(Role role) {
        for (Collection<AccessPolicy> accessPoliciesByScope : role.accessPolicies().values()) {
            for (AccessPolicy accessPolicy : accessPoliciesByScope) {
                if (!this.bindingScopes.isKnownScope(accessPolicy.bindingScope())) {
                    throw new InvalidRoleDefinitionException("Unknown binding scope '" + accessPolicy.bindingScope() + "' defined for " + role);
                }
                accessPolicy.allowedOperations().forEach(resourceOp -> {
                    if (resourceOp.resourceType() == null || resourceOp.resourceType().isEmpty()) {
                        throw new InvalidRoleDefinitionException("Resource type not specified in role definition ops for " + role);
                    }
                    resourceOp.operations().forEach(op -> {
                        if (op.isEmpty()) {
                            throw new InvalidRoleDefinitionException("Operation name not specified in role definition ops for " + role);
                        }
                    });
                });
            }
        }
        if (!this.bindingScopes.isPartialBindingScopePathUnique(role.bindingScopes())) {
            throw new InvalidRoleDefinitionException("Role should have binding scopes that follow a linear path in the tree");
        }
        role.setMostSpecificBindingScope(this.mostSpecificBindingScope(role));
        this.roles.put(role.name(), role);
    }

    public static RbacRoles loadDefaultPolicy(boolean isConfluentCloud) throws InvalidRoleDefinitionException {
        if (isConfluentCloud) {
            ArrayList<String> policyFiles = new ArrayList<String>();
            policyFiles.add(CLOUD_CONTROL_PLANE_POLICY_FILE);
            policyFiles.add(CLOUD_KAFKA_POLICY_FILE);
            policyFiles.add(CLOUD_SDS_SR_POLICY_FILE);
            policyFiles.add(CLOUD_SDS_KSQL_POLICY_FILE);
            return RbacRoles.load(RbacRoles.class.getClassLoader(), policyFiles);
        }
        return RbacRoles.load(RbacRoles.class.getClassLoader(), DEFAULT_POLICY_FILE);
    }

    public static RbacRoles loadDataPlanePolicy() throws InvalidRoleDefinitionException {
        return RbacRoles.load(RbacRoles.class.getClassLoader(), Collections.singletonList(CLOUD_KAFKA_POLICY_FILE));
    }

    public static RbacRoles loadSDSPolicy() throws InvalidRoleDefinitionException {
        ArrayList<String> policyFiles = new ArrayList<String>();
        policyFiles.add(CLOUD_SDS_SR_POLICY_FILE);
        policyFiles.add(CLOUD_SDS_KSQL_POLICY_FILE);
        return RbacRoles.load(RbacRoles.class.getClassLoader(), policyFiles);
    }

    public static RbacRoles loadSDSKsqlPolicy() throws InvalidRoleDefinitionException {
        return RbacRoles.load(RbacRoles.class.getClassLoader(), Collections.singletonList(CLOUD_SDS_KSQL_POLICY_FILE));
    }

    public static RbacRoles loadSDSSchemaRegistryPolicy() throws InvalidRoleDefinitionException {
        return RbacRoles.load(RbacRoles.class.getClassLoader(), Collections.singletonList(CLOUD_SDS_SR_POLICY_FILE));
    }

    public static RbacRoles load(ClassLoader classLoader, String policyResourceName) throws InvalidRoleDefinitionException {
        return RbacRoles.load(classLoader, Collections.singletonList(policyResourceName));
    }

    public static RbacRoles load(ClassLoader classLoader, List<String> policyResourceNames) throws InvalidRoleDefinitionException {
        if (policyResourceNames.size() == 0) {
            throw new InvalidRoleDefinitionException("Invalid RBAC policies");
        }
        if (policyResourceNames.size() == 1) {
            return RbacRoles.parseRbacRoles(classLoader, policyResourceNames.get(0));
        }
        return RbacRoles.mergedRbacRoles(classLoader, policyResourceNames);
    }

    private static RbacRoles mergedRbacRoles(ClassLoader classLoader, List<String> policyResourceNames) {
        List rbacRoles = policyResourceNames.stream().map(policy -> RbacRoles.parseRbacRoles(classLoader, policy)).collect(Collectors.toList());
        RbacRoles mergedRbacRoles = (RbacRoles)rbacRoles.remove(0);
        for (RbacRoles rbacRole : rbacRoles) {
            mergedRbacRoles = RbacRoles.merge(mergedRbacRoles, rbacRole);
        }
        return mergedRbacRoles;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static RbacRoles parseRbacRoles(ClassLoader classLoader, String policyResourceName) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(classLoader.getResourceAsStream(policyResourceName)));){
            RbacRoles rbacRoles = (RbacRoles)JsonMapper.objectMapper().readValue((Reader)reader, RbacRoles.class);
            return rbacRoles;
        }
        catch (IOException e) {
            throw new InvalidRoleDefinitionException("RBAC policies could not be loaded from " + policyResourceName, e);
        }
    }

    private String mostSpecificBindingScope(Role role) {
        Set<String> roleBindingScopes = role.bindingScopes();
        return this.bindingScopes.findMostSpecificBindingScope(roleBindingScopes);
    }

    public static RbacRoles merge(RbacRoles roles1, RbacRoles roles2) {
        if (!roles1.bindingScopes.equals(roles2.bindingScopes)) {
            throw new IllegalArgumentException("bindingScopes are not equal");
        }
        HashMap<String, Role> mergedRolesMap = new HashMap<String, Role>(roles1.rolesMap());
        roles2.rolesMap().forEach((key, value) -> mergedRolesMap.merge((String)key, (Role)value, Role::merge));
        return new RbacRoles(new ArrayList<Role>(mergedRolesMap.values()), roles1.bindingScopes);
    }

    public BindingScopes bindingScopes() {
        return this.bindingScopes;
    }
}

