/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.client.rest;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.confluent.security.auth.client.RestClientConfig;
import io.confluent.security.auth.client.provider.BuiltInAuthProviders;
import io.confluent.security.auth.client.provider.HttpBasicCredentialProvider;
import io.confluent.security.auth.client.provider.HttpCredentialProvider;
import io.confluent.security.auth.client.rest.RequestSender;
import io.confluent.security.auth.client.rest.RestClientThreadFactory;
import io.confluent.security.auth.client.rest.RestRequest;
import io.confluent.security.auth.client.rest.UrlSelector;
import io.confluent.security.auth.client.rest.entities.AuthenticationResponse;
import io.confluent.security.auth.client.rest.entities.ErrorMessage;
import io.confluent.security.auth.client.rest.exceptions.RestClientException;
import io.confluent.security.auth.common.JwtBearerToken;
import io.confluent.security.authorizer.utils.JsonMapper;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.security.auth.SslEngineFactory;
import org.apache.kafka.common.security.oauthbearer.OAuthBearerToken;
import org.apache.kafka.common.security.ssl.DefaultSslEngineFactory;
import org.apache.kafka.common.security.ssl.SslFactory;
import org.apache.kafka.common.utils.ExponentialBackoff;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RestClient
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(RestClient.class);
    protected static final String HTTP_EXCEPTION_MSG = "Unexpected exception sending HTTP Request.";
    private static final int HTTP_CONNECT_TIMEOUT_MS = 60000;
    private static final int HTTP_READ_TIMEOUT_MS = 60000;
    private static final String ACTIVE_NODES_ENDPOINT = "/activenodes/%s";
    private static final String AUTHENTICATE_ENDPOINT = "/authenticate";
    private static final TypeReference<List<String>> ACTIVE_URLS_RESPONSE_TYPE = new TypeReference<List<String>>(){};
    private static final TypeReference<AuthenticationResponse> AUTHENTICATION_RESPONSE_TYPE = new TypeReference<AuthenticationResponse>(){};
    private static final Map<String, String> DEFAULT_REQUEST_PROPERTIES;
    private static ObjectMapper objectMapper;
    private final Time time;
    private final ThreadFactory restClientThreadFactory = new RestClientThreadFactory("restClient");
    private final int requestTimeout;
    private final int httpRequestTimeout;
    private final List<String> bootstrapMetadataServerURLs;
    private volatile List<String> activeMetadataServerURLs;
    private final String protocol;
    private SSLSocketFactory sslSocketFactory;
    private final HostnameVerifier hostNameVerifier;
    private final AtomicReference<HttpCredentialProvider> credentialProvider;
    private ScheduledExecutorService urlRefreshscheduler;
    private RequestSender requestSender = new HTTPRequestSender();

    public RestClient(Map<String, ?> configs) {
        this(configs, Time.SYSTEM);
    }

    public RestClient(Map<String, ?> configs, Time time) {
        this.time = time;
        RestClientConfig rbacClientConfig = new RestClientConfig(configs);
        this.bootstrapMetadataServerURLs = rbacClientConfig.getList("confluent.metadata.bootstrap.server.urls");
        if (this.bootstrapMetadataServerURLs.isEmpty()) {
            throw new ConfigException("Missing required bootstrap metadata server url list.");
        }
        this.protocol = this.protocol(this.bootstrapMetadataServerURLs);
        this.requestTimeout = rbacClientConfig.getInt("confluent.metadata.request.timeout.ms");
        this.httpRequestTimeout = rbacClientConfig.getInt("confluent.metadata.http.request.timeout.ms");
        String credentialProviderName = rbacClientConfig.getString("confluent.metadata.http.auth.credentials.provider");
        this.credentialProvider = new AtomicReference();
        if (credentialProviderName != null && !credentialProviderName.isEmpty()) {
            this.setCredentialProvider(BuiltInAuthProviders.loadHttpCredentialProviders(credentialProviderName));
            this.credentialProvider().configure(configs);
        }
        Map<String, ?> sslClientConfigs = rbacClientConfig.sslClientConfigs();
        if ("https".equals(this.protocol) && sslClientConfigs.get("ssl.truststore.location") != null) {
            this.sslSocketFactory = this.createSslSocketFactory(sslClientConfigs);
        }
        this.hostNameVerifier = this.createHostNameVerifier(sslClientConfigs);
        this.activeMetadataServerURLs = this.bootstrapMetadataServerURLs;
        if (rbacClientConfig.getBoolean("confluent.metadata.enable.server.urls.refresh").booleanValue()) {
            this.scheduleMetadataServiceUrlRefresh(rbacClientConfig);
        }
    }

    private HttpCredentialProvider credentialProvider() {
        return this.credentialProvider.get();
    }

    private String protocol(List<String> bootstrapMetadataServerURLs) {
        try {
            return new URL(bootstrapMetadataServerURLs.get(0)).getProtocol();
        }
        catch (MalformedURLException e) {
            throw new IllegalArgumentException("Error while fetching URL protocol", e);
        }
    }

    private void scheduleMetadataServiceUrlRefresh(final RestClientConfig rbacClientConfig) {
        Long metadataServerUrlsMaxAgeMS = rbacClientConfig.getLong("confluent.metadata.server.urls.max.age.ms");
        this.urlRefreshscheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = this.restClientThreadFactory.newThread(r);
            t.setDaemon(true);
            return t;
        });
        class MetadataServerUrlFetcher
        implements Runnable {
            MetadataServerUrlFetcher() {
            }

            @Override
            public void run() {
                Integer refreshUrlMaxRetries = rbacClientConfig.getInt("confluent.metadata.server.urls.max.retries");
                int retries = 0;
                ExponentialBackoff backoff = new ExponentialBackoff(100L, 2, 5000L, 0.1);
                while (true) {
                    try {
                        RestClient.this.activeMetadataServerURLs = RestClient.this.getActiveMetadataServerURLs();
                        log.debug("Successfully fetched MDS URLs ({})", (Object)RestClient.this.activeMetadataServerURLs);
                        return;
                    }
                    catch (Exception e) {
                        RestClientException restClientException;
                        if (e instanceof RestClientException && (restClientException = (RestClientException)e).errorCode() == 401 && rbacClientConfig.getBoolean("confluent.metadata.server.urls.fail.on.401").booleanValue()) {
                            log.error("Connection attempt failed with bad authorization", (Throwable)restClientException);
                            return;
                        }
                        log.error("Error while refreshing active metadata server urls, retrying", (Throwable)e);
                        if (refreshUrlMaxRetries >= 0 && retries >= refreshUrlMaxRetries) {
                            log.error("Failed to fetch MDS URLs", (Throwable)e);
                            return;
                        }
                        try {
                            Thread.sleep(backoff.backoff((long)retries));
                        }
                        catch (InterruptedException ex) {
                            throw new RuntimeException(ex);
                        }
                        ++retries;
                        continue;
                    }
                    break;
                }
            }
        }
        this.urlRefreshscheduler.scheduleAtFixedRate(new MetadataServerUrlFetcher(), 0L, metadataServerUrlsMaxAgeMS, TimeUnit.MILLISECONDS);
    }

    private SSLSocketFactory createSslSocketFactory(Map<String, ?> sslConfigs) {
        SslFactory sslFactory = new SslFactory(Mode.CLIENT);
        sslFactory.configure(sslConfigs);
        return DefaultSslEngineFactory.castOrThrow((SslEngineFactory)sslFactory.sslEngineFactory()).sslContext().getSocketFactory();
    }

    private HostnameVerifier createHostNameVerifier(Map<String, ?> sslConfigs) {
        String sslEndpointIdentificationAlgo = (String)sslConfigs.get("ssl.endpoint.identification.algorithm");
        if (sslEndpointIdentificationAlgo == null || sslEndpointIdentificationAlgo.isEmpty()) {
            log.debug("SSL Endpoint Identification Algorithm was empty, disabling hostname verification.");
            return (hostname, session) -> true;
        }
        return HttpsURLConnection.getDefaultHostnameVerifier();
    }

    HostnameVerifier getHostNameVerifier() {
        return this.hostNameVerifier;
    }

    List<String> activeMetadataServerURLs() {
        return this.activeMetadataServerURLs;
    }

    List<String> getActiveMetadataServerURLs() throws RestClientException, URISyntaxException {
        RestRequest request = this.newRequest(String.format(ACTIVE_NODES_ENDPOINT, this.protocol));
        request.setCredentialProvider(this.credentialProvider());
        request.setResponse(ACTIVE_URLS_RESPONSE_TYPE);
        return Stream.of((List)this.sendRequest(request), this.bootstrapMetadataServerURLs).flatMap(Collection::stream).distinct().collect(Collectors.toList());
    }

    private void setupSsl(HttpURLConnection connection) {
        if (connection instanceof HttpsURLConnection && this.sslSocketFactory != null) {
            ((HttpsURLConnection)connection).setSSLSocketFactory(this.sslSocketFactory);
        }
        if (connection instanceof HttpsURLConnection && this.hostNameVerifier != null) {
            ((HttpsURLConnection)connection).setHostnameVerifier(this.hostNameVerifier);
        }
    }

    void requestSender(RequestSender requestSender) {
        this.requestSender = requestSender;
    }

    public void setCredentialProvider(HttpCredentialProvider credentialProvider) {
        this.credentialProvider.set(credentialProvider);
    }

    public RestRequest newRequest(String path) {
        RestRequest request = new RestRequest(this.protocol, path);
        request.setCredentialProvider(this.credentialProvider());
        return request;
    }

    public OAuthBearerToken login() throws AuthenticationException {
        return this.login(this.credentialProvider());
    }

    public OAuthBearerToken login(String userInfo) {
        return this.login(new HttpBasicCredentialProvider(userInfo));
    }

    public OAuthBearerToken login(HttpCredentialProvider credentialProvider) {
        RestRequest request = this.newRequest(AUTHENTICATE_ENDPOINT);
        request.setCredentialProvider(credentialProvider);
        request.setResponse(AUTHENTICATION_RESPONSE_TYPE);
        try {
            AuthenticationResponse response = (AuthenticationResponse)this.sendRequest(request);
            return new JwtBearerToken(response.authenticationToken());
        }
        catch (Exception e) {
            throw new AuthenticationException("Failed to authenticate", (Throwable)e);
        }
    }

    public <T> T sendRequest(RestRequest request) throws RestClientException, URISyntaxException {
        long begin = this.time.milliseconds();
        long remainingWaitMs = this.requestTimeout;
        UrlSelector urlSelector = new UrlSelector(this.activeMetadataServerURLs);
        while (true) {
            try {
                URI mds = new URI(urlSelector.current());
                request.setHost(mds.getHost());
                request.setPort(mds.getPort());
                if (urlSelector.failures() > 0) {
                    log.debug("HTTP Request: Failures: {}, Host: {}, Port: {}, URI: {}, requestTimeout: {}", new Object[]{urlSelector.failures(), mds.getHost(), mds.getPort(), mds, this.requestTimeout});
                }
                return this.requestSender.send(request, remainingWaitMs);
            }
            catch (RequestSenderShutdownException e) {
                String msg = "Http request failed. Client has already been closed.";
                log.debug(msg, (Throwable)e);
                throw new RuntimeException(msg);
            }
            catch (IOException | RuntimeException e) {
                log.debug(String.format("Http request to %s failed, selecting next url.", urlSelector.current()), (Throwable)e);
                urlSelector.fail();
                long elapsed = this.time.milliseconds() - begin;
                if (elapsed < (long)this.requestTimeout) {
                    remainingWaitMs = (long)this.requestTimeout - elapsed;
                    int sleepTime = Math.min(500, urlSelector.round() * 20);
                    this.time.sleep((long)sleepTime);
                    continue;
                }
                throw new TimeoutException(String.format("Request aborted due to timeout (%s).", this.requestTimeout));
            }
            break;
        }
    }

    @Override
    public void close() {
        if (this.urlRefreshscheduler != null) {
            this.urlRefreshscheduler.shutdownNow();
        }
        if (this.requestSender != null) {
            Utils.closeQuietly((AutoCloseable)this.requestSender, (String)"requestSender");
        }
    }

    static {
        objectMapper = JsonMapper.objectMapper();
        DEFAULT_REQUEST_PROPERTIES = new HashMap<String, String>();
        DEFAULT_REQUEST_PROPERTIES.put("Content-Type", "application/json");
    }

    private class RequestSenderShutdownException
    extends RuntimeException {
        private RequestSenderShutdownException() {
        }
    }

    private class HTTPRequestSender
    implements RequestSender {
        ExecutorService executor;

        private HTTPRequestSender() {
            this.executor = new ThreadPoolExecutor(0, Integer.MAX_VALUE, 1L, TimeUnit.MINUTES, new SynchronousQueue<Runnable>(), RestClient.this.restClientThreadFactory);
        }

        @Override
        public <T> T send(RestRequest request, long requestTimeout) throws IOException, RestClientException {
            Future<T> f = this.submit(request);
            try {
                return f.get(Math.min(requestTimeout, (long)RestClient.this.httpRequestTimeout), TimeUnit.MILLISECONDS);
            }
            catch (Throwable e) {
                if (e instanceof ExecutionException) {
                    e = e.getCause();
                }
                if (e instanceof RestClientException) {
                    throw (RestClientException)e;
                }
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                throw new RuntimeException(e);
            }
        }

        private <T> Future<T> submit(RestRequest request) {
            if (this.executor.isShutdown()) {
                throw new RequestSenderShutdownException();
            }
            return this.executor.submit(() -> {
                HttpURLConnection connection = null;
                try {
                    int responseCode;
                    URL url = request.build();
                    connection = (HttpURLConnection)url.openConnection();
                    connection.setConnectTimeout(60000);
                    connection.setReadTimeout(60000);
                    RestClient.this.setupSsl(connection);
                    request.configureConnection(connection);
                    connection.setUseCaches(false);
                    for (Map.Entry entry : DEFAULT_REQUEST_PROPERTIES.entrySet()) {
                        connection.setRequestProperty((String)entry.getKey(), (String)entry.getValue());
                    }
                    if (connection.getDoOutput()) {
                        try {
                            Throwable throwable = null;
                            try (OutputStream os = connection.getOutputStream();){
                                request.writeRequestBody(os);
                                os.flush();
                            }
                            catch (Throwable throwable2) {
                                Throwable throwable3 = throwable2;
                                throw throwable2;
                            }
                        }
                        catch (IOException e) {
                            log.error("Failed to send HTTP request to endpoint: " + url, (Throwable)e);
                            throw e;
                        }
                    }
                    if ((responseCode = connection.getResponseCode()) == 200) {
                        InputStream inputStream = connection.getInputStream();
                        Object result = request.readResponse(inputStream);
                        inputStream.close();
                        Object t = result;
                        return t;
                    }
                    if (responseCode == 204) {
                        Object var5_13 = null;
                        return var5_13;
                    }
                    try {
                        ErrorMessage errorMessage = this.processHTTPError(connection);
                        throw new RestClientException(errorMessage.message(), responseCode, errorMessage.errorCode());
                    }
                    catch (Throwable e) {
                        log.error(RestClient.HTTP_EXCEPTION_MSG, e);
                        throw e;
                    }
                }
                finally {
                    if (connection != null) {
                        connection.disconnect();
                    }
                }
            });
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private ErrorMessage processHTTPError(HttpURLConnection connection) throws IOException {
            InputStream es = connection.getErrorStream();
            int responseCode = connection.getResponseCode();
            if (es != null) {
                try {
                    ErrorMessage errorMessage = (ErrorMessage)objectMapper.readValue(es, ErrorMessage.class);
                    return errorMessage;
                }
                catch (JsonProcessingException jsonProcessingException) {
                }
                finally {
                    es.close();
                }
            }
            return new ErrorMessage(responseCode, connection.getResponseMessage());
        }

        @Override
        public void close() {
            this.executor.shutdownNow();
        }
    }
}

