/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.events.exporter.http;

import io.confluent.shaded.io.netty.handler.ssl.SslContext;
import io.confluent.shaded.io.netty.handler.ssl.SslContextBuilder;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.errors.InvalidConfigurationException;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslFactory {
    private static final Logger log = LoggerFactory.getLogger(SslFactory.class);
    private final String kmfAlgorithm;
    private final String tmfAlgorithm;
    private final SecurityStore keystore;
    private final SecurityStore truststore;
    private String protocol;

    public SslFactory(Map<String, Object> configs) {
        this.protocol = (String)configs.get("ssl.protocol");
        this.kmfAlgorithm = (String)configs.get("ssl.keymanager.algorithm");
        this.tmfAlgorithm = (String)configs.get("ssl.trustmanager.algorithm");
        String keystoreType = (String)configs.get("ssl.keystore.type");
        String truststoreType = (String)configs.get("ssl.truststore.type");
        try {
            this.keystore = this.createKeystore(keystoreType, (String)configs.get("ssl.keystore.location"), (Password)configs.get("ssl.keystore.password"), (Password)configs.get("ssl.key.password"), (Password)configs.get("ssl.keystore.key"), (Password)configs.get("ssl.keystore.certificate.chain"));
            this.truststore = SslFactory.createTruststore(truststoreType, (String)configs.get("ssl.truststore.location"), (Password)configs.get("ssl.truststore.password"), (Password)configs.get("ssl.truststore.certificates"));
        }
        catch (Exception e) {
            throw new RuntimeException("Error initializing the ssl context for HTTPExporter", e);
        }
    }

    private static boolean isNotEmpty(CharSequence cs) {
        return !SslFactory.isEmpty(cs);
    }

    private static boolean isEmpty(CharSequence cs) {
        return cs == null || cs.length() == 0;
    }

    private static SecurityStore createTruststore(String type, String path, Password password, Password trustStoreCerts) {
        if (trustStoreCerts != null) {
            return SslFactory.createPemTrustStore(type, path, password, trustStoreCerts);
        }
        if ("PEM".equals(type) && SslFactory.isNotEmpty(path)) {
            if (password != null) {
                throw new InvalidConfigurationException("SSL trust store password cannot be specified for PEM format.");
            }
            return new FileBasedPemStore(path, null, false);
        }
        if (path == null && password != null) {
            throw new InvalidConfigurationException("SSL trust store is not specified, but trust store password is specified.");
        }
        if (SslFactory.isNotEmpty(path)) {
            return new FileBasedStore(type, path, password, null, false);
        }
        return null;
    }

    private static SecurityStore createPemTrustStore(String type, String path, Password password, Password trustStoreCerts) {
        if (!"PEM".equals(type)) {
            throw new InvalidConfigurationException("SSL trust store certs can be specified only for PEM, but trust store type is " + type + ".");
        }
        if (SslFactory.isNotEmpty(path)) {
            throw new InvalidConfigurationException("Both SSL trust store location and separate trust certificates are specified.");
        }
        if (password != null) {
            throw new InvalidConfigurationException("SSL trust store password cannot be specified for PEM format.");
        }
        return new PemStore(trustStoreCerts);
    }

    SecurityStore keyStore() {
        return this.keystore;
    }

    SecurityStore trustStore() {
        return this.truststore;
    }

    public SslContext getSslContext() {
        if (this.truststore == null && this.keystore == null) {
            return null;
        }
        try {
            log.debug("Trying to Create netty SSL context using keystore {}, truststore {}", (Object)this.keystore, (Object)this.truststore);
            String tmfAlgorithm = SslFactory.isNotEmpty(this.tmfAlgorithm) ? this.tmfAlgorithm : TrustManagerFactory.getDefaultAlgorithm();
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
            KeyStore ts = this.truststore == null ? null : this.truststore.get();
            tmf.init(ts);
            SslContextBuilder builder = SslContextBuilder.forClient().trustManager(tmf).protocols(new String[]{this.protocol});
            if (this.keystore != null || SslFactory.isNotEmpty(this.kmfAlgorithm)) {
                String kmfAlgorithm = SslFactory.isNotEmpty(this.kmfAlgorithm) ? this.kmfAlgorithm : KeyManagerFactory.getDefaultAlgorithm();
                KeyManagerFactory kmf = KeyManagerFactory.getInstance(kmfAlgorithm);
                if (this.keystore != null) {
                    kmf.init(this.keystore.get(), this.keystore.keyPassword());
                } else {
                    kmf.init(null, null);
                }
                builder.keyManager(kmf);
            }
            log.debug("Created netty SSL context with keystore {}, truststore {}", (Object)this.keystore, (Object)this.truststore);
            return builder.build();
        }
        catch (Exception e) {
            throw new ConfigException(e.getMessage());
        }
    }

    private SecurityStore createKeystore(String type, String path, Password password, Password keyPassword, Password privateKey, Password certificateChain) {
        if (privateKey != null) {
            return SslFactory.createPemKeyStore(type, path, password, keyPassword, privateKey, certificateChain);
        }
        if (certificateChain != null) {
            throw new InvalidConfigurationException("SSL certificate chain is specified, but private key is not specified");
        }
        if ("PEM".equals(type) && path != null) {
            return SslFactory.createFileBasedPemStore(path, password, keyPassword);
        }
        if (path == null && password != null) {
            throw new InvalidConfigurationException("SSL key store is not specified, but key store password is specified.");
        }
        if (SslFactory.isNotEmpty(path)) {
            if (password == null) {
                throw new InvalidConfigurationException("SSL key store is specified, but key store password is not specified.");
            }
            return new FileBasedStore(type, path, password, keyPassword, true);
        }
        return null;
    }

    private static SecurityStore createPemKeyStore(String type, String path, Password password, Password keyPassword, Password privateKey, Password certificateChain) {
        if (!"PEM".equals(type)) {
            throw new InvalidConfigurationException("SSL private key can be specified only for PEM, but key store type is " + type + ".");
        }
        if (certificateChain == null) {
            throw new InvalidConfigurationException("SSL private key is specified, but certificate chain is not specified.");
        }
        if (path != null) {
            throw new InvalidConfigurationException("Both SSL key store location and separate private key are specified.");
        }
        if (password != null) {
            throw new InvalidConfigurationException("SSL key store password cannot be specified with PEM format, only key password may be specified.");
        }
        return new PemStore(certificateChain, privateKey, keyPassword);
    }

    private static SecurityStore createFileBasedPemStore(String path, Password password, Password keyPassword) {
        if (password != null) {
            throw new InvalidConfigurationException("SSL key store password cannot be specified with PEM format, only key password may be specified");
        }
        return new FileBasedPemStore(path, keyPassword, true);
    }

    static interface SecurityStore {
        public KeyStore get();

        public char[] keyPassword();
    }

    static class FileBasedPemStore
    extends FileBasedStore {
        FileBasedPemStore(String path, Password keyPassword, boolean isKeyStore) {
            super("PEM", path, null, keyPassword, isKeyStore);
        }

        @Override
        protected KeyStore load(boolean isKeyStore) {
            try {
                Password storeContents = new Password(Utils.readFileAsString((String)this.path));
                PemStore pemStore = isKeyStore ? new PemStore(storeContents, storeContents, this.keyPassword) : new PemStore(storeContents);
                return pemStore.keyStore;
            }
            catch (Exception e) {
                throw new InvalidConfigurationException("Failed to load PEM SSL keystore " + this.path, (Throwable)e);
            }
        }
    }

    static class FileBasedStore
    implements SecurityStore {
        protected final String path;
        protected final Password keyPassword;
        private final String type;
        private final Password password;
        private final KeyStore keyStore;

        FileBasedStore(String type, String path, Password password, Password keyPassword, boolean isKeyStore) {
            Objects.requireNonNull(type, "type must not be null");
            this.type = type;
            this.path = path;
            this.password = password;
            this.keyPassword = keyPassword;
            this.keyStore = this.load(isKeyStore);
        }

        @Override
        public KeyStore get() {
            return this.keyStore;
        }

        @Override
        public char[] keyPassword() {
            Password passwd = this.keyPassword != null ? this.keyPassword : this.password;
            return passwd == null ? null : passwd.value().toCharArray();
        }

        protected KeyStore load(boolean isKeyStore) {
            KeyStore keyStore;
            block8: {
                InputStream in = Files.newInputStream(Paths.get(this.path, new String[0]), new OpenOption[0]);
                try {
                    KeyStore ks = KeyStore.getInstance(this.type);
                    char[] passwordChars = this.password != null ? this.password.value().toCharArray() : null;
                    ks.load(in, passwordChars);
                    keyStore = ks;
                    if (in == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (in != null) {
                            try {
                                in.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException | GeneralSecurityException e) {
                        throw new KafkaException("Failed to load SSL keystore " + this.path + " of type " + this.type, (Throwable)e);
                    }
                }
                in.close();
            }
            return keyStore;
        }

        public String toString() {
            return "SecurityStore(path=" + this.path + ")";
        }
    }

    static class PemStore
    implements SecurityStore {
        private static final PemParser CERTIFICATE_PARSER = new PemParser("CERTIFICATE");
        private static final PemParser PRIVATE_KEY_PARSER = new PemParser("PRIVATE KEY");
        private static final List<KeyFactory> KEY_FACTORIES = Arrays.asList(PemStore.keyFactory("RSA"), PemStore.keyFactory("DSA"), PemStore.keyFactory("EC"));
        private final char[] keyPassword;
        private final KeyStore keyStore;

        PemStore(Password certificateChain, Password privateKey, Password keyPassword) {
            this.keyPassword = keyPassword == null ? null : keyPassword.value().toCharArray();
            this.keyStore = this.createKeyStoreFromPem(privateKey.value(), certificateChain.value(), this.keyPassword);
        }

        PemStore(Password trustStoreCerts) {
            this.keyPassword = null;
            this.keyStore = this.createTrustStoreFromPem(trustStoreCerts.value());
        }

        private static KeyFactory keyFactory(String algorithm) {
            try {
                return KeyFactory.getInstance(algorithm);
            }
            catch (Exception e) {
                throw new InvalidConfigurationException("Could not create key factory for algorithm " + algorithm, (Throwable)e);
            }
        }

        @Override
        public KeyStore get() {
            return this.keyStore;
        }

        @Override
        public char[] keyPassword() {
            return this.keyPassword;
        }

        private KeyStore createKeyStoreFromPem(String privateKeyPem, String certChainPem, char[] keyPassword) {
            try {
                KeyStore ks = KeyStore.getInstance("PKCS12");
                ks.load(null, null);
                PrivateKey key = this.privateKey(privateKeyPem, keyPassword);
                Certificate[] certChain = this.certs(certChainPem);
                ks.setKeyEntry("kafka", key, keyPassword, certChain);
                return ks;
            }
            catch (Exception e) {
                throw new InvalidConfigurationException("Invalid PEM keystore configs", (Throwable)e);
            }
        }

        private KeyStore createTrustStoreFromPem(String trustedCertsPem) {
            try {
                KeyStore ts = KeyStore.getInstance("PKCS12");
                ts.load(null, null);
                Certificate[] certs = this.certs(trustedCertsPem);
                for (int i = 0; i < certs.length; ++i) {
                    ts.setCertificateEntry("kafka" + i, certs[i]);
                }
                return ts;
            }
            catch (InvalidConfigurationException e) {
                throw e;
            }
            catch (Exception e) {
                throw new InvalidConfigurationException("Invalid PEM keystore configs", (Throwable)e);
            }
        }

        private Certificate[] certs(String pem) throws GeneralSecurityException {
            List<byte[]> certEntries = CERTIFICATE_PARSER.pemEntries(pem);
            if (certEntries.isEmpty()) {
                throw new InvalidConfigurationException("At least one certificate expected, but none found");
            }
            Certificate[] certs = new Certificate[certEntries.size()];
            for (int i = 0; i < certs.length; ++i) {
                certs[i] = CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(certEntries.get(i)));
            }
            return certs;
        }

        private PrivateKey privateKey(String pem, char[] keyPassword) throws Exception {
            PKCS8EncodedKeySpec keySpec;
            List<byte[]> keyEntries = PRIVATE_KEY_PARSER.pemEntries(pem);
            if (keyEntries.isEmpty()) {
                throw new InvalidConfigurationException("Private key not provided");
            }
            if (keyEntries.size() != 1) {
                throw new InvalidConfigurationException("Expected one private key, but found " + keyEntries.size());
            }
            byte[] keyBytes = keyEntries.get(0);
            if (keyPassword == null) {
                keySpec = new PKCS8EncodedKeySpec(keyBytes);
            } else {
                EncryptedPrivateKeyInfo keyInfo = new EncryptedPrivateKeyInfo(keyBytes);
                String algorithm = keyInfo.getAlgName();
                SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(algorithm);
                SecretKey pbeKey = keyFactory.generateSecret(new PBEKeySpec(keyPassword));
                Cipher cipher = Cipher.getInstance(algorithm);
                cipher.init(2, (Key)pbeKey, keyInfo.getAlgParameters());
                keySpec = keyInfo.getKeySpec(cipher);
            }
            InvalidKeySpecException firstException = null;
            for (KeyFactory factory : KEY_FACTORIES) {
                try {
                    return factory.generatePrivate(keySpec);
                }
                catch (InvalidKeySpecException e) {
                    if (firstException != null) continue;
                    firstException = e;
                }
            }
            throw new InvalidConfigurationException("Private key could not be loaded", firstException);
        }
    }

    static class PemParser {
        private final String name;
        private final Pattern pattern;

        PemParser(String name) {
            this.name = name;
            String beginOrEndFormat = "-+%s\\s*.*%s[^-]*-+\\s+";
            String nameIgnoreSpace = name.replace(" ", "\\s+");
            String encodingParams = "\\s*[^\\r\\n]*:[^\\r\\n]*[\\r\\n]+";
            String base64Pattern = "([a-zA-Z0-9/+=\\s]*)";
            String patternStr = String.format(beginOrEndFormat, "BEGIN", nameIgnoreSpace) + String.format("(?:%s)*", encodingParams) + base64Pattern + String.format(beginOrEndFormat, "END", nameIgnoreSpace);
            this.pattern = Pattern.compile(patternStr);
        }

        private List<byte[]> pemEntries(String pem) {
            Matcher matcher = this.pattern.matcher(pem + "\n");
            ArrayList<byte[]> entries = new ArrayList<byte[]>();
            while (matcher.find()) {
                String base64Str = matcher.group(1).replaceAll("\\s", "");
                entries.add(Base64.getDecoder().decode(base64Str));
            }
            if (entries.isEmpty()) {
                throw new InvalidConfigurationException("No matching " + this.name + " entries in PEM file");
            }
            return entries;
        }
    }
}

