/*
 * Decompiled with CFR 0.152.
 */
package com.squareup.okhttp.internal.ws;

import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.ResponseBody;
import com.squareup.okhttp.internal.ws.WebSocketProtocol;
import com.squareup.okhttp.ws.WebSocket;
import java.io.EOFException;
import java.io.IOException;
import java.net.ProtocolException;
import okio.Buffer;
import okio.BufferedSource;
import okio.Okio;
import okio.Source;
import okio.Timeout;

public final class WebSocketReader {
    private final boolean isClient;
    private final BufferedSource source;
    private final FrameCallback frameCallback;
    private final Source framedMessageSource = new FramedMessageSource();
    private boolean closed;
    private boolean messageClosed;
    private int opcode;
    private long frameLength;
    private long frameBytesRead;
    private boolean isFinalFrame;
    private boolean isControlFrame;
    private boolean isMasked;
    private final byte[] maskKey = new byte[4];
    private final byte[] maskBuffer = new byte[2048];

    public WebSocketReader(boolean isClient, BufferedSource source, FrameCallback frameCallback) {
        if (source == null) {
            throw new NullPointerException("source == null");
        }
        if (frameCallback == null) {
            throw new NullPointerException("frameCallback == null");
        }
        this.isClient = isClient;
        this.source = source;
        this.frameCallback = frameCallback;
    }

    public void processNextFrame() throws IOException {
        this.readHeader();
        if (this.isControlFrame) {
            this.readControlFrame();
        } else {
            this.readMessageFrame();
        }
    }

    private void readHeader() throws IOException {
        boolean reservedFlag3;
        if (this.closed) {
            throw new IOException("closed");
        }
        int b0 = this.source.readByte() & 0xFF;
        this.opcode = b0 & 0xF;
        this.isFinalFrame = (b0 & 0x80) != 0;
        boolean bl = this.isControlFrame = (b0 & 8) != 0;
        if (this.isControlFrame && !this.isFinalFrame) {
            throw new ProtocolException("Control frames must be final.");
        }
        boolean reservedFlag1 = (b0 & 0x40) != 0;
        boolean reservedFlag2 = (b0 & 0x20) != 0;
        boolean bl2 = reservedFlag3 = (b0 & 0x10) != 0;
        if (reservedFlag1 || reservedFlag2 || reservedFlag3) {
            throw new ProtocolException("Reserved flags are unsupported.");
        }
        int b1 = this.source.readByte() & 0xFF;
        boolean bl3 = this.isMasked = (b1 & 0x80) != 0;
        if (this.isMasked == this.isClient) {
            throw new ProtocolException("Client-sent frames must be masked. Server sent must not.");
        }
        this.frameLength = b1 & 0x7F;
        if (this.frameLength == 126L) {
            this.frameLength = (long)this.source.readShort() & 0xFFFFL;
        } else if (this.frameLength == 127L) {
            this.frameLength = this.source.readLong();
            if (this.frameLength < 0L) {
                throw new ProtocolException("Frame length 0x" + Long.toHexString(this.frameLength) + " > 0x7FFFFFFFFFFFFFFF");
            }
        }
        this.frameBytesRead = 0L;
        if (this.isControlFrame && this.frameLength > 125L) {
            throw new ProtocolException("Control frame must be less than 125B.");
        }
        if (this.isMasked) {
            this.source.readFully(this.maskKey);
        }
    }

    private void readControlFrame() throws IOException {
        Buffer buffer = null;
        if (this.frameBytesRead < this.frameLength) {
            buffer = new Buffer();
            if (this.isClient) {
                this.source.readFully(buffer, this.frameLength);
            } else {
                while (this.frameBytesRead < this.frameLength) {
                    int toRead = (int)Math.min(this.frameLength - this.frameBytesRead, (long)this.maskBuffer.length);
                    int read = this.source.read(this.maskBuffer, 0, toRead);
                    if (read == -1) {
                        throw new EOFException();
                    }
                    WebSocketProtocol.toggleMask(this.maskBuffer, read, this.maskKey, this.frameBytesRead);
                    buffer.write(this.maskBuffer, 0, read);
                    this.frameBytesRead += (long)read;
                }
            }
        }
        switch (this.opcode) {
            case 9: {
                this.frameCallback.onPing(buffer);
                break;
            }
            case 10: {
                this.frameCallback.onPong(buffer);
                break;
            }
            case 8: {
                int code = 1000;
                String reason = "";
                if (buffer != null) {
                    long bufferSize = buffer.size();
                    if (bufferSize == 1L) {
                        throw new ProtocolException("Malformed close payload length of 1.");
                    }
                    if (bufferSize != 0L) {
                        code = buffer.readShort();
                        if (code < 1000 || code >= 5000) {
                            throw new ProtocolException("Code must be in range [1000,5000): " + code);
                        }
                        if (code >= 1004 && code <= 1006 || code >= 1012 && code <= 2999) {
                            throw new ProtocolException("Code " + code + " is reserved and may not be used.");
                        }
                        reason = buffer.readUtf8();
                    }
                }
                this.frameCallback.onClose(code, reason);
                this.closed = true;
                break;
            }
            default: {
                throw new ProtocolException("Unknown control opcode: " + Integer.toHexString(this.opcode));
            }
        }
    }

    private void readMessageFrame() throws IOException {
        MediaType type;
        switch (this.opcode) {
            case 1: {
                type = WebSocket.TEXT;
                break;
            }
            case 2: {
                type = WebSocket.BINARY;
                break;
            }
            default: {
                throw new ProtocolException("Unknown opcode: " + Integer.toHexString(this.opcode));
            }
        }
        final BufferedSource source = Okio.buffer(this.framedMessageSource);
        ResponseBody body = new ResponseBody(){

            @Override
            public MediaType contentType() {
                return type;
            }

            @Override
            public long contentLength() throws IOException {
                return -1L;
            }

            @Override
            public BufferedSource source() throws IOException {
                return source;
            }
        };
        this.messageClosed = false;
        this.frameCallback.onMessage(body);
        if (!this.messageClosed) {
            throw new IllegalStateException("Listener failed to call close on message payload.");
        }
    }

    private void readUntilNonControlFrame() throws IOException {
        while (!this.closed) {
            this.readHeader();
            if (!this.isControlFrame) break;
            this.readControlFrame();
        }
    }

    private final class FramedMessageSource
    implements Source {
        private FramedMessageSource() {
        }

        @Override
        public long read(Buffer sink, long byteCount) throws IOException {
            long read;
            if (WebSocketReader.this.closed) {
                throw new IOException("closed");
            }
            if (WebSocketReader.this.messageClosed) {
                throw new IllegalStateException("closed");
            }
            if (WebSocketReader.this.frameBytesRead == WebSocketReader.this.frameLength) {
                if (WebSocketReader.this.isFinalFrame) {
                    return -1L;
                }
                WebSocketReader.this.readUntilNonControlFrame();
                if (WebSocketReader.this.opcode != 0) {
                    throw new ProtocolException("Expected continuation opcode. Got: " + Integer.toHexString(WebSocketReader.this.opcode));
                }
                if (WebSocketReader.this.isFinalFrame && WebSocketReader.this.frameLength == 0L) {
                    return -1L;
                }
            }
            long toRead = Math.min(byteCount, WebSocketReader.this.frameLength - WebSocketReader.this.frameBytesRead);
            if (WebSocketReader.this.isMasked) {
                toRead = Math.min(toRead, (long)WebSocketReader.this.maskBuffer.length);
                read = WebSocketReader.this.source.read(WebSocketReader.this.maskBuffer, 0, (int)toRead);
                if (read == -1L) {
                    throw new EOFException();
                }
                WebSocketProtocol.toggleMask(WebSocketReader.this.maskBuffer, read, WebSocketReader.this.maskKey, WebSocketReader.this.frameBytesRead);
                sink.write(WebSocketReader.this.maskBuffer, 0, (int)read);
            } else {
                read = WebSocketReader.this.source.read(sink, toRead);
                if (read == -1L) {
                    throw new EOFException();
                }
            }
            WebSocketReader.this.frameBytesRead = WebSocketReader.this.frameBytesRead + read;
            return read;
        }

        @Override
        public Timeout timeout() {
            return WebSocketReader.this.source.timeout();
        }

        @Override
        public void close() throws IOException {
            if (WebSocketReader.this.messageClosed) {
                return;
            }
            WebSocketReader.this.messageClosed = true;
            if (WebSocketReader.this.closed) {
                return;
            }
            WebSocketReader.this.source.skip(WebSocketReader.this.frameLength - WebSocketReader.this.frameBytesRead);
            while (!WebSocketReader.this.isFinalFrame) {
                WebSocketReader.this.readUntilNonControlFrame();
                WebSocketReader.this.source.skip(WebSocketReader.this.frameLength);
            }
        }
    }

    public static interface FrameCallback {
        public void onMessage(ResponseBody var1) throws IOException;

        public void onPing(Buffer var1);

        public void onPong(Buffer var1);

        public void onClose(int var1, String var2);
    }
}

