/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.security.providers.httpsign;

import io.helidon.common.pki.KeyConfig;
import io.helidon.security.SecurityEnvironment;
import io.helidon.security.providers.httpsign.HttpSignatureException;
import io.helidon.security.providers.httpsign.InboundClientDefinition;
import io.helidon.security.providers.httpsign.OutboundTargetDefinition;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

class HttpSignature {
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.RFC_1123_DATE_TIME;
    private static final Logger LOGGER = Logger.getLogger(HttpSignature.class.getName());
    private static final List<String> DEFAULT_HEADERS = List.of("date");
    private static final byte[] EMPTY_BYTES = new byte[0];
    private final String keyId;
    private final String algorithm;
    private final List<String> headers;
    private String base64Signature;
    private byte[] signatureBytes;

    HttpSignature(String keyId, String algorithm, List<String> headers) {
        this.keyId = keyId;
        this.algorithm = algorithm;
        this.headers = headers;
    }

    private HttpSignature(String header, String keyId, String algorithm, List<String> headers, String base64Signature) {
        this.keyId = keyId;
        this.algorithm = algorithm;
        this.headers = headers;
        this.base64Signature = base64Signature;
    }

    static HttpSignature fromHeader(String header) {
        int qe;
        String keyId = null;
        String algorithm = null;
        List<String> headers = DEFAULT_HEADERS;
        String signature = null;
        int b = 0;
        do {
            int qb;
            int c = header.indexOf(44, b);
            int eq = header.indexOf(61, b);
            if (eq == -1) {
                return new HttpSignature(header, keyId, algorithm, headers, signature);
            }
            if (eq > c) {
                b = c + 1;
            }
            if ((qb = header.indexOf(34, eq)) == -1) {
                return new HttpSignature(header, keyId, algorithm, headers, signature);
            }
            qe = header.indexOf(34, qb + 1);
            if (qe == -1) {
                return new HttpSignature(header, keyId, algorithm, headers, signature);
            }
            String name = header.substring(b, eq).trim();
            String unquotedValue = header.substring(qb + 1, qe);
            switch (name) {
                case "keyId": {
                    keyId = unquotedValue;
                    break;
                }
                case "algorithm": {
                    algorithm = unquotedValue;
                    break;
                }
                case "signature": {
                    signature = unquotedValue;
                    break;
                }
                case "headers": {
                    headers = Arrays.asList(unquotedValue.split(" "));
                    break;
                }
                default: {
                    LOGGER.finest(() -> "Invalid signature header field: " + name + ": \"" + unquotedValue + "\"");
                }
            }
        } while ((b = qe + 1) < header.length());
        return new HttpSignature(header, keyId, algorithm, headers, signature);
    }

    static HttpSignature sign(SecurityEnvironment env, OutboundTargetDefinition outboundDefinition, Map<String, List<String>> newHeaders) {
        HttpSignature signature = new HttpSignature(outboundDefinition.keyId(), outboundDefinition.algorithm(), outboundDefinition.signedHeadersConfig().headers(env.method(), env.headers()));
        switch (signature.getAlgorithm()) {
            case "rsa-sha256": {
                signature.signatureBytes = signature.signRsaSha256(env, outboundDefinition.keyConfig().orElseThrow(() -> new HttpSignatureException("Private key configuration must be present to use rsa-sha256 algorithm")), newHeaders);
                break;
            }
            case "hmac-sha256": {
                signature.signatureBytes = signature.signHmacSha256(env, outboundDefinition.hmacSharedSecret().orElseThrow(() -> new HttpSignatureException("HMAC shared secret must be configured to use hmac-sha256 algorithm")), newHeaders);
                break;
            }
            default: {
                throw new HttpSignatureException("Unsupported signature algorithm: " + signature.getAlgorithm());
            }
        }
        signature.base64Signature = Base64.getEncoder().encodeToString(signature.signatureBytes);
        return signature;
    }

    String toSignatureHeader() {
        return "keyId=\"" + this.keyId + "\",algorithm=\"" + this.algorithm + "\",headers=\"" + String.join((CharSequence)" ", this.headers) + "\",signature=\"" + this.base64Signature + "\"";
    }

    String getKeyId() {
        return this.keyId;
    }

    String getAlgorithm() {
        return this.algorithm;
    }

    List<String> getHeaders() {
        return Collections.unmodifiableList(this.headers);
    }

    String getBase64Signature() {
        return this.base64Signature;
    }

    Optional<String> validate() {
        ArrayList<Object> problems = new ArrayList<Object>();
        if (null == this.keyId) {
            problems.add("keyId is a mandatory signature header component");
        }
        if (null == this.algorithm) {
            problems.add("algorithm is a mandatory signature header component");
        } else {
            switch (this.algorithm) {
                case "rsa-sha256": 
                case "hmac-sha256": {
                    break;
                }
                default: {
                    problems.add("Unsupported signature algorithm: " + this.algorithm);
                }
            }
        }
        if (null == this.base64Signature) {
            problems.add("signature is a mandatory signature header component");
        }
        try {
            this.signatureBytes = Base64.getDecoder().decode(this.base64Signature);
        }
        catch (Exception e) {
            LOGGER.log(Level.FINEST, "Cannot get bytes from base64: " + this.base64Signature, e);
            problems.add("cannot get bytes from base64 encoded signature: " + e.getMessage());
        }
        if (problems.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of("HttpSignature is not valid. Problems: " + String.join((CharSequence)", ", problems));
    }

    Optional<String> validate(SecurityEnvironment env, InboundClientDefinition clientDefinition, List<String> requiredHeaders) {
        if (!this.algorithm.equalsIgnoreCase(clientDefinition.algorithm())) {
            return Optional.of("Algorithm of signature is " + this.algorithm + ", configured: " + clientDefinition.algorithm());
        }
        for (String requiredHeader : requiredHeaders) {
            if (this.headers.contains(requiredHeader)) continue;
            return Optional.of("Header " + requiredHeader + " is required, yet not signed");
        }
        switch (this.algorithm) {
            case "rsa-sha256": {
                return this.validateRsaSha256(env, clientDefinition);
            }
            case "hmac-sha256": {
                return this.validateHmacSha256(env, clientDefinition);
            }
        }
        return Optional.of("Unsupported algorithm: " + this.algorithm);
    }

    private byte[] signRsaSha256(SecurityEnvironment env, KeyConfig keyConfig, Map<String, List<String>> newHeaders) {
        try {
            Signature signature = Signature.getInstance("SHA256withRSA");
            signature.initSign((PrivateKey)keyConfig.privateKey().orElseThrow(() -> new HttpSignatureException("Private key is required, yet not configured")));
            signature.update(this.getBytesToSign(env, newHeaders));
            return signature.sign();
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | SignatureException e) {
            throw new HttpSignatureException(e);
        }
    }

    private Optional<String> validateRsaSha256(SecurityEnvironment env, InboundClientDefinition clientDefinition) {
        try {
            Signature signature = Signature.getInstance("SHA256withRSA");
            signature.initVerify((PublicKey)clientDefinition.keyConfig().orElseThrow(() -> new HttpSignatureException("RSA public key configuration is required")).publicKey().orElseThrow(() -> new HttpSignatureException("Public key is required, yet not configured")));
            signature.update(this.getBytesToSign(env, null));
            if (!signature.verify(this.signatureBytes)) {
                return Optional.of("Signature is not valid");
            }
            return Optional.empty();
        }
        catch (NoSuchAlgorithmException e) {
            LOGGER.log(Level.FINEST, "SHA256withRSA algorithm not found", e);
            return Optional.of("SHA256withRSA algorithm not found: " + e.getMessage());
        }
        catch (InvalidKeyException e) {
            LOGGER.log(Level.FINEST, "Invalid RSA key", e);
            return Optional.of("Invalid RSA key: " + e.getMessage());
        }
        catch (SignatureException e) {
            LOGGER.log(Level.FINEST, "Signature exception", e);
            return Optional.of("SignatureException: " + e.getMessage());
        }
    }

    private byte[] signHmacSha256(SecurityEnvironment env, byte[] secret, Map<String, List<String>> newHeaders) {
        try {
            String algorithm = "HmacSHA256";
            Mac mac = Mac.getInstance(algorithm);
            SecretKeySpec secretKey = new SecretKeySpec(secret, algorithm);
            mac.init(secretKey);
            return mac.doFinal(this.getBytesToSign(env, newHeaders));
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new HttpSignatureException(e);
        }
    }

    private Optional<String> validateHmacSha256(SecurityEnvironment env, InboundClientDefinition clientDefinition) {
        try {
            byte[] signature = this.signHmacSha256(env, clientDefinition.hmacSharedSecret().orElse(EMPTY_BYTES), null);
            if (!Arrays.equals(signature, this.signatureBytes)) {
                return Optional.of("Signature is not valid");
            }
            return Optional.empty();
        }
        catch (SecurityException e) {
            LOGGER.log(Level.FINEST, "Failed to validate hmac-sha256", e);
            return Optional.of("Failed to validate hmac-sha256: " + e.getMessage());
        }
    }

    private byte[] getBytesToSign(SecurityEnvironment env, Map<String, List<String>> newHeaders) {
        return this.getSignedString(newHeaders, env).getBytes(StandardCharsets.UTF_8);
    }

    String getSignedString(Map<String, List<String>> newHeaders, SecurityEnvironment env) {
        StringBuilder toSign = new StringBuilder();
        Map requestHeaders = env.headers();
        for (String header : this.headers) {
            if ("(request-target)".equals(header)) {
                toSign.append(header).append(": ").append(env.method().toLowerCase()).append(" ").append(env.path().orElse("/")).append('\n');
                continue;
            }
            List<Object> headerValues = (List<String>)requestHeaders.get(header);
            if (null == headerValues && null == newHeaders) {
                throw new HttpSignatureException("Header " + header + " is required for signature, yet not defined in request");
            }
            if (null == headerValues) {
                if ("date".equalsIgnoreCase(header)) {
                    String date = ZonedDateTime.now(ZoneId.of("GMT")).format(DATE_FORMATTER);
                    headerValues = List.of(date);
                    newHeaders.put("date", headerValues);
                    LOGGER.finest(() -> "Added date header to request: " + date);
                } else if ("host".equalsIgnoreCase(header)) {
                    URI uri = env.targetUri();
                    String host = uri.getHost() + ":" + uri.getPort();
                    headerValues = List.of(host);
                    newHeaders.put("host", headerValues);
                    LOGGER.finest(() -> "Added host header to request: " + host);
                } else {
                    throw new HttpSignatureException("Header " + header + " is required for signature, yet not defined in request");
                }
            }
            toSign.append(header).append(": ").append(String.join((CharSequence)" ", (Iterable<? extends CharSequence>)headerValues)).append('\n');
        }
        LOGGER.finest(() -> "Data to sign: " + toSign);
        return toSign.toString();
    }
}

