/**
 * (C) Copyright IBM Corporation 2019, 2020.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.openliberty.tools.common.plugins.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.Watchable;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Scanner;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.StandardLocation;
import javax.tools.ToolProvider;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import com.sun.nio.file.SensitivityWatchEventModifier;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.input.CloseShieldInputStream;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.apache.commons.io.monitor.FileAlterationObserver;

import org.apache.maven.artifact.versioning.ComparableVersion;

import io.openliberty.tools.ant.ServerTask;
import io.openliberty.tools.common.plugins.config.ServerConfigDropinXmlDocument;

/**
 * Utility class for dev mode.
 */
public abstract class DevUtil extends AbstractContainerSupportUtil {

    private static final String START_SERVER_MESSAGE_PREFIX = "CWWKF0011I:";
    private static final String START_APP_MESSAGE_REGEXP = "CWWKZ0001I.*";
    private static final String UPDATED_APP_MESSAGE_REGEXP = "CWWKZ0003I.*";
    private static final String PORT_IN_USE_MESSAGE_PREFIX = "CWWKO0221E:";
    private static final String WEB_APP_AVAILABLE_MESSAGE_PREFIX = "CWWKT0016I:";
    private static final String LISTENING_ON_PORT_MESSAGE_PREFIX = "CWWKO0219I:";
    private static final String HTTP_PREFIX = "http://";
    private static final String HTTP_PREFIX_ESCAPED = "http:\\/\\/";
    private static final String HTTPS_PREFIX = "https://";
    private static final String HTTPS_PREFIX_ESCAPED = "https:\\/\\/";
    private static final String DEVMODE_DIR_NAME = "/devmode";
    public static final String DEVMODE_PROJECT_ROOT = "io.openliberty.tools.projectRoot";
    private static final String GENERATED_HEADER_REGEX = "# Generated by liberty-.*-plugin";
    private static final String DEVMODE_CONTAINER_BASE_NAME = "liberty-dev";
    private static final String DEVMODE_IMAGE_SUFFIX = "-dev-mode";
    public static final String SKIP_BETA_INSTALL_WARNING = "skipBetaInstallFeatureWarning";
    public static final String DEVC_HIDDEN_FOLDER = ".libertyDevc";

    private static final String[] IGNORE_DIRECTORY_PREFIXES = new String[] { "." };
    private static final String[] IGNORE_FILE_PREFIXES = new String[] { "." };
    private static final String[] IGNORE_FILE_POSTFIXES = new String[] {
            // core dumps
            ".dmp",
            // vim
            "~",
            // intellij
            "___jb_tmp___", "___jb_old___" };

    private static final String[] DEFAULT_COMPILER_OPTIONS = new String[] { "-g", "-parameters" };
    private static final int LIBERTY_DEFAULT_HTTP_PORT = 9080;
    private static final int LIBERTY_DEFAULT_HTTPS_PORT = 9443;
    private static final int LIBERTY_DEFAULT_DEBUG_PORT = 7777;
    private static final int DOCKER_TIMEOUT = 20; // seconds

    /**
     * Log debug
     * 
     * @param msg
     */
    public abstract void debug(String msg);

    /**
     * Log debug
     * 
     * @param msg
     * @param e
     */
    public abstract void debug(String msg, Throwable e);

    /**
     * Log debug
     * 
     * @param e
     */
    public abstract void debug(Throwable e);

    /**
     * Log warning
     * 
     * @param msg
     */
    public abstract void warn(String msg);

    /**
     * Log info
     * 
     * @param msg
     */
    public abstract void info(String msg);

    /**
     * Log error
     * 
     * @param msg
     */
    public abstract void error(String msg);

    /**
     * Log error
     * 
     * @param msg
     * @param e
     */
    public abstract void error(String msg, Throwable e);

    /**
     * Returns whether debug is enabled by the current logger
     * 
     * @return whether debug is enabled
     */
    public abstract boolean isDebugEnabled();

    /**
     * Updates artifacts of current project
     */
    public abstract List<String> getArtifacts();

    /**
     * Recompile the build file
     * 
     * @param buildFile
     * @param artifactPaths
     * @param executor      The thread pool executor
     * @throws PluginExecutionException if there was an error when restarting the
     *                                  server
     * @return true if the build file was recompiled with changes
     */
    public abstract boolean recompileBuildFile(File buildFile, List<String> artifactPaths, ThreadPoolExecutor executor)
            throws PluginExecutionException;

    /**
     * Run the unit tests
     * 
     * @throws PluginScenarioException  if unit tests failed
     * @throws PluginExecutionException if unit tests could not be run
     */
    public abstract void runUnitTests() throws PluginScenarioException, PluginExecutionException;

    /**
     * Run the integration tests
     * 
     * @throws PluginScenarioException  if integration tests failed
     * @throws PluginExecutionException if integration tests could not be run
     */
    public abstract void runIntegrationTests() throws PluginScenarioException, PluginExecutionException;

    /**
     * Check the configuration file for new features
     * 
     * @param configFile
     * @param serverDir
     */
    public abstract void checkConfigFile(File configFile, File serverDir);

    /**
     * Compile the specified directory
     * 
     * @param dir
     * @return
     */
    public abstract boolean compile(File dir);

    /**
     * Stop the server
     */
    public abstract void stopServer();

    /**
     * Get the ServerTask to start the server, which can be in either "run" or
     * "debug" mode
     * 
     * @return ServerTask the task to start the server
     * @throws Exception if there was an error copying/creating config files
     */
    public abstract ServerTask getServerTask() throws Exception;

    /**
     * Redeploy the application
     */
    public abstract void redeployApp() throws PluginExecutionException;

    /**
     * Get an example command using the server start timeout parameter. The example
     * command is unique to each plugin.
     *
     * @return String containing the example command
     */
    public abstract String getServerStartTimeoutExample();

    /**
     * Get the name of the current project running dev mode.
     *
     * @return String of the project name
     */
    public abstract String getProjectName();

    /**
     * Is the application deployed as a loose application.
     */
    public abstract boolean isLooseApplication();

    private enum FileTrackMode {
        NOT_SET, FILE_WATCHER, POLLING
    }

    private File serverDirectory;
    private File sourceDirectory;
    private File testSourceDirectory;
    private File configDirectory;
    private File projectDirectory;
    private List<File> resourceDirs;
    private boolean hotTests;
    private Path tempConfigPath;
    private boolean skipTests;
    private boolean skipUTs;
    private boolean skipITs;
    private String applicationId;
    private int appStartupTimeout;
    private int appUpdateTimeout;
    private Thread serverThread;
    private PluginExecutionException serverThreadException;

    /** If user stopped dev mode manually, this is true. If an external process caused dev mode to stop, this is false */
    private AtomicBoolean devStop;
    
    private String hostName;
    private String httpPort;
    private String httpsPort;
    private String containerHttpPort;
    private String containerHttpsPort;
    private final long compileWaitMillis;
    private AtomicBoolean inputUnavailable;
    private int alternativeDebugPort = -1;
    private boolean libertyDebug;
    private int libertyDebugPort;
    private AtomicBoolean detectedAppStarted;
    private long serverStartTimeout;
    private boolean useBuildRecompile;
    private Map<File, Properties> propertyFilesMap;
    final private Set<FileAlterationObserver> fileObservers;
    final private Set<FileAlterationObserver> newFileObservers;
    final private Set<FileAlterationObserver> cancelledFileObservers;
    private AtomicBoolean calledShutdownHook;
    private boolean gradle;
    private long pollingInterval;
    private FileTrackMode trackingMode;
    private final boolean container;
    private String imageName;
    private String containerName;
    private File dockerfile;
    private Path tempDockerfilePath = null;
    private String dockerRunOpts;
    private volatile Process dockerRunProcess;
    private File defaultDockerfile;
    private int dockerBuildTimeout;
    private boolean skipDefaultPorts;
    private boolean keepTempDockerfile;
    protected List<String> srcMount = new ArrayList<String>();
    protected List<String> destMount = new ArrayList<String>();
    private boolean firstStartup = true;
    private Set<Path> dockerfileDirectoriesToWatch = new HashSet<Path>();
    private Set<Path> dockerfileDirectoriesTracked = new HashSet<Path>();
    private Set<WatchKey> dockerfileDirectoriesWatchKeys = new HashSet<WatchKey>();
    private Set<FileAlterationObserver> dockerfileDirectoriesFileObservers = new HashSet<FileAlterationObserver>();
    private final JavaCompilerOptions compilerOptions;
    private final String mavenCacheLocation;
    private AtomicBoolean externalContainerShutdown;
    private AtomicBoolean shownFeaturesShWarning;
    protected AtomicBoolean hasFeaturesSh;
    protected AtomicBoolean serverFullyStarted;
    private final File buildDirectory;

    public DevUtil(File buildDirectory, File serverDirectory, File sourceDirectory, File testSourceDirectory, File configDirectory, File projectDirectory,
            List<File> resourceDirs, boolean hotTests, boolean skipTests, boolean skipUTs, boolean skipITs,
            String applicationId, long serverStartTimeout, int appStartupTimeout, int appUpdateTimeout,
            long compileWaitMillis, boolean libertyDebug, boolean useBuildRecompile, boolean gradle, boolean pollingTest,
            boolean container, File dockerfile, String dockerRunOpts, int dockerBuildTimeout, boolean skipDefaultPorts, 
            JavaCompilerOptions compilerOptions, boolean keepTempDockerfile, String mavenCacheLocation) {
        this.buildDirectory = buildDirectory;
        this.serverDirectory = serverDirectory;
        this.sourceDirectory = sourceDirectory;
        this.testSourceDirectory = testSourceDirectory;
        this.configDirectory = configDirectory;
        this.projectDirectory = projectDirectory;
        this.resourceDirs = resourceDirs;
        this.hotTests = hotTests;
        this.skipTests = skipTests;
        this.skipUTs = skipUTs;
        this.skipITs = skipITs;
        this.applicationId = applicationId;
        this.serverStartTimeout = serverStartTimeout;
        this.appStartupTimeout = appStartupTimeout;
        this.appUpdateTimeout = appUpdateTimeout;
        this.devStop = new AtomicBoolean(false);
        this.compileWaitMillis = compileWaitMillis;
        this.inputUnavailable = new AtomicBoolean(false);
        this.libertyDebug = libertyDebug;
        this.detectedAppStarted = new AtomicBoolean(false);
        this.useBuildRecompile = useBuildRecompile;
        this.calledShutdownHook = new AtomicBoolean(false);
        this.gradle = gradle;
        this.fileObservers = new HashSet<FileAlterationObserver>();
        this.newFileObservers = new HashSet<FileAlterationObserver>();
        this.cancelledFileObservers = new HashSet<FileAlterationObserver>();
        this.pollingInterval = 100;
        if (pollingTest) {
            this.trackingMode = FileTrackMode.POLLING;
        } else {
            this.trackingMode = FileTrackMode.NOT_SET;
        }
        this.container = container;
        this.dockerfile = dockerfile;
        this.dockerRunOpts = dockerRunOpts;
        if (projectDirectory != null) {
            this.defaultDockerfile = new File(projectDirectory, "Dockerfile");
        }
        if (dockerBuildTimeout < 1) {
            this.dockerBuildTimeout = 600;
        } else {
            this.dockerBuildTimeout = dockerBuildTimeout;
        }
        this.skipDefaultPorts = skipDefaultPorts;
        this.compilerOptions = compilerOptions;
        this.keepTempDockerfile = keepTempDockerfile;
        this.mavenCacheLocation = mavenCacheLocation;
        this.externalContainerShutdown = new AtomicBoolean(false);
        this.shownFeaturesShWarning = new AtomicBoolean(false);
        this.hasFeaturesSh = new AtomicBoolean(false);
        this.serverFullyStarted = new AtomicBoolean(false);
    }

    /**
     * Run unit and/or integration tests
     * 
     * @param waitForApplicationUpdate Whether to wait for the application to update
     *                                 before running integration tests
     * @param messageOccurrences       The previous number of times the application
     *                                 updated message has appeared.
     * @param executor                 The thread pool executor
     * @param forceSkipUTs             Whether to force skip the unit tests
     */
    public void runTests(boolean waitForApplicationUpdate, int messageOccurrences, ThreadPoolExecutor executor,
            boolean forceSkipUTs) {
        if (!skipTests) {
            ServerTask serverTask = null;
            try {
                serverTask = getServerTask();
            } catch (Exception e) {
                // not expected since server should already have been started
                error("Could not get the server task for running tests.", e);
            }

            File logFile = getMessagesLogFile(serverTask);

            String regexp = UPDATED_APP_MESSAGE_REGEXP + applicationId;

            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                debug("Thread interrupted while waiting to start tests.", e);
            }

            // if queue size >= 1, it means a newer test has been queued so we
            // should skip this and let that run instead
            if (executor.getQueue().size() >= 1) {
                Runnable head = executor.getQueue().peek();
                boolean manualInvocation = ((TestJob) head).isManualInvocation();

                if (manualInvocation) {
                    debug("Tests were re-invoked before previous tests began. Cancelling previous tests and resubmitting them.");
                } else {
                    debug("Changes were detected before tests began. Cancelling tests and resubmitting them.");
                }
                return;
            }

            // skip unit tests if invoked by Gradle
            if (!gradle && !(skipUTs || forceSkipUTs)) {
                info("Running unit tests...");
                try {
                    runUnitTests();
                    info("Unit tests finished.");
                } catch (PluginScenarioException e) {
                    debug(e);
                    error(e.getMessage());
                    // if unit tests failed, don't run integration tests
                    return;
                } catch (PluginExecutionException e) {
                    error(e.getMessage());
                }
            }

            // if queue size >= 1, it means a newer test has been queued so we
            // should skip this and let that run instead
            if (executor.getQueue().size() >= 1) {
                Runnable head = executor.getQueue().peek();
                boolean manualInvocation = ((TestJob) head).isManualInvocation();

                if (manualInvocation) {
                    info("Tests were invoked while previous tests were running. Restarting tests.");
                } else {
                    info("Changes were detected while tests were running. Restarting tests.");
                }
                return;
            }

            if (!skipITs) {
                if (!detectedAppStarted.get()) {
                    if (appStartupTimeout < 0) {
                        warn("The verifyTimeout (verifyAppStartTimeout) value needs to be an integer greater than or equal to 0.  The default value of 30 seconds will be used.");
                        appStartupTimeout = 30;
                    }
                    long timeout = appStartupTimeout * 1000;

                    // Wait for the app started message in messages.log
                    info("Waiting up to " + appStartupTimeout
                            + " seconds to find the application start up or update message...");
                    String startMessage = serverTask.waitForStringInLog(
                            "(" + START_APP_MESSAGE_REGEXP + "|" + UPDATED_APP_MESSAGE_REGEXP + applicationId + ")",
                            timeout, logFile);
                    if (startMessage == null) {
                        error("Unable to verify if the application was started after " + appStartupTimeout
                                + " seconds.  Consider increasing the verifyTimeout value if this continues to occur.");
                    } else {
                        detectedAppStarted.set(true);
                    }
                } else if (waitForApplicationUpdate) {
                    // wait until application has been updated
                    if (appUpdateTimeout < 0) {
                        appUpdateTimeout = 5;
                    }
                    long timeout = appUpdateTimeout * 1000;
                    serverTask.waitForUpdatedStringInLog(regexp, timeout, logFile, messageOccurrences);
                }

                if (gradle) {
                    info("Running tests...");
                } else {
                    info("Running integration tests...");
                }
                try {
                    runIntegrationTests();
                    if (gradle) {
                        info("Tests finished.");
                    } else {
                        info("Integration tests finished.");
                    }
                } catch (PluginScenarioException e) {
                    debug(e);
                    error(e.getMessage());
                    // if unit tests failed, don't run integration tests
                    return;
                } catch (PluginExecutionException e) {
                    error(e.getMessage());
                }
            }
        }
    }

    /**
     * Get the number of times the application updated message has appeared in the
     * application log
     * 
     * @return the number of times the application has updated
     */
    public int countApplicationUpdatedMessages() {
        int messageOccurrences = -1;
        if (!(skipTests || skipITs)) {
            try {
                ServerTask serverTask = getServerTask();
                File logFile = getMessagesLogFile(serverTask);
                String regexp = UPDATED_APP_MESSAGE_REGEXP + applicationId;
                messageOccurrences = serverTask.countStringOccurrencesInFile(regexp, logFile);
                debug("Message occurrences before compile: " + messageOccurrences);
            } catch (Exception e) {
                debug("Failed to get message occurrences before compile", e);
            }
        }
        return messageOccurrences;
    }

    /**
     * Get the log file from server directory if using container, or from server task otherwise.
     * 
     * @param serverTask the server task
     * @return the messages log file for the server
     */
    private File getMessagesLogFile(ServerTask serverTask) {
        File logFile;
        if (container) {
            logFile = new File(serverDirectory, "logs/messages.log");
        } else {
            logFile = serverTask.getLogFile();           
        }
        return logFile;
    }

    public void startServer() throws PluginExecutionException {
        startServer(true, true);
    }

    /**
     * Start the server and keep it running in a background thread.
     * 
     * @param buildContainer  Force a Docker build when in container mode. Ignored
     *                        otherwise.
     * @param pullParentImage If buildContainer is true, this determines whether the
     *                        Docker build should also pull the latest parent image.
     *                        Ignored otherwise.
     * 
     * @throws PluginExecutionException If the server startup could not be verified
     *                                  within the timeout, or server startup
     *                                  failed.
     */
    public void startServer(boolean buildContainer, boolean pullParentImage) throws PluginExecutionException {
        try {
            final ServerTask serverTask;
            try {
                serverTask = getServerTask();
            } catch (Exception e) {
                throw new PluginExecutionException("An error occurred while starting the server: " + e.getMessage(), e);
            }

            // Set debug variables in server.env if debug enabled
            enableServerDebug();

            if (container) {
                checkDockerVersion();
            }

            // build Docker image if in container mode
            if (container && buildContainer) {
                File dockerfileToUse = getDockerfile();
                debug("Dockerfile to use: " + dockerfileToUse);
                if (dockerfileToUse.exists()) {
                    File tempDockerfile = prepareTempDockerfile(dockerfileToUse);
                    buildDockerImage(tempDockerfile, dockerfileToUse, pullParentImage);
                } else {
                    // this message is mainly for the default dockerfile scenario, since the dockerfile parameter was already validated in Maven/Gradle plugin.
                    throw new PluginExecutionException("No Dockerfile was found at " + dockerfileToUse.getAbsolutePath() + ". Create a Dockerfile at the specified location to use dev mode with container support. For an example of how to configure a Dockerfile, see https://github.com/OpenLiberty/ci.docker");
                }
            }

            String logsDirectory = serverDirectory.getCanonicalPath() + "/logs";
            final File messagesLogFile = new File(logsDirectory + "/messages.log");

            // Watch logs directory if it already exists
            boolean logsExist = new File(logsDirectory).isDirectory();

            // Start server
            serverThread = new Thread(new Runnable() {

                @Override
                public void run() {
                    try {
                        if (container) {
                            startContainer();
                        } else {
                            serverTask.execute();
                        }
                    } catch (RuntimeException e) {
                        // If devStop is true server was stopped with Ctl-c, do not throw exception
                        if (devStop.get() == false) {
                            // If a runtime exception occurred in the server task, log and set the exception field
                            PluginExecutionException e2;
                            if (container) {
                                e2 = new PluginExecutionException("An error occurred while running the container: " + e.getMessage(), e);
                            } else {
                                e2 = new PluginExecutionException("An error occurred while starting the server: " + e.getMessage(), e);
                            }
                            error(e2.getMessage());
                            serverThreadException = e2;
                        }
                    }
                }
            });
            serverThread.start();

            // If the server thread dies at any point after this, allow the error to say
            // that the server stopped
            setDevStop(false);

            // If there were already logs from a previous server run, wait for it to be updated.
            if (logsExist) {
                final AtomicBoolean messagesModified = new AtomicBoolean(false);

                // If logs already exist, then watch the directory to ensure
                // messages.log is modified before continuing.
                FileFilter singleFileFilter = new FileFilter() {
                    @Override
                    public boolean accept(File file) {
                        try {
                            if (file.getCanonicalFile().equals(messagesLogFile.getCanonicalFile())) {
                                return true;
                            }
                        } catch (IOException e) {
                            if (file.equals(messagesLogFile)) {
                                return true;
                            }
                        }
                        return false;
                    }
                };

                FileAlterationObserver observer = new FileAlterationObserver(logsDirectory, singleFileFilter);
                observer.addListener(new FileAlterationListenerAdaptor() {
                    @Override
                    public void onFileCreate(File file) {
                        messagesModified.set(true);
                    }

                    @Override
                    public void onFileChange(File file) {
                        messagesModified.set(true);
                    }
                });

                try {
                    observer.initialize();
                    while (!messagesModified.get()) {
                        checkStopDevMode(false); // stop dev mode if the server thread was terminated
                        observer.checkAndNotify();
                        // wait for the log file to update during server startup
                        Thread.sleep(500);
                    }
                    debug("messages.log has been changed");
                } catch (PluginScenarioException e) {
                    if (serverThreadException != null) {
                        throw serverThreadException;
                    } else {
                        // the server/container failed to start, so wrap this as an execution exception
                        throw new PluginExecutionException(e);
                    }
                } catch (Exception e) {
                    error("An error occured while waiting for the server to update messages.log: " + e.getMessage(), e);
                } finally {
                    try {
                        observer.destroy();
                    } catch (Exception e) {
                        debug("Could not destroy FileAlterationObserver for logs directory " + logsDirectory, e);
                    }
                }
            } else {
                // Wait until log exists
                try {
                    while (!messagesLogFile.exists()) {
                        checkStopDevMode(false); // stop dev mode if the server thread was terminated
                        // wait for the log file to appear during server startup
                        Thread.sleep(500);
                    }
                    debug("messages.log has been created");
                } catch (PluginScenarioException e) {
                    if (serverThreadException != null) {
                        throw serverThreadException;
                    } else {
                        // the server/container failed to start, so wrap this as an execution exception
                        throw new PluginExecutionException(e);
                    }
                } catch (Exception e) {
                    error("An error occured while waiting for the server to create messages.log: " + e.getMessage(), e);
                }
            }
            // Set server start timeout
            if (serverStartTimeout < 0) {
                warn("The serverStartTimeout value needs to be an integer greater than or equal to 0.  The default value of 90 seconds will be used.");
                serverStartTimeout = 90;
            }
            long serverStartTimeoutMillis = serverStartTimeout * 1000;
            // Wait for the server started message in messages.log
            String startMessage = serverTask.waitForStringInLog(START_SERVER_MESSAGE_PREFIX, serverStartTimeoutMillis,
                    messagesLogFile);
            if (startMessage == null) {
                setDevStop(true);
                if (container) {
                    stopContainer();
                } else {
                    stopServer();
                }
                throw new PluginExecutionException("The server has not started within " + serverStartTimeout + " seconds. " +
                        "Consider increasing the server start timeout if this continues to occur. " +
                        "For example, " + getServerStartTimeoutExample());
            } else {
                serverFullyStarted.set(true);
            }
            // Check for port already in use error
            String portError = serverTask.findStringInFile(PORT_IN_USE_MESSAGE_PREFIX, messagesLogFile);
            if (portError != null) {
                error(portError.split(PORT_IN_USE_MESSAGE_PREFIX)[1]);
            }
            // Parse hostname, http, https ports for integration tests to use
            parseHostNameAndPorts(serverTask, messagesLogFile);
        } catch (IOException e) {
            throw new PluginExecutionException("An error occurred while starting the server: " + e.getMessage(), e);
        }
    }

    /**
     * Retrieve the current docker version and compare to a known value.
     * The Maven class ComparableVersion allows for numbers, letters and certain words.
     * Throw an exception if there is a problem with the version.
     */
    private static final String MIN_DOCKER_VERSION = "18.03.0"; // Must use Docker 18.03.0 or higher
    private void checkDockerVersion() throws PluginExecutionException {
        String versionCmd = "docker version --format {{.Client.Version}}";
        String dockerVersion = execDockerCmd(versionCmd, DOCKER_TIMEOUT);
        if (dockerVersion == null) {
            return; // can't tell if the version is valid.
        }
        debug("Detected Docker version >" + dockerVersion);
        ComparableVersion minVer = new ComparableVersion(MIN_DOCKER_VERSION);
        ComparableVersion curVer = new ComparableVersion(dockerVersion);
        if (curVer.compareTo(minVer) < 0) {
            throw new PluginExecutionException("The detected Docker client version number is not supported:" + dockerVersion.trim() + ". Docker version must be 18.03.0 or higher.");
        }
    }

    private File getDockerfile() {
        return dockerfile != null ? dockerfile : defaultDockerfile;
    }

    protected List<String> readDockerfile(File dockerfile) throws PluginExecutionException {
        // Convert Dockerfile to List of strings for each line
        List<String> dockerfileLines = null;
        try {
            dockerfileLines = Files.readAllLines(dockerfile.toPath());
        } catch (IOException e) {
            error("Failed to read Dockerfile located at " + dockerfile);
            throw new PluginExecutionException("Could not read Dockerfile " + dockerfile + ": " + e.getMessage(), e);
        }
        return dockerfileLines;
    }

    /**
     * Get escape character from the escape directive at the top of the Dockerfile.
     * Docker documents a couple of directives, but it seems escape must always be the first line to work.
     */
    protected static char getEscapeCharacter(List<String> dockerfileLines) throws PluginExecutionException {
        if (dockerfileLines.size() > 0) {
            // Remove white space from the beginning and end of the line
            String pendingLine = dockerfileLines.get(0).trim();
            int directiveSymbolIndex = pendingLine.indexOf("#");
            if (directiveSymbolIndex >= 0) {
                String contentAfterSymbol = pendingLine.substring(directiveSymbolIndex + 1, pendingLine.length());
                // trim again after removing preceding symbol
                String directive = contentAfterSymbol.trim();
                String[] split = directive.split("=");
                if (split.length == 2 && split[0].trim().equalsIgnoreCase("escape")) {
                    String escapeChar = split[1].trim();
                    if (escapeChar.length() > 0) {
                        // Get the first char, and don't validate here whether it's a valid escape char
                        // but just let Docker fail the build if it determines that it is invalid.
                        return escapeChar.charAt(0);
                    }
                }
            }
        }
        return '\\';
    }

    /**
     * Trim all lines and get them without comments or empty lines after the first FROM command
     * (so that directives at the beginning of the file are preserved)
     */
    protected static List<String> getCleanedLines(List<String> dockerfileLines) throws PluginExecutionException {
        List<String> result = new ArrayList<String>();
        boolean fromFound = false;
        for (String line : dockerfileLines) {
            // Remove white space from the beginning and end of the line
            String pendingLine = line.trim();
            if (!fromFound) {
                if (pendingLine.startsWith("FROM")) {
                    fromFound = true;
                } else {
                    // until we find a FROM line, just keep all the lines
                    result.add(pendingLine);
                    continue;
                }
            }
            int commentIndex = pendingLine.indexOf("#");
            if (commentIndex >= 0) {
                String contentBeforeSymbol = pendingLine.substring(0, commentIndex);
                // trim again after removing trailing comment
                pendingLine = contentBeforeSymbol.trim();
            }
            if (!pendingLine.isEmpty()) {
                result.add(pendingLine);
            }
        }
        return result;
    }

    /**
     * Combine multi-line commands into single lines. Requires that getCleanedLines() be called first.
     */
    protected static List<String> getCombinedLines(List<String> dockerfileLines, char escape) throws PluginExecutionException {
        List<String> result = new ArrayList<String>();
        int i = 0;
        while (i < dockerfileLines.size()) {
            String pendingLine = dockerfileLines.get(i).trim();
            int multilineIndex;
            int j = i+1;
            while (pendingLine.length() > 0 && !pendingLine.startsWith("#") && (pendingLine.charAt(pendingLine.length() - 1) == escape) && j < dockerfileLines.size()) {
                multilineIndex = pendingLine.length() - 1;
                String contentBeforeSymbol = pendingLine.substring(0, multilineIndex);
                String nextLine = dockerfileLines.get(j);
                String combined = contentBeforeSymbol + nextLine;
                pendingLine = combined.trim(); // trim the combined string to remove whitespace around any further line escapes
                j++;
            }
            result.add(pendingLine);
            i = j;
        }
        return result;
    }

    protected void removeWarFileLines(List<String> dockerfileLines) throws PluginExecutionException {
        List<String> warFileLines = new ArrayList<String>();
        for (String line : dockerfileLines) {
            // Remove white space from the beginning and end of the line
            String trimLine = line.trim();
            if (!trimLine.startsWith("#") && trimLine.toLowerCase().contains(".war")) {
                // Break the Dockerfile line down into segments based on any amount of whitespace.
                // The command must be to the left of any comments.
                String[] cmdSegments = trimLine.split("#")[0].split("\\s+");
                // if the line starts with COPY and the second to last segment ends with ".war", it is a WAR file COPY line
                if (cmdSegments[0].equalsIgnoreCase("COPY") || cmdSegments[0].equalsIgnoreCase("ADD")) {
                    if (cmdSegments.length < 3) {
                        throw new PluginExecutionException("Incorrect syntax on this line in the Dockerfile: '" + line + 
                        "'. There must be at least two arguments for the COPY or ADD command, a source path and a destination path.");
                    }
                    if (cmdSegments[cmdSegments.length - 2].toLowerCase().endsWith(".war")) {
                        warFileLines.add(line);
                    }
                }
            }
        }
        debug("WAR file lines: " + warFileLines.toString());
        dockerfileLines.removeAll(warFileLines);
    }

    /**
     * Disables OpenJ9 SCC to speed up build times by injecting environment variable before "RUN configure.sh" is called.
     * Reference: https://github.com/OpenLiberty/ci.docker#openj9-shared-class-cache-scc
     * <br>
     * Note: lines must have been trimmed and cleaned of comments using getCleanedLines() before calling this.
     */
    protected void disableOpenJ9SCC(List<String> dockerfileLines) {
        final String RUN_CONFIGURE_COMMAND_LOWERCASE = "run configure.sh";
        for (int i=0; i<dockerfileLines.size(); i++) {
            String line = dockerfileLines.get(i);
            // RUN command is case insensitive, so use lowercase matching.
            if (line.toLowerCase().equals(RUN_CONFIGURE_COMMAND_LOWERCASE)) {
                debug("Detected RUN configure.sh command.  Skipping OpenJ9 Shared Class Cache.");
                dockerfileLines.add(i, "ENV OPENJ9_SCC=false");
                return;
            }
        }
    }

    protected void detectFeaturesSh(List<String> dockerfileLines) {
        // Reset features.sh warning flag
        shownFeaturesShWarning.set(false);

        final String FEATURES_SH_COMMAND_LOWERCASE = "run features.sh";
        for (int i=0; i<dockerfileLines.size(); i++) {
            String line = dockerfileLines.get(i);
            // RUN command is case insensitive, so use lowercase matching.
            if (line.toLowerCase().equals(FEATURES_SH_COMMAND_LOWERCASE)) {
                debug("Detected RUN features.sh command.");
                hasFeaturesSh.set(true);
                return;
            }
        }
        // if not detected, reset to false in case the Dockerfile is being rebuilt
        debug("Did not find RUN features.sh command.");
        hasFeaturesSh.set(false);
    }

    protected void processCopyLines(List<String> dockerfileLines, String buildContext) throws PluginExecutionException {
        srcMount.clear();
        destMount.clear();
        for (String line : dockerfileLines) {
            // Remove white space from the beginning and end of the line
            String trimLine = line.trim();
            if (!trimLine.startsWith("#")) {
                // Break the Dockerfile line down into segments based on any amount of whitespace.
                // The command must be to the left of any comments.
                String[] cmdSegments = trimLine.split("#")[0].split("\\s+");
                // If the line starts with COPY or ADD
                if (cmdSegments[0].equalsIgnoreCase("COPY") || cmdSegments[0].equalsIgnoreCase("ADD")) {
                    if (cmdSegments.length < 3) { // preliminary check but some of these segments could be options
                        throw new PluginExecutionException("Incorrect syntax on this line in the Dockerfile: '" + line + 
                        "'. There must be at least two arguments for the COPY or ADD command, a source path and a destination path.");
                    }
                    if (line.contains("$")) {
                        warn("The Dockerfile line '" + line + "' will not be able to be hot deployed to the dev mode container. Dev mode does not currently support environment variables in COPY or ADD commands. If you make changes to files specified by this line, type 'r' and press Enter to rebuild the Docker image and restart the container.");
                        continue;
                    }
                    List<String> srcOrDestArguments = new ArrayList<String>();
                    boolean skipLine = false;
                    for (int i = 1; i < cmdSegments.length; i++) { // start after the word COPY (or ADD)
                        String segment = cmdSegments[i];
                        if (segment.startsWith("--from")) {
                            // multi-stage build, COPY only (not ADD), give a warning
                            warn("The Dockerfile line '" + line + "' will not be able to be hot deployed to the dev mode container. Dev mode does not currently support hot deployment with multi-stage COPY commands.");
                            skipLine = true; // don't mount the dirs in this COPY command
                            break;
                        } else if (segment.startsWith("--")) {
                            continue; // ignore options
                        } else {
                            srcOrDestArguments.add(segment);
                        }
                    }
                    if (skipLine) {
                        continue;
                    }
                    if (srcOrDestArguments.size() < 2) { // proper check for number of src and dest args
                        throw new PluginExecutionException("Incorrect syntax on this line in the Dockerfile: '" + line + 
                        "'. There must be at least two arguments for the COPY or ADD command, a source path and a destination path.");
                    }
                    // dest is the last argument
                    String dest = srcOrDestArguments.get(srcOrDestArguments.size() - 1);
                    List<String> srcArguments = srcOrDestArguments.subList(0, srcOrDestArguments.size() - 1);
                    for (String src : srcArguments) {
                        if (isURL(src)) {
                            debug("COPY/ADD do not watch/mount URL:" + src);
                            continue;
                        }
                        String sourcePath = buildContext + "/" + src;
                        File sourceFile = new File(sourcePath);
                        if (src.contains("*") || src.contains("?")) {
                            warn("The COPY or ADD source " + src + " in the Dockerfile line '" + line + "' will not be able to be hot deployed to the dev mode container. Dev mode does not currently support wildcards in the COPY or ADD commands. If you make changes to files specified by this line, type 'r' and press Enter to rebuild the Docker image and restart the container.");
                        } else if (sourceFile.isDirectory() || cmdSegments[0].equalsIgnoreCase("ADD")) {
                            synchronized(dockerfileDirectoriesToWatch) {
                                try {
                                    dockerfileDirectoriesToWatch.add(sourceFile.getCanonicalFile().toPath());
                                    debug("COPY/ADD line=" + line + ", src=" + sourcePath + ", added to dockerfileDirectoriesToWatch: " + sourceFile);
                                } catch (IOException e) {
                                    // Do not fail here.  Let the Docker build fail instead.
                                    error("Could not resolve the canonical path of the directory specified in the Dockerfile: " + sourcePath, e);
                                }
                            }
                        } else {
                            // No need to validate existence of the file, just let the Docker build fail
                            String destMountString = formatDestMount(dest, sourceFile);
                            srcMount.add(sourcePath);
                            destMount.add(destMountString);
                            debug("COPY line=" + line + ", src=" + sourcePath + ", dest=" + destMountString);
                        }
                    }
                }
            }
        }
    }

    private String formatDestMount(String destMountString, File srcMountFile) {
        // Cannot mount a file onto a directory, so must add a filename to the end of the destination argument for mounting
        if (destMountString.endsWith("/") || destMountString.endsWith("\\")) {
            destMountString += srcMountFile.getName();
        }
        return destMountString;
    }

    /**
     * Check the name used in Dockerfile for URL e.g. ADD https://repo.maven.apache.org/maven2/postgres9.jar /lib/
     * @param name the source name in a copy or add command
     * @return true if the name is a URL
     */
    private boolean isURL(String name) {
        try {
            URL url = new URL(name);
        } catch (MalformedURLException m) {
            return false;
        }
        return true;
    }

    protected File prepareTempDockerfile(File dockerfile) throws PluginExecutionException {
        // Create a temp Dockerfile to build image from

        List<String> dockerfileLines = readDockerfile(dockerfile);
        char escape = getEscapeCharacter(dockerfileLines);
        dockerfileLines = getCleanedLines(dockerfileLines);
        dockerfileLines = getCombinedLines(dockerfileLines, escape);
        removeWarFileLines(dockerfileLines);
        processCopyLines(dockerfileLines, dockerfile.getParent());
        detectFeaturesSh(dockerfileLines);
        disableOpenJ9SCC(dockerfileLines);
        for (String line : dockerfileLines) {
            debug(line);
        }

        File tempDockerfile = null;
        try {
            debug("Creating temp Dockerfile...");
            tempDockerfile = File.createTempFile("tempDockerfile", "");
            debug("temp Dockerfile: " + tempDockerfile);
            tempDockerfilePath = tempDockerfile.toPath(); // save name to clean up later
            if (keepTempDockerfile) {
                info("Keeping temporary Dockerfile: "+tempDockerfilePath);
            } else {
                // set the tempDockerfile to be deleted when the JVM exits
                tempDockerfile.deleteOnExit();
            }
            Files.write(tempDockerfile.toPath(), dockerfileLines, StandardCharsets.UTF_8);
        } catch (IOException e) {
            error("Failed to create temp Dockerfile");
            throw new PluginExecutionException("Could not create temp Dockerfile: " + e.getMessage(), e);
        }
        return tempDockerfile;
    }

    private void buildDockerImage(File tempDockerfile, File userDockerfile, boolean pullParentImage) throws PluginExecutionException {
        try {
            info("Building Docker image...");
            imageName = getProjectName() + DEVMODE_IMAGE_SUFFIX;
            // Name rules: may contain lowercase letters, digits and a period, one or two underscores, or one or more dashes. Cannot start with dash.
            imageName = imageName.replaceAll("[^a-zA-Z0-9]", "-").replaceAll("^[\\-]+", "").toLowerCase();
            // The image is built using the tempDockerfile, but the build context comes from the user's Dockerfile location
            debug("Docker build context: " + userDockerfile.getParent());
            StringBuilder sb = new StringBuilder();
            sb.append("docker build ");
            if (pullParentImage) {
                sb.append("--pull ");
            }
            sb.append("-f " + tempDockerfile + " -t " + imageName + " " + userDockerfile.getParent());
            String buildCmd = sb.toString();
            info(buildCmd);
            if (hasFeaturesSh.get()) {
                info("The RUN features.sh command is detected in the Dockerfile and extra time may be necessary when installing features.");
            }
            long startTime = System.currentTimeMillis();
            execDockerCmdAndLog(getRunProcess(buildCmd), dockerBuildTimeout);
            checkDockerBuildTime(startTime, userDockerfile.getParentFile());
            info("Completed building Docker image.");
        } catch (IllegalThreadStateException  e) {
            // the timeout was too short and the docker command has not yet completed.
            debug("IllegalThreadStateException, message="+e.getMessage());
            throw new PluginExecutionException("The docker build command did not complete within the timeout period: " + dockerBuildTimeout + " seconds. " +
                "Use the dockerBuildTimeout option to specify a longer period or " +
                "add files not needed in the container to the .dockerignore file", e);
        } catch (IOException e) {
            error("Input or output error building Docker image: " + e.getMessage());
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            debug("Thread InterruptedException while building the Docker image: " + e.getMessage());
            throw new PluginExecutionException("Could not build Docker image using Dockerfile: " +
                userDockerfile.getAbsolutePath() + ". Address the following docker build error and then start dev mode again: " + e.getMessage(), e);
        } catch (RuntimeException r) {
            debug("RuntimeException building Docker image: " + r.getMessage());
            throw new PluginExecutionException("Could not build Docker image using Dockerfile: " + 
                userDockerfile.getAbsolutePath() + ". Address the following docker build error and then start dev mode again: " + r.getMessage(), r);
        }
    }

    // Suggest a performance improvement if docker build takes too long.
    private static final long DOCKER_BUILD_SOFT_TIMEOUT = 30000; // millis
    private void checkDockerBuildTime(long startTime, File dockerBuildContext) {
        if (System.currentTimeMillis() - startTime < DOCKER_BUILD_SOFT_TIMEOUT) {
            return;
        }
        debug("checkDockerBuildTime, dockerBuildContext=" + dockerBuildContext.getAbsolutePath());
        File dockerIgnore = new File(dockerBuildContext, ".dockerignore");
        if (!dockerIgnore.exists()) { // provide some advice
            String buildContextPath;
            try {
                buildContextPath = dockerBuildContext.getCanonicalPath();
            } catch (IOException e) {
                buildContextPath = dockerBuildContext.getAbsolutePath();
            }
            warn("The docker build command took longer than " + DOCKER_BUILD_SOFT_TIMEOUT / 1000 + " seconds. " +
                "You may increase performance by adding unneeded files and directories " +
                "such as any Liberty runtime directories to a .dockerignore file in " +
                buildContextPath + ".");
        }
    }

    private void startContainer() {
        try {
            if (OSUtil.isLinux()) {
                // Allow the server to write to the log files. If we don't create it here docker daemon will create it as root.
                runCmd("mkdir -p " + serverDirectory + "/logs");
            }

            info("Starting Docker container...");
            String startContainerCommand = getContainerCommand();
            info(startContainerCommand);
            dockerRunProcess = getRunProcess(startContainerCommand);
            execDockerCmdAndLog(dockerRunProcess, 0);
        } catch (IOException e) {
            error("Error starting container: " + e.getMessage());
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            error("Thread was interrupted while starting the container: " + e.getMessage());
        } catch (RuntimeException r) {
            try {
                // remove container in case of an error trying to run the container because the docker run --rm will not rm the container
                String dockerRmCmd = "docker container rm " + containerName;
                execDockerCmd(dockerRmCmd, DOCKER_TIMEOUT);
            } catch (Exception e) {
                // do not report the "docker container rm" error so that we can instead report the startContainer() error
                debug("Exception running docker container rm:", e);
            }
            throw r;
        }
    }

    private Process getRunProcess(String command) throws IOException {
        ProcessBuilder processBuilder = new ProcessBuilder();
        processBuilder.command(getCommandTokens(command));
        if (!OSUtil.isLinux()){
            Map<String, String> env = processBuilder.environment();
            if (!env.keySet().contains("DOCKER_BUILDKIT")) { // don't touch if already set
                env.put("DOCKER_BUILDKIT", "0"); // must set 0 on Windows VMs
                debug("Generating environment for docker build & run: DOCKER_BUILDKIT=0");
            }
        }
        return processBuilder.start();
    }

    private void execDockerCmdAndLog(final Process startingProcess, int timeout) throws InterruptedException {
        Thread logCopyInputThread = new Thread(new Runnable() {
            @Override
            public void run() {
                copyStreamToBuildLog(startingProcess.getInputStream(), true);
            }
        });
        logCopyInputThread.start();

        final StringBuilder firstErrorLine = new StringBuilder();
        Thread logCopyErrorThread = new Thread(new Runnable() {
            @Override
            public void run() {
                firstErrorLine.append(copyStreamToBuildLog(startingProcess.getErrorStream(), false));
            }
        });
        logCopyErrorThread.start();

        if (timeout == 0) {
            startingProcess.waitFor();
        } else {
            startingProcess.waitFor(timeout, TimeUnit.SECONDS);
        }
        if (startingProcess.exitValue() != 0 && !devStop.get()) { // if there was an error and the user didn't choose to stop dev mode
            // return code 143 corresponds to 'docker stop xxx' so assume user requested to stop
            if (startingProcess.exitValue() == 143) {
                setDevStop(true); // indicate intentional shutdown
                externalContainerShutdown.set(true); // container shut down by external command
            }
            debug("Unexpected exit running docker command, return value=" + startingProcess.exitValue());
            // show first message from standard err
            String errorMessage = new String(firstErrorLine).trim() + " RC=" + startingProcess.exitValue();
            throw new RuntimeException(errorMessage);
        }
    }

    private String runCmd(String cmd) throws IOException, InterruptedException {
        String result = null;
        Process p = Runtime.getRuntime().exec(cmd);
        p.waitFor(5, TimeUnit.SECONDS);
        if (p.exitValue() != 0) {
            error("Error running command:" + cmd + ", return value=" + p.exitValue());
        } else {
            result = readStdOut(p);
        }
        return result;
    }

    /**
     * Copies the process output to the Maven/Gradle logs
     * 
     * @param stream The stream to copy
     * @param info If true, log as info. Else log as error.
     * @throws RuntimeException if there was an error reading the process output
     * @return The first line from the stream
     */
    private String copyStreamToBuildLog(InputStream stream, boolean info) {
        String firstLine = null;
        BufferedReader inputReader = new BufferedReader(new InputStreamReader(stream));
        try {
            for (String line; (line = inputReader.readLine()) != null;) {
                if (firstLine == null) {
                    firstLine = line;
                }
                if (info) {
                    info(line);
                } else {
                    error(line);

                    // Look for JVM version error in the line
                    alertOnServerError(line, "JVMCFRE003",
                            "Java classes were compiled with a higher version of Java than the JVM in the container. To resolve this issue, set the source and target Java versions in your Gradle build to correspond to the Java version used in your Dockerfile or its parent image, then restart dev mode.",
                            // Maven project should be cleaned before restarting dev mode, otherwise compile does not realize Java version settings have changed
                            "Java classes were compiled with a higher version of Java than the JVM in the container. To resolve this issue, set the source and target Java versions in your Maven build to correspond to the Java version used in your Dockerfile or its parent image, then clean the project output and restart dev mode.",
                            false);

                    // Look for features not available message during server startup if features.sh was not defined in Dockerfile
                    if (!serverFullyStarted.get() && !hasFeaturesSh.get() && !shownFeaturesShWarning.get()) {
                        String errMsg = "Feature definitions were not found in the container. To install features to the container, specify 'RUN features.sh' in your Dockerfile. For an example of how to configure a Dockerfile, see https://github.com/OpenLiberty/ci.docker";
                        shownFeaturesShWarning.set(alertOnServerError(line, "CWWKF0001E", errMsg, errMsg, true));
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException("Error reading container output: " + e.getMessage());
        } finally {
            try {
                inputReader.close();
            } catch (IOException e) {
                // nothing to do
            }
        }
        return firstLine;
    }

    private boolean alertOnServerError(String line, String errorCode, String gradleMessage, String mavenMessage, boolean warning) {
        if (container && line.contains(errorCode)) {
            if (gradle) {
                // Gradle doesn't show errors in an obvious way, so use some formatting to make it stand out more
                if (warning) {
                    warn("***** [ WARNING ] ***** " + gradleMessage);
                } else {
                    error("***** [ ERROR ] ***** " + gradleMessage);
                }
            } else {
                if (warning) {
                    warn(mavenMessage);
                } else {
                    error(mavenMessage);
                }
            }
            return true;
        }
        return false;
    }

    private String[] getCommandTokens(String command) {
        StringTokenizer stringTokenizer = new StringTokenizer(command);
        String[] commandTokens = new String[stringTokenizer.countTokens()];
        for (int i = 0; stringTokenizer.hasMoreTokens(); i++) {
            commandTokens[i] = stringTokenizer.nextToken();
        }
        return commandTokens;
    }

    private void stopContainer() {
        try {
            serverFullyStarted.set(false);

            // see if docker run command (container) is still running before trying to stop it.
            if (dockerRunProcess != null && dockerRunProcess.isAlive()) {
                info("Stopping container...");
                String dockerStopCmd = "docker stop " + containerName;
                debug("Stopping container " + containerName);
                execDockerCmd(dockerStopCmd, DOCKER_TIMEOUT + 20); // allow extra time for server shutdown
            }
        } catch (RuntimeException r) {
            error("Error stopping container: " + r.getMessage());
            throw r;
        } finally {
            dockerRunProcess = null;
        }
    }

    private String execDockerCmd(String command, int timeout) {
        return execDockerCmd(command, timeout, true);
    }

    /**
     * Build a docker run command with all the ports and directories required to run Open Liberty 
     * inside a container. Also included is the image name and the server run command to override
     * the CMD attribute of the Open Liberty docker image. 
     * @return the command string to use to start the container
     */
    private String getContainerCommand() {
        StringBuilder command = new StringBuilder("docker run --rm");
        if (!skipDefaultPorts) {
            int httpPortToUse, httpsPortToUse;
            try {
                httpPortToUse = findAvailablePort(LIBERTY_DEFAULT_HTTP_PORT, false);
                httpsPortToUse = findAvailablePort(LIBERTY_DEFAULT_HTTPS_PORT, false);
            } catch (IOException x) {
                error("An error occurred while trying to find an available network port. Using default port numbers.", x);
                httpPortToUse = LIBERTY_DEFAULT_HTTP_PORT;
                httpsPortToUse = LIBERTY_DEFAULT_HTTPS_PORT;
            }
            command.append(" -p ").append(httpPortToUse).append(":").append(LIBERTY_DEFAULT_HTTP_PORT);
            command.append(" -p ").append(httpsPortToUse).append(":").append(LIBERTY_DEFAULT_HTTPS_PORT);
        }
        
        if (libertyDebug) {
            // map debug port
            int containerDebugPort, hostDebugPort;
            try {
                if (alternativeDebugPort == -1) {
                    // it is possible another JVM has grabbed our port since dev mode last checked
                    hostDebugPort = findAvailablePort(libertyDebugPort, true);
                    containerDebugPort = libertyDebugPort;
                } else {
                    // dev mode has already selected an ephemeral port
                    containerDebugPort = hostDebugPort = alternativeDebugPort;
                }
            } catch (IOException x) {
                containerDebugPort = hostDebugPort = libertyDebugPort;
            }
            command.append(" -p " + hostDebugPort + ":" + containerDebugPort);
            // set environment variables in the container to ensure debug mode does not suspend the server, and to enable a custom debug port to be used
            command.append(" -e WLP_DEBUG_SUSPEND=n -e WLP_DEBUG_ADDRESS=" + containerDebugPort);
        }

        // mount potential directories containing .war.xml from devc specific folder - override /config/apps and /config/dropins
        command.append(" -v " + buildDirectory + "/" + DEVC_HIDDEN_FOLDER + "/apps:/config/apps");
        command.append(" -v " + buildDirectory + "/" + DEVC_HIDDEN_FOLDER + "/dropins:/config/dropins");

        // mount the loose application resources in the container
        command.append(" -v " + projectDirectory.getAbsolutePath() + ":" + DEVMODE_DIR_NAME);

        // mount the server logs directory over the /logs used by the open liberty container as defined by the LOG_DIR env. var.
        command.append(" -v " + serverDirectory.getAbsolutePath() + "/logs:/logs");

        // mount the Maven .m2 cache directory for featureUtility to use. For now, featureUtility does not support Gradle cache.
        command.append(" -v " + mavenCacheLocation + ":/devmode-maven-cache");

        // mount all files from COPY commands in the Dockerfile to allow for hot deployment
        command.append(getCopiedFiles());

        // Add a --user option when running Linux
        command.append(getUserId());

        // Do not generate a name if the user has specified a name
        String name = getDockerOption("--name");
        if (name == null || name.isEmpty()) {
            if (name != null && name.isEmpty()) {
                error("The Docker option --name is specified with an unsupported value: empty string.");
                // now generate a name so that the Docker errors make some sense to the user.
            }
            containerName = generateNewContainerName();
            command.append(" --name " +  containerName);
        } else {
            containerName = name;
        }
        debug("containerName: " + containerName + ".");

        // Allow the user to add their own options to this command via a system property.
        if (dockerRunOpts != null) {
            command.append(" "+dockerRunOpts);
        }

        // Options must preceed this in any order. Image name and command code follows.
        command.append(" " + imageName);
        // Command to start the server
        command.append(" /opt/ol/wlp/bin/server" + ((libertyDebug) ? " debug " : " run ")  + "defaultServer");
        // All the Liberty variable definitions must appear after the -- option.
        // Important: other Liberty options must appear before --
        command.append(" -- --"+DEVMODE_PROJECT_ROOT+"="+DEVMODE_DIR_NAME);

        return command.toString();
    }

    /**
     * Obtain a given Docker run option from the dockerRunOpts parameter
     * @param optionName the name of the option to extract from the dockerRunOpts
     * @return a string representation of the value of the option or null
     *
     * The option of interest must not use a quoted string.
     */
    private String getDockerOption(String optionName) {
        if (dockerRunOpts == null || dockerRunOpts.isEmpty()) {
            return null;
        }
        String[] options = dockerRunOpts.split("\\s+"); // split on whitespace
        for (int i = 0; i < options.length; i++) {
            if (options[i].equals(optionName)) { // --name ABC format
                return (i < options.length - 1) ? options[i+1] : null;
            } else if (options[i].startsWith(optionName + "=")) { // --name=ABC format
                return options[i].substring(optionName.length()+1); // could be empty string
            }
        }
        return null;
    }

    private String generateNewContainerName() {
        String dockerContNamesCmd = "docker ps -a --format \"{{.Names}}\"";
        debug("docker container names list command: " + dockerContNamesCmd);
        String result = execDockerCmd(dockerContNamesCmd, DOCKER_TIMEOUT);
        if (result == null) {
            return DEVMODE_CONTAINER_BASE_NAME;
        }
        String[] containerNames = result.split(" ");
        int highestNum = -1;
        for(int i = 0; i < containerNames.length; i++) {
            String name = removeSurroundingQuotes(containerNames[i]);
            int num = -1;
            if (name.equals(DEVMODE_CONTAINER_BASE_NAME)) {
                num = 0;
            } else if (name.startsWith(DEVMODE_CONTAINER_BASE_NAME + "-")) {
                String[] nameSegments = name.split("-");
                // if DEVMODE_CONTAINER_BASE_NAME changes, the logic below may need to change
                if (nameSegments.length == 3) {
                    String lastSegment = nameSegments[nameSegments.length - 1];
                    try {
                        num = Integer.parseInt(lastSegment);
                    } catch (NumberFormatException e) {
                        debug("Last segment of container name is not a number.");
                    }
                }
            }
            if (num > highestNum) {
                highestNum = num;
            }
        }
        
        return DEVMODE_CONTAINER_BASE_NAME + ((highestNum != -1) ? "-" + ++highestNum : "");
    }

    /**
     * Retrieves all the networks a container is connected to
     * @param contName name of the container to check for networks
     * @return a String array containing the names of the networks the specified container is connected to
     */
    private String[] getContainerNetworks(String contName) {
        String dockerNetworkCmd = "docker inspect -f '{{.NetworkSettings.Networks}}' " + contName;
        String cmdResult = execDockerCmd(dockerNetworkCmd, DOCKER_TIMEOUT, false);
        if (cmdResult == null || cmdResult.contains(" RC=")) { // RC is added in execDockerCmd if there is an error
            warn("Unable to retrieve container networks.");
            return null;
        }
        else {
            return parseNetworks(removeSurroundingQuotes(cmdResult.trim()));
        }
    }

     /**
     * Parses Docker network names from a "docker inspect" command result on a container.
     * @param dockerResult the result from the command "docker inspect -f '{{.NetworkSettings.Networks}}' containerName"
     * -> dockerResult must not contain surrounding quotes or leading/trailing whitespace
     * @return a String array containing the names of the networks contained in the dockerResult parameter
     */
    protected static String[] parseNetworks(String dockerResult) {
        // Example dockerResult value: map[bridge:0xc000622000 myNet:0xc0006220c0 otherNet:0xc000622180]
        if (!dockerResult.matches("map\\[(.*?)\\]")) {
            return null;
        }
        String networkMap = dockerResult.substring(dockerResult.indexOf("[")+1, dockerResult.indexOf("]"));
        String[] networkHex = networkMap.split(" ");
        String[] networks = new String[networkHex.length];
        for (int i=0; i < networkHex.length; i++) {
            networks[i] = networkHex[i].split(":")[0];
        }
        return networks;
    }

    private String getContainerIPAddress(String contName, String network) {
        String dockerIPAddressCmd = "docker inspect -f '{{.NetworkSettings.Networks." + network + ".IPAddress}}' " + contName;
        String result = execDockerCmd(dockerIPAddressCmd, DOCKER_TIMEOUT, false);
        if (result == null || result.contains(" RC=")) { // RC is added in execDockerCmd if there is an error
            warn("Unable to retrieve container IP address for network '" + network + "'.");
            return "<no value>"; // this is what Docker displays when an IP address it not found for a network
        }
        return removeSurroundingQuotes(result.trim());
    }

    protected static String removeSurroundingQuotes(String str) {
        if (str != null && str.length() >= 2 && ((str.startsWith("\"") && str.endsWith("\"")) || (str.startsWith("\'") && str.endsWith("\'")))) {
            return str.substring(1, str.length()-1);
        }
        return str;
    }

    // Read all the files from the array list.
    private String getCopiedFiles() {
        StringBuilder param = new StringBuilder(256); // estimate of size needed
        for (int i=0; i < srcMount.size(); i++) {
            if (new File(srcMount.get(i)).exists()) { // only Files are in this list
                param.append(" -v ").append(srcMount.get(i)).append(":").append(destMount.get(i));
            } else {
                error("A file referenced by the Dockerfile is not found: " + srcMount.get(i) +
                    ". Update the Dockerfile or ensure the file is in the correct location.");
            }
        }
        return param.toString();
    }

    private String getUserId() {
        if (OSUtil.isLinux()) {
            try {
                String id = runCmd("id -u");
                if (id != null) {
                    return " --user " + id;
                }
            } catch (IOException e) {
                // can't get user id. runCmd has printed an error message.
            } catch (InterruptedException e) {
                // can't get user id. runCmd has printed an error message.
            }
        }
        return "";
    }

    public abstract void libertyCreate() throws PluginExecutionException;

    public abstract void libertyDeploy() throws PluginExecutionException;

    /**
     * Install features in regular dev mode. This method should not be used in container mode.
     * @throws PluginExecutionException
     */
    public abstract void libertyInstallFeature() throws PluginExecutionException;

    public void restartServer() throws PluginExecutionException {
        restartServer(false);
    }

    /**
     * Stop the server, set up Liberty and restart it.
     * @param buildContainer  Force a Docker build when in container mode. Ignored otherwise.
     */
    public void restartServer(boolean buildContainer) throws PluginExecutionException {
        info("Restarting server...");
        setDevStop(true);
        if (container) {
            stopContainer(); // this command is synchronous
        } else {
            stopServer();
            if (serverThread != null) {
                final long threadShutdownTimeoutSeconds = 30;
                try {
                    serverThread.join(threadShutdownTimeoutSeconds * 1000);
                    if (serverThread.isAlive()) {
                        throw new PluginExecutionException("Could not stop the server after " + threadShutdownTimeoutSeconds
                                + " seconds.  Ensure that the server has been stopped, then start dev mode again.");
                    }
                } catch (InterruptedException e) {
                    if (serverThread.isAlive()) {
                        throw new PluginExecutionException(
                                "Could not stop the server.  Ensure that the server has been stopped, then start dev mode again.",
                                e);
                    } else {
                        // the thread was interrupted, but the server thread is already stopped
                        debug(e);
                    }
                }
            }
        }
        // suppress install feature warning
        System.setProperty(SKIP_BETA_INSTALL_WARNING, Boolean.TRUE.toString());
        libertyCreate();
        // Skip installing features on container during restart, since the Dockerfile should have 'RUN features.sh'
        if (!container) {
            libertyInstallFeature();
        }
        libertyDeploy();
        startServer(buildContainer, false);
        setDevStop(false);
        info("The server has been restarted.");
        printDevModeMessages(inputUnavailable.get(), true);
    }

    private void parseHostNameAndPorts(final ServerTask serverTask, File messagesLogFile)
            throws PluginExecutionException {
        String webAppMessage = serverTask.findStringInFile(WEB_APP_AVAILABLE_MESSAGE_PREFIX, messagesLogFile);
        debug("Web app available message: " + webAppMessage);
        if (webAppMessage != null) {
            int portPrefixIndex = parseHostName(webAppMessage);
            parseHttpPort(webAppMessage, portPrefixIndex);
        }
        List<String> listeningOnPortMessages = serverTask.findStringsInFile(LISTENING_ON_PORT_MESSAGE_PREFIX,
                messagesLogFile);
        if (listeningOnPortMessages != null) {
            parseHttpsPort(listeningOnPortMessages);
        }
    }

    protected int parseHostName(String webAppMessage) throws PluginExecutionException {
        // Get unescaped HTTP protocol and hostname index
        int protocolIndex = webAppMessage.indexOf(HTTP_PREFIX);
        int hostNameIndex = protocolIndex + HTTP_PREFIX.length();

        // if not found, get escaped HTTP protocol and hostname index
        if (protocolIndex < 0) {
            protocolIndex = webAppMessage.indexOf(HTTP_PREFIX_ESCAPED);
            hostNameIndex = protocolIndex + HTTP_PREFIX_ESCAPED.length();
        }

        // if not found, get unescaped HTTPS protocol and hostname index
        if (protocolIndex < 0) {
            protocolIndex = webAppMessage.indexOf(HTTPS_PREFIX);
            hostNameIndex = protocolIndex + HTTPS_PREFIX.length();
        }

        // if not found, get escaped HTTPS protocol and hostname index
        if (protocolIndex < 0) {
            protocolIndex = webAppMessage.indexOf(HTTPS_PREFIX_ESCAPED);
            hostNameIndex = protocolIndex + HTTPS_PREFIX_ESCAPED.length();
        }

        if (protocolIndex < 0) {
            throw new PluginExecutionException(
                    "Could not parse the host name from the log message: " + webAppMessage);
        }

        int portPrefixIndex = webAppMessage.indexOf(":", hostNameIndex);
        if (portPrefixIndex < 0) {
            throw new PluginExecutionException(
                    "Could not parse the port number from the log message: " + webAppMessage);
        }
        if (container) {
            hostName = "localhost";
        } else {
            hostName = webAppMessage.substring(hostNameIndex, portPrefixIndex);
        }
        debug("Parsed host name: " + hostName);
        return portPrefixIndex;
    }

    protected void parseHttpPort(String webAppMessage, int portPrefixIndex) {
        if (!webAppMessage.contains(HTTP_PREFIX)) {
            return;
        }
        int portIndex = portPrefixIndex + 1;
        int portEndIndex = webAppMessage.indexOf("/", portIndex);
        if (portEndIndex < 0) {
            // if no ending slash, the port ends at the end of the message
            portEndIndex = webAppMessage.length();
        }
        String parsedHttpPort = webAppMessage.substring(portIndex, portEndIndex);
        debug("Parsed http port: " + parsedHttpPort);
        if (container) {
            httpPort = findLocalPort(parsedHttpPort);
            containerHttpPort = parsedHttpPort;
        }
        else {
            httpPort = parsedHttpPort;
        }
    }

    protected void parseHttpsPort(List<String> messages) throws PluginExecutionException {
        for (String message : messages) {
            debug("Looking for https port in message: " + message);
            String httpsMessageContents = message.split(LISTENING_ON_PORT_MESSAGE_PREFIX)[1];
            String[] messageTokens = httpsMessageContents.split(" ");
            // Look for endpoint with name containing "-ssl"
            for (String token : messageTokens) {
                if (token.contains("-ssl")) {
                    String parsedHttpsPort = getPortFromMessageTokens(messageTokens);
                    if (parsedHttpsPort != null) {
                        debug("Parsed https port: " + parsedHttpsPort);
                        if (container) {
                            httpsPort = findLocalPort(parsedHttpsPort);
                            containerHttpsPort = parsedHttpsPort;
                        }
                        else {
                            httpsPort = parsedHttpsPort;
                        }
                        return;
                    } else {
                        throw new PluginExecutionException(
                                "Could not parse the https port number from the log message: " + message);
                    }
                }
            }
        }
        debug("Could not find https port. The server might not be configured for https.");
    }

    private String getPortFromMessageTokens(String[] messageTokens) throws PluginExecutionException {
        // For each space-separated token, keep only the numeric parts.
        // The port is the last numeric token which is a number <= 65535.
        for (int i = messageTokens.length - 1; i >= 0; i--) {
            String numericToken = messageTokens[i].replaceAll("[^\\d]", "");
            if (numericToken.length() > 0) {
                try {
                    int parsedPort = Integer.parseInt(numericToken);
                    if (parsedPort <= 65535) {
                        return numericToken;
                    }
                } catch (NumberFormatException e) {
                    // If the token is not parseable for some reason, then it's probably not a port
                    // number
                    debug("Could not parse integer from numeric token " + numericToken + " from message token "
                            + messageTokens[i], e);
                }
            }
        }
        return null;
    }

    private String findLocalPort(String internalContainerPort) {
        String dockerPortCmd = "docker port " + containerName + " " + internalContainerPort;
        String cmdResult = execDockerCmd(dockerPortCmd, DOCKER_TIMEOUT, false);
        if (cmdResult == null) {
            warn("Unable to retrieve locally mapped port.");
            return null;
        }
        if (cmdResult.contains(" RC=")) { // This piece of the string is added in execDockerCmd if there is an error
            warn("Unable to retrieve locally mapped port. Docker result: \"" + cmdResult.split(" RC=")[0] + "\". Ensure the Docker ports are mapped correctly.");
            return null;
        }
        String[] cmdResultSplit = cmdResult.split(":");
        String localPort = cmdResultSplit[cmdResultSplit.length - 1].trim();
        debug("Local port: " + localPort);
        return localPort;
    }

    public void cleanUpServerEnv() {
        // clean up server.env file
        File serverEnvFile;
        File serverEnvBackup;
        try {
            serverEnvBackup = new File(serverDirectory.getCanonicalPath() + "/server.env.bak");
            serverEnvFile = new File(serverDirectory.getCanonicalPath() + "/server.env");
            if (serverEnvBackup.exists()) {
                // Restore original server.env file
                try {
                    Files.copy(serverEnvBackup.toPath(), serverEnvFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                } catch (IOException e) {
                    error("Could not restore server.env: " + e.getMessage());
                }
                serverEnvBackup.delete();
            } else {
                // Delete server.env file
                serverEnvFile.delete();
            }
        } catch (IOException e) {
            error("Could not retrieve server.env: " + e.getMessage());
        }
    }

    public void cleanUpTempConfig() {
        if (this.tempConfigPath != null) {
            File tempConfig = this.tempConfigPath.toFile();
            if (tempConfig.exists()) {
                try {
                    FileUtils.deleteDirectory(tempConfig);
                    debug("Successfully deleted liberty:dev temporary configuration folder");
                } catch (IOException e) {
                    warn("Could not delete liberty:dev temporary configuration folder: " + e.getMessage());
                }
            }
        }
    }

    public void cleanUpTempDockerfile() {
        if (!keepTempDockerfile && tempDockerfilePath != null) {
            File tempDockerfile = tempDockerfilePath.toFile();
            if (tempDockerfile.exists()) {
                try {
                    Files.delete(tempDockerfilePath);
                    debug("Successfully deleted dev mode temporary Dockerfile");
                } catch (IOException e) {
                    warn("Could not delete dev mode temporary Dockerfile: " + e.getMessage());
                }
            }
        }
    }

    /**
     * Whether dev mode intentionally caused the server to stop.
     * 
     * @param devStop If true, stopping the server will not cause dev mode to print
     *                an error message.
     */
    public void setDevStop(boolean devStop) {
        this.devStop.set(devStop);
    }

    public void addShutdownHook(final ThreadPoolExecutor executor) {
        // shutdown hook to stop server when dev mode is terminated
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                runShutdownHook(executor);
            }
        });
    }

    private void runShutdownHook(final ThreadPoolExecutor executor) {
        if (!calledShutdownHook.getAndSet(true)) {

            if (trackingMode == FileTrackMode.POLLING || trackingMode == FileTrackMode.NOT_SET) {
                disablePolling();
            }

            setDevStop(true);
            cleanUpTempConfig();
            cleanUpServerEnv();

            if (hotkeyReader != null) {
                hotkeyReader.shutdown();
            }

            // shutdown tests
            executor.shutdown();

            // stopping server
            if (container) {
                cleanUpTempDockerfile();
                stopContainer();
            } else {
                stopServer();
            }
        }
    }

    private void disablePolling() {
        synchronized (newFileObservers) {
            consolidateFileObservers();
            for (FileAlterationObserver observer : fileObservers) {
                try {
                    observer.destroy();
                } catch (Exception e) {
                    debug("Could not destroy file observer", e);
                }
            }
        }
    }

    /**
     * Gets a map of the environment variables to set for debug mode.
     * 
     * @param libertyDebugPort the debug port to use
     */
    public Map<String, String> getDebugEnvironmentVariables() throws IOException {
        Map<String, String> map = new HashMap<String, String>();
        map.put("WLP_DEBUG_SUSPEND", "n");
        map.put("WLP_DEBUG_ADDRESS", String.valueOf(findAvailablePort(libertyDebugPort, true)));
        return map;
    }

    /**
     * Enable server debug variables in server.env, using the user specified debug
     * port if it's available, otherwise uses a random available port.
     * 
     * @throws IOException if there was an IO exception when reading or writing the
     *                     server.env
     */
    public void enableServerDebug() throws IOException {
        enableServerDebug(true);
    }

    private void enableServerDebug(boolean doFindPort) throws IOException {
        if (!libertyDebug) {
            return;
        }

        String serverEnvPath = serverDirectory.getCanonicalPath() + "/server.env";
        File serverEnvFile = new File(serverEnvPath);
        StringBuilder sb = new StringBuilder();
        File serverEnvBackup = new File(serverEnvPath + ".bak");
        if (serverEnvFile.exists()) {
            debug("server.env already exists");

            Files.copy(serverEnvFile.toPath(), serverEnvBackup.toPath(), StandardCopyOption.REPLACE_EXISTING);
            boolean deleted = serverEnvFile.delete();
            if (!deleted) {
                error("Could not move existing server.env file");
            }

            BufferedReader reader = new BufferedReader(new FileReader(serverEnvBackup));
            try {
                String line;
                while ((line = reader.readLine()) != null) {
                    sb.append(line);
                    sb.append("\n");
                }
            } finally {
                reader.close();
            }
        } else {
            // if server.env does not exist, clean up any backup file
            serverEnvBackup.delete();
        }

        debug("Creating server.env file: " + serverEnvFile.getCanonicalPath());
        sb.append("WLP_DEBUG_SUSPEND=n\n");
        sb.append("WLP_DEBUG_ADDRESS=");
        if (doFindPort) {
            sb.append(findAvailablePort(libertyDebugPort, true));
        } else {
            sb.append(alternativeDebugPort == -1 ? libertyDebugPort : alternativeDebugPort);
        }
        sb.append("\n");

        BufferedWriter writer = new BufferedWriter(new FileWriter(serverEnvFile));
        try {
            writer.write(sb.toString());
        } finally {
            writer.close();
        }

        if (serverEnvFile.exists()) {
            debug("Successfully created liberty:dev server.env file");
        }
    }

    /**
     * Finds an available port to use. There are two semantics. If looking for a port
     * for the server debug connection and the port is in use then return an
     * ephemeral port. If looking for a port for the server http connection then
     * try sequential port numbers.
     * 
     * In the case of the server debug connection, if the preferred port is not
     * available, return a random available port and cache the result which will
     * override the preferredPort if this method is called again.
     * 
     * @param  The number of the port to start the search for an available port.
     * @param  Whether to choose an ephemeral port. True to choose an ephemeral port,
     *         false to search sequentially.
     * @return An available port.
     * @throws IOException if it could not find any available port, or there was an
     *                     error when opening a server socket regardless of port.
     */
    public int findAvailablePort(int preferredPort, boolean isDebugPort) throws IOException {
        int portToTry = preferredPort;
        if (isDebugPort && alternativeDebugPort != -1) {
            portToTry = alternativeDebugPort;
        }

        ServerSocket serverSocket = null;
        while (portToTry < 65535) {
            if (OSUtil.isWindows()) {
                try {
                    // try binding to the portToTry
                    serverSocket = new ServerSocket(portToTry);
                    return serverSocket.getLocalPort();
                } catch (IOException e) {
                    if (serverSocket != null) {
                        serverSocket.close();
                    }
                    if (isDebugPort) {
                        // if binding failed, try binding to a random port
                        serverSocket = new ServerSocket(0);
                        int availablePort = serverSocket.getLocalPort();
                        processAvailableDebugPort(preferredPort, portToTry, availablePort);
                        return availablePort;
                    } else {
                        debug("findAvailablePort found port is in use: " + portToTry);
                        ++portToTry;
                    }
                } finally {
                    if (serverSocket != null) {
                        serverSocket.close();
                    }
                }
            } else {
                try {
                    serverSocket = new ServerSocket();
                    serverSocket.setReuseAddress(false);
                    // try binding to the loopback address at the port to try
                    serverSocket.bind(new InetSocketAddress(InetAddress.getByName(null), portToTry), 1);
                    return serverSocket.getLocalPort();
                } catch (IOException e) {
                    if (serverSocket != null) {
                        if (isDebugPort) {
                            // if binding failed, try binding to a random port
                            serverSocket.bind(null, 1);
                            int availablePort = serverSocket.getLocalPort();
                            processAvailableDebugPort(preferredPort, portToTry, availablePort);
                            return availablePort;
                        } else {
                            debug("findAvailablePort found port is in use: " + portToTry);
                            ++portToTry;
                        }
                    } else {
                        throw new IOException("Could not create a server socket.", e);
                    }
                } finally {
                    if (serverSocket != null) {
                        serverSocket.close();
                    }
                }
            }
        }
        return preferredPort; // usual return is from the try or the catch
    }

    private void processAvailableDebugPort(int preferredPort, int portToTry, int availablePort) {
        if (portToTry == preferredPort) {
            warn("The debug port " + preferredPort + " is not available.  Using " + availablePort
                    + " as the debug port instead.");
        } else {
            debug("The previous debug port " + alternativeDebugPort + " is no longer available.  Using "
                    + availablePort + " as the debug port instead.");
        }
        alternativeDebugPort = availablePort;
    }

    private HotkeyReader hotkeyReader = null;

    /**
     * Run a hotkey reader thread. If the thread is already running, re-prints the
     * message about pressing enter to run tests.
     * 
     * @param executor the test thread executor
     */
    public void runHotkeyReaderThread(ThreadPoolExecutor executor) {
        if (inputUnavailable.get()) {
            return;
        }
        boolean startedNewHotkeyReader = false;
        if (hotkeyReader == null) {
            hotkeyReader = new HotkeyReader(executor);
            new Thread(hotkeyReader).start();
            debug("Started hotkey reader.");
            startedNewHotkeyReader = true;
        }
        if (!skipTests) {
            synchronized (inputUnavailable) {
                try {
                    if (startedNewHotkeyReader) {
                        // if new hotkey reader started, wait for it to try getting the input to see if
                        // it's available
                        inputUnavailable.wait(500);
                    }
                    printDevModeMessages(inputUnavailable.get(), firstStartup);
                    firstStartup = false;
                } catch (InterruptedException e) {
                    debug("Interrupted while waiting to determine whether input can be read", e);
                }
            }
        }
    }

    /**
     * Print the dev mode startup and/or run tests messages.
     * 
     * @param inputUnavailable If true, indicates that the console is non-interactive so hotkey messages should not be printed.
     * @param startup If true, include attention barriers (asterisks lines) and overall dev mode startup messages such as list of hotkeys and ports.
     */
    private void printDevModeMessages(boolean inputUnavailable, boolean startup) {
        // the following will be printed only on startup or restart
        if (startup) {
            // print barrier header
            info(formatAttentionBarrier());

            info(formatAttentionTitle("Liberty is running in dev mode."));
        }

        if (!inputUnavailable) {
            // the following will be printed on startup and every time after the tests run
            if (hotTests) {
                String message = "Tests will run automatically when changes are detected. You can also press the Enter key to run tests on demand.";
                info(startup ? formatAttentionMessage(message) : message);
            } else {
                String message = "To run tests on demand, press Enter.";
                info(startup ? formatAttentionMessage(message) : message);
            }

            // the following will be printed only on startup or restart
            if (startup) {
                if (container) {
                    info(formatAttentionMessage("To rebuild the Docker image and restart the container, type 'r' and press Enter."));
                } else {
                    info(formatAttentionMessage("To restart the server, type 'r' and press Enter."));
                }

                info(formatAttentionMessage("To stop the server and quit dev mode, press Ctrl-C or type 'q' and press Enter."));
            }
        } else {
            debug("Cannot read user input, setting hotTests to true.");
            String message = "Tests will run automatically when changes are detected.";
            info(startup ? formatAttentionMessage(message) : message);
            hotTests = true;
        }
        if (startup) {
            if (container) {
                boolean nonDefaultHttpPortUsed = !skipDefaultPorts && !String.valueOf(LIBERTY_DEFAULT_HTTP_PORT).equals(httpPort);
                boolean nonDefaultHttpsPortUsed = !skipDefaultPorts && !String.valueOf(LIBERTY_DEFAULT_HTTPS_PORT).equals(httpsPort);
                boolean nonDefaultDebugPortUsed = alternativeDebugPort != -1; // this is set when a random ephemeral port is selected
                if (containerHttpPort != null || containerHttpsPort != null || libertyDebug) {
                    info(formatAttentionMessage(""));
                    info(formatAttentionTitle("Liberty container port information:"));
                }
                if ((containerHttpPort != null && httpPort != null && nonDefaultHttpPortUsed)
                        || (containerHttpsPort != null && httpsPort != null && nonDefaultHttpsPortUsed)
                        || (libertyDebug && nonDefaultDebugPortUsed)) {
                    warn(formatAttentionMessage("The Liberty container is using non-default host ports to avoid port conflict errors."));
                }
                if (containerHttpPort != null) {
                    if (httpPort != null) {
                        if (!nonDefaultHttpPortUsed) {
                            info(formatAttentionMessage("Internal container HTTP port [ " + containerHttpPort + " ] is mapped to Docker host port [ " + httpPort + " ]"));
                        } else {
                            info(formatAttentionMessage("Internal container HTTP port [ " + containerHttpPort + " ] is mapped to Docker host port [ " + httpPort + " ] <"));
                        }
                    } else {
                        info(formatAttentionMessage("Internal container HTTP port: [ " + containerHttpPort + " ]"));
                    }
                }
                if (containerHttpsPort != null) {
                    if (httpsPort != null) {
                        if (!nonDefaultHttpsPortUsed) {
                            info(formatAttentionMessage("Internal container HTTPS port [ " + containerHttpsPort + " ] is mapped to Docker host port [ " + httpsPort + " ]"));
                        } else {
                            info(formatAttentionMessage("Internal container HTTPS port [ " + containerHttpsPort + " ] is mapped to Docker host port [ " + httpsPort + " ] <"));
                        }
                    } else {
                        info(formatAttentionMessage("Internal container HTTPS port: [ " + containerHttpsPort + " ]"));
                    }
                }
                if (libertyDebug) {
                    int debugPort = (alternativeDebugPort == -1 ? libertyDebugPort : alternativeDebugPort);
                    if (!nonDefaultDebugPortUsed) {
                        info(formatAttentionMessage("Liberty debug port mapped to Docker host port: [ " + debugPort + " ]"));
                    } else {
                        info(formatAttentionMessage("Liberty debug port mapped to Docker host port: [ " + debugPort + " ] <"));
                    }
                }
                info(formatAttentionMessage(""));
                info(formatAttentionTitle("Docker network information:"));
                info(formatAttentionMessage("Container name: [ " + containerName + " ]"));

                String[] networks = getContainerNetworks(containerName);
                if (networks != null) {
                    for (String network : networks) {
                        info(formatAttentionMessage("IP address [ " + getContainerIPAddress(containerName, network) + " ] on Docker network [ " + network + " ]"));
                    }
                }
            }
            else {
                if (httpPort != null || httpsPort != null || libertyDebug) {
                    info(formatAttentionMessage(""));
                    info(formatAttentionTitle("Liberty server port information:"));
                }
                if (httpPort != null) {
                    info(formatAttentionMessage("Liberty server HTTP port: [ " + httpPort + " ]"));
                }
                if (httpsPort != null) {
                    info(formatAttentionMessage("Liberty server HTTPS port: [ " + httpsPort + " ]"));
                }
                if (libertyDebug) {
                    int debugPort = (alternativeDebugPort == -1 ? libertyDebugPort : alternativeDebugPort);
                    info(formatAttentionMessage("Liberty debug port: [ " + debugPort + " ]"));
                }
            }
            // print barrier footer
            info(formatAttentionBarrier());
        }
    }

    private String formatAttentionBarrier() {
        return "************************************************************************";
    }

    private String formatAttentionTitle(String message) {
        return "*    " + message;
    }

    private String formatAttentionMessage(String message) {
        return "*        " + message;
    }

    private class HotkeyReader implements Runnable {
        private Scanner scanner;
        private ThreadPoolExecutor executor;
        private boolean shutdown = false;

        public HotkeyReader(ThreadPoolExecutor executor) {
            this.executor = executor;
        }

        @Override
        public void run() {
            debug("Running hotkey reader thread");
            scanner = new Scanner(new CloseShieldInputStream(System.in)); // shield allows us to close the scanner without closing System.in.
            try {
                readInput();
            } finally {
                scanner.close();
            }
        }

        public void shutdown() {
            shutdown = true;
        }

        private void readInput() {
            if (scanner.hasNextLine()) {
                synchronized (inputUnavailable) {
                    inputUnavailable.notify();
                }
                while (!shutdown) {
                    debug("Waiting for Enter key to run tests");
                    if (!scanner.hasNextLine()) {
                        break;
                    }
                    String line = scanner.nextLine();
                    if (line != null && (line.trim().equalsIgnoreCase("q") || line.trim().equalsIgnoreCase("quit")
                            || line.trim().equalsIgnoreCase("exit"))) {
                        debug("Detected exit command");
                        runShutdownHook(executor);
                    } else if (line != null && line.trim().equalsIgnoreCase("r")) {
                        debug("Detected restart command");
                        try {
                            restartServer(true);
                        } catch (PluginExecutionException e) {
                            debug("Exiting dev mode due to server restart failure");
                            error("Could not restart the server.", e);
                            runShutdownHook(executor);
                        }
                    } else {
                        debug("Detected Enter key. Running tests...");
                        runTestThread(false, executor, -1, false, true);
                    }
                }
            } else {
                synchronized (inputUnavailable) {
                    inputUnavailable.set(true);
                    inputUnavailable.notify();
                }
            }
        }
    }

    Collection<File> recompileJavaSources;
    Collection<File> recompileJavaTests;
    Collection<File> deleteJavaSources;
    Collection<File> deleteJavaTests;
    Collection<File> failedCompilationJavaSources;
    Collection<File> failedCompilationJavaTests;
    long lastJavaSourceChange;
    long lastJavaTestChange;
    boolean triggerJavaSourceRecompile;
    boolean triggerJavaTestRecompile;
    File outputDirectory;
    File serverXmlFile;
    File serverXmlFileParent;
    File bootstrapPropertiesFile;
    File bootstrapPropertiesFileParent;
    File jvmOptionsFile;
    File jvmOptionsFileParent;
    File buildFile;
    File dockerfileUsed;
    List<String> artifactPaths;
    WatchService watcher;

    // The serverXmlFile parameter can be null when using the server.xml from the
    // configDirectory, which has a default value.
    public void watchFiles(File buildFile, File outputDirectory, File testOutputDirectory,
            final ThreadPoolExecutor executor, List<String> artifactPaths, File serverXmlFile,
            File bootstrapPropertiesFile, File jvmOptionsFile) throws Exception {
        this.buildFile = buildFile;
        this.outputDirectory = outputDirectory;
        this.serverXmlFile = serverXmlFile;
        this.bootstrapPropertiesFile = bootstrapPropertiesFile;
        this.jvmOptionsFile = jvmOptionsFile;
        this.artifactPaths = artifactPaths;
        this.dockerfileUsed = null;

        try {
            watcher = FileSystems.getDefault().newWatchService();
            serverXmlFileParent = null;
            if (serverXmlFile != null && serverXmlFile.exists()) {
                serverXmlFileParent = serverXmlFile.getParentFile();
            }

            bootstrapPropertiesFileParent = null;
            if (bootstrapPropertiesFile != null && bootstrapPropertiesFile.exists()) {
                bootstrapPropertiesFileParent = bootstrapPropertiesFile.getParentFile();
            }

            jvmOptionsFileParent = null;
            if (jvmOptionsFile != null && jvmOptionsFile.exists()) {
                jvmOptionsFileParent = jvmOptionsFile.getParentFile();
            }

            Path srcPath = this.sourceDirectory.getCanonicalFile().toPath();
            Path testSrcPath = this.testSourceDirectory.getCanonicalFile().toPath();
            Path configPath = this.configDirectory.getCanonicalFile().toPath();

            boolean sourceDirRegistered = false;
            boolean testSourceDirRegistered = false;
            boolean configDirRegistered = false;
            boolean serverXmlFileRegistered = false;
            boolean bootstrapPropertiesFileRegistered = false;
            boolean jvmOptionsFileRegistered = false;

            if (this.sourceDirectory.exists()) {
                registerAll(srcPath, executor);
                sourceDirRegistered = true;
            }

            if (this.testSourceDirectory.exists()) {
                registerAll(testSrcPath, executor);
                testSourceDirRegistered = true;
            }

            if (this.configDirectory.exists()) {
                registerAll(configPath, executor);
                configDirRegistered = true;
            }

            if (serverXmlFile != null && serverXmlFile.exists() && serverXmlFileParent.exists()) {
                Path serverXmlFilePath = serverXmlFileParent.getCanonicalFile().toPath();
                registerAll(serverXmlFilePath, executor);
                serverXmlFileRegistered = true;
            }

            if (bootstrapPropertiesFile != null && bootstrapPropertiesFile.exists() && bootstrapPropertiesFileParent.exists()) {
                Path bootstrapPropertiesFilePath = bootstrapPropertiesFileParent.getCanonicalFile().toPath();
                registerAll(bootstrapPropertiesFilePath, executor);
                bootstrapPropertiesFileRegistered = true;
            }

            if (jvmOptionsFile != null && jvmOptionsFile.exists() && jvmOptionsFileParent.exists()) {
                Path jvmOptionsFilePath = jvmOptionsFileParent.getCanonicalFile().toPath();
                registerAll(jvmOptionsFilePath, executor);
                jvmOptionsFileRegistered = true;
            }

            if (container) {
                dockerfileUsed = getDockerfile();
                registerSingleFile(dockerfileUsed, executor);
            }

            HashMap<File, Boolean> resourceMap = new HashMap<File, Boolean>();
            for (File resourceDir : resourceDirs) {
                resourceMap.put(resourceDir, false);
                if (resourceDir.exists()) {
                    registerAll(resourceDir.getCanonicalFile().toPath(), executor);
                    resourceMap.put(resourceDir, true);
                }
            }

            registerSingleFile(buildFile, executor);

            if (propertyFilesMap != null) {
                for (File f : propertyFilesMap.keySet()) {
                    registerSingleFile(f, executor);
                }
            }

            initWatchLoop();

            while (true) {
                // Check the server and stop dev mode by throwing an exception if the server stopped.
                checkStopDevMode(true);

                if (container) {
                    synchronized(dockerfileDirectoriesToWatch) {
                        if (!dockerfileDirectoriesToWatch.isEmpty()) {
                            for (Path path : dockerfileDirectoriesToWatch) {
                                File f = path.toFile();
                                if (f.isDirectory()) {
                                    debug("Registering path from dockerfileDirectoriesToWatch: " + path);
                                    registerAll(path, executor, true);
                                } else {
                                    debug("Registering file path from dockerfileDirectoriesToWatch: " + path);
                                    registerSingleFile(f, executor, true);
                                }
                                dockerfileDirectoriesTracked.add(path);
                            }
                            dockerfileDirectoriesToWatch.clear();
                        }
                    }
                }

                processJavaCompilation(outputDirectory, testOutputDirectory, executor, artifactPaths);

                // check if javaSourceDirectory has been added
                if (!sourceDirRegistered && this.sourceDirectory.exists()
                        && this.sourceDirectory.listFiles().length > 0) {
                    compile(this.sourceDirectory);
                    registerAll(srcPath, executor);
                    debug("Registering Java source directory: " + this.sourceDirectory);
                    sourceDirRegistered = true;
                } else if (sourceDirRegistered && !this.sourceDirectory.exists()) {
                    cleanTargetDir(outputDirectory);
                    sourceDirRegistered = false;
                }

                // check if testSourceDirectory has been added
                if (!testSourceDirRegistered && this.testSourceDirectory.exists()
                        && this.testSourceDirectory.listFiles().length > 0) {
                    compile(this.testSourceDirectory);
                    registerAll(testSrcPath, executor);
                    debug("Registering Java test directory: " + this.testSourceDirectory);
                    runTestThread(false, executor, -1, false, false);
                    testSourceDirRegistered = true;

                } else if (testSourceDirRegistered && !this.testSourceDirectory.exists()) {
                    cleanTargetDir(testOutputDirectory);
                    testSourceDirRegistered = false;
                }

                // check if configDirectory has been added
                if (!configDirRegistered && this.configDirectory.exists()) {
                    configDirRegistered = true;
                    if (serverXmlFile != null && !serverXmlFile.exists()) {
                        registerAll(configPath, executor);
                        debug("Registering configuration directory: " + this.configDirectory);
                    } else {
                        warn("The server configuration directory " + configDirectory
                                + " has been added. Restart dev mode for it to take effect.");
                    }
                }

                // check if serverXmlFile has been added
                if (!serverXmlFileRegistered && serverXmlFile != null && serverXmlFile.exists()) {
                    serverXmlFileRegistered = true;
                    debug("Server configuration file has been added: " + serverXmlFile);
                    warn("The server configuration file " + serverXmlFile
                            + " has been added. Restart dev mode for it to take effect.");
                }

                if (!bootstrapPropertiesFileRegistered && bootstrapPropertiesFile != null && bootstrapPropertiesFile.exists()) {
                    bootstrapPropertiesFileRegistered = true;
                    debug("Bootstrap properties file has been added: " + bootstrapPropertiesFile);
                    warn("The bootstrap properties file " + bootstrapPropertiesFile
                            + " has been added. Restart dev mode for it to take effect.");
                }

                if (!jvmOptionsFileRegistered && jvmOptionsFile != null && jvmOptionsFile.exists()) {
                    jvmOptionsFileRegistered = true;
                    debug("JVM Options file has been added: " + jvmOptionsFile);
                    warn("The JVM Options file " + jvmOptionsFile
                            + " has been added. Restart dev mode for it to take effect.");
                }

                // check if resourceDirectory has been added
                for (File resourceDir : resourceDirs) {
                    if (!resourceMap.get(resourceDir) && resourceDir.exists()) {
                        // added resource directory
                        registerAll(resourceDir.getCanonicalFile().toPath(), executor);
                        resourceMap.put(resourceDir, true);
                    } else if (resourceMap.get(resourceDir) && !resourceDir.exists()) {
                        // deleted resource directory
                        warn("The resource directory " + resourceDir
                                + " was deleted.  Restart liberty:dev mode for it to take effect.");
                        resourceMap.put(resourceDir, false);
                    }
                }

                if (trackingMode == FileTrackMode.FILE_WATCHER || trackingMode == FileTrackMode.NOT_SET) {
                    try {
                        final WatchKey wk = watcher.poll(100, TimeUnit.MILLISECONDS);
                        final Watchable watchable = wk.watchable();
                        final Path directory = (Path) watchable;

                        List<WatchEvent<?>> events = wk.pollEvents();

                        for (WatchEvent<?> event : events) {
                            if (trackingMode == FileTrackMode.NOT_SET) {
                                trackingMode = FileTrackMode.FILE_WATCHER;
                                disablePolling();
                            }
                            final Path changed = (Path) event.context();
                            debug("Processing events for watched directory: " + directory);

                            File fileChanged = new File(directory.toString(), changed.toString());
                            if (ignoreFileOrDir(fileChanged)) {
                                // skip this file or directory, and continue to the next file or directory
                                continue;
                            }
                            debug("Changed: " + changed + "; " + event.kind());

                            ChangeType changeType = null;
                            if (event.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                                changeType = ChangeType.CREATE;
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_MODIFY) {
                                changeType = ChangeType.MODIFY;
                            } else if (event.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                                changeType = ChangeType.DELETE;
                            }
                            processFileChanges(executor, fileChanged, outputDirectory, false, changeType);

                        }
                        // reset the key
                        boolean valid = wk.reset();
                        if (!valid) {
                            debug("WatchService key has been unregistered for " + directory);
                        }
                    } catch (InterruptedException | NullPointerException e) {
                        // do nothing let loop continue
                    }
                }
                if (trackingMode == FileTrackMode.POLLING || trackingMode == FileTrackMode.NOT_SET) {
                    synchronized (newFileObservers) {
                        consolidateFileObservers();
                    }
                    // iterate through file observers
                    for (FileAlterationObserver observer : fileObservers) {
                        if (!cancelledFileObservers.contains(observer)) {
                            observer.checkAndNotify();
                        }
                    }
                    synchronized (cancelledFileObservers) {
                        removeCancelledFileObservers();
                    }

                    Thread.sleep(pollingInterval);
                }
            }
        } finally {
            if (watcher != null) {
                try {
                    watcher.close();
                } catch (IOException e) {
                    error("An error occurred attempting to close the file watcher. " + e.getMessage(), e);
                }
            }
        }
    }

    /**
     * Consolidate new file observers into the main observers set
     */
    private void consolidateFileObservers() {
        fileObservers.addAll(newFileObservers);
        newFileObservers.clear();
    }

    /**
     * Remove cancelled file observers from the main observers set
     */
    private void removeCancelledFileObservers() {
        fileObservers.removeAll(cancelledFileObservers);
        cancelledFileObservers.clear();
    }

    private void registerSingleFile(final File registerFile, final ThreadPoolExecutor executor) throws IOException {
        registerSingleFile(registerFile, executor, false);
    }

    /**
     * Register a single file with the WatchService using a file filter.
     * 
     * @param registerFile             the file of interest
     * @param executor                 the test thread executor
     * @param removeOnContainerRebuild whether the files should be unwatched if the container is rebuilt
     * @throws IOException unable to read the canonical path name
     */
    private void registerSingleFile(final File registerFile, final ThreadPoolExecutor executor, boolean removeOnContainerRebuild) throws IOException {
        if (trackingMode == FileTrackMode.POLLING || trackingMode == FileTrackMode.NOT_SET) {
            String parentPath = registerFile.getParentFile().getCanonicalPath();

            debug("Registering single file polling for " + registerFile.toString());

            // synchronize on the new observer set since only those are being updated in separate threads
            synchronized (newFileObservers) {
                Set<FileAlterationObserver> tempCombinedObservers = new HashSet<FileAlterationObserver>();
                tempCombinedObservers.addAll(fileObservers);
                tempCombinedObservers.addAll(newFileObservers);

                // if this path is already observed, ignore it
                for (FileAlterationObserver observer : tempCombinedObservers) {
                    if (parentPath.equals(observer.getDirectory().getCanonicalPath())) {
                        debug("Skipping single file polling for " + registerFile.toString() + " since its parent directory is already being observed");
                        return;
                    }
                }

                FileFilter singleFileFilter = new FileFilter() {
                    @Override
                    public boolean accept(File file) {
                        try {
                            if (file.getCanonicalFile().equals(registerFile.getCanonicalFile())) {
                                return true;
                            }
                        } catch (IOException e) {
                            if (file.equals(registerFile)) {
                                return true;
                            }
                        }
                        return false;
                    }
                };

                try {
                    debug("Adding single file observer for: " + registerFile.toString());
                    FileAlterationObserver observer = addFileAlterationObserver(executor, parentPath, singleFileFilter);
                    if (removeOnContainerRebuild) {
                        debug("Adding file to dockerfileDirectoriesFileObservers: " + registerFile.toString());
                        dockerfileDirectoriesFileObservers.add(observer);
                    }
                } catch (Exception e) {
                    error("Could not observe single file " + registerFile.toString(), e);
                }
            }
        }
        if (trackingMode == FileTrackMode.FILE_WATCHER || trackingMode == FileTrackMode.NOT_SET) {
            debug("Adding directory to WatchService " + registerFile.getParentFile().toPath() + " for single file " + registerFile.getName());
            WatchKey key = registerFile.getParentFile().toPath().register(
                watcher, 
                new WatchEvent.Kind[] { StandardWatchEventKinds.ENTRY_MODIFY,
                        StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_CREATE },
                SensitivityWatchEventModifier.HIGH);
            if (removeOnContainerRebuild) {
                debug("Adding file to dockerfileDirectoriesWatchKeys: " + registerFile.getName());
                dockerfileDirectoriesWatchKeys.add(key);
            }
        }
    }

    private FileAlterationObserver addFileAlterationObserver(final ThreadPoolExecutor executor, String parentPath, FileFilter filter)
            throws Exception {
        FileAlterationObserver observer = getFileAlterationObserver(executor, parentPath, filter);
        observer.initialize();
        newFileObservers.add(observer);
        return observer;
    }

    private FileAlterationObserver getFileAlterationObserver(final ThreadPoolExecutor executor, final String parentPath, FileFilter filter) {
        FileAlterationObserver observer = new FileAlterationObserver(parentPath, filter);
        observer.addListener(new FileAlterationListenerAdaptor() {
            @Override
            public void onDirectoryCreate(File file) {
                onAlteration(executor, parentPath, file, true, ChangeType.CREATE);
            }

            @Override
            public void onDirectoryDelete(File file) {
                onAlteration(executor, parentPath, file, true, ChangeType.DELETE);
            }

            @Override
            public void onDirectoryChange(File file) {
                onAlteration(executor, parentPath, file, true, ChangeType.MODIFY);
            }

            @Override
            public void onFileCreate(File file) {
                onAlteration(executor, parentPath, file, false, ChangeType.CREATE);
            }

            @Override
            public void onFileDelete(File file) {
                onAlteration(executor, parentPath, file, false, ChangeType.DELETE);
            }

            @Override
            public void onFileChange(File file) {
                onAlteration(executor, parentPath, file, false, ChangeType.MODIFY);
            }

            private void onAlteration(final ThreadPoolExecutor executor, final String parentPath, File file,
                    boolean isDirectory, ChangeType changeType) {
                if (trackingMode == FileTrackMode.NOT_SET) {
                    try {
                        WatchKey wk = null;
                        if (watcher != null) {
                            wk = watcher.poll(100, TimeUnit.MILLISECONDS);
                        }
                        List<WatchEvent<?>> events = null;
                        if (wk != null) {
                            events = wk.pollEvents();
                        }
                        if ((events == null) || events.isEmpty()) {
                            trackingMode = FileTrackMode.POLLING;
                            if (watcher != null) {
                                watcher.close();
                            }
                        } else {
                            trackingMode = FileTrackMode.FILE_WATCHER;
                            disablePolling();
                        }
                    } catch (Exception e) {
                        error("An error occured attempting to retrieve the watch key or close the file watcher. " + e.getMessage(), e);
                    }
                }
                try {
                    processFileChanges(executor, file, outputDirectory, isDirectory, changeType);
                } catch(Exception e) {
                    debug(e);
                    error("Could not file process changes for " + file.getAbsolutePath() + ": " + e.getMessage());
                }
            }
        });
        return observer;
    }

    private void processJavaCompilation(File outputDirectory, File testOutputDirectory, final ThreadPoolExecutor executor,
            List<String> artifactPaths) throws IOException, PluginExecutionException {
        // process java source files if no changes detected after the compile wait time
        boolean processSources = System.currentTimeMillis() > lastJavaSourceChange + compileWaitMillis;
        boolean processTests = System.currentTimeMillis() > lastJavaTestChange + compileWaitMillis;
        if (processSources) {
            // delete before recompiling, so if a file is in both lists, its class will be
            // deleted then recompiled
            if (!deleteJavaSources.isEmpty()) {
                debug("Deleting Java source files: " + deleteJavaSources);
                for (File file : deleteJavaSources) {
                    deleteJavaFile(file, outputDirectory, this.sourceDirectory);
                }
            }
            if (!recompileJavaSources.isEmpty() || triggerJavaSourceRecompile) {
                // try to recompile java files that previously did not compile successfully
                if (!failedCompilationJavaSources.isEmpty()) {
                    recompileJavaSources.addAll(failedCompilationJavaSources);
                }
                if (recompileJavaSource(recompileJavaSources, artifactPaths, executor, outputDirectory,
                        testOutputDirectory)) {
                    // successful compilation so we can clear failedCompilation list
                    failedCompilationJavaSources.clear();
                } else {
                    failedCompilationJavaSources.addAll(recompileJavaSources);
                }
            }
            // additionally, process java test files if no changes detected after a
            // different timeout
            // (but source timeout takes precedence i.e. don't recompile tests if someone
            // keeps changing the source)
            if (processTests) {
                // delete before recompiling, so if a file is in both lists, its class will be
                // deleted then recompiled
                if (!deleteJavaTests.isEmpty()) {
                    debug("Deleting Java test files: " + deleteJavaTests);
                    for (File file : deleteJavaTests) {
                        deleteJavaFile(file, testOutputDirectory, this.testSourceDirectory);
                    }
                }
                if (!recompileJavaTests.isEmpty() || triggerJavaTestRecompile) {
                    debug("Recompiling Java test files: " + recompileJavaTests);
                    if (!failedCompilationJavaTests.isEmpty()) {
                        recompileJavaTests.addAll(failedCompilationJavaTests);
                    }
                    if (recompileJavaTest(recompileJavaTests, artifactPaths, executor, outputDirectory,
                            testOutputDirectory)) {
                        // successful compilation so we can clear failedCompilation list
                        failedCompilationJavaTests.clear();
                    } else {
                        failedCompilationJavaTests.addAll(recompileJavaTests);
                    }
                }
            }

            // run tests if files were deleted without any other changes, since
            // recompileJavaSource won't run (which normally handles tests)
            if (!deleteJavaSources.isEmpty() && recompileJavaSources.isEmpty()) {
                // run tests after waiting for app update since app changed
                int numApplicationUpdatedMessages = countApplicationUpdatedMessages();
                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
            } else if (processTests && !deleteJavaTests.isEmpty() && recompileJavaTests.isEmpty()) {
                // run all tests without waiting for app update since only tests changed
                runTestThread(false, executor, -1, false, false);
            }

            deleteJavaSources.clear();
            recompileJavaSources.clear();
            triggerJavaTestRecompile = false;
            triggerJavaSourceRecompile = false;

            if (processTests) {
                deleteJavaTests.clear();
                recompileJavaTests.clear();
            }
        }
    }
 
    private void checkStopDevMode(boolean skipOnRestart) throws PluginScenarioException {
        // stop dev mode if the server has been stopped by another process
        if (serverThread == null || serverThread.getState().equals(Thread.State.TERMINATED)) {
            // server is restarting if devStop was set to true and we have not called the shutdown hook
            boolean restarting = devStop.get() && !calledShutdownHook.get();
            if (skipOnRestart && restarting && !externalContainerShutdown.get()) {
                debug("Server is restarting. Allowing dev mode to continue.");
                return;
            }
            if (!devStop.get()) {
                // an external situation caused the server to stop
                if (container) {
                    throw new PluginScenarioException("The container has stopped. Exiting dev mode.");
                }
                throw new PluginScenarioException("The server has stopped. Exiting dev mode.");
            } else {
                // server was stopped by dev mode
                throw new PluginScenarioException();
            }
        }
    }

    private void initWatchLoop() throws IOException {
        recompileJavaSources = new HashSet<File>();
        recompileJavaTests = new HashSet<File>();
        deleteJavaSources = new HashSet<File>();
        deleteJavaTests = new HashSet<File>();
        failedCompilationJavaSources = new HashSet<File>();
        failedCompilationJavaTests = new HashSet<File>();
        lastJavaSourceChange = System.currentTimeMillis();
        lastJavaTestChange = System.currentTimeMillis();
        triggerJavaSourceRecompile = false;
        triggerJavaTestRecompile = false;

        // initial source and test compile
        if (this.sourceDirectory.exists()) {
            Collection<File> allJavaSources = FileUtils.listFiles(this.sourceDirectory.getCanonicalFile(),
                    new String[] { "java" }, true);
            recompileJavaSources.addAll(allJavaSources);
        }
        if (this.testSourceDirectory.exists()) {
            Collection<File> allJavaTestSources = FileUtils.listFiles(this.testSourceDirectory.getCanonicalFile(),
                    new String[] { "java" }, true);
            recompileJavaTests.addAll(allJavaTestSources);
        }
    }

    private void processFileChanges(
        final ThreadPoolExecutor executor, File fileChanged, File outputDirectory,
        boolean isDirectory, ChangeType changeType) throws IOException, PluginExecutionException {

        if (ignoreFileOrDir(fileChanged)) {
            // skip this file or directory, and continue to the next file or directory
            return;
        }

        debug("Processing file changes for " + fileChanged + ", change type " + changeType);

        Path srcPath = this.sourceDirectory.getCanonicalFile().toPath();
        Path testSrcPath = this.testSourceDirectory.getCanonicalFile().toPath();
        Path configPath = this.configDirectory.getCanonicalFile().toPath();

        Path directory = fileChanged.getParentFile().toPath();

        // resource file check
        File resourceParent = null;
        for (File resourceDir : resourceDirs) {
            if (directory.startsWith(resourceDir.getCanonicalFile().toPath())) {
                resourceParent = resourceDir;
                break;
            }
        }

        if (fileChanged.isDirectory()) {
            // if new directory added, watch the entire directory
            if (changeType == ChangeType.CREATE) {
                registerAll(fileChanged.toPath(), executor);
            }
            // otherwise if a directory was modified, just continue to the next entry
            // (if delete, can't tell if it was a directory since it doesn't exist anymore)
            return;
        }

        int numApplicationUpdatedMessages = countApplicationUpdatedMessages();

        // reset this property in case it had been set to true
        System.setProperty(SKIP_BETA_INSTALL_WARNING, Boolean.FALSE.toString());

        // src/main/java directory
        if (directory.startsWith(srcPath)) {
            ArrayList<File> javaFilesChanged = new ArrayList<File>();
            javaFilesChanged.add(fileChanged);
            if (fileChanged.exists() && fileChanged.getName().endsWith(".java")
                    && (changeType == ChangeType.MODIFY
                            || changeType == ChangeType.CREATE)) {
                debug("Java source file modified: " + fileChanged.getName()
                        + ". Adding to list for processing.");
                lastJavaSourceChange = System.currentTimeMillis();
                recompileJavaSources.add(fileChanged);
            } else if (changeType == ChangeType.DELETE) {
                debug("Java file deleted: " + fileChanged.getName()
                        + ". Adding to list for processing.");
                lastJavaSourceChange = System.currentTimeMillis();
                deleteJavaSources.add(fileChanged);
            }
        } else if (directory.startsWith(testSrcPath)) { // src/main/test
            ArrayList<File> javaFilesChanged = new ArrayList<File>();
            javaFilesChanged.add(fileChanged);
            if (fileChanged.exists() && fileChanged.getName().endsWith(".java")
                    && (changeType == ChangeType.MODIFY
                            || changeType == ChangeType.CREATE)) {
                debug("Java test file modified: " + fileChanged.getName()
                        + ". Adding to list for processing.");
                lastJavaTestChange = System.currentTimeMillis();
                recompileJavaTests.add(fileChanged);
            } else if (changeType == ChangeType.DELETE) {
                debug("Java test file deleted: " + fileChanged.getName()
                        + ". Adding to list for processing.");
                lastJavaTestChange = System.currentTimeMillis();
                deleteJavaTests.add(fileChanged);
            }
        } else if (directory.startsWith(configPath)
                && !isGeneratedConfigFile(fileChanged, configDirectory, serverDirectory)) { // config
                                                                                            // files
            if (fileChanged.exists() && (changeType == ChangeType.MODIFY
                    || changeType == ChangeType.CREATE)) {
                // suppress install feature warning - property must be set before calling copyConfigFolder
                System.setProperty(SKIP_BETA_INSTALL_WARNING, Boolean.TRUE.toString());
                copyConfigFolder(fileChanged, configDirectory, null);
                copyFile(fileChanged, configDirectory, serverDirectory, null);

                if (isDockerfileDirectoryChanged(serverDirectory, fileChanged)) {
                    untrackDockerfileDirectoriesAndRestart();
                } else {
                    if (changeType == ChangeType.CREATE) {
                        redeployApp();
                    }
                    if (fileChanged.getName().equals("server.env")) {
                        // re-enable debug variables in server.env
                        enableServerDebug(false);
                    } else if ((fileChanged.getName().equals("bootstrap.properties") && bootstrapPropertiesFileParent == null)
                         || (fileChanged.getName().equals("jvm.options") && jvmOptionsFileParent == null)) {
                        // restart server to load new properties
                        restartServer(false);
                    }
                }
                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
            } else if (changeType == ChangeType.DELETE) {
                info("Config file deleted: " + fileChanged.getName());
                deleteFile(fileChanged, configDirectory, serverDirectory, null);
                if (isDockerfileDirectoryChanged(serverDirectory, fileChanged)) {
                    untrackDockerfileDirectoriesAndRestart();
                } else {
                    if (fileChanged.getName().equals("server.env")) {
                        // re-enable debug variables in server.env
                        enableServerDebug(false);
                    }
                    if (container && OSUtil.isLinux()) {
                        info("Restarting the container for this change to take effect.");
                        // Allow a 1 second grace period to replace the file in case the user changes the file with a script or a tool like vim.
                        try {
                            Thread.sleep(1000);
                        } catch (InterruptedException e) {
                            debug("Unexpected InterruptedException handling config file deletion.", e);
                        }
                        restartServer(false);
                    }    
                }
                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
            }
        } else if (serverXmlFileParent != null
                && directory.equals(serverXmlFileParent.getCanonicalFile().toPath())
                && fileChanged.getCanonicalPath().endsWith(serverXmlFile.getName())) {
            if (fileChanged.exists() && (changeType == ChangeType.MODIFY || changeType == ChangeType.CREATE)) {
                // suppress install feature warning - property must be set before calling copyConfigFolder
                System.setProperty(SKIP_BETA_INSTALL_WARNING, Boolean.TRUE.toString());
                copyConfigFolder(fileChanged, serverXmlFileParent, "server.xml");
                copyFile(fileChanged, serverXmlFileParent, serverDirectory, "server.xml");
                if (isDockerfileDirectoryChanged(serverDirectory, fileChanged)) {
                    untrackDockerfileDirectoriesAndRestart();
                } else if (changeType == ChangeType.CREATE) {
                    redeployApp();
                }
                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);

            } else if (changeType == ChangeType.DELETE) {
                info("Config file deleted: " + fileChanged.getName());
                deleteFile(fileChanged, configDirectory, serverDirectory, "server.xml");
                // Let this restart if needed for container mode.  Otherwise, nothing else needs to be done for config file delete.
                if (isDockerfileDirectoryChanged(serverDirectory, fileChanged)) {
                    untrackDockerfileDirectoriesAndRestart();
                }
                runTestThread(true, executor, numApplicationUpdatedMessages, true, false);
            }
        } else if (bootstrapPropertiesFileParent != null
                   && directory.equals(bootstrapPropertiesFileParent.getCanonicalFile().toPath())
                   && fileChanged.getCanonicalPath().endsWith(bootstrapPropertiesFile.getName())) {
            // This is for bootstrap.properties outside of the config folder
            // restart server to load new properties
            if (isDockerfileDirectoryChanged(fileChanged)) {
                untrackDockerfileDirectoriesAndRestart();
            } else {
                restartServer(false);
            }
        } else if (jvmOptionsFileParent != null
                && directory.equals(jvmOptionsFileParent.getCanonicalFile().toPath())
                && fileChanged.getCanonicalPath().endsWith(jvmOptionsFile.getName())) {
            // This is for jvm.options outside of the config folder
            // restart server to load new options
            if (isDockerfileDirectoryChanged(fileChanged)) {
                untrackDockerfileDirectoriesAndRestart();
            } else {
                restartServer(false);
            }
        } else if (resourceParent != null
                && directory.startsWith(resourceParent.getCanonicalFile().toPath())) { // resources
            debug("Resource dir: " + resourceParent.toString());
            if (fileChanged.exists() && (changeType == ChangeType.MODIFY
                    || changeType == ChangeType.CREATE)) {
                copyFile(fileChanged, resourceParent, outputDirectory, null);

                // run all tests on resource change
                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
            } else if (changeType == ChangeType.DELETE) {
                debug("Resource file deleted: " + fileChanged.getName());
                deleteFile(fileChanged, resourceParent, outputDirectory, null);
                // run all tests on resource change
                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
            }
        } else if (fileChanged.equals(buildFile)
                && directory.startsWith(buildFile.getParentFile().getCanonicalFile().toPath())
                && changeType == ChangeType.MODIFY) { // pom.xml

            boolean recompiledBuild = recompileBuildFile(buildFile, artifactPaths, executor);
            // run all tests on build file change
            if (recompiledBuild) {
                // trigger java source recompile if there are compilation errors
                if (!failedCompilationJavaSources.isEmpty()) {
                    triggerJavaSourceRecompile = true;
                }
                // trigger java test recompile if there are compilation errors
                if (!failedCompilationJavaTests.isEmpty()) {
                    triggerJavaTestRecompile = true;
                }
                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
            }
        } else if (fileChanged.equals(dockerfileUsed)
                && directory.startsWith(dockerfileUsed.getParentFile().getCanonicalFile().toPath())
                && changeType == ChangeType.MODIFY) { // dockerfile
            untrackDockerfileDirectoriesAndRestart(); // untrack all Dockerfile directories, then rebuild container and restart
        } else if (propertyFilesMap != null && propertyFilesMap.keySet().contains(fileChanged)) { // properties file
            boolean reloadedPropertyFile = reloadPropertyFile(fileChanged);
            // run all tests on properties file change
            if (reloadedPropertyFile) {
                runTestThread(true, executor, numApplicationUpdatedMessages, false, false);
            }
        } else if (isDockerfileDirectoryChanged(fileChanged)) {
            // If contents within a directory specified in a Dockerfile COPY command were changed, and not already processed by one of the other conditions above.
            untrackDockerfileDirectoriesAndRestart();
        }
    }

    /**
     * Unwatches all directories that were specified in Dockerfile COPY commands, then does a container
     * rebuild and restart.
     * 
     * @throws PluginExecutionException
     */
    private void untrackDockerfileDirectoriesAndRestart() throws PluginExecutionException {
        // Cancel and clear any WatchKeys that were added for to the Dockerfile directories
        for (WatchKey key : dockerfileDirectoriesWatchKeys) {
            key.cancel();
        }
        dockerfileDirectoriesWatchKeys.clear();

        // Cancel and clear any FileAlterationObservers that were added for the Dockerfile directories
        synchronized (cancelledFileObservers) {
            for (FileAlterationObserver observer : dockerfileDirectoriesFileObservers) {
                // add the observer to be cancelled 
                cancelledFileObservers.add(observer);
                try {
                    // destroy the observer
                    observer.destroy();
                } catch (Exception e) {
                    debug("Could not destroy file observer", e);
                }
            }
        }
        dockerfileDirectoriesFileObservers.clear();

        // Untrack the directories
        dockerfileDirectoriesTracked.clear();

        restartServer(true);
    }

    /**
     * If container mode, check if any of the files are within a directory specified in one of the Dockerfile's
     * COPY commands.  If not container mode, does nothing.
     * 
     * @param file The files to check, in the same order.
     * @return true if container mode and any of the files are within a directory specified in one of the Dockerfile's COPY commands.
     * @throws IOException if there was an error getting canonical paths
     */
    private boolean isDockerfileDirectoryChanged(File... files) throws IOException {
        // Check for directory content changes from directories specified in Dockerfile
        if (container && !dockerfileDirectoriesTracked.isEmpty()) {
            for (Path trackedPath : dockerfileDirectoriesTracked) {
                Path logsPath = new File(serverDirectory, "logs").getCanonicalFile().toPath();

                for (File file : files) {
                    // if the file's path is a child of the tracked path, except for the server logs folder or if it's the loose application itself
                    Path filePath = file.getCanonicalFile().toPath();
                    if (filePath.startsWith(trackedPath) && !filePath.startsWith(logsPath) && !filePath.toString().endsWith(".war.xml")) {
                        debug("isDockerfileDirectoryChanged=true for directory " + trackedPath + " with file " + file);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Determines if the corresponding target config file was generated by a Liberty
     * plugin
     * 
     * @param fileChanged the file that was changed
     * @param srcDir      the directory of the file changed
     * @param targetDir   the target directory
     * @throws IOException unable to resolve canonical path
     */
    protected boolean isGeneratedConfigFile(File fileChanged, File srcDir, File targetDir) throws IOException {
        return (fileChanged.getName().equals("bootstrap.properties") || fileChanged.getName().equals("jvm.options"))
                && isGeneratedTargetFile(fileChanged, srcDir, targetDir);
    }

    private boolean isGeneratedTargetFile(File fileChanged, File srcDir, File targetDir) throws IOException {
        File targetFile = getTargetFile(fileChanged, srcDir, targetDir, null);
        try (FileReader fileReader = new FileReader(targetFile);
                BufferedReader bufferedReader = new BufferedReader(fileReader)) {
            String line = bufferedReader.readLine();
            return line == null ? false : line.matches(GENERATED_HEADER_REGEX);
        } catch (IOException e) {
            // If the target file could not be read, assume it was not a generated file
            debug("Could not read the target file " + targetFile + ". It will be replaced by the contents of "
                    + fileChanged, e);
        }
        return false;
    }

    /**
     * Reads the file to a String
     * 
     * @param file
     * @return String representation of the file
     * @throws IOException unable to read file to string
     */
    public String readFile(File file) throws IOException {
        return FileUtils.readFileToString(file, StandardCharsets.UTF_8);
    }

    /**
     * Creates a temporary copy of the configuration file and checks the configFile
     * in the temporary directory to avoid install-feature timing issues
     * 
     * @param fileChanged    the file that was changed
     * @param srcDir         the directory of the file changed
     * @param targetFileName if not null renames the fileChanged to targetFileName
     *                       in the targetDir
     * @throws IOException creating and copying to tempConfig directory
     */
    public void copyConfigFolder(File fileChanged, File srcDir, String targetFileName) throws IOException {
        this.tempConfigPath = Files.createTempDirectory("tempConfig");
        File tempConfig = tempConfigPath.toFile();
        debug("Temporary configuration folder created: " + tempConfig);

        FileUtils.copyDirectory(serverDirectory, tempConfig, new FileFilter() {
            public boolean accept(File pathname) {
                String name = pathname.getName();
                // skip:
                // - ignore list
                // - workarea and logs dirs from the server directory, since those can be
                // changing
                boolean skip = ignoreFileOrDir(pathname)
                        || (pathname.isDirectory() && (name.equals("workarea") || name.equals("logs")));
                return !skip;
            }
        }, true);
        copyFile(fileChanged, srcDir, tempConfig, targetFileName);
        checkConfigFile(fileChanged, tempConfig);
        cleanUpTempConfig();
    }

    /**
     * Whether dev mode should ignore a file or directory.
     * 
     * @param file File or directory
     * @return true if the file or directory should be ignored, false otherwise
     */
    private boolean ignoreFileOrDir(File file) {
        String name = file.getName();
        if (file.isDirectory()) {
            for (String prefix : IGNORE_DIRECTORY_PREFIXES) {
                if (name.startsWith(prefix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }
        } else {
            for (String prefix : IGNORE_FILE_PREFIXES) {
                if (name.startsWith(prefix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }
            for (String postfix : IGNORE_FILE_POSTFIXES) {
                if (name.endsWith(postfix)) {
                    debug("Ignoring " + name);
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Copies the fileChanged from the srcDir to the targetDir.
     * 
     * @param fileChanged    the file that was changed
     * @param srcDir         the directory of the file changed
     * @param targetDir      the target directory
     * @param targetFileName if not null renames the fileChanged to targetFileName
     *                       in the targetDir
     * @throws IOException unable to resolve canonical path
     */
    public void copyFile(File fileChanged, File srcDir, File targetDir, String targetFileName) throws IOException {
        File targetResource = getTargetFile(fileChanged, srcDir, targetDir, targetFileName);

        try {
            FileUtils.copyFile(fileChanged, targetResource);
            info("Copied file: " + fileChanged.getCanonicalPath() + " to: " + targetResource.getCanonicalPath());
        } catch (FileNotFoundException ex) {
            debug("Failed to copy file: " + fileChanged.getCanonicalPath());
        } catch (Exception ex) {
            debug(ex);
        }
    }

    private File getTargetFile(File fileChanged, File srcDir, File targetDir, String targetFileName)
            throws IOException {
        String relPath = fileChanged.getCanonicalPath().substring(
                fileChanged.getCanonicalPath().indexOf(srcDir.getCanonicalPath()) + srcDir.getCanonicalPath().length());
        if (targetFileName != null) {
            relPath = relPath.substring(0, relPath.indexOf(fileChanged.getName())) + targetFileName;
        }
        File targetResource = new File(targetDir.getCanonicalPath() + relPath);
        return targetResource;
    }

    /**
     * Deletes the corresponding file in the targetDir.
     * 
     * @param deletedFile    the file that was deleted
     * @param dir            the directory of the deletedFile
     * @param targetDir      the corresponding targetDir of the deletedFile
     * @param targetFileName if not null deletes the targetFile with this name
     * @throws IOException unable to resolve canonical path
     */
    protected void deleteFile(File deletedFile, File dir, File targetDir, String targetFileName) throws IOException {
        File targetFile = getTargetFile(deletedFile, dir, targetDir, targetFileName);
        if (targetFile.exists()) {
            if (targetFile.isDirectory()) {
                try {
                    FileUtils.deleteDirectory(targetFile);
                    info("The directory " + targetFile.getCanonicalPath() + " was deleted.");
                } catch (IllegalArgumentException e) {
                    debug("Could not delete the directory " + targetFile.getCanonicalPath() + ". " + e.getMessage());
                } catch (IOException e) {
                    error("An error encountered while deleting the directory " + targetFile.getCanonicalPath()
                            + ". " + e.getMessage());
                }
            } else {
                if (targetFile.delete()) {
                    info("The file " + targetFile.getCanonicalPath() + " was deleted.");
                } else {
                    error("Could not delete the file " + targetFile.getCanonicalPath() + ".");
                }
            }
        }
    }

    /**
     * Delete all the Java class files within the specified directory. If the
     * directory is empty, deletes the directory as well.
     * 
     * @param outputDirectory the directory for compiled classes
     */
    protected void cleanTargetDir(File outputDirectory) {
        File[] fList = outputDirectory.listFiles();
        if (fList != null) {
            for (File file : fList) {
                if (file.isFile() && file.getName().toLowerCase().endsWith(".class")) {
                    file.delete();
                    info("Deleted Java class file: " + file);
                } else if (file.isDirectory()) {
                    cleanTargetDir(file);
                }
            }
        }
        if (outputDirectory.listFiles().length == 0) {
            outputDirectory.delete();
        }
    }

    private enum ChangeType {
        CREATE,
        DELETE,
        MODIFY
    };

    /**
     * Register the parent directory and all sub-directories with the WatchService
     * 
     * @param start   parent directory
     * @param executor the test thread executor
     * @throws IOException unable to walk through file tree
     */
    protected void registerAll(final Path start, final ThreadPoolExecutor executor) throws IOException {
        registerAll(start, executor, false);
    }

    /**
     * Register the parent directory and all sub-directories with the WatchService
     * 
     * @param start   parent directory
     * @param executor the test thread executor
     * @param removeOnContainerRebuild whether the files should be unwatched if the container is rebuilt
     * @throws IOException unable to walk through file tree
     */
    protected void registerAll(final Path start, final ThreadPoolExecutor executor, final boolean removeOnContainerRebuild) throws IOException {
        debug("Registering all files in directory: " + start.toString());

        // register directory and sub-directories
        Files.walkFileTree(start, new SimpleFileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(final Path dir, BasicFileAttributes attrs) throws IOException {
                if (trackingMode == FileTrackMode.POLLING || trackingMode == FileTrackMode.NOT_SET) {
                    // synchronize on the new observer set since only those are being updated in separate threads
                    synchronized (newFileObservers) {
                        Set<FileAlterationObserver> tempCombinedObservers = new HashSet<FileAlterationObserver>();
                        tempCombinedObservers.addAll(fileObservers);
                        tempCombinedObservers.addAll(newFileObservers);
        
                        // if this path is already observed, ignore it
                        for (FileAlterationObserver observer : tempCombinedObservers) {
                            if (dir.equals(observer.getDirectory().getCanonicalFile().toPath())) {
                                debug("Skipping subdirectory " + dir.toString() + " since it already being observed");
                                return FileVisitResult.CONTINUE;
                            }
                        }
        
                        FileFilter singleDirectoryFilter = new FileFilter() {
                            @Override
                            public boolean accept(File file) {
                                try {
                                    // if it's a direct child of this path
                                    if (dir.equals(file.getParentFile().getCanonicalFile().toPath())) {
                                        return true;
                                    }
                                } catch (IOException e) {
                                    return false;
                                }
                                return false;
                            }
                        };
            
                        try {
                            debug("Adding subdirectory to file observers: " + dir.toString());
                            FileAlterationObserver observer = addFileAlterationObserver(executor, dir.toString(), singleDirectoryFilter);
                            if (removeOnContainerRebuild) {
                                debug("Adding to dockerfileDirectoriesFileObservers: " + dir);
                                dockerfileDirectoriesFileObservers.add(observer);
                            }
                        } catch (Exception e) {
                            error("Could not observe directory " + dir.toString(), e);
                        }
                    }
                } 
                if (trackingMode == FileTrackMode.FILE_WATCHER || trackingMode == FileTrackMode.NOT_SET) {
                    debug("Adding subdirectory to WatchService: " + dir.toString());
                    WatchKey key = dir.register(watcher,
                            new WatchEvent.Kind[] { StandardWatchEventKinds.ENTRY_MODIFY,
                                    StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_CREATE },
                            SensitivityWatchEventModifier.HIGH);
                    if (removeOnContainerRebuild) {
                        debug("Adding to dockerfileDirectoriesWatchKeys: " + dir);
                        dockerfileDirectoriesWatchKeys.add(key);
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });

    }

    /**
     * Get the file from the configDirectory if it exists
     * 
     * @param file 
     * @return file or null if it does not exist
     */
    protected File getFileFromConfigDirectory(String file) {
        File f = new File(configDirectory, file);
        if (configDirectory != null && f.exists()) {
            return f;
        }
        return null;
    }

    /**
     * Given the fileChanged delete the corresponding Java class or directory
     * 
     * @param fileChanged       Java file changed
     * @param classesDir        the directory for compiled classes
     * @param compileSourceRoot the source directory for the Java classes
     * @throws IOException unable to resolve canonical path
     */
    protected void deleteJavaFile(File fileChanged, File classesDir, File compileSourceRoot) throws IOException {
        String fileName = fileChanged.getName();
        File parentFile = fileChanged.getParentFile();

        boolean javaFile = fileName.endsWith(".java");
        String relPath;
        if (javaFile) {
            fileName = fileName.substring(0, fileChanged.getName().indexOf(".java"));
            relPath = parentFile.getCanonicalPath()
                    .substring(parentFile.getCanonicalPath().indexOf(compileSourceRoot.getCanonicalPath())
                            + compileSourceRoot.getCanonicalPath().length())
                    + "/" + fileName + ".class";
        } else {
            relPath = parentFile.getCanonicalPath()
                    .substring(parentFile.getCanonicalPath().indexOf(compileSourceRoot.getCanonicalPath())
                            + compileSourceRoot.getCanonicalPath().length())
                    + "/" + fileName;
        }

        File targetFile = new File(classesDir.getCanonicalPath() + relPath);
        if (targetFile.exists()) {
            if (targetFile.isDirectory()) {
                try {
                    FileUtils.deleteDirectory(targetFile);
                    info("The target directory " + targetFile.getCanonicalPath() + " was deleted.");
                } catch (IllegalArgumentException e) {
                    debug("Could not delete directory " + targetFile.getCanonicalPath() + ". " + e.getMessage());
                } catch (IOException e) {
                    error("There was an error encountered while deleting the directory " + targetFile.getCanonicalPath()
                            + ". " + e.getMessage());
                }
            } else {
                if (targetFile.delete()) {
                    info("The java class " + targetFile.getCanonicalPath() + " was deleted.");
                } else {
                    error("Could not delete the file " + targetFile.getCanonicalPath() + ". ");
                }
            }
        } else {
            warn("File deleted but could not find corresponding file or folder in the target directory: "
                    + fileChanged.getCanonicalPath() + ".");
        }
    }

    /**
     * Recompile Java source files and run tests after application update
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJavaSource(Collection<File> javaFilesChanged, List<String> artifactPaths,
            ThreadPoolExecutor executor, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        return recompileJava(javaFilesChanged, artifactPaths, executor, false, outputDirectory, testOutputDirectory);
    }

    /**
     * Recompile test source files and run tests immediately
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJavaTest(Collection<File> javaFilesChanged, List<String> artifactPaths,
            ThreadPoolExecutor executor, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        return recompileJava(javaFilesChanged, artifactPaths, executor, true, outputDirectory, testOutputDirectory);
    }

    /**
     * Recompile source files
     * 
     * @param javaFilesChanged collection of Java files changed
     * @param artifactPaths list of project artifact paths for building the classpath
     * @param executor the test thread executor
     * @param tests indicates whether the files changed were test files
     * @param outputDirectory the directory for compiled classes
     * @param testOutputDirectory the directory for compiled test classes
     * @throws PluginExecutionException if the classes output directory doesn't exist and can't be created
     */
    protected boolean recompileJava(Collection<File> javaFilesChanged, List<String> artifactPaths, ThreadPoolExecutor executor,
            boolean tests, File outputDirectory, File testOutputDirectory) throws PluginExecutionException {
        try {
            int messageOccurrences = countApplicationUpdatedMessages();
            boolean compileResult;
            
            if (useBuildRecompile) {
                compileResult = compile(tests ? testSourceDirectory : sourceDirectory);
            } else {
                // source root is src/main/java or src/test/java
                File classesDir = tests ? testOutputDirectory : outputDirectory;
                if (!classesDir.exists()) {
                    if (!classesDir.mkdirs()) {
                        throw new PluginExecutionException("The classes output directory " + classesDir.getAbsolutePath()
                                + " does not exist and cannot be created.");
                    } else if (classesDir.exists() && Objects.equals(classesDir.getCanonicalFile(), outputDirectory.getCanonicalFile())) {
                        // redeploy application when class directory has been created
                        redeployApp();
                    }
                }

                List<String> combinedCompilerOptions = new ArrayList<>(Arrays.asList(DEFAULT_COMPILER_OPTIONS));
                if (compilerOptions != null) {
                    combinedCompilerOptions.addAll(compilerOptions.getOptions());
                }
                debug("Compiler options: " + combinedCompilerOptions);

                List<File> outputDirs = new ArrayList<File>();

                if (tests) {
                    outputDirs.add(outputDirectory);
                    outputDirs.add(testOutputDirectory);
                } else {
                    outputDirs.add(outputDirectory);
                }
                Set<File> classPathElems = getClassPath(artifactPaths, outputDirs);

                JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
                StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);

                fileManager.setLocation(StandardLocation.CLASS_PATH, classPathElems);
                fileManager.setLocation(StandardLocation.CLASS_OUTPUT, Collections.singleton(classesDir));

                Collection<JavaFileObject> compilationUnits = new HashSet<JavaFileObject>();
                for (File file : javaFilesChanged) {
                    if (file.exists() && file.isFile()) {
                        for (JavaFileObject o : fileManager.getJavaFileObjects(file)) {
                            compilationUnits.add(o);
                        }    
                    } else {
                        debug("The Java file " + file + " does not exist and will not be compiled.");
                    }
                }

                JavaCompiler.CompilationTask task = compiler.getTask(null, fileManager, null, combinedCompilerOptions, null,
                        compilationUnits);

                compileResult = task.call();
            }
            if (compileResult) {
                if (tests) {
                    info("Tests compilation was successful.");
                } else {
                    // redeploy app after compilation if not loose application
                    if (!isLooseApplication()) {
                        redeployApp();
                    }

                    info("Source compilation was successful.");
                }

                // run tests after successful compile
                if (tests) {
                    // if only tests were compiled, don't need to wait for
                    // app to update
                    runTestThread(false, executor, -1, false, false);
                } else {
                    runTestThread(true, executor, messageOccurrences, false, false);
                }
                return true;
            } else {
                if (tests) {
                    info("Tests compilation had errors.");
                } else {
                    info("Source compilation had errors.");
                }
                return false;
            }
        } catch (Exception e) {
            error("Error compiling Java files: " + e.getMessage());
            debug(e);
            return false;
        }
    }

    /**
     * Gets the class path for the specified artifactPaths and outputDirs.
     * 
     * @param artifactPaths list of artifacts for the current project
     * @param outputDirs list of output directories for the current project
     * @return set of classpath files
     * @throws IOException unable to resolve canonical path
     */
    protected Set<File> getClassPath(List<String> artifactPaths, List<File> outputDirs) throws IOException {
        Set<String> parsedFiles = new HashSet<>();
        Deque<String> toParse = new ArrayDeque<>();

        for (String artifactPath : artifactPaths) {
            toParse.add(new File(artifactPath).getCanonicalPath());
        }

        Set<File> classPathElements = new HashSet<>();
        classPathElements.addAll(outputDirs);
        while (!toParse.isEmpty()) {
            String s = toParse.poll();
            if (!parsedFiles.contains(s)) {
                parsedFiles.add(s);
                File file = new File(s);
                if (file.exists() && file.getName().endsWith(".jar")) {
                    classPathElements.add(file);
                    if (!file.isDirectory()) {
                        try (JarFile jar = new JarFile(file)) {
                            Manifest mf = jar.getManifest();
                            if (mf == null || mf.getMainAttributes() == null) {
                                continue;
                            }
                            Object classPath = mf.getMainAttributes().get(Attributes.Name.CLASS_PATH);
                            if (classPath != null) {
                                for (String i : classPath.toString().split(" ")) {
                                    File f;
                                    try {
                                        URL u = new URL(i);
                                        f = new File(u.getPath());
                                    } catch (MalformedURLException e) {
                                        f = new File(file.getParentFile(), i);
                                    }
                                    if (f.exists()) {
                                        toParse.add(f.getCanonicalPath());
                                    }
                                }
                            }
                        } catch (Exception e) {
                            throw new RuntimeException("Failed to open class path file " + file, e);
                        }
                    }
                }
            }
        }
        return classPathElements;
    }

    /**
     * Run tests in a new thread.
     * 
     * @param waitForApplicationUpdate whether it should wait for the application to
     *                                 update before running integration tests
     * @param executor                 the thread pool executor
     * @param messageOccurrences       how many times the application updated
     *                                 message has occurred in the log
     * @param forceSkipUTs             whether to force skip the unit tests
     * @param manualInvocation         whether the tests were manually invoked
     */
    public void runTestThread(boolean waitForApplicationUpdate, ThreadPoolExecutor executor, int messageOccurrences,
            boolean forceSkipUTs, boolean manualInvocation) {
        try {
            if (manualInvocation || hotTests) {
                executor.execute(new TestJob(waitForApplicationUpdate, messageOccurrences, executor, forceSkipUTs,
                        manualInvocation));
            }
        } catch (RejectedExecutionException e) {
            debug("Cannot add thread since max threads reached", e);
        }
    }

    public class TestJob implements Runnable {
        private boolean waitForApplicationUpdate;
        private int messageOccurrences;
        private ThreadPoolExecutor executor;
        private boolean forceSkipUTs;
        private boolean manualInvocation;

        public TestJob(boolean waitForApplicationUpdate, int messageOccurrences, ThreadPoolExecutor executor, boolean forceSkipUTs, boolean manualInvocation) {
            this.waitForApplicationUpdate = waitForApplicationUpdate;
            this.messageOccurrences = messageOccurrences;
            this.executor = executor;
            this.forceSkipUTs = forceSkipUTs;
            this.manualInvocation = manualInvocation;
        }

        @Override
        public void run() {
            try {
                runTests(waitForApplicationUpdate, messageOccurrences, executor, forceSkipUTs);
            } finally {
                // start watching for hotkey presses if not already started, or re-print message if thread already running
                runHotkeyReaderThread(executor);
            }
        }

        public boolean isManualInvocation() {
            return manualInvocation;
        }
    }

    /**
     * Gets the Liberty server's host name.
     * @return hostName the host name, or null if the server is not started
     */
    public String getHostName() {
        return hostName;
    }

    /**
     * Gets the Liberty server's http port.
     * @return httpPort the http port, or null if the server is not started or there is no http port bound
     */
    public String getHttpPort() {
        return httpPort;
    }

    /**
     * Gets the Liberty server's https port.
     * @return httpsPort the https port, or null if the server is not started or there is no https port bound
     */
    public String getHttpsPort() {
        return httpsPort;
    }

    /**
     * Sets the preferred debug port.
     * 
     * @param libertyDebugPort the preferred debug port
     */
    public void setLibertyDebugPort(int libertyDebugPort) {
        this.libertyDebugPort = libertyDebugPort;
    }

    /**
     * Reload the property file by restarting the server if there were changes.
     * 
     * @param propertyFile The property file that was changed.
     * @throws PluginExecutionException if there was an error when reloading the file
     * @return true if the property file was reloaded with changes
     */
    private boolean reloadPropertyFile(File propertyFile) throws PluginExecutionException {
        Properties properties = readPropertiesFromFile(propertyFile);
        if (!Objects.equals(properties, propertyFilesMap.get(propertyFile))) {
            debug("Properties file " + propertyFile.getAbsolutePath() + " has changed. Restarting server...");
            propertyFilesMap.put(propertyFile, properties);    
            restartServer();
            return true;
        } else {
            debug("No changes detected in properties file " + propertyFile.getAbsolutePath());
            return false;
        }
    }

    /**
     * This is needed for Gradle only.
     * 
     * Sets additional property files that may be used by the build.
     * Loads the properties for later comparison of changes.
     * 
     * @param propertyFiles list of property files
     */
    public void setPropertyFiles(List<File> propertyFiles) {
        if (propertyFiles == null) {
            return;
        }
        if (propertyFilesMap == null) {
            propertyFilesMap = new HashMap<File, Properties>(propertyFiles.size());
        }
        for (File propertyFile : propertyFiles) {
            Properties properties = readPropertiesFromFile(propertyFile);
            propertyFilesMap.put(propertyFile, properties);
        }
    }

    /**
     * Read properties from file.  If file does not exist or an IO exception occurred, returns null.
     */
    private Properties readPropertiesFromFile(File propertyFile) {
        Properties properties = null;
        if (propertyFile.exists()) {
            InputStream inputStream = null;
            try {
                debug("Loading properties from file: " + propertyFile);
                inputStream = new FileInputStream(propertyFile);
                properties = new Properties();
                properties.load(inputStream);
            } catch (IOException e) {
                error("Could not read properties file " + propertyFile.getAbsolutePath(), e);
            } finally {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (IOException e) {
                        // nothing to do
                    }
                }
            }
        }
        return properties;
    }

    public String getContainerName() {
        return containerName;
    }

}
