/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.scheduler.runtime;

import com.cronutils.model.Cron;
import com.cronutils.model.CronType;
import com.cronutils.model.definition.CronDefinition;
import com.cronutils.model.definition.CronDefinitionBuilder;
import com.cronutils.model.time.ExecutionTime;
import com.cronutils.parser.CronParser;
import io.quarkus.runtime.StartupEvent;
import io.quarkus.scheduler.Scheduled;
import io.quarkus.scheduler.ScheduledExecution;
import io.quarkus.scheduler.Scheduler;
import io.quarkus.scheduler.SkippedExecution;
import io.quarkus.scheduler.Trigger;
import io.quarkus.scheduler.runtime.ScheduledInvoker;
import io.quarkus.scheduler.runtime.ScheduledMethodMetadata;
import io.quarkus.scheduler.runtime.SchedulerContext;
import io.quarkus.scheduler.runtime.SchedulerRuntimeConfig;
import io.quarkus.scheduler.runtime.SkipConcurrentExecutionInvoker;
import io.quarkus.scheduler.runtime.util.SchedulerUtils;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import javax.annotation.Priority;
import javax.enterprise.event.Event;
import javax.enterprise.event.Observes;
import javax.enterprise.inject.Typed;
import javax.inject.Singleton;
import org.jboss.logging.Logger;
import org.jboss.threads.JBossScheduledThreadPoolExecutor;

@Typed(value={Scheduler.class})
@Singleton
public class SimpleScheduler
implements Scheduler {
    private static final Logger LOG = Logger.getLogger(SimpleScheduler.class);
    private static final long CHECK_PERIOD = 1000L;
    private final ScheduledExecutorService scheduledExecutor;
    private final ExecutorService executor;
    private volatile boolean running = true;
    private final List<ScheduledTask> scheduledTasks;
    private final boolean enabled;

    public SimpleScheduler(SchedulerContext context, SchedulerRuntimeConfig schedulerRuntimeConfig, Event<SkippedExecution> skippedExecutionEvent) {
        this.enabled = schedulerRuntimeConfig.enabled;
        this.scheduledTasks = new ArrayList<ScheduledTask>();
        this.executor = context.getExecutor();
        if (!schedulerRuntimeConfig.enabled) {
            this.scheduledExecutor = null;
            LOG.info((Object)"Simple scheduler is disabled by config property and will not be started");
        } else if (context.getScheduledMethods().isEmpty()) {
            this.scheduledExecutor = null;
            LOG.info((Object)"No scheduled business methods found - Simple scheduler will not be started");
        } else {
            this.scheduledExecutor = new JBossScheduledThreadPoolExecutor(1, new Runnable(){

                @Override
                public void run() {
                }
            });
            CronDefinition definition = CronDefinitionBuilder.instanceDefinitionFor((CronType)context.getCronType());
            CronParser parser = new CronParser(definition);
            for (ScheduledMethodMetadata method : context.getScheduledMethods()) {
                int nameSequence = 0;
                for (Scheduled scheduled : method.getSchedules()) {
                    Optional<SimpleTrigger> trigger = this.createTrigger(method.getInvokerClassName(), parser, scheduled, ++nameSequence);
                    if (!trigger.isPresent()) continue;
                    ScheduledInvoker invoker = context.createInvoker(method.getInvokerClassName());
                    if (scheduled.concurrentExecution() == Scheduled.ConcurrentExecution.SKIP) {
                        invoker = new SkipConcurrentExecutionInvoker(invoker, skippedExecutionEvent);
                    }
                    this.scheduledTasks.add(new ScheduledTask(trigger.get(), invoker));
                }
            }
        }
    }

    void start(@Observes @Priority(value=0) StartupEvent event) {
        if (this.scheduledExecutor == null) {
            return;
        }
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime trunc = now.plusSeconds(1L).truncatedTo(ChronoUnit.SECONDS);
        this.scheduledExecutor.scheduleAtFixedRate(this::checkTriggers, ChronoUnit.MILLIS.between(now, trunc), 1000L, TimeUnit.MILLISECONDS);
    }

    @PreDestroy
    void stop() {
        try {
            if (this.scheduledExecutor != null) {
                this.scheduledExecutor.shutdownNow();
            }
        }
        catch (Exception e) {
            LOG.warn((Object)"Unable to shutdown the scheduler executor", (Throwable)e);
        }
    }

    void checkTriggers() {
        if (!this.running) {
            LOG.trace((Object)"Skip all triggers - scheduler paused");
            return;
        }
        ZonedDateTime now = ZonedDateTime.now();
        LOG.tracef("Check triggers at %s", (Object)now);
        for (ScheduledTask task : this.scheduledTasks) {
            task.execute(now, this.executor);
        }
    }

    @Override
    public void pause() {
        if (!this.enabled) {
            LOG.warn((Object)"Scheduler is disabled and cannot be paused");
        } else {
            this.running = false;
        }
    }

    @Override
    public void pause(String identity) {
        Objects.requireNonNull(identity, "Cannot pause - identity is null");
        if (identity.isEmpty()) {
            LOG.warn((Object)"Cannot pause - identity is empty");
            return;
        }
        String parsedIdentity = SchedulerUtils.lookUpPropertyValue(identity);
        for (ScheduledTask task : this.scheduledTasks) {
            if (!parsedIdentity.equals(task.trigger.id)) continue;
            task.trigger.setRunning(false);
            return;
        }
    }

    @Override
    public void resume() {
        if (!this.enabled) {
            LOG.warn((Object)"Scheduler is disabled and cannot be resumed");
        } else {
            this.running = true;
        }
    }

    @Override
    public void resume(String identity) {
        Objects.requireNonNull(identity, "Cannot resume - identity is null");
        if (identity.isEmpty()) {
            LOG.warn((Object)"Cannot resume - identity is empty");
            return;
        }
        String parsedIdentity = SchedulerUtils.lookUpPropertyValue(identity);
        for (ScheduledTask task : this.scheduledTasks) {
            if (!parsedIdentity.equals(task.trigger.id)) continue;
            task.trigger.setRunning(true);
            return;
        }
    }

    @Override
    public boolean isRunning() {
        return this.enabled && this.running;
    }

    Optional<SimpleTrigger> createTrigger(String invokerClass, CronParser parser, Scheduled scheduled, int nameSequence) {
        String cron;
        Object id = SchedulerUtils.lookUpPropertyValue(scheduled.identity());
        if (((String)id).isEmpty()) {
            id = nameSequence + "_" + invokerClass;
        }
        ZonedDateTime start = ZonedDateTime.now().truncatedTo(ChronoUnit.SECONDS);
        Long millisToAdd = null;
        if (scheduled.delay() > 0L) {
            millisToAdd = scheduled.delayUnit().toMillis(scheduled.delay());
        } else if (!scheduled.delayed().isEmpty()) {
            millisToAdd = SchedulerUtils.parseDelayedAsMillis(scheduled);
        }
        if (millisToAdd != null) {
            start = start.toInstant().plusMillis(millisToAdd).atZone(start.getZone());
        }
        if (!(cron = SchedulerUtils.lookUpPropertyValue(scheduled.cron())).isEmpty()) {
            Cron cronExpr;
            if (SchedulerUtils.isOff(cron)) {
                return Optional.empty();
            }
            try {
                cronExpr = parser.parse(cron);
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("Cannot parse cron expression: " + cron, e);
            }
            return Optional.of(new CronTrigger((String)id, start, cronExpr));
        }
        if (!scheduled.every().isEmpty()) {
            OptionalLong everyMillis = SchedulerUtils.parseEveryAsMillis(scheduled);
            if (!everyMillis.isPresent()) {
                return Optional.empty();
            }
            return Optional.of(new IntervalTrigger((String)id, start, everyMillis.getAsLong()));
        }
        throw new IllegalArgumentException("Invalid schedule configuration: " + scheduled);
    }

    static class SimpleScheduledExecution
    implements ScheduledExecution {
        private final ZonedDateTime fireTime;
        private final ZonedDateTime scheduledFireTime;
        private final Trigger trigger;

        public SimpleScheduledExecution(ZonedDateTime fireTime, ZonedDateTime scheduledFireTime, SimpleTrigger trigger) {
            this.fireTime = fireTime;
            this.scheduledFireTime = scheduledFireTime;
            this.trigger = trigger;
        }

        @Override
        public Trigger getTrigger() {
            return this.trigger;
        }

        @Override
        public Instant getFireTime() {
            return this.fireTime.toInstant();
        }

        @Override
        public Instant getScheduledFireTime() {
            return this.scheduledFireTime.toInstant();
        }
    }

    static class CronTrigger
    extends SimpleTrigger {
        private final Cron cron;
        private final ExecutionTime executionTime;

        public CronTrigger(String id, ZonedDateTime start, Cron cron) {
            super(id, start);
            this.cron = cron;
            this.executionTime = ExecutionTime.forCron((Cron)cron);
            this.lastFireTime = ZonedDateTime.now();
        }

        @Override
        public Instant getNextFireTime() {
            Optional nextFireTime = this.executionTime.nextExecution(ZonedDateTime.now());
            return nextFireTime.isPresent() ? ((ZonedDateTime)nextFireTime.get()).toInstant() : null;
        }

        @Override
        public Instant getPreviousFireTime() {
            Optional prevFireTime = this.executionTime.lastExecution(ZonedDateTime.now());
            return prevFireTime.isPresent() ? ((ZonedDateTime)prevFireTime.get()).toInstant() : null;
        }

        @Override
        ZonedDateTime evaluate(ZonedDateTime now) {
            ZonedDateTime lastTruncated;
            if (now.isBefore(this.start)) {
                return null;
            }
            Optional lastExecution = this.executionTime.lastExecution(now);
            if (lastExecution.isPresent() && now.isAfter(lastTruncated = ((ZonedDateTime)lastExecution.get()).truncatedTo(ChronoUnit.SECONDS)) && this.lastFireTime.isBefore(lastTruncated)) {
                LOG.tracef("%s fired, last=", (Object)this, (Object)lastTruncated);
                this.lastFireTime = now;
                return lastTruncated;
            }
            return null;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("CronTrigger [id=").append(this.getId()).append(", cron=").append(this.cron.asString()).append("]");
            return builder.toString();
        }
    }

    static class IntervalTrigger
    extends SimpleTrigger {
        private final long interval;

        public IntervalTrigger(String id, ZonedDateTime start, long interval) {
            super(id, start);
            this.interval = interval;
        }

        @Override
        ZonedDateTime evaluate(ZonedDateTime now) {
            if (now.isBefore(this.start)) {
                return null;
            }
            if (this.lastFireTime == null) {
                this.lastFireTime = now.truncatedTo(ChronoUnit.SECONDS);
                return now;
            }
            long diff = ChronoUnit.MILLIS.between(this.lastFireTime, now);
            if (diff >= this.interval) {
                ZonedDateTime scheduledFireTime = this.lastFireTime.plus(Duration.ofMillis(this.interval));
                this.lastFireTime = now.truncatedTo(ChronoUnit.SECONDS);
                LOG.tracef("%s fired, diff=%s ms", (Object)this, (Object)diff);
                return scheduledFireTime;
            }
            return null;
        }

        @Override
        public Instant getNextFireTime() {
            return this.lastFireTime.plus(Duration.ofMillis(this.interval)).toInstant();
        }

        @Override
        public Instant getPreviousFireTime() {
            return this.lastFireTime.toInstant();
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("IntervalTrigger [id=").append(this.getId()).append(", interval=").append(this.interval).append("]");
            return builder.toString();
        }
    }

    static abstract class SimpleTrigger
    implements Trigger {
        private final String id;
        private volatile boolean running;
        protected final ZonedDateTime start;
        protected volatile ZonedDateTime lastFireTime;

        public SimpleTrigger(String id, ZonedDateTime start) {
            this.id = id;
            this.start = start;
            this.running = true;
        }

        abstract ZonedDateTime evaluate(ZonedDateTime var1);

        @Override
        public String getId() {
            return this.id;
        }

        public synchronized boolean isRunning() {
            return this.running;
        }

        public synchronized void setRunning(boolean running) {
            this.running = running;
        }
    }

    static class ScheduledTask {
        final SimpleTrigger trigger;
        final ScheduledInvoker invoker;

        ScheduledTask(SimpleTrigger trigger, ScheduledInvoker invoker) {
            this.trigger = trigger;
            this.invoker = invoker;
        }

        void execute(final ZonedDateTime now, ExecutorService executor) {
            if (!this.trigger.isRunning()) {
                return;
            }
            final ZonedDateTime scheduledFireTime = this.trigger.evaluate(now);
            if (scheduledFireTime != null) {
                try {
                    executor.execute(new Runnable(){

                        @Override
                        public void run() {
                            try {
                                invoker.invoke(new SimpleScheduledExecution(now, scheduledFireTime, trigger));
                            }
                            catch (Throwable t) {
                                LOG.errorf(t, "Error occured while executing task for trigger %s", (Object)trigger);
                            }
                        }
                    });
                }
                catch (RejectedExecutionException e) {
                    LOG.warnf("Rejected execution of a scheduled task for trigger %s", (Object)this.trigger);
                }
            }
        }
    }
}

