package io.smallrye.graphql.client.typesafe.impl.json;

import static io.smallrye.graphql.client.typesafe.impl.json.JsonUtils.isListOf;
import static java.util.stream.Collectors.toList;

import java.util.List;
import java.util.Optional;

import javax.json.JsonArray;
import javax.json.JsonNumber;
import javax.json.JsonObject;
import javax.json.JsonString;
import javax.json.JsonValue;

import io.smallrye.graphql.client.typesafe.api.ErrorOr;
import io.smallrye.graphql.client.typesafe.api.GraphQLClientError;
import io.smallrye.graphql.client.typesafe.api.GraphQLClientException;
import io.smallrye.graphql.client.typesafe.impl.reflection.FieldInfo;
import io.smallrye.graphql.client.typesafe.impl.reflection.TypeInfo;

public class JsonReader extends Reader<JsonValue> {
    public static Object readJson(String description, TypeInfo type, JsonValue value, FieldInfo field) {
        return readJson(new Location(type, description), type, value, field);
    }

    static Object readJson(Location location, TypeInfo type, JsonValue value, FieldInfo field) {
        return new JsonReader(type, location, value, field).read();
    }

    private JsonReader(TypeInfo type, Location location, JsonValue value, FieldInfo field) {
        super(type, location, value, field);
    }

    @Override
    Object read() {
        if (type.isOptional())
            return Optional.ofNullable(readJson(location, type.getItemType(), value, field));
        if (type.isErrorOr())
            return readErrorOr();
        if (isListOfErrors(value) && !isGraphQlErrorsType())
            throw cantApplyErrors(readGraphQlClientErrors());
        return reader(location).read();
    }

    private ErrorOr<Object> readErrorOr() {
        if (isListOfErrors(value))
            return ErrorOr.ofErrors(readGraphQlClientErrors());
        return ErrorOr.of(readJson(location, type.getItemType(), value, field));
    }

    private List<GraphQLClientError> readGraphQlClientErrors() {
        return value.asJsonArray().stream()
                .map(item -> (GraphQLClientError) readJson(location, TypeInfo.of(GraphQLClientErrorImpl.class), item, field))
                .collect(toList());
    }

    private boolean isListOfErrors(JsonValue jsonValue) {
        return isListOf(jsonValue, ErrorOr.class.getSimpleName());
    }

    private boolean isGraphQlErrorsType() {
        return GraphQLClientError.class.isAssignableFrom(type.getRawType());
    }

    private GraphQLClientException cantApplyErrors(List<GraphQLClientError> errors) {
        return new GraphQLClientException("errors from service (and we can't apply them to a " + location + "; see ErrorOr)",
                errors);
    }

    private Reader<?> reader(Location location) {
        switch (value.getValueType()) {
            case ARRAY:
                return new JsonArrayReader(type, location, (JsonArray) value, field);
            case OBJECT:
                return new JsonObjectReader(type, location, (JsonObject) value, field);
            case STRING:
                return new JsonStringReader(type, location, (JsonString) value, field);
            case NUMBER:
                return new JsonNumberReader(type, location, (JsonNumber) value, field);
            case TRUE:
            case FALSE:
                return new JsonBooleanReader(type, location, value, field);
            case NULL:
                return new JsonNullReader(type, location, value, field);
        }
        throw new GraphQLClientException("unreachable code");
    }
}
