/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-config-js/config_retriever */
var utils = require('vertx-js/util/utils');
var Vertx = require('vertx-js/vertx');
var ReadStream = require('vertx-js/read_stream');
var Future = require('vertx-js/future');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JConfigRetriever = Java.type('io.vertx.config.ConfigRetriever');
var ConfigRetrieverOptions = Java.type('io.vertx.config.ConfigRetrieverOptions');
var ConfigChange = Java.type('io.vertx.config.ConfigChange');

/**
 Defines a configuration retriever that read configuration from
 @class
*/
var ConfigRetriever = function(j_val) {

  var j_configRetriever = j_val;
  var that = this;

  var __super_create = this.create;
  var __super_create = this.create;
  var __super_getConfigAsFuture = this.getConfigAsFuture;
  var __super_getConfig = this.getConfig;
  var __super_close = this.close;
  var __super_getCachedConfig = this.getCachedConfig;
  var __super_listen = this.listen;
  var __super_setBeforeScanHandler = this.setBeforeScanHandler;
  var __super_setConfigurationProcessor = this.setConfigurationProcessor;
  var __super_configStream = this.configStream;
  /**
   Reads the configuration from the different 
   and computes the final configuration.

   @public
   @param completionHandler {function} handler receiving the computed configuration, or a failure if the configuration cannot be retrieved 
   */
  this.getConfig =  function(completionHandler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_configRetriever["getConfig(io.vertx.core.Handler)"](function(ar) {
        if (ar.succeeded()) {
          __args[0](utils.convReturnJson(ar.result()), null);
        } else {
          __args[0](null, ar.cause());
        }
      });
    } else if (typeof __super_getConfig != 'undefined') {
      return __super_getConfig.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Closes the retriever.

   @public

   */
  this.close =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      j_configRetriever["close()"]();
    } else if (typeof __super_close != 'undefined') {
      return __super_close.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Gets the last computed configuration.

   @public

   @return {Object} the last configuration
   */
  this.getCachedConfig =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnJson(j_configRetriever["getCachedConfig()"]()) ;
    } else if (typeof __super_getCachedConfig != 'undefined') {
      return __super_getCachedConfig.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Registers a listener receiving configuration changes. This method cannot only be called if
   the configuration is broadcasted.

   @public
   @param listener {function} the listener 
   */
  this.listen =  function(listener) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_configRetriever["listen(io.vertx.core.Handler)"](function(jVal) {
        __args[0](utils.convReturnDataObject(jVal));
      });
    } else if (typeof __super_listen != 'undefined') {
      return __super_listen.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Registers a handler called before every scan. This method is mostly used for logging purpose.

   @public
   @param function {function} the function, must not be <code>null</code> 
   @return {ConfigRetriever} the current config retriever
   */
  this.setBeforeScanHandler =  function(function) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_configRetriever["setBeforeScanHandler(io.vertx.core.Handler)"](__args[0]) ;
      return that;
    } else if (typeof __super_setBeforeScanHandler != 'undefined') {
      return __super_setBeforeScanHandler.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Registers a handler that process the configuration before being injected into {@link ConfigRetriever#getConfig} or {@link ConfigRetriever#listen}. This allows
   the code to customize the configuration.

   @public
   @param processor {function} the processor, must not be <code>null</code>. The method must not return <code>null</code>. The returned configuration is used. If the processor does not update the configuration, it must return the input configuration. If the processor throws an exception, the failure is passed to the {@link ConfigRetriever#getConfig} handler. 
   @return {ConfigRetriever} the current config retriever
   */
  this.setConfigurationProcessor =  function(processor) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_configRetriever["setConfigurationProcessor(java.util.function.Function)"](function(jVal) {
        var jRet = __args[0](utils.convReturnJson(jVal));
        return utils.convParamJsonObject(jRet);
      }) ;
      return that;
    } else if (typeof __super_setConfigurationProcessor != 'undefined') {
      return __super_setConfigurationProcessor.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  /**

   @public

   @return {ReadStream} the stream of configurations. It's single stream (unicast) and that delivers the last known config and the successors periodically.
   */
  this.configStream =  function() {
    var __args = arguments;
    if (__args.length === 0) {
      if (that.cachedconfigStream == null) {
        that.cachedconfigStream = utils.convReturnVertxGen(ReadStream, j_configRetriever["configStream()"](), undefined);
      }
      return that.cachedconfigStream;
    } else if (typeof __super_configStream != 'undefined') {
      return __super_configStream.apply(this, __args);
    }
    else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_configRetriever;
};

ConfigRetriever._jclass = utils.getJavaClass("io.vertx.config.ConfigRetriever");
ConfigRetriever._jtype = {accept: function(obj) {
    return ConfigRetriever._jclass.isInstance(obj._jdel);
  },wrap: function(jdel) {
    var obj = Object.create(ConfigRetriever.prototype, {});
    ConfigRetriever.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
ConfigRetriever._create = function(jdel) {var obj = Object.create(ConfigRetriever.prototype, {});
  ConfigRetriever.apply(obj, arguments);
  return obj;
}
/**
 Creates an instance of the default implementation of the {@link ConfigRetriever}, using the default
 settings (json file, system properties and environment variables).

 @memberof module:vertx-config-js/config_retriever
 @param vertx {Vertx} the vert.x instance 
 @return {ConfigRetriever} the created instance.
 */
ConfigRetriever.create =  function() {
  var __args = arguments;
  if (__args.length === 2 && typeof __args[0] === 'object' && __args[0]._jdel && (typeof __args[1] === 'object' && __args[1] != null)) {
    return utils.convReturnVertxGen(ConfigRetriever, JConfigRetriever["create(io.vertx.core.Vertx,io.vertx.config.ConfigRetrieverOptions)"](__args[0]._jdel, __args[1]  != null ? new ConfigRetrieverOptions(new JsonObject(Java.asJSONCompatible(__args[1]))) : null)) ;
  } else if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(ConfigRetriever, JConfigRetriever["create(io.vertx.core.Vertx)"](__args[0]._jdel)) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

/**
 Same as {@link ConfigRetriever#getConfig}, but returning a  object. The result is a
 .

 @memberof module:vertx-config-js/config_retriever
 @param retriever {ConfigRetriever} the config retrieve 
 @return {Future} the future completed when the configuration is retrieved
 */
ConfigRetriever.getConfigAsFuture =  function(retriever) {
  var __args = arguments;
  if (__args.length === 1 && typeof __args[0] === 'object' && __args[0]._jdel) {
    return utils.convReturnVertxGen(Future, JConfigRetriever["getConfigAsFuture(io.vertx.config.ConfigRetriever)"](__args[0]._jdel), undefined) ;
  }else throw new TypeError('function invoked with invalid arguments');
};

module.exports = ConfigRetriever;