/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.core.eventbus;

import java.util.Map;
import rx.Observable;
import rx.Single;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.rxjava.core.metrics.Measured;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * A Vert.x event-bus is a light-weight distributed messaging system which allows different parts of your application,
 * or different applications and services to communicate with each in a loosely coupled way.
 * <p>
 * An event-bus supports publish-subscribe messaging, point-to-point messaging and request-response messaging.
 * <p>
 * Message delivery is best-effort and messages can be lost if failure of all or part of the event bus occurs.
 * <p>
 * Please refer to the documentation for more information on the event bus.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.eventbus.EventBus original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rxjava.RxGen(io.vertx.core.eventbus.EventBus.class)
public class EventBus implements Measured {

  public static final io.vertx.lang.rxjava.TypeArg<EventBus> __TYPE_ARG = new io.vertx.lang.rxjava.TypeArg<>(
    obj -> new EventBus((io.vertx.core.eventbus.EventBus) obj),
    EventBus::getDelegate
  );

  private final io.vertx.core.eventbus.EventBus delegate;
  
  public EventBus(io.vertx.core.eventbus.EventBus delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.eventbus.EventBus getDelegate() {
    return delegate;
  }

  /**
   * Whether the metrics are enabled for this measured object
   * @return true if the metrics are enabled
   */
  public boolean isMetricsEnabled() { 
    boolean ret = delegate.isMetricsEnabled();
    return ret;
  }

  /**
   * Sends a message.
   * <p>
   * The message will be delivered to at most one of the handlers registered to the address.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus send(String address, Object message) { 
    delegate.send(address, message);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @param replyHandler reply handler will be called when any reply from the recipient is received, may be <code>null</code>
   * @return a reference to this, so the API can be used fluently
   */
  public <T> EventBus send(String address, Object message, Handler<AsyncResult<Message<T>>> replyHandler) { 
    delegate.send(address, message, new Handler<AsyncResult<io.vertx.core.eventbus.Message<T>>>() {
      public void handle(AsyncResult<io.vertx.core.eventbus.Message<T>> ar) {
        if (ar.succeeded()) {
          replyHandler.handle(io.vertx.core.Future.succeededFuture(Message.newInstance(ar.result(), io.vertx.lang.rxjava.TypeArg.unknown())));
        } else {
          replyHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @return 
   * @deprecated use {@link #rxSend} instead
   */
  @Deprecated()
  public <T> Observable<Message<T>> sendObservable(String address, Object message) { 
    io.vertx.rx.java.ObservableFuture<Message<T>> replyHandler = io.vertx.rx.java.RxHelper.observableFuture();
    send(address, message, replyHandler.toHandler());
    return replyHandler;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @return 
   */
  public <T> Single<Message<T>> rxSend(String address, Object message) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      send(address, message, fut);
    }));
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying <code>options</code> that can be used to configure the delivery.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @param options delivery options
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus send(String address, Object message, DeliveryOptions options) { 
    delegate.send(address, message, options);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @param options delivery options
   * @param replyHandler reply handler will be called when any reply from the recipient is received, may be <code>null</code>
   * @return a reference to this, so the API can be used fluently
   */
  public <T> EventBus send(String address, Object message, DeliveryOptions options, Handler<AsyncResult<Message<T>>> replyHandler) { 
    delegate.send(address, message, options, new Handler<AsyncResult<io.vertx.core.eventbus.Message<T>>>() {
      public void handle(AsyncResult<io.vertx.core.eventbus.Message<T>> ar) {
        if (ar.succeeded()) {
          replyHandler.handle(io.vertx.core.Future.succeededFuture(Message.newInstance(ar.result(), io.vertx.lang.rxjava.TypeArg.unknown())));
        } else {
          replyHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @param options delivery options
   * @return 
   * @deprecated use {@link #rxSend} instead
   */
  @Deprecated()
  public <T> Observable<Message<T>> sendObservable(String address, Object message, DeliveryOptions options) { 
    io.vertx.rx.java.ObservableFuture<Message<T>> replyHandler = io.vertx.rx.java.RxHelper.observableFuture();
    send(address, message, options, replyHandler.toHandler());
    return replyHandler;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#send} but specifying a <code>replyHandler</code> that will be called if the recipient
   * subsequently replies to the message.
   * @param address the address to send it to
   * @param message the message, may be <code>null</code>
   * @param options delivery options
   * @return 
   */
  public <T> Single<Message<T>> rxSend(String address, Object message, DeliveryOptions options) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      send(address, message, options, fut);
    }));
  }

  /**
   * Publish a message.<p>
   * The message will be delivered to all handlers registered to the address.
   * @param address the address to publish it to
   * @param message the message, may be <code>null</code>
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus publish(String address, Object message) { 
    delegate.publish(address, message);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#publish} but specifying <code>options</code> that can be used to configure the delivery.
   * @param address the address to publish it to
   * @param message the message, may be <code>null</code>
   * @param options the delivery options
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus publish(String address, Object message, DeliveryOptions options) { 
    delegate.publish(address, message, options);
    return this;
  }

  /**
   * Create a message consumer against the specified address.
   * <p>
   * The returned consumer is not yet registered
   * at the address, registration will be effective when {@link io.vertx.rxjava.core.eventbus.MessageConsumer#handler}
   * is called.
   * @param address the address that it will register it at
   * @return the event bus message consumer
   */
  public <T> MessageConsumer<T> consumer(String address) { 
    MessageConsumer<T> ret = MessageConsumer.newInstance(delegate.consumer(address), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Create a consumer and register it against the specified address.
   * @param address the address that will register it at
   * @param handler the handler that will process the received messages
   * @return the event bus message consumer
   */
  public <T> MessageConsumer<T> consumer(String address, Handler<Message<T>> handler) { 
    MessageConsumer<T> ret = MessageConsumer.newInstance(delegate.consumer(address, new Handler<io.vertx.core.eventbus.Message<T>>() {
      public void handle(io.vertx.core.eventbus.Message<T> event) {
        handler.handle(Message.newInstance(event, io.vertx.lang.rxjava.TypeArg.unknown()));
      }
    }), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#consumer} but the address won't be propagated across the cluster.
   * @param address the address to register it at
   * @return the event bus message consumer
   */
  public <T> MessageConsumer<T> localConsumer(String address) { 
    MessageConsumer<T> ret = MessageConsumer.newInstance(delegate.localConsumer(address), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#consumer} but the address won't be propagated across the cluster.
   * @param address the address that will register it at
   * @param handler the handler that will process the received messages
   * @return the event bus message consumer
   */
  public <T> MessageConsumer<T> localConsumer(String address, Handler<Message<T>> handler) { 
    MessageConsumer<T> ret = MessageConsumer.newInstance(delegate.localConsumer(address, new Handler<io.vertx.core.eventbus.Message<T>>() {
      public void handle(io.vertx.core.eventbus.Message<T> event) {
        handler.handle(Message.newInstance(event, io.vertx.lang.rxjava.TypeArg.unknown()));
      }
    }), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Create a message sender against the specified address.
   * <p>
   * The returned sender will invoke the {@link io.vertx.rxjava.core.eventbus.EventBus#send}
   * method when the stream {@link io.vertx.rxjava.core.streams.WriteStream#write} method is called with the sender
   * address and the provided data.
   * @param address the address to send it to
   * @return The sender
   */
  public <T> MessageProducer<T> sender(String address) { 
    MessageProducer<T> ret = MessageProducer.newInstance(delegate.sender(address), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#sender} but specifying delivery options that will be used for configuring the delivery of
   * the message.
   * @param address the address to send it to
   * @param options the delivery options
   * @return The sender
   */
  public <T> MessageProducer<T> sender(String address, DeliveryOptions options) { 
    MessageProducer<T> ret = MessageProducer.newInstance(delegate.sender(address, options), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Create a message publisher against the specified address.
   * <p>
   * The returned publisher will invoke the {@link io.vertx.rxjava.core.eventbus.EventBus#publish}
   * method when the stream {@link io.vertx.rxjava.core.streams.WriteStream#write} method is called with the publisher
   * address and the provided data.
   * @param address The address to publish it to
   * @return The publisher
   */
  public <T> MessageProducer<T> publisher(String address) { 
    MessageProducer<T> ret = MessageProducer.newInstance(delegate.publisher(address), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.eventbus.EventBus#publisher} but specifying delivery options that will be used for configuring the delivery of
   * the message.
   * @param address the address to publish it to
   * @param options the delivery options
   * @return The publisher
   */
  public <T> MessageProducer<T> publisher(String address, DeliveryOptions options) { 
    MessageProducer<T> ret = MessageProducer.newInstance(delegate.publisher(address, options), io.vertx.lang.rxjava.TypeArg.unknown());
    return ret;
  }

  /**
   * Add an interceptor that will be called whenever a message is sent from Vert.x
   * @param interceptor the interceptor
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus addInterceptor(Handler<SendContext> interceptor) { 
    EventBus ret = EventBus.newInstance(delegate.addInterceptor(new Handler<io.vertx.core.eventbus.SendContext>() {
      public void handle(io.vertx.core.eventbus.SendContext event) {
        interceptor.handle(SendContext.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Remove an interceptor
   * @param interceptor the interceptor
   * @return a reference to this, so the API can be used fluently
   */
  public EventBus removeInterceptor(Handler<SendContext> interceptor) { 
    EventBus ret = EventBus.newInstance(delegate.removeInterceptor(new Handler<io.vertx.core.eventbus.SendContext>() {
      public void handle(io.vertx.core.eventbus.SendContext event) {
        interceptor.handle(SendContext.newInstance(event));
      }
    }));
    return ret;
  }


  public static EventBus newInstance(io.vertx.core.eventbus.EventBus arg) {
    return arg != null ? new EventBus(arg) : null;
  }
}
