/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.file;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.core.file.OpenOptions;
import java.util.List;
import io.vertx.core.file.CopyOptions;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Contains a broad set of operations for manipulating files on the file system.
 * <p>
 * A (potential) blocking and non blocking version of each operation is provided.
 * <p>
 * The non blocking versions take a handler which is called when the operation completes or an error occurs.
 * <p>
 * The blocking versions are named <code>xxxBlocking</code> and return the results, or throw exceptions directly.
 * In many cases, depending on the operating system and file system some of the potentially blocking operations
 * can return quickly, which is why we provide them, but it's highly recommended that you test how long they take to
 * return in your particular application before using them on an event loop.
 * <p>
 * Please consult the documentation for more information on file system support.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.file.FileSystem original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rx.RxGen(io.vertx.core.file.FileSystem.class)
public class FileSystem {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    FileSystem that = (FileSystem) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.rx.TypeArg<FileSystem> __TYPE_ARG = new io.vertx.lang.rx.TypeArg<>(    obj -> new FileSystem((io.vertx.core.file.FileSystem) obj),
    FileSystem::getDelegate
  );

  private final io.vertx.core.file.FileSystem delegate;
  
  public FileSystem(io.vertx.core.file.FileSystem delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.file.FileSystem getDelegate() {
    return delegate;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The copy will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem copy(String from, String to, Handler<AsyncResult<Void>> handler) { 
    delegate.copy(from, to, handler);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The copy will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxCopy(String from, String to) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      copy(from, to, handler);
    });
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem copy(String from, String to, CopyOptions options, Handler<AsyncResult<Void>> handler) { 
    delegate.copy(from, to, options, handler);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxCopy(String from, String to, CopyOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      copy(from, to, options, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#copy}
   * @param from 
   * @param to 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem copyBlocking(String from, String to) { 
    delegate.copyBlocking(from, to);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * If <code>recursive</code> is <code>true</code> and <code>from</code> represents a directory, then the directory and its contents
   * will be copied recursively to the destination <code>to</code>.
   * <p>
   * The copy will fail if the destination if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem copyRecursive(String from, String to, boolean recursive, Handler<AsyncResult<Void>> handler) { 
    delegate.copyRecursive(from, to, recursive, handler);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * If <code>recursive</code> is <code>true</code> and <code>from</code> represents a directory, then the directory and its contents
   * will be copied recursively to the destination <code>to</code>.
   * <p>
   * The copy will fail if the destination if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxCopyRecursive(String from, String to, boolean recursive) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      copyRecursive(from, to, recursive, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#copyRecursive}
   * @param from 
   * @param to 
   * @param recursive 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem copyRecursiveBlocking(String from, String to, boolean recursive) { 
    delegate.copyRecursiveBlocking(from, to, recursive);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The move will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem move(String from, String to, Handler<AsyncResult<Void>> handler) { 
    delegate.move(from, to, handler);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The move will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMove(String from, String to) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      move(from, to, handler);
    });
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem move(String from, String to, CopyOptions options, Handler<AsyncResult<Void>> handler) { 
    delegate.move(from, to, options, handler);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMove(String from, String to, CopyOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      move(from, to, options, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#move}
   * @param from 
   * @param to 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem moveBlocking(String from, String to) { 
    delegate.moveBlocking(from, to);
    return this;
  }

  /**
   * Truncate the file represented by <code>path</code> to length <code>len</code> in bytes, asynchronously.
   * <p>
   * The operation will fail if the file does not exist or <code>len</code> is less than <code>zero</code>.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem truncate(String path, long len, Handler<AsyncResult<Void>> handler) { 
    delegate.truncate(path, len, handler);
    return this;
  }

  /**
   * Truncate the file represented by <code>path</code> to length <code>len</code> in bytes, asynchronously.
   * <p>
   * The operation will fail if the file does not exist or <code>len</code> is less than <code>zero</code>.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxTruncate(String path, long len) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      truncate(path, len, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#truncate}
   * @param path 
   * @param len 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem truncateBlocking(String path, long len) { 
    delegate.truncateBlocking(path, len);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.
   * <p>
   * The permission String takes the form rwxr-x--- as
   * specified <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * @param path the path to the file
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem chmod(String path, String perms, Handler<AsyncResult<Void>> handler) { 
    delegate.chmod(path, perms, handler);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.
   * <p>
   * The permission String takes the form rwxr-x--- as
   * specified <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * @param path the path to the file
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxChmod(String path, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      chmod(path, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem #chmod(String, String, Handler)}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chmodBlocking(String path, String perms) { 
    delegate.chmodBlocking(path, perms);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.<p>
   * The permission String takes the form rwxr-x--- as
   * specified in {<a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>}.
   * <p>
   * If the file is directory then all contents will also have their permissions changed recursively. Any directory permissions will
   * be set to <code>dirPerms</code>, whilst any normal file permissions will be set to <code>perms</code>.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem chmodRecursive(String path, String perms, String dirPerms, Handler<AsyncResult<Void>> handler) { 
    delegate.chmodRecursive(path, perms, dirPerms, handler);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.<p>
   * The permission String takes the form rwxr-x--- as
   * specified in {<a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>}.
   * <p>
   * If the file is directory then all contents will also have their permissions changed recursively. Any directory permissions will
   * be set to <code>dirPerms</code>, whilst any normal file permissions will be set to <code>perms</code>.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxChmodRecursive(String path, String perms, String dirPerms) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      chmodRecursive(path, perms, dirPerms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#chmodRecursive}
   * @param path 
   * @param perms 
   * @param dirPerms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chmodRecursiveBlocking(String path, String perms, String dirPerms) { 
    delegate.chmodRecursiveBlocking(path, perms, dirPerms);
    return this;
  }

  /**
   * Change the ownership on the file represented by <code>path</code> to <code>user</code> and {code group}, asynchronously.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem chown(String path, String user, String group, Handler<AsyncResult<Void>> handler) { 
    delegate.chown(path, user, group, handler);
    return this;
  }

  /**
   * Change the ownership on the file represented by <code>path</code> to <code>user</code> and {code group}, asynchronously.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxChown(String path, String user, String group) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      chown(path, user, group, handler);
    });
  }

  /**
   * Blocking version of 
   *
   * @param path 
   * @param user 
   * @param group 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chownBlocking(String path, String user, String group) { 
    delegate.chownBlocking(path, user, group);
    return this;
  }

  /**
   * Obtain properties for the file represented by <code>path</code>, asynchronously.
   * <p>
   * If the file is a link, the link will be followed.
   * @param path the path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem props(String path, Handler<AsyncResult<io.vertx.reactivex.core.file.FileProps>> handler) { 
    delegate.props(path, new Handler<AsyncResult<io.vertx.core.file.FileProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.file.FileProps.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Obtain properties for the file represented by <code>path</code>, asynchronously.
   * <p>
   * If the file is a link, the link will be followed.
   * @param path the path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.file.FileProps> rxProps(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      props(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#props}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileProps propsBlocking(String path) { 
    io.vertx.reactivex.core.file.FileProps ret = io.vertx.reactivex.core.file.FileProps.newInstance(delegate.propsBlocking(path));
    return ret;
  }

  /**
   * Obtain properties for the link represented by <code>path</code>, asynchronously.
   * <p>
   * The link will not be followed.
   * @param path the path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem lprops(String path, Handler<AsyncResult<io.vertx.reactivex.core.file.FileProps>> handler) { 
    delegate.lprops(path, new Handler<AsyncResult<io.vertx.core.file.FileProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.file.FileProps.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Obtain properties for the link represented by <code>path</code>, asynchronously.
   * <p>
   * The link will not be followed.
   * @param path the path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.file.FileProps> rxLprops(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      lprops(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#lprops}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileProps lpropsBlocking(String path) { 
    io.vertx.reactivex.core.file.FileProps ret = io.vertx.reactivex.core.file.FileProps.newInstance(delegate.lpropsBlocking(path));
    return ret;
  }

  /**
   * Create a hard link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem link(String link, String existing, Handler<AsyncResult<Void>> handler) { 
    delegate.link(link, existing, handler);
    return this;
  }

  /**
   * Create a hard link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxLink(String link, String existing) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      link(link, existing, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#link}
   * @param link 
   * @param existing 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem linkBlocking(String link, String existing) { 
    delegate.linkBlocking(link, existing);
    return this;
  }

  /**
   * Create a symbolic link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem symlink(String link, String existing, Handler<AsyncResult<Void>> handler) { 
    delegate.symlink(link, existing, handler);
    return this;
  }

  /**
   * Create a symbolic link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxSymlink(String link, String existing) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      symlink(link, existing, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#link}
   * @param link 
   * @param existing 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem symlinkBlocking(String link, String existing) { 
    delegate.symlinkBlocking(link, existing);
    return this;
  }

  /**
   * Unlinks the link on the file system represented by the path <code>link</code>, asynchronously.
   * @param link the link
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem unlink(String link, Handler<AsyncResult<Void>> handler) { 
    delegate.unlink(link, handler);
    return this;
  }

  /**
   * Unlinks the link on the file system represented by the path <code>link</code>, asynchronously.
   * @param link the link
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxUnlink(String link) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      unlink(link, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#unlink}
   * @param link 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem unlinkBlocking(String link) { 
    delegate.unlinkBlocking(link);
    return this;
  }

  /**
   * Returns the path representing the file that the symbolic link specified by <code>link</code> points to, asynchronously.
   * @param link the link
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem readSymlink(String link, Handler<AsyncResult<String>> handler) { 
    delegate.readSymlink(link, handler);
    return this;
  }

  /**
   * Returns the path representing the file that the symbolic link specified by <code>link</code> points to, asynchronously.
   * @param link the link
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxReadSymlink(String link) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      readSymlink(link, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readSymlink}
   * @param link 
   * @return 
   */
  public String readSymlinkBlocking(String link) { 
    String ret = delegate.readSymlinkBlocking(link);
    return ret;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem delete(String path, Handler<AsyncResult<Void>> handler) { 
    delegate.delete(path, handler);
    return this;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxDelete(String path) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      delete(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#delete}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem deleteBlocking(String path) { 
    delegate.deleteBlocking(path);
    return this;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * If the path represents a directory and <code>recursive = true</code> then the directory and its contents will be
   * deleted recursively.
   * @param path path to the file
   * @param recursive delete recursively?
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem deleteRecursive(String path, boolean recursive, Handler<AsyncResult<Void>> handler) { 
    delegate.deleteRecursive(path, recursive, handler);
    return this;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * If the path represents a directory and <code>recursive = true</code> then the directory and its contents will be
   * deleted recursively.
   * @param path path to the file
   * @param recursive delete recursively?
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxDeleteRecursive(String path, boolean recursive) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      deleteRecursive(path, recursive, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#deleteRecursive}
   * @param path 
   * @param recursive 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem deleteRecursiveBlocking(String path, boolean recursive) { 
    delegate.deleteRecursiveBlocking(path, recursive);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem mkdir(String path, Handler<AsyncResult<Void>> handler) { 
    delegate.mkdir(path, handler);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMkdir(String path) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      mkdir(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdir}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirBlocking(String path) { 
    delegate.mkdirBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem mkdir(String path, String perms, Handler<AsyncResult<Void>> handler) { 
    delegate.mkdir(path, perms, handler);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMkdir(String path, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      mkdir(path, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdir}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirBlocking(String path, String perms) { 
    delegate.mkdirBlocking(path, perms);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirs(String path, Handler<AsyncResult<Void>> handler) { 
    delegate.mkdirs(path, handler);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMkdirs(String path) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      mkdirs(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdirs}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirsBlocking(String path) { 
    delegate.mkdirsBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirs(String path, String perms, Handler<AsyncResult<Void>> handler) { 
    delegate.mkdirs(path, perms, handler);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxMkdirs(String path, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      mkdirs(path, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdirs}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirsBlocking(String path, String perms) { 
    delegate.mkdirsBlocking(path, perms);
    return this;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem readDir(String path, Handler<AsyncResult<List<String>>> handler) { 
    delegate.readDir(path, handler);
    return this;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Single<List<String>> rxReadDir(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      readDir(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readDir}
   * @param path 
   * @return 
   */
  public List<String> readDirBlocking(String path) { 
    List<String> ret = delegate.readDirBlocking(path);
    return ret;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The parameter <code>filter</code> is a regular expression. If <code>filter</code> is specified then only the paths that
   * match  @{filter}will be returned.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the directory
   * @param filter the filter expression
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem readDir(String path, String filter, Handler<AsyncResult<List<String>>> handler) { 
    delegate.readDir(path, filter, handler);
    return this;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The parameter <code>filter</code> is a regular expression. If <code>filter</code> is specified then only the paths that
   * match  @{filter}will be returned.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the directory
   * @param filter the filter expression
   * @return a reference to this, so the API can be used fluently
   */
  public Single<List<String>> rxReadDir(String path, String filter) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      readDir(path, filter, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readDir}
   * @param path 
   * @param filter 
   * @return 
   */
  public List<String> readDirBlocking(String path, String filter) { 
    List<String> ret = delegate.readDirBlocking(path, filter);
    return ret;
  }

  /**
   * Reads the entire file as represented by the path <code>path</code> as a , asynchronously.
   * <p>
   * Do not use this method to read very large files or you risk running out of available RAM.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem readFile(String path, Handler<AsyncResult<io.vertx.reactivex.core.buffer.Buffer>> handler) { 
    delegate.readFile(path, new Handler<AsyncResult<io.vertx.core.buffer.Buffer>>() {
      public void handle(AsyncResult<io.vertx.core.buffer.Buffer> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.buffer.Buffer.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Reads the entire file as represented by the path <code>path</code> as a , asynchronously.
   * <p>
   * Do not use this method to read very large files or you risk running out of available RAM.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.buffer.Buffer> rxReadFile(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      readFile(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readFile}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.buffer.Buffer readFileBlocking(String path) { 
    io.vertx.reactivex.core.buffer.Buffer ret = io.vertx.reactivex.core.buffer.Buffer.newInstance(delegate.readFileBlocking(path));
    return ret;
  }

  /**
   * Creates the file, and writes the specified <code>Buffer data</code> to the file represented by the path <code>path</code>,
   * asynchronously.
   * @param path path to the file
   * @param data 
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem writeFile(String path, io.vertx.reactivex.core.buffer.Buffer data, Handler<AsyncResult<Void>> handler) { 
    delegate.writeFile(path, data.getDelegate(), handler);
    return this;
  }

  /**
   * Creates the file, and writes the specified <code>Buffer data</code> to the file represented by the path <code>path</code>,
   * asynchronously.
   * @param path path to the file
   * @param data 
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxWriteFile(String path, io.vertx.reactivex.core.buffer.Buffer data) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      writeFile(path, data, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#writeFile}
   * @param path 
   * @param data 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem writeFileBlocking(String path, io.vertx.reactivex.core.buffer.Buffer data) { 
    delegate.writeFileBlocking(path, data.getDelegate());
    return this;
  }

  /**
   * Open the file represented by <code>path</code>, asynchronously.
   * <p>
   * The file is opened for both reading and writing. If the file does not already exist it will be created.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem open(String path, OpenOptions options, Handler<AsyncResult<io.vertx.reactivex.core.file.AsyncFile>> handler) { 
    delegate.open(path, options, new Handler<AsyncResult<io.vertx.core.file.AsyncFile>>() {
      public void handle(AsyncResult<io.vertx.core.file.AsyncFile> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.file.AsyncFile.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Open the file represented by <code>path</code>, asynchronously.
   * <p>
   * The file is opened for both reading and writing. If the file does not already exist it will be created.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.file.AsyncFile> rxOpen(String path, OpenOptions options) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      open(path, options, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#open}
   * @param path 
   * @param options 
   * @return 
   */
  public io.vertx.reactivex.core.file.AsyncFile openBlocking(String path, OpenOptions options) { 
    io.vertx.reactivex.core.file.AsyncFile ret = io.vertx.reactivex.core.file.AsyncFile.newInstance(delegate.openBlocking(path, options));
    return ret;
  }

  /**
   * Creates an empty file with the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createFile(String path, Handler<AsyncResult<Void>> handler) { 
    delegate.createFile(path, handler);
    return this;
  }

  /**
   * Creates an empty file with the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxCreateFile(String path) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      createFile(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createFile}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem createFileBlocking(String path) { 
    delegate.createFileBlocking(path);
    return this;
  }

  /**
   * Creates an empty file with the specified <code>path</code> and permissions <code>perms</code>, asynchronously.
   * @param path path to the file
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createFile(String path, String perms, Handler<AsyncResult<Void>> handler) { 
    delegate.createFile(path, perms, handler);
    return this;
  }

  /**
   * Creates an empty file with the specified <code>path</code> and permissions <code>perms</code>, asynchronously.
   * @param path path to the file
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Completable rxCreateFile(String path, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultCompletable.toCompletable(handler -> {
      createFile(path, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createFile}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem createFileBlocking(String path, String perms) { 
    delegate.createFileBlocking(path, perms);
    return this;
  }

  /**
   * Determines whether the file as specified by the path <code>path</code> exists, asynchronously.
   * @param path path to the file
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem exists(String path, Handler<AsyncResult<Boolean>> handler) { 
    delegate.exists(path, handler);
    return this;
  }

  /**
   * Determines whether the file as specified by the path <code>path</code> exists, asynchronously.
   * @param path path to the file
   * @return a reference to this, so the API can be used fluently
   */
  public Single<Boolean> rxExists(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      exists(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#exists}
   * @param path 
   * @return 
   */
  public boolean existsBlocking(String path) { 
    boolean ret = delegate.existsBlocking(path);
    return ret;
  }

  /**
   * Returns properties of the file-system being used by the specified <code>path</code>, asynchronously.
   * @param path path to anywhere on the filesystem
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem fsProps(String path, Handler<AsyncResult<io.vertx.reactivex.core.file.FileSystemProps>> handler) { 
    delegate.fsProps(path, new Handler<AsyncResult<io.vertx.core.file.FileSystemProps>>() {
      public void handle(AsyncResult<io.vertx.core.file.FileSystemProps> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.file.FileSystemProps.newInstance(ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Returns properties of the file-system being used by the specified <code>path</code>, asynchronously.
   * @param path path to anywhere on the filesystem
   * @return a reference to this, so the API can be used fluently
   */
  public Single<io.vertx.reactivex.core.file.FileSystemProps> rxFsProps(String path) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      fsProps(path, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#fsProps}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystemProps fsPropsBlocking(String path) { 
    io.vertx.reactivex.core.file.FileSystemProps ret = io.vertx.reactivex.core.file.FileSystemProps.newInstance(delegate.fsPropsBlocking(path));
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempDirectory(String prefix, Handler<AsyncResult<String>> handler) { 
    delegate.createTempDirectory(prefix, handler);
    return this;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempDirectory(String prefix) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempDirectory(prefix, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param prefix 
   * @return 
   */
  public String createTempDirectoryBlocking(String prefix) { 
    String ret = delegate.createTempDirectoryBlocking(prefix);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempDirectory(String prefix, String perms, Handler<AsyncResult<String>> handler) { 
    delegate.createTempDirectory(prefix, perms, handler);
    return this;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempDirectory(String prefix, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempDirectory(prefix, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param prefix 
   * @param perms 
   * @return 
   */
  public String createTempDirectoryBlocking(String prefix, String perms) { 
    String ret = delegate.createTempDirectoryBlocking(prefix, perms);
    return ret;
  }

  /**
   * Creates a new directory in the directory provided by the path <code>path</code>, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempDirectory(String dir, String prefix, String perms, Handler<AsyncResult<String>> handler) { 
    delegate.createTempDirectory(dir, prefix, perms, handler);
    return this;
  }

  /**
   * Creates a new directory in the directory provided by the path <code>path</code>, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempDirectory(String dir, String prefix, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempDirectory(dir, prefix, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param dir 
   * @param prefix 
   * @param perms 
   * @return 
   */
  public String createTempDirectoryBlocking(String dir, String prefix, String perms) { 
    String ret = delegate.createTempDirectoryBlocking(dir, prefix, perms);
    return ret;
  }

  /**
   * Creates a new file in the default temporary-file directory, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempFile(String prefix, String suffix, Handler<AsyncResult<String>> handler) { 
    delegate.createTempFile(prefix, suffix, handler);
    return this;
  }

  /**
   * Creates a new file in the default temporary-file directory, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempFile(String prefix, String suffix) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempFile(prefix, suffix, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param prefix 
   * @param suffix 
   * @return 
   */
  public String createTempFileBlocking(String prefix, String suffix) { 
    String ret = delegate.createTempFileBlocking(prefix, suffix);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempFile(String prefix, String suffix, String perms, Handler<AsyncResult<String>> handler) { 
    delegate.createTempFile(prefix, suffix, perms, handler);
    return this;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempFile(String prefix, String suffix, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempFile(prefix, suffix, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public String createTempFileBlocking(String prefix, String suffix, String perms) { 
    String ret = delegate.createTempFileBlocking(prefix, suffix, perms);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @param handler the handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.file.FileSystem createTempFile(String dir, String prefix, String suffix, String perms, Handler<AsyncResult<String>> handler) { 
    delegate.createTempFile(dir, prefix, suffix, perms, handler);
    return this;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return a reference to this, so the API can be used fluently
   */
  public Single<String> rxCreateTempFile(String dir, String prefix, String suffix, String perms) { 
    return io.vertx.reactivex.impl.AsyncResultSingle.toSingle(handler -> {
      createTempFile(dir, prefix, suffix, perms, handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param dir 
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public String createTempFileBlocking(String dir, String prefix, String suffix, String perms) { 
    String ret = delegate.createTempFileBlocking(dir, prefix, suffix, perms);
    return ret;
  }


  public static  FileSystem newInstance(io.vertx.core.file.FileSystem arg) {
    return arg != null ? new FileSystem(arg) : null;
  }
}
