/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import io.vertx.reactivex.RxHelper;
import io.vertx.reactivex.ObservableHelper;
import io.vertx.reactivex.FlowableHelper;
import io.vertx.reactivex.impl.AsyncResultMaybe;
import io.vertx.reactivex.impl.AsyncResultSingle;
import io.vertx.reactivex.impl.AsyncResultCompletable;
import io.vertx.reactivex.WriteStreamObserver;
import io.vertx.reactivex.WriteStreamSubscriber;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * An asynchronous HTTP client.
 * <p>
 * It allows you to make requests to HTTP servers, and a single client can make requests to any server.
 * <p>
 * It also allows you to open WebSockets to servers.
 * <p>
 * The client can also pool HTTP connections.
 * <p>
 * For pooling to occur, keep-alive must be true on the {@link io.vertx.core.http.HttpClientOptions} (default is true).
 * In this case connections will be pooled and re-used if there are pending HTTP requests waiting to get a connection,
 * otherwise they will be closed.
 * <p>
 * This gives the benefits of keep alive when the client is loaded but means we don't keep connections hanging around
 * unnecessarily when there would be no benefits anyway.
 * <p>
 * The client also supports pipe-lining of requests. Pipe-lining means another request is sent on the same connection
 * before the response from the preceding one has returned. Pipe-lining is not appropriate for all requests.
 * <p>
 * To enable pipe-lining, it must be enabled on the {@link io.vertx.core.http.HttpClientOptions} (default is false).
 * <p>
 * When pipe-lining is enabled the connection will be automatically closed when all in-flight responses have returned
 * and there are no outstanding pending requests to write.
 * <p>
 * The client is designed to be reused between requests.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClient original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.http.HttpClient.class)
public class HttpClient implements io.vertx.reactivex.core.metrics.Measured {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClient that = (HttpClient) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<HttpClient> __TYPE_ARG = new TypeArg<>(    obj -> new HttpClient((io.vertx.core.http.HttpClient) obj),
    HttpClient::getDelegate
  );

  private final io.vertx.core.http.HttpClient delegate;
  
  public HttpClient(io.vertx.core.http.HttpClient delegate) {
    this.delegate = delegate;
  }

  public HttpClient(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpClient)delegate;
  }

  public io.vertx.core.http.HttpClient getDelegate() {
    return delegate;
  }

  /**
   * Whether the metrics are enabled for this measured object
   * @return <code>true</code> if metrics are enabled
   */
  public boolean isMetricsEnabled() { 
    boolean ret = delegate.isMetricsEnabled();
    return ret;
  }

  /**
   * Create an HTTP request to send to the server. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param options the request options
   * @param handler the handler called when the request is ready to be sent
   */
  public void request(io.vertx.core.http.RequestOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.request(options, new Handler<AsyncResult<io.vertx.core.http.HttpClientRequest>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientRequest> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Create an HTTP request to send to the server. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param options the request options
   */
  public void request(io.vertx.core.http.RequestOptions options) {
    request(options, ar -> { });
  }

  /**
   * Create an HTTP request to send to the server. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param options the request options
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest> rxRequest(io.vertx.core.http.RequestOptions options) { 
    return AsyncResultSingle.toSingle($handler -> {
      request(options, $handler);
    });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and <code>port</code>. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param handler the handler called when the request is ready to be sent
   */
  public void request(io.vertx.core.http.HttpMethod method, int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.request(method, port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientRequest>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientRequest> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and <code>port</code>. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   */
  public void request(io.vertx.core.http.HttpMethod method, int port, String host, String requestURI) {
    request(method, port, host, requestURI, ar -> { });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and <code>port</code>. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest> rxRequest(io.vertx.core.http.HttpMethod method, int port, String host, String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      request(method, port, host, requestURI, $handler);
    });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and default port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @param handler the handler called when the request is ready to be sent
   */
  public void request(io.vertx.core.http.HttpMethod method, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.request(method, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientRequest>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientRequest> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and default port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   */
  public void request(io.vertx.core.http.HttpMethod method, String host, String requestURI) {
    request(method, host, requestURI, ar -> { });
  }

  /**
   * Create an HTTP request to send to the server at the <code>host</code> and default port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest> rxRequest(io.vertx.core.http.HttpMethod method, String host, String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      request(method, host, requestURI, $handler);
    });
  }

  /**
   * Create an HTTP request to send to the server at the default host and port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @param handler the handler called when the request is ready to be sent
   */
  public void request(io.vertx.core.http.HttpMethod method, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.request(method, requestURI, new Handler<AsyncResult<io.vertx.core.http.HttpClientRequest>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpClientRequest> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Create an HTTP request to send to the server at the default host and port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param requestURI the relative URI
   */
  public void request(io.vertx.core.http.HttpMethod method, String requestURI) {
    request(method, requestURI, ar -> { });
  }

  /**
   * Create an HTTP request to send to the server at the default host and port. The <code>handler</code>
   * is called when the request is ready to be sent.
   * @param method the HTTP method
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest> rxRequest(io.vertx.core.http.HttpMethod method, String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      request(method, requestURI, $handler);
    });
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @param handler handler that will be called with the WebSocket when connected
   */
  public void webSocket(int port, String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(port, host, requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance((io.vertx.core.http.WebSocket)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   */
  public void webSocket(int port, String host, String requestURI) {
    webSocket(port, host, requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket to the specified port, host and relative request URI
   * @param port the port
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(int port, String host, String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      webSocket(port, host, requestURI, $handler);
    });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @param handler handler that will be called with the WebSocket when connected
   */
  public void webSocket(String host, String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(host, requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance((io.vertx.core.http.WebSocket)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   */
  public void webSocket(String host, String requestURI) {
    webSocket(host, requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket to the host and relative request URI and default port
   * @param host the host
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(String host, String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      webSocket(host, requestURI, $handler);
    });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @param handler handler that will be called with the WebSocket when connected
   */
  public void webSocket(String requestURI, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(requestURI, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance((io.vertx.core.http.WebSocket)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   */
  public void webSocket(String requestURI) {
    webSocket(requestURI, ar -> { });
  }

  /**
   * Connect a WebSocket at the relative request URI using the default host and port
   * @param requestURI the relative URI
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(String requestURI) { 
    return AsyncResultSingle.toSingle($handler -> {
      webSocket(requestURI, $handler);
    });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   * @param handler 
   */
  public void webSocket(io.vertx.core.http.WebSocketConnectOptions options, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocket(options, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance((io.vertx.core.http.WebSocket)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   */
  public void webSocket(io.vertx.core.http.WebSocketConnectOptions options) {
    webSocket(options, ar -> { });
  }

  /**
   * Connect a WebSocket with the specified options.
   * @param options the request options
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocket(io.vertx.core.http.WebSocketConnectOptions options) { 
    return AsyncResultSingle.toSingle($handler -> {
      webSocket(options, $handler);
    });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified WebSocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the WebSocket version
   * @param subProtocols the subprotocols to use
   * @param handler handler that will be called if WebSocket connection fails
   */
  public void webSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, io.vertx.core.http.WebsocketVersion version, List<String> subProtocols, Handler<AsyncResult<io.vertx.reactivex.core.http.WebSocket>> handler) { 
    delegate.webSocketAbs(url, headers.getDelegate(), version, subProtocols, new Handler<AsyncResult<io.vertx.core.http.WebSocket>>() {
      public void handle(AsyncResult<io.vertx.core.http.WebSocket> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.WebSocket.newInstance((io.vertx.core.http.WebSocket)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified WebSocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the WebSocket version
   * @param subProtocols the subprotocols to use
   */
  public void webSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, io.vertx.core.http.WebsocketVersion version, List<String> subProtocols) {
    webSocketAbs(url, headers, version, subProtocols, ar -> { });
  }

  /**
   * Connect a WebSocket with the specified absolute url, with the specified headers, using
   * the specified version of WebSockets, and the specified WebSocket sub protocols.
   * @param url the absolute url
   * @param headers the headers
   * @param version the WebSocket version
   * @param subProtocols the subprotocols to use
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.WebSocket> rxWebSocketAbs(String url, io.vertx.reactivex.core.MultiMap headers, io.vertx.core.http.WebsocketVersion version, List<String> subProtocols) { 
    return AsyncResultSingle.toSingle($handler -> {
      webSocketAbs(url, headers, version, subProtocols, $handler);
    });
  }

  /**
   * Set a connection handler for the client. This handler is called when a new connection is established.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient connectionHandler(Handler<io.vertx.reactivex.core.http.HttpConnection> handler) { 
    delegate.connectionHandler(new Handler<io.vertx.core.http.HttpConnection>() {
      public void handle(io.vertx.core.http.HttpConnection event) {
        handler.handle(io.vertx.reactivex.core.http.HttpConnection.newInstance((io.vertx.core.http.HttpConnection)event));
      }
    });
    return this;
  }

  /**
   * Set a redirect handler for the http client.
   * <p>
   * The redirect handler is called when a <code>3xx</code> response is received and the request is configured to
   * follow redirects with {@link io.vertx.reactivex.core.http.HttpClientRequest#setFollowRedirects}.
   * <p>
   * The redirect handler is passed the {@link io.vertx.reactivex.core.http.HttpClientResponse}, it can return an {@link io.vertx.reactivex.core.http.HttpClientRequest} or <code>null</code>.
   * <ul>
   *   <li>when null is returned, the original response is processed by the original request response handler</li>
   *   <li>when a new <code>Future<HttpClientRequest></code> is returned, the client will send this new request</li>
   * </ul>
   * The new request will get a copy of the previous request headers unless headers are set. In this case,
   * the client assumes that the redirect handler exclusively managers the headers of the new request.
   * <p>
   * The handler must return a <code>Future<HttpClientRequest></code> unsent so the client can further configure it and send it.
   * @param handler the new redirect handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient redirectHandler(Function<io.vertx.reactivex.core.http.HttpClientResponse, io.vertx.core.Future<io.vertx.core.http.RequestOptions>> handler) { 
    delegate.redirectHandler(new Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.RequestOptions>>() {
      public io.vertx.core.Future<io.vertx.core.http.RequestOptions> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.vertx.core.Future<io.vertx.core.http.RequestOptions> ret = handler.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)arg));
        return ret;
      }
    });
    return this;
  }

  /**
   * Set a redirect handler for the http client.
   * <p>
   * The redirect handler is called when a <code>3xx</code> response is received and the request is configured to
   * follow redirects with {@link io.vertx.reactivex.core.http.HttpClientRequest#setFollowRedirects}.
   * <p>
   * The redirect handler is passed the {@link io.vertx.reactivex.core.http.HttpClientResponse}, it can return an {@link io.vertx.reactivex.core.http.HttpClientRequest} or <code>null</code>.
   * <ul>
   *   <li>when null is returned, the original response is processed by the original request response handler</li>
   *   <li>when a new <code>Future<HttpClientRequest></code> is returned, the client will send this new request</li>
   * </ul>
   * The new request will get a copy of the previous request headers unless headers are set. In this case,
   * the client assumes that the redirect handler exclusively managers the headers of the new request.
   * <p>
   * The handler must return a <code>Future<HttpClientRequest></code> unsent so the client can further configure it and send it.
   * @param handler the new redirect handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClient redirectHandler(io.reactivex.functions.Function<io.vertx.reactivex.core.http.HttpClientResponse, io.reactivex.Single<io.vertx.core.http.RequestOptions>> handler) { 
    delegate.redirectHandler(new Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.RequestOptions>>() {
      public io.vertx.core.Future<io.vertx.core.http.RequestOptions> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.reactivex.Single<io.vertx.core.http.RequestOptions> ret;
        try {
          ret = handler.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)arg));
        } catch (Exception e) {
          return io.vertx.core.Future.failedFuture(e);
        }
        return io.vertx.reactivex.SingleHelper.toFuture(ret, obj -> obj);
      }
    });
    return this;
  }

  /**
   * Close the client. Closing will close down any pooled connections.
   * Clients should always be closed after use.
   * @param handler 
   */
  public void close(Handler<AsyncResult<Void>> handler) { 
    delegate.close(handler);
  }

  /**
   * Close the client. Closing will close down any pooled connections.
   * Clients should always be closed after use.
   */
  public void close() {
    close(ar -> { });
  }

  /**
   * Close the client. Closing will close down any pooled connections.
   * Clients should always be closed after use.
   * @return 
   */
  public io.reactivex.Completable rxClose() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      close($handler);
    });
  }

  public static HttpClient newInstance(io.vertx.core.http.HttpClient arg) {
    return arg != null ? new HttpClient(arg) : null;
  }

}
