/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.ext.web.client;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.reactivex.core.buffer.Buffer;
import io.vertx.core.http.HttpMethod;
import io.vertx.reactivex.ext.web.codec.BodyCodec;
import io.vertx.reactivex.core.MultiMap;
import io.vertx.reactivex.core.streams.ReadStream;
import io.vertx.core.json.JsonObject;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * A client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.reactivex.ext.web.client.WebClient} instance, via one of the methods corresponding to the specific
 * HTTP methods such as {@link io.vertx.reactivex.ext.web.client.WebClient#get}, etc...
 * <p>
 * The request shall be configured prior sending, the request is immutable and when a mutator method
 * is called, a new request is returned allowing to expose the request in a public API and apply further customization.
 * <p>
 * After the request has been configured, the methods
 * <ul>
 *   <li>{@link io.vertx.reactivex.ext.web.client.HttpRequest#send}</li>
 *   <li>{@link io.vertx.reactivex.ext.web.client.HttpRequest#sendStream}</li>
 *   <li>{@link io.vertx.reactivex.ext.web.client.HttpRequest#sendJson} ()}</li>
 *   <li>{@link io.vertx.reactivex.ext.web.client.HttpRequest#sendForm}</li>
 * </ul>
 * can be called.
 * The <code>sendXXX</code> methods perform the actual request, they can be called multiple times to perform the same HTTP
 * request at different points in time.
 * <p>
 * The handler is called back with
 * <ul>
 *   <li>an {@link io.vertx.reactivex.ext.web.client.HttpResponse} instance when the HTTP response has been received</li>
 *   <li>a failure when the HTTP request failed (like a connection error) or when the HTTP response could
 *   not be obtained (like connection or unmarshalling errors)</li>
 * </ul>
 * <p>
 * Most of the time, this client will buffer the HTTP response fully unless a specific  is used
 * such as .
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.ext.web.client.HttpRequest original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.reactivex.RxGen(io.vertx.ext.web.client.HttpRequest.class)
public class HttpRequest<T> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpRequest that = (HttpRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.reactivex.TypeArg<HttpRequest> __TYPE_ARG = new io.vertx.lang.reactivex.TypeArg<>(
    obj -> new HttpRequest((io.vertx.ext.web.client.HttpRequest) obj),
    HttpRequest::getDelegate
  );

  private final io.vertx.ext.web.client.HttpRequest delegate;
  public final io.vertx.lang.reactivex.TypeArg<T> __typeArg_0;
  
  public HttpRequest(io.vertx.ext.web.client.HttpRequest delegate) {
    this.delegate = delegate;
    this.__typeArg_0 = io.vertx.lang.reactivex.TypeArg.unknown();
  }

  public HttpRequest(io.vertx.ext.web.client.HttpRequest delegate, io.vertx.lang.reactivex.TypeArg<T> typeArg_0) {
    this.delegate = delegate;
    this.__typeArg_0 = typeArg_0;
  }

  public io.vertx.ext.web.client.HttpRequest getDelegate() {
    return delegate;
  }

  /**
   * Configure the request to use a new method <code>value</code>.
   * @param value 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> method(HttpMethod value) { 
    delegate.method(value);
    return this;
  }

  /**
   * Configure the request to use a new port <code>value</code>.
   * @param value 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> port(int value) { 
    delegate.port(value);
    return this;
  }

  /**
   * Configure the request to decode the response with the <code>responseCodec</code>.
   * @param responseCodec the response codec
   * @return a reference to this, so the API can be used fluently
   */
  public <U> HttpRequest<U> as(BodyCodec<U> responseCodec) { 
    HttpRequest<U> ret = HttpRequest.newInstance(delegate.as(responseCodec.getDelegate()), responseCodec.__typeArg_0);
    return ret;
  }

  /**
   * Configure the request to use a new host <code>value</code>.
   * @param value 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> host(String value) { 
    delegate.host(value);
    return this;
  }

  /**
   * Configure the request to use a new request URI <code>value</code>.
   * <p>
   * When the uri has query parameters, they are set in the {@link io.vertx.reactivex.ext.web.client.HttpRequest#queryParams} multimap, overwritting
   * any parameters previously set.
   * @param value 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> uri(String value) { 
    delegate.uri(value);
    return this;
  }

  /**
   * Configure the request to add a new HTTP header.
   * @param name the header name
   * @param value the header value
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return The HTTP headers
   */
  public MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    MultiMap ret = MultiMap.newInstance(delegate.headers());
    cached_0 = ret;
    return ret;
  }

  public HttpRequest<T> ssl(boolean value) { 
    delegate.ssl(value);
    return this;
  }

  /**
   * Configures the amount of time in milliseconds after which if the request does not return any data within the timeout
   * period an {@link java.util.concurrent.TimeoutException} fails the request.
   * <p>
   * Setting zero or a negative <code>value</code> disables the timeout.
   * @param value The quantity of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> timeout(long value) { 
    delegate.timeout(value);
    return this;
  }

  /**
   * Add a query parameter to the request.
   * @param paramName the param name
   * @param paramValue the param value
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> addQueryParam(String paramName, String paramValue) { 
    delegate.addQueryParam(paramName, paramValue);
    return this;
  }

  /**
   * Set a query parameter to the request.
   * @param paramName the param name
   * @param paramValue the param value
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> setQueryParam(String paramName, String paramValue) { 
    delegate.setQueryParam(paramName, paramValue);
    return this;
  }

  /**
   * Set wether or not to follow the directs for the request.
   * @param value true if redirections should be followed
   * @return a reference to this, so the API can be used fluently
   */
  public HttpRequest<T> followRedirects(boolean value) { 
    delegate.followRedirects(value);
    return this;
  }

  /**
   * Return the current query parameters.
   * @return the current query parameters
   */
  public MultiMap queryParams() { 
    MultiMap ret = MultiMap.newInstance(delegate.queryParams());
    return ret;
  }

  /**
   * Copy this request
   * @return a copy of this request
   */
  public HttpRequest<T> copy() { 
    HttpRequest<T> ret = HttpRequest.newInstance(delegate.copy(), __typeArg_0);
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @param handler 
   */
  public void sendStream(ReadStream<Buffer> body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendStream(body.getDelegate(), new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendStream(ReadStream<Buffer> body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendStream(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @param handler 
   */
  public void sendStream(Flowable<Buffer> body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendStream(io.vertx.reactivex.core.impl.ReadStreamSubscriber.asReadStream(body,obj -> (io.vertx.core.buffer.Buffer)obj.getDelegate()).resume(), new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendStream(Flowable<Buffer> body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendStream(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @param handler 
   */
  public void sendStream(Observable<Buffer> body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendStream(io.vertx.reactivex.core.impl.ReadStreamSubscriber.asReadStream(body,obj -> (io.vertx.core.buffer.Buffer)obj.getDelegate()).resume(), new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> stream.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendStream(Observable<Buffer> body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendStream(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> buffer.
   * @param body the body
   * @param handler 
   */
  public void sendBuffer(Buffer body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendBuffer(body.getDelegate(), new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> buffer.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendBuffer(Buffer body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendBuffer(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   * set to <code>application/json</code>.
   * @param body the body
   * @param handler 
   */
  public void sendJsonObject(JsonObject body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendJsonObject(body, new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   * set to <code>application/json</code>.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendJsonObject(JsonObject body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendJsonObject(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   * set to <code>application/json</code>.
   * @param body the body
   * @param handler 
   */
  public void sendJson(Object body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendJson(body, new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> object encoded as json and the content type
   * set to <code>application/json</code>.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendJson(Object body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendJson(body, handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> multimap encoded as form and the content type
   * set to <code>application/x-www-form-urlencoded</code>.
   * <p>
   * When the content type header is previously set to <code>multipart/form-data</code> it will be used instead.
   * @param body the body
   * @param handler 
   */
  public void sendForm(MultiMap body, Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.sendForm(body.getDelegate(), new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Like {@link io.vertx.reactivex.ext.web.client.HttpRequest#send} but with an HTTP request <code>body</code> multimap encoded as form and the content type
   * set to <code>application/x-www-form-urlencoded</code>.
   * <p>
   * When the content type header is previously set to <code>multipart/form-data</code> it will be used instead.
   * @param body the body
   * @return 
   */
  public Single<HttpResponse<T>> rxSendForm(MultiMap body) { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      sendForm(body, handler);
    });
  }

  /**
   * Send a request, the <code>handler</code> will receive the response as an {@link io.vertx.reactivex.ext.web.client.HttpResponse}.
   * @param handler 
   */
  public void send(Handler<AsyncResult<HttpResponse<T>>> handler) { 
    delegate.send(new Handler<AsyncResult<io.vertx.ext.web.client.HttpResponse<T>>>() {
      public void handle(AsyncResult<io.vertx.ext.web.client.HttpResponse<T>> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(HttpResponse.newInstance(ar.result(), __typeArg_0)));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
  }

  /**
   * Send a request, the <code>handler</code> will receive the response as an {@link io.vertx.reactivex.ext.web.client.HttpResponse}.
   * @return 
   */
  public Single<HttpResponse<T>> rxSend() { 
    return new io.vertx.reactivex.core.impl.AsyncResultSingle<HttpResponse<T>>(handler -> {
      send(handler);
    });
  }

  private MultiMap cached_0;

  public static <T>HttpRequest<T> newInstance(io.vertx.ext.web.client.HttpRequest arg) {
    return arg != null ? new HttpRequest<T>(arg) : null;
  }

  public static <T>HttpRequest<T> newInstance(io.vertx.ext.web.client.HttpRequest arg, io.vertx.lang.reactivex.TypeArg<T> __typeArg_T) {
    return arg != null ? new HttpRequest<T>(arg, __typeArg_T) : null;
  }
}
