/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.interceptors;

import EDU.oswego.cs.dl.util.concurrent.ConcurrentReaderHashMap;
import org.jboss.cache.TreeCache;
import org.jboss.cache.loader.CacheLoader;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ConcurrentModificationException;

/**
 * asbtract superclass for cache loader and cache store interceptors.
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 */
public class BaseCacheLoaderInterceptor extends Interceptor
{
    protected CacheLoader loader = null;
    private Map lockMap = new ConcurrentReaderHashMap();

    public void setCache(TreeCache cache)
    {
        super.setCache(cache);
        this.loader = cache.getCacheLoader();
    }

    /**
     * Simplistic locking mechanism that uses a single synchronized lock per
     * fqn accessed.  Nothing complex here - no isolation levels or
     * consideration for the operation being performed for now.
     */
    protected void obtainLoaderLock(Object lock)
    {
        Thread current = Thread.currentThread();

        synchronized(this)
        {
            while (lockMap.containsKey(lock) && !lockMap.get(lock).equals(current))
            {
                try
                {
                    this.wait();
                }
                catch (InterruptedException e)
                {
                }
            }

            if (lockMap.containsKey(lock) && !lockMap.get(lock).equals(current))
            {
                throw new ConcurrentModificationException("Loader lock " + lock + " is already held by someone else.");
            }

            lockMap.put(lock, current);
        }
    }

    /**
     * Releases simplistic loader lock.
     */
    protected void releaseLoaderLock(Object lock)
    {
        synchronized (this)
        {
            lockMap.remove(lock);
            this.notify();
        }
    }

    protected void releaseLoaderLocks(List locks)
    {
        Iterator it = locks.iterator();
        while (it.hasNext())
        {
            releaseLoaderLock(it.next());
        }
    }

    protected void obtainLoaderLocks(List locks)
    {
        Iterator it = locks.iterator();
        while (it.hasNext())
        {
            obtainLoaderLock(it.next());
        }
    }
}