/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.interceptors;

import org.jboss.cache.*;
import org.jboss.cache.marshall.MethodDeclarations;
import org.jboss.cache.marshall.JBCMethodCall;
import org.jboss.cache.optimistic.TransactionWorkspace;
import org.jboss.cache.optimistic.WorkspaceNode;
import org.jboss.cache.optimistic.DefaultDataVersion;
import org.jgroups.blocks.MethodCall;

import javax.transaction.Transaction;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * Validates the data in the transaction workspace against data in the actual
 * cache (versions only), and then performs commits if necessary.  Does not
 * pass on prepare/commit/rollbacks to the other interceptors.
 * <p/>
 * Currently uses simplistic integer based versioning and validation. Plans are
 * to have this configurable as there will always be a performance/complexity
 * tradeoff.
 * <p/>
 * On the commit it applies the changes in the workspace to the real nodes in
 * the cache.
 * <p/>
 * On rollback clears the nodes in the workspace.
 *
 * @author Manik Surtani (<a href="mailto:manik@jboss.org">manik@jboss.org</a>)
 * @author Steve Woodcock (<a href="mailto:stevew@jofti.com">stevew@jofti.com</a>)
 */
public class OptimisticValidatorInterceptor extends OptimisticInterceptor
{

    public void setCache(TreeCache cache)
    {
        super.setCache(cache);
    }

    public Object invoke(MethodCall call) throws Throwable
    {
        JBCMethodCall m = (JBCMethodCall) call;
        InvocationContext ctx = getInvocationContext();
        Transaction tx = ctx.getTransaction();
        GlobalTransaction gtx = ctx.getGlobalTransaction();
        Object retval = null;
        Method meth = m.getMethod();

        if (tx == null)
            throw new CacheException("Not in a transaction");

        // Methods we are interested in are prepare/commit
        // They do not go further than this interceptor
        switch (m.getMethodId()) 
        {
           case MethodDeclarations.optimisticPrepareMethod_id:
              // should pass in a different prepare here
              validateNodes(gtx);
              break;
           case MethodDeclarations.commitMethod_id:
              commit(gtx);
              break;
           case MethodDeclarations.rollbackMethod_id:
              rollBack(gtx);
              break;
           default :
              retval = super.invoke(m);
              break;
        }
        return retval;
    }


    private void validateNodes(GlobalTransaction gtx) throws CacheException
    {
        TransactionWorkspace workspace;

        try
        {
            workspace = getTransactionWorkspace(gtx);
        }
        catch (CacheException e)
        {
            throw new CacheException("unable to retrieve workspace", e);
        }

        // should be an ordered list - get the set of nodes
        Collection nodes = workspace.getNodes().values();

        boolean validated;
        //we have all locks here so lets try and validate
        log.debug("validating nodes");
        simpleValidate(nodes);
        log.debug("validated nodes");
    }

    private void simpleValidate(Collection nodes)
        throws CacheException
    {
        WorkspaceNode workspaceNode;

        boolean trace = log.isTraceEnabled();
        for (Iterator it = nodes.iterator(); it.hasNext();)
        {
            workspaceNode = (WorkspaceNode) it.next();
            Fqn fqn = workspaceNode.getFqn();
            if (trace) log.trace("validating version for node " + fqn);
            OptimisticTreeNode realNode = (OptimisticTreeNode) cache._get(fqn);

            // if this is a newly created node then we expect the underlying node to be null.
            // if not, we have a problem...
            if (realNode == null && !workspaceNode.isCreated())
            {
                throw new CacheException("Real node for " + fqn + " is null, and this wasn't newly created in this tx!");
            }
            if (!workspaceNode.isCreated() && realNode.getVersion().newerThan( workspaceNode.getVersion() ))
            {
                // we have an out of date node here
                throw new CacheException("DataNode [" + fqn + "] version " + ((OptimisticTreeNode)workspaceNode.getNode()).getVersion() + " is newer than workspace node " + workspaceNode.getVersion());
            }
        }
    }


    private void commit(GlobalTransaction gtx)
    {
        TransactionWorkspace workspace;

        try
        {
            workspace = getTransactionWorkspace(gtx);
        }
        catch (CacheException e)
        {
            log.trace("we can't rollback", e);
            return;
        }

        log.debug("commiting validated changes ");
        // should be an ordered list
        Collection nodes = workspace.getNodes().values();

        boolean trace = log.isTraceEnabled();
        for (Iterator it = nodes.iterator(); it.hasNext();)
        {
            WorkspaceNode wrappedNode = (WorkspaceNode) it.next();
            // short circuit if this node is deleted?
            if (wrappedNode.isDeleted())
            {
                // handle notifications

                if (trace) log.trace("Workspace node " + wrappedNode.getFqn() + " deleted; removing");
                DataNode dNode = wrappedNode.getNode();
                cache.notifyNodeRemove(dNode.getFqn(), true);

                if (dNode.getFqn().isRoot())
                {
                    log.warn("Attempted to delete the root node");
                }
                else
                {
                    DataNode parent = (DataNode) dNode.getParent();
                    parent.removeChild( dNode.getName() );
                }
                cache.notifyNodeRemoved(dNode.getFqn());
                cache.notifyNodeRemove(dNode.getFqn(), false);
            }
            else
            {
                // "Will somebody please think of the children!!"
                // if (wrappedNode.hasCreatedOrRemovedChildren() handleChildNodes(wrappedNode);
                if (wrappedNode.isDirty())
                {
                    cache.notifyNodeModify(wrappedNode.getFqn(), true);
                    OptimisticTreeNode current = (OptimisticTreeNode) wrappedNode.getNode();
                    Map mergedChildren = wrappedNode.getMergedChildren();

                    // this could be done better to account for more subtle merges
                    current.setChildren(mergedChildren);

                    if (log.isTraceEnabled()) log.trace("inserting merged data " + wrappedNode.getMergedData());
                    Map mergedData = wrappedNode.getMergedData();
                    current.put(mergedData, true);
                    if (workspace.isVersioningImplicit())
                    {
                        current.setVersion(((DefaultDataVersion)wrappedNode.getVersion()).increment());
                    }
                    else
                    {
                        if (trace) log.trace("Versioning is explicit; not attempting an increment.");
                        current.setVersion(wrappedNode.getVersion());
                    }
                    if (trace) log.trace("Setting version of node " + current.getName() + " from " + wrappedNode.getVersion() + " to " + current.getVersion());
                    cache.notifyNodeModified(wrappedNode.getFqn());
                    cache.notifyNodeModify(wrappedNode.getFqn(), false);
                }
                else
                {
                    if (trace) log.trace("Merging node " + wrappedNode.getFqn() + " not necessary since the node is not dirty");
                    cache.notifyNodeVisited(wrappedNode.getFqn());
                }
            }
        }

    }

    private void rollBack(GlobalTransaction gtx)
    {
        TransactionWorkspace workspace;
        try
        {
            workspace = getTransactionWorkspace(gtx);
            Map nodes = workspace.getNodes();
            nodes.clear();
        }
        catch (CacheException e)
        {
            log.info("Unable to roll back", e);
        }
    }
}
