/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.interceptors;

import org.jboss.cache.DataNode;
import org.jboss.cache.Fqn;
import org.jboss.cache.GlobalTransaction;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.TransactionEntry;
import org.jboss.cache.TransactionTable;
import org.jboss.cache.TreeCache;
import org.jboss.cache.lock.IdentityLock;
import org.jboss.cache.lock.IsolationLevel;
import org.jboss.cache.lock.LockingException;
import org.jboss.cache.lock.TimeoutException;
import org.jboss.cache.marshall.JBCMethodCall;
import org.jboss.cache.marshall.MethodDeclarations;
import org.jgroups.blocks.MethodCall;

import javax.transaction.Transaction;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * An interceptor that handles locking. When a TX is associated, we register
 * for TX completion and unlock the locks acquired within the scope of the TX.
 * When no TX is present, we keep track of the locks acquired during the
 * current method and unlock when the method returns.
 *
 * @author Bela Ban
 * @version $Id: PessimisticLockInterceptor.java 2043 2006-06-06 10:20:05Z msurtani $
 */
public class PessimisticLockInterceptor extends Interceptor {
   TransactionTable           tx_table=null;

   /** Map<Object, java.util.List>. Keys = threads, values = lists of locks held by that thread */
   Map                        lock_table;
   private long               lock_acquisition_timeout;


   public void setCache(TreeCache cache) {
      super.setCache(cache);
      tx_table=cache.getTransactionTable();
      lock_table=cache.getLockTable();
      lock_acquisition_timeout=cache.getLockAcquisitionTimeout();
   }



   public Object invoke(MethodCall call) throws Throwable {
      JBCMethodCall m = (JBCMethodCall) call;
      Fqn               fqn=null;
      int               lock_type=DataNode.LOCK_TYPE_NONE;
      long              lock_timeout=lock_acquisition_timeout;
      Object[]          args=m.getArgs();
      InvocationContext ctx = getInvocationContext();

       if (log.isTraceEnabled()) log.trace("PessimisticLockInterceptor invoked for method " + m);
       if (ctx.getOptionOverrides() != null && ctx.getOptionOverrides().isSuppressLocking())
       {
          log.trace("Suppressing locking");
          switch (m.getMethodId())
          {
             case MethodDeclarations.putDataMethodLocal_id:
             case MethodDeclarations.putDataEraseMethodLocal_id:
             case MethodDeclarations.putKeyValMethodLocal_id:
             case MethodDeclarations.putFailFastKeyValueMethodLocal_id:
                log.trace("Creating nodes if necessary");
                createNodes((Fqn) args[1], ctx.getGlobalTransaction());
                break;
          }
           
          return super.invoke(m);
       }

       /** List<IdentityLock> locks. Locks acquired during the current method; will be released later by UnlockInterceptor.
   *  This list is only populated when there is no TX, otherwise the TransactionTable maintains the locks
   * (keyed by TX) */
     // List locks=null;

      boolean recursive=false;
      boolean createIfNotExists=false;


      // 1. Determine the type of lock (read, write, or none) depending on the method. If no lock is required, invoke
      //    the method, then return immediately
      //    Set the Fqn
      switch (m.getMethodId())
      {
         case MethodDeclarations.putDataMethodLocal_id:
         case MethodDeclarations.putDataEraseMethodLocal_id:
         case MethodDeclarations.putKeyValMethodLocal_id:
         case MethodDeclarations.putFailFastKeyValueMethodLocal_id:
            createIfNotExists=true;
            fqn=(Fqn)args[1];
            lock_type=DataNode.LOCK_TYPE_WRITE;
            if(m.getMethodId() == MethodDeclarations.putFailFastKeyValueMethodLocal_id)
               lock_timeout=((Long)args[5]).longValue();
            break;
         case MethodDeclarations.removeNodeMethodLocal_id:
            fqn=(Fqn)args[1];
            lock_type=DataNode.LOCK_TYPE_WRITE;
            recursive=true; // remove node and *all* child nodes
            break;
         case MethodDeclarations.removeKeyMethodLocal_id:
         case MethodDeclarations.removeDataMethodLocal_id:
         case MethodDeclarations.addChildMethodLocal_id:
            fqn=(Fqn)args[1];
            lock_type=DataNode.LOCK_TYPE_WRITE;
            break;
         case MethodDeclarations.evictNodeMethodLocal_id:
            fqn=(Fqn)args[0];
            lock_type=DataNode.LOCK_TYPE_WRITE;
            break;
         case MethodDeclarations.getKeyValueMethodLocal_id:
         case MethodDeclarations.getNodeMethodLocal_id:
         case MethodDeclarations.getKeysMethodLocal_id:
         case MethodDeclarations.getChildrenNamesMethodLocal_id:
         case MethodDeclarations.releaseAllLocksMethodLocal_id:
         case MethodDeclarations.printMethodLocal_id:
            fqn=(Fqn)args[0];
            lock_type=DataNode.LOCK_TYPE_READ;
            break;
         case MethodDeclarations.lockMethodLocal_id:
            fqn=(Fqn)args[0];
            lock_type=((Integer)args[1]).intValue();
            recursive=((Boolean)args[2]).booleanValue();
            break;
         case MethodDeclarations.commitMethod_id:
            // commit propagated up from the tx interceptor
            commit(ctx.getGlobalTransaction());
            break;
         case MethodDeclarations.rollbackMethod_id:
            // rollback propagated up from the tx interceptor
            rollback(ctx.getGlobalTransaction());
            break;
         default :
            if (isOnePhaseCommitPrepareMehod(m)) 
            {
               // commit propagated up from the tx interceptor
               commit(ctx.getGlobalTransaction());
            }
            break;
      }

      // Lock the node (must be either read or write if we get here)
      // If no TX: add each acquired lock to the list of locks for this method (locks)
      // If TX: [merge code from TransactionInterceptor]: register with TxManager, on commit/rollback,
      // release the locks for the given TX
      if(fqn != null) {
         if(createIfNotExists) {
            do {
               lock(fqn, ctx.getGlobalTransaction(), lock_type, recursive, lock_timeout, createIfNotExists);
            }
            while(!cache.exists(fqn)); // keep trying until we have the lock (fixes concurrent remove())
                                               // terminates successfully, or with (Timeout)Exception
         }
         else
            lock(fqn, ctx.getGlobalTransaction(), lock_type, recursive, lock_timeout, createIfNotExists);
      }
      else {
         if(log.isTraceEnabled())
            log.trace("bypassed locking as method " + m.getName() + "() doesn't require locking");
      }
      if(m.getMethodId() == MethodDeclarations.lockMethodLocal_id)
         return null;
      return super.invoke(m);
   }



   /**
    * Locks a given node.
    * @param fqn
    * @param gtx
    * @param lock_type DataNode.LOCK_TYPE_READ, DataNode.LOCK_TYPE_WRITE or DataNode.LOCK_TYPE_NONE
    * @param recursive Lock children recursively
    */
   private void lock(Fqn fqn, GlobalTransaction gtx, int lock_type, boolean recursive,
                     long lock_timeout, boolean createIfNotExists)
         throws TimeoutException, LockingException, InterruptedException {
      DataNode       n;
      DataNode       child_node;
      Object         child_name;
      Thread         currentThread = Thread.currentThread();
      Object         owner = (gtx != null) ? (Object)gtx : currentThread;
      int            treeNodeSize;
      boolean        acquired=false;


       if (log.isTraceEnabled()) log.trace("Attempting to lock node " + fqn + " for owner " + owner);

      if(fqn == null) {
         log.error("fqn is null - this should not be the case");
         return;
      }

      if((treeNodeSize=fqn.size()) == 0)
         return;

      if(cache.getIsolationLevelClass() == IsolationLevel.NONE)
         lock_type=DataNode.LOCK_TYPE_NONE;

      n=cache.getRoot();
      for(int i=0; i < treeNodeSize; i++) {
         child_name=fqn.get(i);
         child_node=(DataNode) n.getOrCreateChild(child_name, gtx, createIfNotExists);
         if(child_node == null) {
            if(log.isTraceEnabled())
               log.trace("failed to find or create child " + child_name + " of node " + n.getFqn());
            return;
         }

         if(lock_type == DataNode.LOCK_TYPE_NONE) {
            // acquired=false;
            n=child_node;
            continue;
         }
         else {
            if(lock_type == DataNode.LOCK_TYPE_WRITE && i == (treeNodeSize - 1)) {
               acquired=child_node.acquire(owner, lock_timeout, DataNode.LOCK_TYPE_WRITE);
            }
            else {
               acquired=child_node.acquire(owner, lock_timeout, DataNode.LOCK_TYPE_READ);
            }
         }


         if(acquired) {
            if(gtx != null) {
               // add the lock to the list of locks maintained for this transaction
               // (needed for release of locks on commit or rollback)
               cache.getTransactionTable().addLock(gtx, child_node.getLock());
            }
            else {
               IdentityLock l=child_node.getLock();
               List locks = getLocks(currentThread);
               if(!locks.contains(l))
                  locks.add(l);
            }
         }

         if(recursive && i == (treeNodeSize - 1)) {
            Set acquired_locks=child_node.acquireAll(owner, lock_timeout, lock_type);
            if(acquired_locks.size() > 0) {
               if(gtx != null) {
                  cache.getTransactionTable().addLocks(gtx, acquired_locks);
               }
               else {
                  List locks = getLocks(currentThread);
                  locks.addAll(acquired_locks);
               }
            }
         }
         n=child_node;
      }
   }

   private List getLocks(Thread currentThread) {
      // This sort of looks like a get/put race condition, but
      // since we key off the Thread, it's not
      List locks = (List)lock_table.get(currentThread);
      if (locks == null) {
        locks = Collections.synchronizedList(new LinkedList());
        lock_table.put(currentThread, locks);
      }
      return locks;
   }


    private void createNodes(Fqn fqn, GlobalTransaction gtx)
    {
        int treeNodeSize;
        if ((treeNodeSize=fqn.size()) == 0) return;
        DataNode n=cache.getRoot();
        for(int i=0; i < treeNodeSize; i++)
        {
            Object child_name=fqn.get(i);
            DataNode child_node=(DataNode) n.getOrCreateChild(child_name, gtx, true);
            if(child_node == null)
            {
                if(log.isTraceEnabled()) log.trace("failed to find or create child " + child_name + " of node " + n.getFqn());
                return;
            }
            n = child_node;
        }
    }


   /**
    * Remove all locks held by <tt>tx</tt>, remove the transaction from the transaction table
    * @param gtx
    */
   private void commit(GlobalTransaction gtx) {
      if(log.isTraceEnabled())
         log.trace("committing cache with gtx " + gtx);

      TransactionEntry entry=tx_table.get(gtx);
      if(entry == null) {
         log.error("entry for transaction " + gtx + " not found (maybe already committed)");
         return;
      }

      // Let's do it in stack style, LIFO
      entry.releaseAllLocksLIFO(gtx);

      Transaction ltx=entry.getTransaction();
      if(log.isTraceEnabled())
         log.trace("removing local transaction " + ltx + " and global transaction " + gtx);
      tx_table.remove(ltx);
      tx_table.remove(gtx);
   }


   /**
     * Revert all changes made inside this TX: invoke all method calls of the undo-ops
     * list. Then release all locks and remove the TX from the transaction table.
     * <ol>
     * <li>Revert all modifications done in the current TX<li/>
     * <li>Release all locks held by the current TX</li>
     * <li>Remove all temporary nodes created by the current TX</li>
     * </ol>
     *
     * @param tx
     */
   private void rollback(GlobalTransaction tx) {
      List undo_ops;
      TransactionEntry entry=tx_table.get(tx);
      MethodCall undo_op;

      if(log.isTraceEnabled())
         log.trace("called to rollback cache with GlobalTransaction=" + tx);

      if(entry == null) {
         log.error("entry for transaction " + tx + " not found (transaction has possibly already been rolled back)");
         return;
      }

      // 1. Revert the modifications by running the undo-op list in reverse. This *cannot* throw any exceptions !
      entry.undoOperations(cache);

      // This was removed as we don't use temporary nodes anymore; we now create undo-operations on put(), e.g.
      // put(/a/b/c) on /a, create b and c, plus undo operations _remove(a/b/c) and _remove(/a/b)

      // 2. Remove all temporary nodes. Need to do it backwards since node is LIFO.
//      for(ListIterator it=new LinkedList(entry.getNodes()).listIterator(entry.getNodes().size());
//          it.hasPrevious();) {
//         node_name=(Fqn)it.previous();
//         try {
//            cache._remove(tx, node_name, false);
//         }
//         catch(Throwable t) {
//            log.error("failed removing node \"" + node_name + "\"", t);
//         }
//      }


      // 3. Finally, release all locks held by this TX
      // Let's do it in stack style, LIFO
      // Note that the lock could have been released already so don't panic.
      entry.releaseAllLocksLIFO(tx);

      Transaction ltx=entry.getTransaction();
      if(log.isTraceEnabled())
         log.trace("removing local transaction " + ltx + " and global transaction " + tx);
      tx_table.remove(ltx);
      tx_table.remove(tx);
   }

}
