package name.remal.gradle_plugins.dsl.extensions

import name.remal.escapeRegex
import org.gradle.api.artifacts.ArtifactRepositoryContainer.DEFAULT_MAVEN_CENTRAL_REPO_NAME
import org.gradle.api.artifacts.ArtifactRepositoryContainer.MAVEN_CENTRAL_URL
import org.gradle.api.artifacts.dsl.RepositoryHandler
import org.gradle.api.artifacts.repositories.ArtifactRepository
import org.gradle.api.artifacts.repositories.MavenArtifactRepository
import org.gradle.api.internal.artifacts.dsl.DefaultRepositoryHandler.BINTRAY_JCENTER_URL
import org.gradle.api.internal.artifacts.dsl.DefaultRepositoryHandler.DEFAULT_BINTRAY_JCENTER_REPO_NAME

private fun String?.hostEndsWith(host: String) = this != null && (this == host || endsWith(".$host"))

val RepositoryHandler.hasRepositoryWithDynamicVersionsSupport: Boolean
    get() = any { repo ->
        if (repo is MavenArtifactRepository) {
            val url = repo.url ?: return@any false
            val scheme = url.scheme
            val host = url.host
            if (scheme == "http" || scheme == "https") {
                if (host.hostEndsWith("bintray.com")
                    || host.hostEndsWith("jfrog.org")
                    || host.hostEndsWith("maven.org")
                    || host.hostEndsWith("maven.apache.org")
                    || host.hostEndsWith("sonatype.org")
                ) {
                    return@any true
                }
            }
        }
        return@any false
    }


private val jcenterRepositoryNameRegex = Regex(escapeRegex(DEFAULT_BINTRAY_JCENTER_REPO_NAME) + "\\d*")
private val jcenterRepositoryUri = BINTRAY_JCENTER_URL
private val ArtifactRepository.isJCenter: Boolean
    get() = jcenterRepositoryNameRegex.matches(name)
        && this is MavenArtifactRepository
        && url.toString() == jcenterRepositoryUri

fun RepositoryHandler.forJCenter(configurer: (repository: MavenArtifactRepository) -> Unit) {
    withType(MavenArtifactRepository::class.java).matching(ArtifactRepository::isJCenter).all(configurer)
}

val RepositoryHandler.isJCenterAdded: Boolean
    get() = any(ArtifactRepository::isJCenter)

fun RepositoryHandler.jcenterIfNotAdded() {
    if (isJCenterAdded) return
    jcenter()
}

fun RepositoryHandler.jcenterIfNotAdded(repositoryName: String, configurer: (repository: MavenArtifactRepository) -> Unit = {}) {
    if (isJCenterAdded) return

    if (jcenterRepositoryNameRegex.matches(repositoryName)) {
        throw IllegalArgumentException("Repository name can't match '$jcenterRepositoryNameRegex'")
    }

    maven {
        it.name = repositoryName
        it.setUrl(BINTRAY_JCENTER_URL)
        it.apply(configurer)
    }
}


private val mavenCentralRepositoryNameRegex = Regex(escapeRegex(DEFAULT_MAVEN_CENTRAL_REPO_NAME) + "\\d*")
private val mavenCentralRepositoryUri = MAVEN_CENTRAL_URL
private val ArtifactRepository.isMavenCentral: Boolean
    get() = mavenCentralRepositoryNameRegex.matches(name)
        && this is MavenArtifactRepository
        && url.toString() == mavenCentralRepositoryUri

fun RepositoryHandler.forMavenCentral(configurer: (repository: MavenArtifactRepository) -> Unit) {
    withType(MavenArtifactRepository::class.java).matching(ArtifactRepository::isMavenCentral).all(configurer)
}

val RepositoryHandler.isMavenCentralAdded: Boolean
    get() = any(ArtifactRepository::isMavenCentral)

fun RepositoryHandler.mavenCentralIfNotAdded() {
    if (isMavenCentralAdded) return
    mavenCentral()
}

fun RepositoryHandler.mavenCentralIfNotAdded(repositoryName: String, configurer: (repository: MavenArtifactRepository) -> Unit = {}) {
    if (isMavenCentralAdded) return

    if (mavenCentralRepositoryNameRegex.matches(repositoryName)) {
        throw IllegalArgumentException("Repository name can't match '$mavenCentralRepositoryNameRegex'")
    }

    maven {
        it.name = repositoryName
        it.setUrl(MAVEN_CENTRAL_URL)
        it.apply(configurer)
    }
}
