/*
 * Copyright 2013-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.logstash.logback.composite;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;
import java.util.Objects;
import java.util.ServiceConfigurationError;

import net.logstash.logback.dataformat.DataFormatFactory;
import net.logstash.logback.dataformat.json.JsonDataFormatFactory;
import net.logstash.logback.decorate.CompositeJsonGeneratorDecorator;
import net.logstash.logback.decorate.CompositeMapperBuilderDecorator;
import net.logstash.logback.decorate.CompositeTokenStreamFactoryBuilderDecorator;
import net.logstash.logback.decorate.Decorator;
import net.logstash.logback.decorate.JsonGeneratorDecorator;
import net.logstash.logback.decorate.MapperBuilderDecorator;
import net.logstash.logback.decorate.TokenStreamFactoryBuilderDecorator;
import net.logstash.logback.util.ProxyOutputStream;
import net.logstash.logback.util.ThreadLocalHolder;

import ch.qos.logback.access.common.spi.IAccessEvent;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.spi.ContextAware;
import ch.qos.logback.core.spi.ContextAwareBase;
import ch.qos.logback.core.spi.DeferredProcessingAware;
import ch.qos.logback.core.spi.LifeCycle;
import ch.qos.logback.core.util.CloseUtil;
import tools.jackson.core.JsonEncoding;
import tools.jackson.core.JsonGenerator;
import tools.jackson.core.StreamWriteFeature;
import tools.jackson.core.TSFBuilder;
import tools.jackson.core.TokenStreamFactory;
import tools.jackson.databind.ObjectMapper;
import tools.jackson.databind.SerializationFeature;
import tools.jackson.databind.cfg.MapperBuilder;

/**
 * Formats logstash Events as JSON using {@link JsonProvider}s.
 *
 * <p>The {@link AbstractCompositeJsonFormatter} starts the JSON object ('{'),
 * then delegates writing the contents of the object to the {@link JsonProvider}s,
 * and then ends the JSON object ('}').
 *
 * <p>Jackson {@link JsonGenerator} are initially created with a "disconnected" output stream so they can be
 * reused multiple times with different target output stream.
 * 
 * <p>{@link JsonGenerator} instances are *not* reused after they threw an exception. This is to prevent
 * reusing an instance whose internal state may be unpredictable.
 * 
 * @param <Event> type of event ({@link ILoggingEvent} or {@link IAccessEvent}).
 * 
 * @author brenuart
 */
public abstract class AbstractCompositeJsonFormatter<Event extends DeferredProcessingAware>
        extends ContextAwareBase implements LifeCycle {

    /**
     * Built-in data formats.
     */
    private static final Map<String, String> DATA_FORMATS = Map.of(
            DataFormatFactory.JSON, "net.logstash.logback.dataformat.json.JsonDataFormatFactory",
            DataFormatFactory.YAML, "net.logstash.logback.dataformat.yaml.YamlDataFormatFactory",
            DataFormatFactory.CBOR, "net.logstash.logback.dataformat.cbor.CborDataFormatFactory",
            DataFormatFactory.SMILE, "net.logstash.logback.dataformat.smile.SmileDataFormatFactory");

    /**
     * Factory for creating builders for a data format supported by jackson.
     */
    private DataFormatFactory dataFormatFactory = new JsonDataFormatFactory();

    /**
     * Decorates the builder that builds the {@link #objectMapper}.
     * Allows customization of the {@link #objectMapper}.
     */
    private final CompositeMapperBuilderDecorator mapperBuilderDecorator = new CompositeMapperBuilderDecorator();

    /**
     * Decorates the builder that builds the {@link TokenStreamFactory} used by the {@link #objectMapper}.
     * Allows customization of the {@link TokenStreamFactory}.
     */
    private final CompositeTokenStreamFactoryBuilderDecorator tokenStreamFactoryBuilderDecorator = new CompositeTokenStreamFactoryBuilderDecorator();

    /**
     * Decorates the generators generated by the {@link #objectMapper}.
     * Allows customization of the generators.
     */
    private final CompositeJsonGeneratorDecorator jsonGeneratorDecorator = new CompositeJsonGeneratorDecorator();

    /**
     * Used to create the {@link JsonGenerator}s for generating output.
     */
    private ObjectMapper objectMapper;

    /**
     * The providers that are used to populate the output JSON object.
     */
    private JsonProviders<Event> jsonProviders = new JsonProviders<>();

    private JsonEncoding encoding = JsonEncoding.UTF8;

    private boolean findAndRegisterJacksonModules = true;

    private volatile boolean started;

    private ThreadLocalHolder<JsonFormatter> threadLocalJsonFormatter;


    public AbstractCompositeJsonFormatter(ContextAware declaredOrigin) {
        super(declaredOrigin);
    }

    @Override
    public void start() {
        if (isStarted()) {
            return;
        }
        if (jsonProviders.getProviders().isEmpty()) {
            addError("No providers configured");
        }

        objectMapper = createObjectMapper();

        jsonProviders.setContext(context);
        jsonProviders.setObjectMapper(objectMapper);
        jsonProviders.start();
        
        threadLocalJsonFormatter = new ThreadLocalHolder<>(this::createJsonFormatter);
        started = true;
    }

    @Override
    public void stop() {
        if (isStarted()) {
            threadLocalJsonFormatter.close();
            jsonProviders.stop();
            objectMapper = null;
            started = false;
        }
    }

    @Override
    public boolean isStarted() {
        return started;
    }

    private ObjectMapper createObjectMapper() {
        TokenStreamFactory tokenStreamFactory = createTokenStreamFactory();

        MapperBuilder mapperBuilder = dataFormatFactory.createMapperBuilder(tokenStreamFactory)
                /*
                 * Assume empty beans are ok.
                 */
                .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);

        if (findAndRegisterJacksonModules) {
            try {
                mapperBuilder = mapperBuilder.findAndAddModules();
            } catch (ServiceConfigurationError serviceConfigurationError) {
                addError("Error occurred while dynamically loading jackson modules", serviceConfigurationError);
            }
        }

        mapperBuilder = (MapperBuilder) this.mapperBuilderDecorator.decorate(mapperBuilder);

        return mapperBuilder.build();
    }

    private TokenStreamFactory createTokenStreamFactory() {
        return ((TSFBuilder) tokenStreamFactoryBuilderDecorator.decorate(dataFormatFactory.createTokenStreamFactoryBuilder()))
                /*
                 * Don't let the JsonGenerator close the underlying outputStream and let the
                 * encoder managed it.
                 */
                .disable(StreamWriteFeature.AUTO_CLOSE_TARGET)
                .build();
    }

    /**
     * Write an event in the given output stream.
     * 
     * @param event the event to write
     * @param outputStream the output stream to write the event into
     * @throws IOException thrown upon failure to write the event
     */
    public void writeEvent(Event event, OutputStream outputStream) throws IOException {
        Objects.requireNonNull(outputStream);
        if (!isStarted()) {
            throw new IllegalStateException("Formatter is not started");
        }
        
        try (JsonFormatter formatter = this.threadLocalJsonFormatter.acquire()) {
            formatter.writeEvent(outputStream, event);
        }
    }
    
    
    /**
     * Create a reusable {@link JsonFormatter} bound to a {@link DisconnectedOutputStream}.
     * 
     * @return {@link JsonFormatter} writing JSON content in the output stream
     */
    private JsonFormatter createJsonFormatter() {
        try {
            DisconnectedOutputStream outputStream = new DisconnectedOutputStream();
            JsonGenerator generator = createGenerator(outputStream);
            return new JsonFormatter(outputStream, generator);
        } catch (IOException e) {
            throw new IllegalStateException("Unable to initialize Jackson JSON layer", e);
        }
        
    }
    
    private class JsonFormatter implements ThreadLocalHolder.Lifecycle, Closeable {
        private final JsonGenerator generator;
        private final DisconnectedOutputStream stream;
        private boolean recyclable = true;
        
        JsonFormatter(DisconnectedOutputStream outputStream, JsonGenerator generator) {
            this.stream = Objects.requireNonNull(outputStream);
            this.generator = Objects.requireNonNull(generator);
        }
        
        public void writeEvent(OutputStream outputStream, Event event) throws IOException {
            try {
                this.stream.connect(outputStream);
                writeEventToGenerator(generator, event);
                
            } catch (IOException | RuntimeException e) {
                this.recyclable = false;
                throw e;
                
            } finally {
                this.stream.disconnect();
            }
        }
        
        @Override
        public boolean recycle() {
            return this.recyclable;
        }
        
        @Override
        public void dispose() {
            CloseUtil.closeQuietly(this.generator);
            
            // Note:
            //   The stream is disconnected at this point.
            //   Closing the JsonGenerator may throw additional exception if it is flagged as not recyclable,
            //   meaning it already threw a exception earlier during the writeEvent() method. The generator
            //   is disposed here and won't be reused anymore - we can safely ignore these new exceptions
            //   here.
        }
        
        @Override
        public void close() {
            AbstractCompositeJsonFormatter.this.threadLocalJsonFormatter.release();
        }
    }
    
    private static class DisconnectedOutputStream extends ProxyOutputStream {
        DisconnectedOutputStream() {
            super(null);
        }
        
        public void connect(OutputStream out) {
            this.delegate = out;
        }
        
        public void disconnect() {
            this.delegate = null;
        }
    }
    
    protected void writeEventToGenerator(JsonGenerator generator, Event event) throws IOException {
        generator.writeStartObject();
        jsonProviders.writeTo(generator, event);
        generator.writeEndObject();
        generator.flush();
    }

    protected void prepareForDeferredProcessing(Event event) {
        event.prepareForDeferredProcessing();
        jsonProviders.prepareForDeferredProcessing(event);
    }

    private JsonGenerator createGenerator(OutputStream outputStream) throws IOException {
        return this.jsonGeneratorDecorator.decorate(objectMapper.createGenerator(outputStream, encoding));
    }

    /**
     * Gets the name of the currently configured data format.
     *
     * @return the name of the currently configured data format.
     */
    public String getDataFormat() {
        return dataFormatFactory.getName();
    }

    /**
     * Sets the dataformat to one of the build-in data formats (
     * {@value DataFormatFactory#JSON},
     * {@value DataFormatFactory#YAML},
     * {@value DataFormatFactory#CBOR}, or
     * {@value DataFormatFactory#SMILE}).
     *
     * <p>For non-JSON data formats, the appropriate Jackson data format module must be on the classpath.</p>
     *
     * <p>To set a custom (non-builtin) data format, use {@link #setDataFormatFactory(DataFormatFactory)}</p>
     *
     * @param dataFormat {@value DataFormatFactory#JSON},
     *                   {@value DataFormatFactory#YAML},
     *                   {@value DataFormatFactory#CBOR}, or
     *                    {@value DataFormatFactory#SMILE}
     * @see #setDataFormatFactory(DataFormatFactory)
     */
    public void setDataFormat(String dataFormat) {
        String dataFormatClassName = DATA_FORMATS.get(dataFormat);
        if (dataFormatClassName == null) {
            throw new IllegalArgumentException(String.format("Unknown data format: %s.  Known data formats are: %s", dataFormat, String.join(",", DATA_FORMATS.keySet())));
        }
        try {
            setDataFormatFactory((DataFormatFactory) Class.forName(dataFormatClassName).getConstructor().newInstance());
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException(String.format("Unable to instantiate %s. Ensure the appropriate jackson dataformat module is on the classpath.", dataFormatClassName), e);
        } catch (InstantiationException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            throw new IllegalArgumentException(String.format("Unable to instantiate %s.", dataFormatClassName), e);
        }
    }

    /**
     * Gets the currently configured {@link DataFormatFactory}.
     *
     * @return the currently configured {@link DataFormatFactory}.
     */
    public DataFormatFactory getDataFormatFactory() {
        return dataFormatFactory;
    }

    /**
     * Sets the {@link DataFormatFactory} to be used for output.
     *
     * <p>For built-in formats, prefer using  {@link #setDataFormat(String)}</p>
     *
     * @param dataFormatFactory the {@link DataFormatFactory} to be used for output.
     */
    public void setDataFormatFactory(DataFormatFactory dataFormatFactory) {
        this.dataFormatFactory = Objects.requireNonNull(dataFormatFactory);
    }

    /**
     * Adds the given decorator of type
     * {@link MapperBuilderDecorator},
     * {@link TokenStreamFactoryBuilderDecorator}, or
     * {@link JsonGeneratorDecorator}.
     *
     * <p>These decorators decorate Jackson objects.</p>
     *
     * @param decorator the decorator to add (of type {@link MapperBuilderDecorator}, {@link TokenStreamFactoryBuilderDecorator}, or {@link JsonGeneratorDecorator})
     * @throws IllegalArgumentException if an unsupported decorator is added
     */
    public void addDecorator(Decorator<?> decorator) {
        Objects.requireNonNull(decorator);
        if (decorator instanceof MapperBuilderDecorator mapperBuilderDecoratorToAdd) {
            addMapperBuilderDecorator(mapperBuilderDecoratorToAdd);
        } else if (decorator instanceof TokenStreamFactoryBuilderDecorator tokenStreamFactoryBuilderDecoratorToAdd) {
            addTokenStreamFactoryBuilderDecorator(tokenStreamFactoryBuilderDecoratorToAdd);
        } else if (decorator instanceof JsonGeneratorDecorator jsonGeneratorDecoratorToAdd) {
            addJsonGeneratorDecorator(jsonGeneratorDecoratorToAdd);
        } else {
            throw new IllegalArgumentException("Unknown decorator type:" + decorator.getClass().getName());
        }
    }

    public void addJsonGeneratorDecorator(JsonGeneratorDecorator jsonGeneratorDecoratorToAdd) {
        jsonGeneratorDecorator.addDecorator(jsonGeneratorDecoratorToAdd);
    }

    public void addTokenStreamFactoryBuilderDecorator(TokenStreamFactoryBuilderDecorator tokenStreamFactoryBuilderDecoratorToAdd) {
        tokenStreamFactoryBuilderDecorator.addDecorator(tokenStreamFactoryBuilderDecoratorToAdd);
    }

    public void addMapperBuilderDecorator(MapperBuilderDecorator mapperBuilderDecoratorToAdd) {
        mapperBuilderDecorator.addDecorator(mapperBuilderDecoratorToAdd);
    }

    /**
     * Gets a mutable {@link CompositeTokenStreamFactoryBuilderDecorator}
     * containing all the currently configured {@link TokenStreamFactoryBuilderDecorator}s.
     *
     * @return a mutable {@link CompositeTokenStreamFactoryBuilderDecorator}
     *         containing all the currently configured {@link TokenStreamFactoryBuilderDecorator}s.
     */
    public CompositeTokenStreamFactoryBuilderDecorator getTokenStreamFactoryBuilderDecorator() {
        return tokenStreamFactoryBuilderDecorator;
    }

    /**
     * Gets a mutable {@link CompositeMapperBuilderDecorator}
     * containing all the currently configured {@link MapperBuilderDecorator}s.
     *
     * @return a mutable {@link CompositeMapperBuilderDecorator}
     *         containing all the currently configured {@link MapperBuilderDecorator}s.
     */
    public CompositeMapperBuilderDecorator getMapperBuilderDecorator() {
        return mapperBuilderDecorator;
    }

    /**
     * Gets a mutable {@link CompositeJsonGeneratorDecorator}
     * containing all the currently configured {@link JsonGeneratorDecorator}s.
     *
     * @return a mutable {@link CompositeJsonGeneratorDecorator}
     *         containing all the currently configured {@link JsonGeneratorDecorator}s.
     */
    public CompositeJsonGeneratorDecorator getJsonGeneratorDecorator() {
        return jsonGeneratorDecorator;
    }

    public JsonProviders<Event> getProviders() {
        return jsonProviders;
    }

    public String getEncoding() {
        return encoding.getJavaName();
    }

    public void setEncoding(String encodingName) {
        for (JsonEncoding encoding: JsonEncoding.values()) {
            if (encoding.getJavaName().equalsIgnoreCase(encodingName) || encoding.name().equalsIgnoreCase(encodingName)) {
                this.encoding = encoding;
                return;
            }
        }
        throw new IllegalArgumentException("Unknown encoding " + encodingName);
    }

    public void setProviders(JsonProviders<Event> jsonProviders) {
        this.jsonProviders = Objects.requireNonNull(jsonProviders);
    }

    public boolean isFindAndRegisterJacksonModules() {
        return findAndRegisterJacksonModules;
    }

    public void setFindAndRegisterJacksonModules(boolean findAndRegisterJacksonModules) {
        this.findAndRegisterJacksonModules = findAndRegisterJacksonModules;
    }
}
