package net.openhft.chronicle.wire;

import net.openhft.chronicle.bytes.Bytes;
import net.openhft.chronicle.core.Jvm;
import net.openhft.chronicle.core.io.IORuntimeException;
import net.openhft.chronicle.core.util.ObjectUtils;
import net.openhft.chronicle.core.util.ReadResolvable;

import java.io.Externalizable;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.*;

/**
 * Created by peter on 10/05/16.
 */
public enum SerializationStrategies implements SerializationStrategy {
    MARSHALLABLE {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            ((ReadMarshallable) o).readMarshallable(in.wireIn());
            return o;
        }

        @Override
        public Class type() {
            return Marshallable.class;
        }
    },
    ANY_OBJECT {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            return in.objectWithInferredType(o, ANY_NESTED, null);
        }

        @Override
        public Class type() {
            return Object.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.UNKNOWN;
        }
    },

    ANY_SCALAR {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            return in.objectWithInferredType(o, ANY_NESTED, null);
        }

        @Override
        public Class type() {
            return Object.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.NONE;
        }
    },
    ENUM {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            return in.objectWithInferredType(o, ANY_NESTED, null);
        }

        @Override
        public Class type() {
            return Enum.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.NONE;
        }
    },
    ANY_NESTED {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            Wires.readMarshallable(o, in.wireIn(), true);
            return o;
        }

        @Override
        public Class type() {
            return Object.class;
        }


    },
    DEMARSHALLABLE {
        @Override
        public Object readUsing(Object using, ValueIn in) {
            final DemarshallableWrapper wrapper = (DemarshallableWrapper) using;
            wrapper.demarshallable = Demarshallable.newInstance(wrapper.type, in.wireIn());
            return wrapper;
        }

        @Override
        public Class type() {
            return Demarshallable.class;
        }

        @Override
        public Object newInstance(Class type) {
            return new DemarshallableWrapper(type);
        }
    },
    SERIALIZABLE {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            if (o instanceof Externalizable)
                EXTERNALIZABLE.readUsing(o, in);
            else
                ANY_OBJECT.readUsing(o, in);
            return o;
        }

        @Override
        public Class type() {
            return Serializable.class;
        }
    },
    EXTERNALIZABLE {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            try {
                ((Externalizable) o).readExternal(in.wireIn().objectInput());
            } catch (IOException | ClassNotFoundException e) {
                throw new IORuntimeException(e);
            }
            return o;
        }

        @Override
        public Class type() {
            return Externalizable.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.SEQ;
        }
    },
    MAP {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            Map<Object, Object> map = (Map<Object, Object>) o;
            final WireIn wireIn = in.wireIn();
            long pos = wireIn.bytes().readPosition();
            while (in.hasNext()) {
                Object key = wireIn.readEvent(Object.class);
                map.put(key, in.object());

                // make sure we are progressing.
                long pos2 = wireIn.bytes().readPosition();
                if (pos2 <= pos)
                    if (!Jvm.isDebug())
                        throw new IllegalStateException(wireIn.bytes().toDebugString());
                pos = pos2;
            }
            return o;
        }

        @Override
        public Object newInstance(Class type) {
            return new LinkedHashMap<>();
        }

        @Override
        public Class type() {
            return Map.class;
        }
    },
    SET {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            Set<Object> set = (Set<Object>) o;
            final WireIn wireIn = in.wireIn();
            final Bytes<?> bytes = wireIn.bytes();
            long pos = bytes.readPosition();
            while (in.hasNextSequenceItem()) {
                final Object object = in.object();
                set.add(object);

                // make sure we are progressing.
                long pos2 = bytes.readPosition();
                if (pos2 <= pos)
                    if (!Jvm.isDebug())
                        throw new IllegalStateException(bytes.toDebugString());
                pos = pos2;
            }
            return o;
        }

        @Override
        public Object newInstance(Class type) {
            return new LinkedHashSet<>();
        }

        @Override
        public Class type() {
            return Set.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.SEQ;
        }
    },
    LIST {
        @Override
        public Object readUsing(Object o, ValueIn in) {
            List<Object> list = (List<Object>) o;
            final WireIn wireIn = in.wireIn();
            long pos = wireIn.bytes().readPosition();
            while (in.hasNextSequenceItem()) {
                list.add(in.object());

                // make sure we are progressing.
                long pos2 = wireIn.bytes().readPosition();
                if (pos2 <= pos)
                    if (!Jvm.isDebug())
                        throw new IllegalStateException(wireIn.bytes().toDebugString());
                pos = pos2;
            }
            return o;
        }

        @Override
        public Object newInstance(Class type) {
            return new ArrayList<>();
        }

        @Override
        public Class type() {
            return List.class;
        }

        @Override
        public BracketType bracketType() {
            return BracketType.SEQ;
        }
    },
    ARRAY {
        @Override
        public Object readUsing(Object using, ValueIn in) {
            ArrayWrapper wrapper = (ArrayWrapper) using;
            final Class componentType = wrapper.type.getComponentType();
            List list = new ArrayList<>();
            while (in.hasNextSequenceItem())
                list.add(in.object(componentType));
            wrapper.array = list.toArray((Object[]) Array.newInstance(componentType, list.size()));
            return wrapper;
        }

        @Override
        public Class type() {
            return Object[].class;
        }

        @Override
        public Object newInstance(Class type) {
            return new ArrayWrapper(type);
        }

        @Override
        public BracketType bracketType() {
            return BracketType.SEQ;
        }
    };


    @Override
    public Object newInstance(Class type) {
        return ObjectUtils.newInstance(type);
    }

    @Override
    public BracketType bracketType() {
        return BracketType.MAP;
    }

    static class ArrayWrapper implements ReadResolvable<Object[]> {
        final Class type;
        Object[] array;

        ArrayWrapper(Class type) {
            this.type = type;
        }

        @Override
        public Object[] readResolve() {
            return array;
        }
    }

    static class DemarshallableWrapper implements ReadResolvable<Demarshallable> {
        final Class type;
        Demarshallable demarshallable;

        DemarshallableWrapper(Class type) {
            this.type = type;
        }

        @Override
        public Demarshallable readResolve() {
            return demarshallable;
        }
    }
}
