/*
 * Copyright 2016 higherfrequencytrading.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.chronicle.wire;

import net.openhft.chronicle.bytes.Bytes;
import net.openhft.chronicle.bytes.BytesStore;
import net.openhft.chronicle.bytes.VanillaBytes;
import net.openhft.chronicle.core.ClassLocal;
import net.openhft.chronicle.core.annotation.ForceInline;
import net.openhft.chronicle.core.io.IORuntimeException;
import net.openhft.chronicle.core.pool.ClassAliasPool;
import net.openhft.chronicle.core.pool.StringBuilderPool;
import net.openhft.chronicle.core.threads.ThreadLocalHelper;
import net.openhft.chronicle.core.util.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.Externalizable;
import java.io.File;
import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Function;

/**
 * Created by peter on 31/08/15.
 */
public enum Wires {
    ;
    public static final int LENGTH_MASK = -1 >>> 2;
    public static final int NOT_COMPLETE = 1 << 31;
    @Deprecated
    public static final int NOT_READY = NOT_COMPLETE;
    public static final int META_DATA = 1 << 30;
    public static final int UNKNOWN_LENGTH = 0x0;
    public static final int MAX_LENGTH = (1 << 30) - 1;

    // value to use when the message is not ready and of an unknown length
    public static final int NOT_COMPLETE_UNKNOWN_LENGTH = NOT_COMPLETE | UNKNOWN_LENGTH;
    // value to use when no more data is possible e.g. on a roll.
    public static final int END_OF_DATA = NOT_COMPLETE | META_DATA | UNKNOWN_LENGTH;
    public static final int NOT_INITIALIZED = 0x0;
    public static final Bytes<?> NO_BYTES = new VanillaBytes<>(BytesStore.empty());
    public static final WireIn EMPTY = new BinaryWire(NO_BYTES);
    public static final int SPB_HEADER_SIZE = 4;
    public static final List<Function<Class, SerializationStrategy>> CLASS_STRATEGY_FUNCTIONS = new CopyOnWriteArrayList<>();
    public static final ClassLocal<SerializationStrategy> CLASS_STRATEGY = ClassLocal.withInitial(c -> {
        for (@NotNull Function<Class, SerializationStrategy> func : CLASS_STRATEGY_FUNCTIONS) {
            final SerializationStrategy strategy = func.apply(c);
            if (strategy != null)
                return strategy;
        }
        return SerializationStrategies.ANY_OBJECT;
    });
    static final ClassLocal<FieldInfoPair> FIELD_INFOS = ClassLocal.withInitial(VanillaFieldInfo::lookupClass);
    static final StringBuilderPool SBP = new StringBuilderPool();

    static {
        CLASS_STRATEGY_FUNCTIONS.add(SerializeEnum.INSTANCE);
        CLASS_STRATEGY_FUNCTIONS.add(SerializeJavaLang.INSTANCE);
        CLASS_STRATEGY_FUNCTIONS.add(SerializeMarshallables.INSTANCE);
        CLASS_STRATEGY_FUNCTIONS.add(SerializeBytes.INSTANCE);
        ClassAliasPool.CLASS_ALIASES.addAlias(VanillaFieldInfo.class, "FieldInfo");
    }

    @Nullable
    public static <T> T read(@NotNull Class<T> tClass, ValueIn in) {
        final SerializationStrategy<T> strategy = CLASS_STRATEGY.get(tClass);
        return strategy.read(in, tClass);
    }

    /**
     * This decodes some Bytes where the first 4-bytes is the length.  e.g. Wire.writeDocument wrote
     * it. <a href="https://github.com/OpenHFT/RFC/tree/master/Size-Prefixed-Blob">Size Prefixed
     * Blob</a>
     *
     * @param bytes to decode
     * @return as String
     */
    public static String fromSizePrefixedBlobs(@NotNull Bytes bytes) {
        return WireDumper.of(bytes).asString();
    }

    public static String fromSizePrefixedBlobs(@NotNull Bytes bytes, long position) {
        final long limit = bytes.readLimit();
        if (position > limit)
            return "";
        return WireDumper.of(bytes).asString(position, limit - position);
    }

    public static String fromSizePrefixedBlobs(@NotNull DocumentContext dc) {
        Wire wire = dc.wire();
        Bytes<?> bytes = wire.bytes();
        if (wire instanceof TextWire) {
            return bytes.toString();
        }

        long headerPosition;

        long length;
        if ("BufferedTailer".equals(dc.getClass().getSimpleName())) {
            length = wire.bytes().readLimit();
            int metaDataBit = dc.isMetaData() ? Wires.META_DATA : 0;
            int header = metaDataBit | toIntU30(length, "Document length %,d out of 30-bit int range.");

            Bytes tempBytes = Bytes.allocateElasticDirect();
            try {
                tempBytes.writeOrderedInt(header);
                tempBytes.write(((BinaryReadDocumentContext) dc).wire.bytes, 0, ((BinaryReadDocumentContext) dc).wire.bytes.readLimit());

                final WireType wireType = WireType.valueOf(wire);

                assert wireType != null;
                Wire tempWire = wireType.apply(tempBytes);

                return WireDumper.of(tempWire).asString(0, length + 4);

            } finally {
                tempBytes.release();
            }

        } else {
            if (dc instanceof BinaryReadDocumentContext) {
                long start = ((BinaryReadDocumentContext) dc).lastStart;
                if (start != -1)
                    headerPosition = start;
                else
                    headerPosition = bytes.readPosition() - 4;
            } else
                headerPosition = bytes.readPosition() - 4;

            length = Wires.lengthOf(bytes.readInt(headerPosition));
        }

        return WireDumper.of(wire).asString(headerPosition, (long) (length + 4));
    }

    public static String fromSizePrefixedBlobs(@NotNull WireIn wireIn) {
        return WireDumper.of(wireIn).asString();
    }

    @Nullable
    public static CharSequence asText(@NotNull WireIn wireIn) {
        long pos = wireIn.bytes().readPosition();
        try {
            Bytes bytes = acquireBytes();
            wireIn.copyTo(new TextWire(bytes).addTimeStamps(true));
            return bytes;
        } finally {
            wireIn.bytes().readPosition(pos);
        }
    }

    public static StringBuilder acquireStringBuilder() {
        return SBP.acquireStringBuilder();
    }

    public static int lengthOf(int len) {
        final int len0 = len & LENGTH_MASK;
//        if (len0 > 1 << 20)
//            System.out.println("len: " + len0);
        return len0;
    }

    public static boolean isReady(int header) {
        return (header & NOT_COMPLETE) == 0 && header != 0;
    }

    public static boolean isNotComplete(int header) {
        return (header & NOT_COMPLETE) != 0 || header == 0;
    }

    public static boolean isReadyData(int header) {
        return ((header & (META_DATA | NOT_COMPLETE)) == 0) && (header != 0);
    }

    @Deprecated
    public static boolean isData(long len) {
        return isData((int) len);
    }

    public static boolean isData(int len) {
        return (len & META_DATA) == 0;
    }

    public static boolean isReadyMetaData(int len) {
        return (len & (META_DATA | NOT_COMPLETE)) == META_DATA;
    }

    public static boolean isKnownLength(int len) {
        return (len & (META_DATA | LENGTH_MASK)) != UNKNOWN_LENGTH;
    }

    public static boolean isNotInitialized(int len) {
        return len == NOT_INITIALIZED;
    }

    public static int toIntU30(long l, @NotNull String error) {
        if (l < 0 || l > LENGTH_MASK)
            throw new IllegalStateException(String.format(error, l));
        return (int) l;
    }

    public static boolean acquireLock(@NotNull BytesStore store, long position) {
        return store.compareAndSwapInt(position, NOT_INITIALIZED, NOT_COMPLETE);
    }

    public static boolean exceedsMaxLength(long length) {
        return length > LENGTH_MASK;
    }

    @ForceInline
    public static <T extends WriteMarshallable> long writeData(
            @NotNull WireOut wireOut,
            @NotNull T writer) {
        return WireInternal.writeData(wireOut, false, false, writer);
    }

    @ForceInline
    public static long readWire(@NotNull WireIn wireIn, long size, @NotNull ReadMarshallable readMarshallable) {
        @NotNull final Bytes<?> bytes = wireIn.bytes();
        final long limit0 = bytes.readLimit();
        final long limit = bytes.readPosition() + size;
        try {
            bytes.readLimit(limit);
            readMarshallable.readMarshallable(wireIn);
        } finally {
            bytes.readLimit(limit0);
            bytes.readPosition(limit);
        }

        return bytes.readPosition();
    }

    @Nullable
    public static Bytes acquireBytes() {
        Bytes bytes = ThreadLocalHelper.getTL(WireInternal.BYTES_TL, Bytes::allocateElasticDirect);
        bytes.clear();
        return bytes;
    }

    @Nullable
    public static Wire acquireBinaryWire() {
        Wire wire = ThreadLocalHelper.getTL(WireInternal.BINARY_WIRE_TL, () -> new BinaryWire(acquireBytes()));
        wire.clear();
        return wire;
    }

    @Nullable
    public static Bytes acquireAnotherBytes() {
        Bytes bytes = ThreadLocalHelper.getTL(WireInternal.BYTES_TL, Bytes::allocateElasticDirect);
        bytes.clear();
        return bytes;
    }

    public static String fromSizePrefixedBlobs(@NotNull Bytes<?> bytes, long position, long length) {
        return WireDumper.of(bytes).asString(position, length);
    }

    public static void readMarshallable(@NotNull Object marshallable, @NotNull WireIn wire, boolean overwrite) {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(marshallable.getClass());
        wm.readMarshallable(marshallable, wire, wm.defaultValue(), overwrite);
    }

    public static void writeMarshallable(@NotNull Object marshallable, @NotNull WireOut wire) {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(marshallable.getClass());
        wm.writeMarshallable(marshallable, wire);
    }

    public static void writeMarshallable(@NotNull Object marshallable, @NotNull WireOut wire, boolean writeDefault) {
        WireMarshaller marshaller = WireMarshaller.WIRE_MARSHALLER_CL.get(marshallable.getClass());
        if (writeDefault)
            marshaller.writeMarshallable(marshallable, wire);
        else
            marshaller.writeMarshallable(marshallable, wire, marshaller.defaultValue(), false);
    }

    public static void writeMarshallable(@NotNull Object marshallable, @NotNull WireOut wire, @NotNull Object previous, boolean copy) {
        assert marshallable.getClass() == previous.getClass();
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(marshallable.getClass());
        wm.writeMarshallable(marshallable, wire, previous, copy);
    }

    public static void writeKey(@NotNull Object marshallable, Bytes bytes) {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(marshallable.getClass());
        wm.writeKey(marshallable, bytes);
    }

    @NotNull
    public static <T extends Marshallable> T deepCopy(@NotNull T marshallable) {
        Wire wire = acquireBinaryWire();
        marshallable.writeMarshallable(wire);
        @NotNull T t = (T) ObjectUtils.newInstance(marshallable.getClass());
        t.readMarshallable(wire);
        return t;
    }

    @NotNull
    public static <T> T copyTo(Object source, @NotNull T target) {
        Wire wire = acquireBinaryWire();
        wire.getValueOut().object(source);
        wire.getValueIn().typePrefix(); // drop the type prefix.
        wire.getValueIn().object(target, target.getClass());
        return target;
    }

    @NotNull
    public static <T> T project(Class<T> tClass, Object source) {
        T target = ObjectUtils.newInstance(tClass);
        Wires.copyTo(source, target);
        return target;
    }

    public static boolean isEquals(@NotNull Object o1, @NotNull Object o2) {
        if (o1.getClass() != o2.getClass())
            return false;
        return WireMarshaller.WIRE_MARSHALLER_CL.get(o1.getClass())
                .isEqual(o1, o2);
    }

    @NotNull
    public static List<FieldInfo> fieldInfos(@NotNull Class aClass) {
        return FIELD_INFOS.get(aClass).list;
    }

    public static FieldInfo fieldInfo(@NotNull Class aClass, String name) {
        return FIELD_INFOS.get(aClass).map.get(name);
    }

    public static boolean isEndOfFile(int num) {
        return num == END_OF_DATA;
    }

    @Nullable
    public static <T> T getField(@NotNull Object o, String name, Class<T> tClass) throws NoSuchFieldException {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(o.getClass());
        Object value = wm.getField(o, name);
        return ObjectUtils.convertTo(tClass, value);
    }

    public static void setField(@NotNull Object o, String name, Object value) throws NoSuchFieldException {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(o.getClass());
        wm.setField(o, name, value);
    }

    public static void reset(@NotNull Object o) {
        WireMarshaller wm = WireMarshaller.WIRE_MARSHALLER_CL.get(o.getClass());
        wm.reset(o);
    }

    enum SerializeBytes implements Function<Class, SerializationStrategy> {
        INSTANCE;

        @Override
        public SerializationStrategy apply(@NotNull Class aClass) {
            switch (aClass.getName()) {
                case "net.openhft.chronicle.bytes.BytesStore":
                    return ScalarStrategy.of(BytesStore.class, (o, in) -> in.bytesStore());
                default:
                    return null;
            }
        }
    }

    enum SerializeEnum implements Function<Class, SerializationStrategy> {
        INSTANCE;

        @Nullable
        static SerializationStrategy getSerializationStrategy(@NotNull Class aClass) {
            if (Enum.class.isAssignableFrom(aClass))
                return SerializationStrategies.ENUM;
            return null;
        }

        @Nullable
        @Override
        public SerializationStrategy apply(@NotNull Class aClass) {
            return getSerializationStrategy(aClass);
        }
    }

    enum SerializeJavaLang implements Function<Class, SerializationStrategy> {
        INSTANCE;

        @Override
        public SerializationStrategy apply(@NotNull Class aClass) {
            switch (aClass.getName()) {
                case "[B":
                    return ScalarStrategy.of(byte[].class, (o, in) -> in.bytes());

                case "java.lang.StringBuilder":
                    return ScalarStrategy.of(StringBuilder.class, (o, in) -> {
                        StringBuilder builder = (o == null)
                                ? acquireStringBuilder()
                                : o;
                        in.textTo(builder);
                        return o;
                    });

                case "java.lang.String":
                    return ScalarStrategy.of(String.class, (o, in) -> in.text());

                case "java.lang.Object":
                    return SerializationStrategies.ANY_OBJECT;

                case "java.lang.Class":
                    return ScalarStrategy.of(Class.class, (o, in) -> {
                        try {
                            return ClassAliasPool.CLASS_ALIASES.forName(in.text());
                        } catch (ClassNotFoundException e) {
                            throw new IORuntimeException(e);
                        }
                    });

                case "java.lang.Boolean":
                    return ScalarStrategy.of(Boolean.class, (o, in) -> in.bool());

                case "java.lang.Byte":
                    return ScalarStrategy.of(Byte.class, (o, in) -> in.int8());

                case "java.lang.Short":
                    return ScalarStrategy.of(Short.class, (o, in) -> in.int16());

                case "java.lang.Character":
                    return ScalarStrategy.of(Character.class, (o, in) -> {
                        //noinspection unchecked
                        @Nullable final String text = in.text();
                        if (text == null || text.length() == 0)
                            return null;
                        return text.charAt(0);
                    });

                case "java.lang.Integer":
                    return ScalarStrategy.of(Integer.class, (o, in) -> in.int32());

                case "java.lang.Float":
                    return ScalarStrategy.of(Float.class, (o, in) -> in.float32());

                case "java.lang.Long":
                    return ScalarStrategy.of(Long.class, (o, in) -> in.int64());

                case "java.lang.Double":
                    return ScalarStrategy.of(Double.class, (o, in) -> in.float64());

                case "java.time.LocalTime":
                    return ScalarStrategy.of(LocalTime.class, (o, in) -> in.time());

                case "java.time.LocalDate":
                    return ScalarStrategy.of(LocalDate.class, (o, in) -> in.date());

                case "java.time.LocalDateTime":
                    return ScalarStrategy.of(LocalDateTime.class, (o, in) -> in.dateTime());

                case "java.time.ZonedDateTime":
                    return ScalarStrategy.of(ZonedDateTime.class, (o, in) -> in.zonedDateTime());

                case "java.io.File":
                    return ScalarStrategy.text(File.class, File::new);

                case "java.util.UUID":
                    return ScalarStrategy.of(UUID.class, (o, in) -> in.uuid());

                case "java.math.BigInteger":
                    return ScalarStrategy.text(BigInteger.class, BigInteger::new);

                case "java.math.BigDecimal":
                    return ScalarStrategy.text(BigDecimal.class, BigDecimal::new);

                default:
                    if (aClass.isPrimitive())
                        return SerializationStrategies.ANY_SCALAR;
                    if (aClass.isArray()) {
                        final Class componentType = aClass.getComponentType();
                        if (componentType.isPrimitive())
                            return SerializationStrategies.PRIM_ARRAY;
                        return SerializationStrategies.ARRAY;
                    }
                    if (Enum.class.isAssignableFrom(aClass)) {
                        @Nullable final SerializationStrategy ss = SerializeMarshallables.getSerializationStrategy(aClass);
                        return ss == null ? SerializationStrategies.ENUM : ss;
                    }
                    return null;
            }
        }
    }

    enum SerializeMarshallables implements Function<Class, SerializationStrategy> {
        INSTANCE;

        @Nullable
        static SerializationStrategy getSerializationStrategy(@NotNull Class aClass) {
            if (Demarshallable.class.isAssignableFrom(aClass))
                return SerializationStrategies.DEMARSHALLABLE;
            if (ReadMarshallable.class.isAssignableFrom(aClass))
                return SerializationStrategies.MARSHALLABLE;
            return null;
        }

        @Override
        public SerializationStrategy apply(@NotNull Class aClass) {
            @Nullable SerializationStrategy x = getSerializationStrategy(aClass);
            if (x != null) return x;
            if (Map.class.isAssignableFrom(aClass))
                return SerializationStrategies.MAP;
            if (Set.class.isAssignableFrom(aClass))
                return SerializationStrategies.SET;
            if (List.class.isAssignableFrom(aClass))
                return SerializationStrategies.LIST;
            if (Externalizable.class.isAssignableFrom(aClass))
                return SerializationStrategies.EXTERNALIZABLE;
            if (Serializable.class.isAssignableFrom(aClass))
                return SerializationStrategies.ANY_NESTED;
            return null;
        }
    }

    static class FieldInfoPair {
        static final FieldInfoPair EMPTY = new FieldInfoPair(Collections.emptyList(), Collections.emptyMap());

        @NotNull
        final List<FieldInfo> list;
        @NotNull
        final Map<String, FieldInfo> map;

        public FieldInfoPair(@NotNull List<FieldInfo> list, @NotNull Map<String, FieldInfo> map) {
            this.list = list;
            this.map = map;
        }
    }
}
