////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2013 Saxonica Limited.
// This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0.
// If a copy of the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
// This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public License, v. 2.0.
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

package net.sf.saxon.regex;

import net.sf.saxon.serialize.charcode.XMLCharacterData;
import net.sf.saxon.z.*;

import java.util.HashMap;
import java.util.Map;

/**
 * Data for Regular expression character categories. This is derived from Unicode 6.0.0. Since Saxon 9.4,
 * we no longer make use of Java's support for character categories since there are too many differences
 * from Unicode.
 */
public class Categories {


    private final static HashMap<String, IntPredicate> CATEGORIES = new HashMap<String, IntPredicate>(30);

    static {


        CATEGORIES.put("Cc", pred(new int[]{
                0x1, 0x1F,
                0x7F, 0x9F
        }));


        CATEGORIES.put("Cf", pred(new int[]{
                0xAD, 0xAD,
                0x600, 0x603,
                0x6DD, 0x6DD,
                0x70F, 0x70F,
                0x17B4, 0x17B5,
                0x200B, 0x200F,
                0x202A, 0x202E,
                0x2060, 0x2064,
                0x206A, 0x206F,
                0xFEFF, 0xFEFF,
                0xFFF9, 0xFFFB,
                0x110BD, 0x110BD,
                0x1D173, 0x1D17A,
                0xE0001, 0xE0001,
                0xE0020, 0xE007F
        }));

        CATEGORIES.put("Cn", pred(new int[]{
                0x0378, 0x0379,
                0x037F, 0x0383,
                0x038B, 0x038B,
                0x038D, 0x038D,
                0x03A2, 0x03A2,
                0x0528, 0x0530,
                0x0557, 0x0558,
                0x0560, 0x0560,
                0x0588, 0x0588,
                0x058B, 0x058F,
                0x0590, 0x0590,
                0x05C8, 0x05CF,
                0x05EB, 0x05EF,
                0x05F5, 0x05FF,
                0x0604, 0x0605,
                0x061C, 0x061D,
                0x070E, 0x070E,
                0x074B, 0x074C,
                0x07B2, 0x07BF,
                0x07FB, 0x07FF,
                0x082E, 0x082F,
                0x083F, 0x083F,
                0x085C, 0x085D,
                0x085F, 0x08FF,
                0x0978, 0x0978,
                0x0980, 0x0980,
                0x0984, 0x0984,
                0x098D, 0x098E,
                0x0991, 0x0992,
                0x09A9, 0x09A9,
                0x09B1, 0x09B1,
                0x09B3, 0x09B5,
                0x09BA, 0x09BB,
                0x09C5, 0x09C6,
                0x09C9, 0x09CA,
                0x09CF, 0x09D6,
                0x09D8, 0x09DB,
                0x09DE, 0x09DE,
                0x09E4, 0x09E5,
                0x09FC, 0x0A00,
                0x0A04, 0x0A04,
                0x0A0B, 0x0A0E,
                0x0A11, 0x0A12,
                0x0A29, 0x0A29,
                0x0A31, 0x0A31,
                0x0A34, 0x0A34,
                0x0A37, 0x0A37,
                0x0A3A, 0x0A3B,
                0x0A3D, 0x0A3D,
                0x0A43, 0x0A46,
                0x0A49, 0x0A4A,
                0x0A4E, 0x0A50,
                0x0A52, 0x0A58,
                0x0A5D, 0x0A5D,
                0x0A5F, 0x0A65,
                0x0A76, 0x0A80,
                0x0A84, 0x0A84,
                0x0A8E, 0x0A8E,
                0x0A92, 0x0A92,
                0x0AA9, 0x0AA9,
                0x0AB1, 0x0AB1,
                0x0AB4, 0x0AB4,
                0x0ABA, 0x0ABB,
                0x0AC6, 0x0AC6,
                0x0ACA, 0x0ACA,
                0x0ACE, 0x0ACF,
                0x0AD1, 0x0ADF,
                0x0AE4, 0x0AE5,
                0x0AF0, 0x0AF0,
                0x0AF2, 0x0B00,
                0x0B04, 0x0B04,
                0x0B0D, 0x0B0E,
                0x0B11, 0x0B12,
                0x0B29, 0x0B29,
                0x0B31, 0x0B31,
                0x0B34, 0x0B34,
                0x0B3A, 0x0B3B,
                0x0B45, 0x0B46,
                0x0B49, 0x0B4A,
                0x0B4E, 0x0B55,
                0x0B58, 0x0B5B,
                0x0B5E, 0x0B5E,
                0x0B64, 0x0B65,
                0x0B78, 0x0B81,
                0x0B84, 0x0B84,
                0x0B8B, 0x0B8D,
                0x0B91, 0x0B91,
                0x0B96, 0x0B98,
                0x0B9B, 0x0B9B,
                0x0B9D, 0x0B9D,
                0x0BA0, 0x0BA2,
                0x0BA5, 0x0BA7,
                0x0BAB, 0x0BAD,
                0x0BBA, 0x0BBD,
                0x0BC3, 0x0BC5,
                0x0BC9, 0x0BC9,
                0x0BCE, 0x0BCF,
                0x0BD1, 0x0BD6,
                0x0BD8, 0x0BE5,
                0x0BFB, 0x0C00,
                0x0C04, 0x0C04,
                0x0C0D, 0x0C0D,
                0x0C11, 0x0C11,
                0x0C29, 0x0C29,
                0x0C34, 0x0C34,
                0x0C3A, 0x0C3C,
                0x0C45, 0x0C45,
                0x0C49, 0x0C49,
                0x0C4E, 0x0C54,
                0x0C57, 0x0C57,
                0x0C5A, 0x0C5F,
                0x0C64, 0x0C65,
                0x0C70, 0x0C77,
                0x0C80, 0x0C81,
                0x0C84, 0x0C84,
                0x0C8D, 0x0C8D,
                0x0C91, 0x0C91,
                0x0CA9, 0x0CA9,
                0x0CB4, 0x0CB4,
                0x0CBA, 0x0CBB,
                0x0CC5, 0x0CC5,
                0x0CC9, 0x0CC9,
                0x0CCE, 0x0CD4,
                0x0CD7, 0x0CDD,
                0x0CDF, 0x0CDF,
                0x0CE4, 0x0CE5,
                0x0CF0, 0x0CF0,
                0x0CF3, 0x0D01,
                0x0D04, 0x0D04,
                0x0D0D, 0x0D0D,
                0x0D11, 0x0D11,
                0x0D3B, 0x0D3C,
                0x0D45, 0x0D45,
                0x0D49, 0x0D49,
                0x0D4F, 0x0D56,
                0x0D58, 0x0D5F,
                0x0D64, 0x0D65,
                0x0D76, 0x0D78,
                0x0D80, 0x0D81,
                0x0D84, 0x0D84,
                0x0D97, 0x0D99,
                0x0DB2, 0x0DB2,
                0x0DBC, 0x0DBC,
                0x0DBE, 0x0DBF,
                0x0DC7, 0x0DC9,
                0x0DCB, 0x0DCE,
                0x0DD5, 0x0DD5,
                0x0DD7, 0x0DD7,
                0x0DE0, 0x0DF1,
                0x0DF5, 0x0E00,
                0x0E3B, 0x0E3E,
                0x0E5C, 0x0E80,
                0x0E83, 0x0E83,
                0x0E85, 0x0E86,
                0x0E89, 0x0E89,
                0x0E8B, 0x0E8C,
                0x0E8E, 0x0E93,
                0x0E98, 0x0E98,
                0x0EA0, 0x0EA0,
                0x0EA4, 0x0EA4,
                0x0EA6, 0x0EA6,
                0x0EA8, 0x0EA9,
                0x0EAC, 0x0EAC,
                0x0EBA, 0x0EBA,
                0x0EBE, 0x0EBF,
                0x0EC5, 0x0EC5,
                0x0EC7, 0x0EC7,
                0x0ECE, 0x0ECF,
                0x0EDA, 0x0EDB,
                0x0EDE, 0x0EFF,
                0x0F48, 0x0F48,
                0x0F6D, 0x0F70,
                0x0F98, 0x0F98,
                0x0FBD, 0x0FBD,
                0x0FCD, 0x0FCD,
                0x0FDB, 0x0FFF,
                0x10C6, 0x10CF,
                0x10FD, 0x10FF,
                0x1249, 0x1249,
                0x124E, 0x124F,
                0x1257, 0x1257,
                0x1259, 0x1259,
                0x125E, 0x125F,
                0x1289, 0x1289,
                0x128E, 0x128F,
                0x12B1, 0x12B1,
                0x12B6, 0x12B7,
                0x12BF, 0x12BF,
                0x12C1, 0x12C1,
                0x12C6, 0x12C7,
                0x12D7, 0x12D7,
                0x1311, 0x1311,
                0x1316, 0x1317,
                0x135B, 0x135C,
                0x137D, 0x137F,
                0x139A, 0x139F,
                0x13F5, 0x13FF,
                0x169D, 0x169F,
                0x16F1, 0x16FF,
                0x170D, 0x170D,
                0x1715, 0x171F,
                0x1737, 0x173F,
                0x1754, 0x175F,
                0x176D, 0x176D,
                0x1771, 0x1771,
                0x1774, 0x177F,
                0x17DE, 0x17DF,
                0x17EA, 0x17EF,
                0x17FA, 0x17FF,
                0x180F, 0x180F,
                0x181A, 0x181F,
                0x1878, 0x187F,
                0x18AB, 0x18AF,
                0x18F6, 0x18FF,
                0x191D, 0x191F,
                0x192C, 0x192F,
                0x193C, 0x193F,
                0x1941, 0x1943,
                0x196E, 0x196F,
                0x1975, 0x197F,
                0x19AC, 0x19AF,
                0x19CA, 0x19CF,
                0x19DB, 0x19DD,
                0x1A1C, 0x1A1D,
                0x1A5F, 0x1A5F,
                0x1A7D, 0x1A7E,
                0x1A8A, 0x1A8F,
                0x1A9A, 0x1A9F,
                0x1AAE, 0x1AFF,
                0x1B4C, 0x1B4F,
                0x1B7D, 0x1B7F,
                0x1BAB, 0x1BAD,
                0x1BBA, 0x1BBF,
                0x1BF4, 0x1BFB,
                0x1C38, 0x1C3A,
                0x1C4A, 0x1C4C,
                0x1C80, 0x1CCF,
                0x1CF3, 0x1CFF,
                0x1DE7, 0x1DFB,
                0x1F16, 0x1F17,
                0x1F1E, 0x1F1F,
                0x1F46, 0x1F47,
                0x1F4E, 0x1F4F,
                0x1F58, 0x1F58,
                0x1F5A, 0x1F5A,
                0x1F5C, 0x1F5C,
                0x1F5E, 0x1F5E,
                0x1F7E, 0x1F7F,
                0x1FB5, 0x1FB5,
                0x1FC5, 0x1FC5,
                0x1FD4, 0x1FD5,
                0x1FDC, 0x1FDC,
                0x1FF0, 0x1FF1,
                0x1FF5, 0x1FF5,
                0x1FFF, 0x1FFF,
                0x2065, 0x2069,
                0x2072, 0x2073,
                0x208F, 0x208F,
                0x209D, 0x209F,
                0x20BA, 0x20CF,
                0x20F1, 0x20FF,
                0x218A, 0x218F,
                0x23F4, 0x23FF,
                0x2427, 0x243F,
                0x244B, 0x245F,
                0x2700, 0x2700,
                0x27CB, 0x27CB,
                0x27CD, 0x27CD,
                0x2B4D, 0x2B4F,
                0x2B5A, 0x2BFF,
                0x2C2F, 0x2C2F,
                0x2C5F, 0x2C5F,
                0x2CF2, 0x2CF8,
                0x2D26, 0x2D2F,
                0x2D66, 0x2D6E,
                0x2D71, 0x2D7E,
                0x2D97, 0x2D9F,
                0x2DA7, 0x2DA7,
                0x2DAF, 0x2DAF,
                0x2DB7, 0x2DB7,
                0x2DBF, 0x2DBF,
                0x2DC7, 0x2DC7,
                0x2DCF, 0x2DCF,
                0x2DD7, 0x2DD7,
                0x2DDF, 0x2DDF,
                0x2E32, 0x2E7F,
                0x2E9A, 0x2E9A,
                0x2EF4, 0x2EFF,
                0x2FD6, 0x2FEF,
                0x2FFC, 0x2FFF,
                0x3040, 0x3040,
                0x3097, 0x3098,
                0x3100, 0x3104,
                0x312E, 0x3130,
                0x318F, 0x318F,
                0x31BB, 0x31BF,
                0x31E4, 0x31EF,
                0x321F, 0x321F,
                0x32FF, 0x32FF,
                0x4DB6, 0x4DBF,
                0x9FCC, 0x9FFF,
                0xA48D, 0xA48F,
                0xA4C7, 0xA4CF,
                0xA62C, 0xA63F,
                0xA674, 0xA67B,
                0xA698, 0xA69F,
                0xA6F8, 0xA6FF,
                0xA78F, 0xA78F,
                0xA792, 0xA79F,
                0xA7AA, 0xA7F9,
                0xA82C, 0xA82F,
                0xA83A, 0xA83F,
                0xA878, 0xA87F,
                0xA8C5, 0xA8CD,
                0xA8DA, 0xA8DF,
                0xA8FC, 0xA8FF,
                0xA954, 0xA95E,
                0xA97D, 0xA97F,
                0xA9CE, 0xA9CE,
                0xA9DA, 0xA9DD,
                0xA9E0, 0xA9FF,
                0xAA37, 0xAA3F,
                0xAA4E, 0xAA4F,
                0xAA5A, 0xAA5B,
                0xAA7C, 0xAA7F,
                0xAAC3, 0xAADA,
                0xAAE0, 0xAB00,
                0xAB07, 0xAB08,
                0xAB0F, 0xAB10,
                0xAB17, 0xAB1F,
                0xAB27, 0xAB27,
                0xAB2F, 0xABBF,
                0xABEE, 0xABEF,
                0xABFA, 0xABFF,
                0xD7A4, 0xD7AF,
                0xD7C7, 0xD7CA,
                0xD7FC, 0xD7FF,
                0xFA2E, 0xFA2F,
                0xFA6E, 0xFA6F,
                0xFADA, 0xFAFF,
                0xFB07, 0xFB12,
                0xFB18, 0xFB1C,
                0xFB37, 0xFB37,
                0xFB3D, 0xFB3D,
                0xFB3F, 0xFB3F,
                0xFB42, 0xFB42,
                0xFB45, 0xFB45,
                0xFBC2, 0xFBD2,
                0xFD40, 0xFD4F,
                0xFD90, 0xFD91,
                0xFDC8, 0xFDCF,
                0xFDFE, 0xFDFF,
                0xFE1A, 0xFE1F,
                0xFE27, 0xFE2F,
                0xFE53, 0xFE53,
                0xFE67, 0xFE67,
                0xFE6C, 0xFE6F,
                0xFE75, 0xFE75,
                0xFEFD, 0xFEFE,
                0xFF00, 0xFF00,
                0xFFBF, 0xFFC1,
                0xFFC8, 0xFFC9,
                0xFFD0, 0xFFD1,
                0xFFD8, 0xFFD9,
                0xFFDD, 0xFFDF,
                0xFFE7, 0xFFE7,
                0xFFEF, 0xFFEF,
                0xFFF0, 0xFFF8,
                0x1000C, 0x1000C,
                0x10027, 0x10027,
                0x1003B, 0x1003B,
                0x1003E, 0x1003E,
                0x1004E, 0x1004F,
                0x1005E, 0x1007F,
                0x100FB, 0x100FF,
                0x10103, 0x10106,
                0x10134, 0x10136,
                0x1018B, 0x1018F,
                0x1019C, 0x101CF,
                0x101FE, 0x1027F,
                0x1029D, 0x1029F,
                0x102D1, 0x102FF,
                0x1031F, 0x1031F,
                0x10324, 0x1032F,
                0x1034B, 0x1037F,
                0x1039E, 0x1039E,
                0x103C4, 0x103C7,
                0x103D6, 0x103FF,
                0x1049E, 0x1049F,
                0x104AA, 0x107FF,
                0x10806, 0x10807,
                0x10809, 0x10809,
                0x10836, 0x10836,
                0x10839, 0x1083B,
                0x1083D, 0x1083E,
                0x10856, 0x10856,
                0x10860, 0x108FF,
                0x1091C, 0x1091E,
                0x1093A, 0x1093E,
                0x10940, 0x109FF,
                0x10A04, 0x10A04,
                0x10A07, 0x10A0B,
                0x10A14, 0x10A14,
                0x10A18, 0x10A18,
                0x10A34, 0x10A37,
                0x10A3B, 0x10A3E,
                0x10A48, 0x10A4F,
                0x10A59, 0x10A5F,
                0x10A80, 0x10AFF,
                0x10B36, 0x10B38,
                0x10B56, 0x10B57,
                0x10B73, 0x10B77,
                0x10B80, 0x10BFF,
                0x10C49, 0x10E5F,
                0x10E7F, 0x10FFF,
                0x1104E, 0x11051,
                0x11070, 0x1107F,
                0x110C2, 0x11FFF,
                0x1236F, 0x123FF,
                0x12463, 0x1246F,
                0x12474, 0x12FFF,
                0x1342F, 0x167FF,
                0x16A39, 0x1AFFF,
                0x1B002, 0x1CFFF,
                0x1D0F6, 0x1D0FF,
                0x1D127, 0x1D128,
                0x1D1DE, 0x1D1FF,
                0x1D246, 0x1D2FF,
                0x1D357, 0x1D35F,
                0x1D372, 0x1D3FF,
                0x1D455, 0x1D455,
                0x1D49D, 0x1D49D,
                0x1D4A0, 0x1D4A1,
                0x1D4A3, 0x1D4A4,
                0x1D4A7, 0x1D4A8,
                0x1D4AD, 0x1D4AD,
                0x1D4BA, 0x1D4BA,
                0x1D4BC, 0x1D4BC,
                0x1D4C4, 0x1D4C4,
                0x1D506, 0x1D506,
                0x1D50B, 0x1D50C,
                0x1D515, 0x1D515,
                0x1D51D, 0x1D51D,
                0x1D53A, 0x1D53A,
                0x1D53F, 0x1D53F,
                0x1D545, 0x1D545,
                0x1D547, 0x1D549,
                0x1D551, 0x1D551,
                0x1D6A6, 0x1D6A7,
                0x1D7CC, 0x1D7CD,
                0x1D800, 0x1E7FF,
                0x1E800, 0x1EFFF,
                0x1F02C, 0x1F02F,
                0x1F094, 0x1F09F,
                0x1F0AF, 0x1F0B0,
                0x1F0BF, 0x1F0C0,
                0x1F0D0, 0x1F0D0,
                0x1F0E0, 0x1F0FF,
                0x1F10B, 0x1F10F,
                0x1F12F, 0x1F12F,
                0x1F16A, 0x1F16F,
                0x1F19B, 0x1F1E5,
                0x1F203, 0x1F20F,
                0x1F23B, 0x1F23F,
                0x1F249, 0x1F24F,
                0x1F252, 0x1F2FF,
                0x1F321, 0x1F32F,
                0x1F336, 0x1F336,
                0x1F37D, 0x1F37F,
                0x1F394, 0x1F39F,
                0x1F3C5, 0x1F3C5,
                0x1F3CB, 0x1F3DF,
                0x1F3F1, 0x1F3FF,
                0x1F43F, 0x1F43F,
                0x1F441, 0x1F441,
                0x1F4F8, 0x1F4F8,
                0x1F4FD, 0x1F4FF,
                0x1F53E, 0x1F54F,
                0x1F568, 0x1F5FA,
                0x1F600, 0x1F600,
                0x1F611, 0x1F611,
                0x1F615, 0x1F615,
                0x1F617, 0x1F617,
                0x1F619, 0x1F619,
                0x1F61B, 0x1F61B,
                0x1F61F, 0x1F61F,
                0x1F626, 0x1F627,
                0x1F62C, 0x1F62C,
                0x1F62E, 0x1F62F,
                0x1F634, 0x1F634,
                0x1F641, 0x1F644,
                0x1F650, 0x1F67F,
                0x1F6C6, 0x1F6FF,
                0x1F774, 0x1FFFD,
                0x2A6D7, 0x2A6FF,
                0x2B735, 0x2B73F,
                0x2B81E, 0x2F7FF,
                0x2FA1E, 0x2FFFD,
                0x30000, 0x3FFFD,
                0x40000, 0x4FFFD,
                0x50000, 0x5FFFD,
                0x60000, 0x6FFFD,
                0x70000, 0x7FFFD,
                0x80000, 0x8FFFD,
                0x90000, 0x9FFFD,
                0xA0000, 0xAFFFD,
                0xB0000, 0xBFFFD,
                0xC0000, 0xCFFFD,
                0xD0000, 0xDFFFD,
                0xE0000, 0xE0000,
                0xE0002, 0xE001F,
                0xE0080, 0xE00FF,
                0xE01F0, 0xE0FFF,
                0xE1000, 0xEFFFD

        }));

        CATEGORIES.put("Co", pred(new int[]{        //added by hand
                0xE000, 0xF8FF,
                0xF0000, 0xFFFFD,
                0x100000, 0x10FFFD
        }));


        CATEGORIES.put("Ll", pred(new int[]{
                0x61, 0x7A,
                0xAA, 0xAA,
                0xB5, 0xB5,
                0xBA, 0xBA,
                0xDF, 0xF6,
                0xF8, 0xFF,
                0x101, 0x101,
                0x103, 0x103,
                0x105, 0x105,
                0x107, 0x107,
                0x109, 0x109,
                0x10B, 0x10B,
                0x10D, 0x10D,
                0x10F, 0x10F,
                0x111, 0x111,
                0x113, 0x113,
                0x115, 0x115,
                0x117, 0x117,
                0x119, 0x119,
                0x11B, 0x11B,
                0x11D, 0x11D,
                0x11F, 0x11F,
                0x121, 0x121,
                0x123, 0x123,
                0x125, 0x125,
                0x127, 0x127,
                0x129, 0x129,
                0x12B, 0x12B,
                0x12D, 0x12D,
                0x12F, 0x12F,
                0x131, 0x131,
                0x133, 0x133,
                0x135, 0x135,
                0x137, 0x138,
                0x13A, 0x13A,
                0x13C, 0x13C,
                0x13E, 0x13E,
                0x140, 0x140,
                0x142, 0x142,
                0x144, 0x144,
                0x146, 0x146,
                0x148, 0x149,
                0x14B, 0x14B,
                0x14D, 0x14D,
                0x14F, 0x14F,
                0x151, 0x151,
                0x153, 0x153,
                0x155, 0x155,
                0x157, 0x157,
                0x159, 0x159,
                0x15B, 0x15B,
                0x15D, 0x15D,
                0x15F, 0x15F,
                0x161, 0x161,
                0x163, 0x163,
                0x165, 0x165,
                0x167, 0x167,
                0x169, 0x169,
                0x16B, 0x16B,
                0x16D, 0x16D,
                0x16F, 0x16F,
                0x171, 0x171,
                0x173, 0x173,
                0x175, 0x175,
                0x177, 0x177,
                0x17A, 0x17A,
                0x17C, 0x17C,
                0x17E, 0x180,
                0x183, 0x183,
                0x185, 0x185,
                0x188, 0x188,
                0x18C, 0x18D,
                0x192, 0x192,
                0x195, 0x195,
                0x199, 0x19B,
                0x19E, 0x19E,
                0x1A1, 0x1A1,
                0x1A3, 0x1A3,
                0x1A5, 0x1A5,
                0x1A8, 0x1A8,
                0x1AA, 0x1AB,
                0x1AD, 0x1AD,
                0x1B0, 0x1B0,
                0x1B4, 0x1B4,
                0x1B6, 0x1B6,
                0x1B9, 0x1BA,
                0x1BD, 0x1BF,
                0x1C6, 0x1C6,
                0x1C9, 0x1C9,
                0x1CC, 0x1CC,
                0x1CE, 0x1CE,
                0x1D0, 0x1D0,
                0x1D2, 0x1D2,
                0x1D4, 0x1D4,
                0x1D6, 0x1D6,
                0x1D8, 0x1D8,
                0x1DA, 0x1DA,
                0x1DC, 0x1DD,
                0x1DF, 0x1DF,
                0x1E1, 0x1E1,
                0x1E3, 0x1E3,
                0x1E5, 0x1E5,
                0x1E7, 0x1E7,
                0x1E9, 0x1E9,
                0x1EB, 0x1EB,
                0x1ED, 0x1ED,
                0x1EF, 0x1F0,
                0x1F3, 0x1F3,
                0x1F5, 0x1F5,
                0x1F9, 0x1F9,
                0x1FB, 0x1FB,
                0x1FD, 0x1FD,
                0x1FF, 0x1FF,
                0x201, 0x201,
                0x203, 0x203,
                0x205, 0x205,
                0x207, 0x207,
                0x209, 0x209,
                0x20B, 0x20B,
                0x20D, 0x20D,
                0x20F, 0x20F,
                0x211, 0x211,
                0x213, 0x213,
                0x215, 0x215,
                0x217, 0x217,
                0x219, 0x219,
                0x21B, 0x21B,
                0x21D, 0x21D,
                0x21F, 0x21F,
                0x221, 0x221,
                0x223, 0x223,
                0x225, 0x225,
                0x227, 0x227,
                0x229, 0x229,
                0x22B, 0x22B,
                0x22D, 0x22D,
                0x22F, 0x22F,
                0x231, 0x231,
                0x233, 0x239,
                0x23C, 0x23C,
                0x23F, 0x240,
                0x242, 0x242,
                0x247, 0x247,
                0x249, 0x249,
                0x24B, 0x24B,
                0x24D, 0x24D,
                0x24F, 0x293,
                0x295, 0x2AF,
                0x371, 0x371,
                0x373, 0x373,
                0x377, 0x377,
                0x37B, 0x37D,
                0x390, 0x390,
                0x3AC, 0x3CE,
                0x3D0, 0x3D1,
                0x3D5, 0x3D7,
                0x3D9, 0x3D9,
                0x3DB, 0x3DB,
                0x3DD, 0x3DD,
                0x3DF, 0x3DF,
                0x3E1, 0x3E1,
                0x3E3, 0x3E3,
                0x3E5, 0x3E5,
                0x3E7, 0x3E7,
                0x3E9, 0x3E9,
                0x3EB, 0x3EB,
                0x3ED, 0x3ED,
                0x3EF, 0x3F3,
                0x3F5, 0x3F5,
                0x3F8, 0x3F8,
                0x3FB, 0x3FC,
                0x430, 0x45F,
                0x461, 0x461,
                0x463, 0x463,
                0x465, 0x465,
                0x467, 0x467,
                0x469, 0x469,
                0x46B, 0x46B,
                0x46D, 0x46D,
                0x46F, 0x46F,
                0x471, 0x471,
                0x473, 0x473,
                0x475, 0x475,
                0x477, 0x477,
                0x479, 0x479,
                0x47B, 0x47B,
                0x47D, 0x47D,
                0x47F, 0x47F,
                0x481, 0x481,
                0x48B, 0x48B,
                0x48D, 0x48D,
                0x48F, 0x48F,
                0x491, 0x491,
                0x493, 0x493,
                0x495, 0x495,
                0x497, 0x497,
                0x499, 0x499,
                0x49B, 0x49B,
                0x49D, 0x49D,
                0x49F, 0x49F,
                0x4A1, 0x4A1,
                0x4A3, 0x4A3,
                0x4A5, 0x4A5,
                0x4A7, 0x4A7,
                0x4A9, 0x4A9,
                0x4AB, 0x4AB,
                0x4AD, 0x4AD,
                0x4AF, 0x4AF,
                0x4B1, 0x4B1,
                0x4B3, 0x4B3,
                0x4B5, 0x4B5,
                0x4B7, 0x4B7,
                0x4B9, 0x4B9,
                0x4BB, 0x4BB,
                0x4BD, 0x4BD,
                0x4BF, 0x4BF,
                0x4C2, 0x4C2,
                0x4C4, 0x4C4,
                0x4C6, 0x4C6,
                0x4C8, 0x4C8,
                0x4CA, 0x4CA,
                0x4CC, 0x4CC,
                0x4CE, 0x4CF,
                0x4D1, 0x4D1,
                0x4D3, 0x4D3,
                0x4D5, 0x4D5,
                0x4D7, 0x4D7,
                0x4D9, 0x4D9,
                0x4DB, 0x4DB,
                0x4DD, 0x4DD,
                0x4DF, 0x4DF,
                0x4E1, 0x4E1,
                0x4E3, 0x4E3,
                0x4E5, 0x4E5,
                0x4E7, 0x4E7,
                0x4E9, 0x4E9,
                0x4EB, 0x4EB,
                0x4ED, 0x4ED,
                0x4EF, 0x4EF,
                0x4F1, 0x4F1,
                0x4F3, 0x4F3,
                0x4F5, 0x4F5,
                0x4F7, 0x4F7,
                0x4F9, 0x4F9,
                0x4FB, 0x4FB,
                0x4FD, 0x4FD,
                0x4FF, 0x4FF,
                0x501, 0x501,
                0x503, 0x503,
                0x505, 0x505,
                0x507, 0x507,
                0x509, 0x509,
                0x50B, 0x50B,
                0x50D, 0x50D,
                0x50F, 0x50F,
                0x511, 0x511,
                0x513, 0x513,
                0x515, 0x515,
                0x517, 0x517,
                0x519, 0x519,
                0x51B, 0x51B,
                0x51D, 0x51D,
                0x51F, 0x51F,
                0x521, 0x521,
                0x523, 0x523,
                0x525, 0x525,
                0x527, 0x527,
                0x561, 0x587,
                0x1D00, 0x1D2B,
                0x1D62, 0x1D77,
                0x1D79, 0x1D9A,
                0x1E01, 0x1E01,
                0x1E03, 0x1E03,
                0x1E05, 0x1E05,
                0x1E07, 0x1E07,
                0x1E09, 0x1E09,
                0x1E0B, 0x1E0B,
                0x1E0D, 0x1E0D,
                0x1E0F, 0x1E0F,
                0x1E11, 0x1E11,
                0x1E13, 0x1E13,
                0x1E15, 0x1E15,
                0x1E17, 0x1E17,
                0x1E19, 0x1E19,
                0x1E1B, 0x1E1B,
                0x1E1D, 0x1E1D,
                0x1E1F, 0x1E1F,
                0x1E21, 0x1E21,
                0x1E23, 0x1E23,
                0x1E25, 0x1E25,
                0x1E27, 0x1E27,
                0x1E29, 0x1E29,
                0x1E2B, 0x1E2B,
                0x1E2D, 0x1E2D,
                0x1E2F, 0x1E2F,
                0x1E31, 0x1E31,
                0x1E33, 0x1E33,
                0x1E35, 0x1E35,
                0x1E37, 0x1E37,
                0x1E39, 0x1E39,
                0x1E3B, 0x1E3B,
                0x1E3D, 0x1E3D,
                0x1E3F, 0x1E3F,
                0x1E41, 0x1E41,
                0x1E43, 0x1E43,
                0x1E45, 0x1E45,
                0x1E47, 0x1E47,
                0x1E49, 0x1E49,
                0x1E4B, 0x1E4B,
                0x1E4D, 0x1E4D,
                0x1E4F, 0x1E4F,
                0x1E51, 0x1E51,
                0x1E53, 0x1E53,
                0x1E55, 0x1E55,
                0x1E57, 0x1E57,
                0x1E59, 0x1E59,
                0x1E5B, 0x1E5B,
                0x1E5D, 0x1E5D,
                0x1E5F, 0x1E5F,
                0x1E61, 0x1E61,
                0x1E63, 0x1E63,
                0x1E65, 0x1E65,
                0x1E67, 0x1E67,
                0x1E69, 0x1E69,
                0x1E6B, 0x1E6B,
                0x1E6D, 0x1E6D,
                0x1E6F, 0x1E6F,
                0x1E71, 0x1E71,
                0x1E73, 0x1E73,
                0x1E75, 0x1E75,
                0x1E77, 0x1E77,
                0x1E79, 0x1E79,
                0x1E7B, 0x1E7B,
                0x1E7D, 0x1E7D,
                0x1E7F, 0x1E7F,
                0x1E81, 0x1E81,
                0x1E83, 0x1E83,
                0x1E85, 0x1E85,
                0x1E87, 0x1E87,
                0x1E89, 0x1E89,
                0x1E8B, 0x1E8B,
                0x1E8D, 0x1E8D,
                0x1E8F, 0x1E8F,
                0x1E91, 0x1E91,
                0x1E93, 0x1E93,
                0x1E95, 0x1E9D,
                0x1E9F, 0x1E9F,
                0x1EA1, 0x1EA1,
                0x1EA3, 0x1EA3,
                0x1EA5, 0x1EA5,
                0x1EA7, 0x1EA7,
                0x1EA9, 0x1EA9,
                0x1EAB, 0x1EAB,
                0x1EAD, 0x1EAD,
                0x1EAF, 0x1EAF,
                0x1EB1, 0x1EB1,
                0x1EB3, 0x1EB3,
                0x1EB5, 0x1EB5,
                0x1EB7, 0x1EB7,
                0x1EB9, 0x1EB9,
                0x1EBB, 0x1EBB,
                0x1EBD, 0x1EBD,
                0x1EBF, 0x1EBF,
                0x1EC1, 0x1EC1,
                0x1EC3, 0x1EC3,
                0x1EC5, 0x1EC5,
                0x1EC7, 0x1EC7,
                0x1EC9, 0x1EC9,
                0x1ECB, 0x1ECB,
                0x1ECD, 0x1ECD,
                0x1ECF, 0x1ECF,
                0x1ED1, 0x1ED1,
                0x1ED3, 0x1ED3,
                0x1ED5, 0x1ED5,
                0x1ED7, 0x1ED7,
                0x1ED9, 0x1ED9,
                0x1EDB, 0x1EDB,
                0x1EDD, 0x1EDD,
                0x1EDF, 0x1EDF,
                0x1EE1, 0x1EE1,
                0x1EE3, 0x1EE3,
                0x1EE5, 0x1EE5,
                0x1EE7, 0x1EE7,
                0x1EE9, 0x1EE9,
                0x1EEB, 0x1EEB,
                0x1EED, 0x1EED,
                0x1EEF, 0x1EEF,
                0x1EF1, 0x1EF1,
                0x1EF3, 0x1EF3,
                0x1EF5, 0x1EF5,
                0x1EF7, 0x1EF7,
                0x1EF9, 0x1EF9,
                0x1EFB, 0x1EFB,
                0x1EFD, 0x1EFD,
                0x1EFF, 0x1F07,
                0x1F10, 0x1F15,
                0x1F20, 0x1F27,
                0x1F30, 0x1F37,
                0x1F40, 0x1F45,
                0x1F50, 0x1F57,
                0x1F60, 0x1F67,
                0x1F70, 0x1F7D,
                0x1F80, 0x1F87,
                0x1F90, 0x1F97,
                0x1FA0, 0x1FA7,
                0x1FB0, 0x1FB4,
                0x1FB6, 0x1FB7,
                0x1FBE, 0x1FBE,
                0x1FC2, 0x1FC4,
                0x1FC6, 0x1FC7,
                0x1FD0, 0x1FD3,
                0x1FD6, 0x1FD7,
                0x1FE0, 0x1FE7,
                0x1FF2, 0x1FF4,
                0x1FF6, 0x1FF7,
                0x210A, 0x210A,
                0x210E, 0x210F,
                0x2113, 0x2113,
                0x212F, 0x212F,
                0x2134, 0x2134,
                0x2139, 0x2139,
                0x213C, 0x213D,
                0x2146, 0x2149,
                0x214E, 0x214E,
                0x2184, 0x2184,
                0x2C30, 0x2C5E,
                0x2C61, 0x2C61,
                0x2C65, 0x2C66,
                0x2C68, 0x2C68,
                0x2C6A, 0x2C6A,
                0x2C6C, 0x2C6C,
                0x2C71, 0x2C71,
                0x2C73, 0x2C74,
                0x2C76, 0x2C7C,
                0x2C81, 0x2C81,
                0x2C83, 0x2C83,
                0x2C85, 0x2C85,
                0x2C87, 0x2C87,
                0x2C89, 0x2C89,
                0x2C8B, 0x2C8B,
                0x2C8D, 0x2C8D,
                0x2C8F, 0x2C8F,
                0x2C91, 0x2C91,
                0x2C93, 0x2C93,
                0x2C95, 0x2C95,
                0x2C97, 0x2C97,
                0x2C99, 0x2C99,
                0x2C9B, 0x2C9B,
                0x2C9D, 0x2C9D,
                0x2C9F, 0x2C9F,
                0x2CA1, 0x2CA1,
                0x2CA3, 0x2CA3,
                0x2CA5, 0x2CA5,
                0x2CA7, 0x2CA7,
                0x2CA9, 0x2CA9,
                0x2CAB, 0x2CAB,
                0x2CAD, 0x2CAD,
                0x2CAF, 0x2CAF,
                0x2CB1, 0x2CB1,
                0x2CB3, 0x2CB3,
                0x2CB5, 0x2CB5,
                0x2CB7, 0x2CB7,
                0x2CB9, 0x2CB9,
                0x2CBB, 0x2CBB,
                0x2CBD, 0x2CBD,
                0x2CBF, 0x2CBF,
                0x2CC1, 0x2CC1,
                0x2CC3, 0x2CC3,
                0x2CC5, 0x2CC5,
                0x2CC7, 0x2CC7,
                0x2CC9, 0x2CC9,
                0x2CCB, 0x2CCB,
                0x2CCD, 0x2CCD,
                0x2CCF, 0x2CCF,
                0x2CD1, 0x2CD1,
                0x2CD3, 0x2CD3,
                0x2CD5, 0x2CD5,
                0x2CD7, 0x2CD7,
                0x2CD9, 0x2CD9,
                0x2CDB, 0x2CDB,
                0x2CDD, 0x2CDD,
                0x2CDF, 0x2CDF,
                0x2CE1, 0x2CE1,
                0x2CE3, 0x2CE4,
                0x2CEC, 0x2CEC,
                0x2CEE, 0x2CEE,
                0x2D00, 0x2D25,
                0xA641, 0xA641,
                0xA643, 0xA643,
                0xA645, 0xA645,
                0xA647, 0xA647,
                0xA649, 0xA649,
                0xA64B, 0xA64B,
                0xA64D, 0xA64D,
                0xA64F, 0xA64F,
                0xA651, 0xA651,
                0xA653, 0xA653,
                0xA655, 0xA655,
                0xA657, 0xA657,
                0xA659, 0xA659,
                0xA65B, 0xA65B,
                0xA65D, 0xA65D,
                0xA65F, 0xA65F,
                0xA661, 0xA661,
                0xA663, 0xA663,
                0xA665, 0xA665,
                0xA667, 0xA667,
                0xA669, 0xA669,
                0xA66B, 0xA66B,
                0xA66D, 0xA66D,
                0xA681, 0xA681,
                0xA683, 0xA683,
                0xA685, 0xA685,
                0xA687, 0xA687,
                0xA689, 0xA689,
                0xA68B, 0xA68B,
                0xA68D, 0xA68D,
                0xA68F, 0xA68F,
                0xA691, 0xA691,
                0xA693, 0xA693,
                0xA695, 0xA695,
                0xA697, 0xA697,
                0xA723, 0xA723,
                0xA725, 0xA725,
                0xA727, 0xA727,
                0xA729, 0xA729,
                0xA72B, 0xA72B,
                0xA72D, 0xA72D,
                0xA72F, 0xA731,
                0xA733, 0xA733,
                0xA735, 0xA735,
                0xA737, 0xA737,
                0xA739, 0xA739,
                0xA73B, 0xA73B,
                0xA73D, 0xA73D,
                0xA73F, 0xA73F,
                0xA741, 0xA741,
                0xA743, 0xA743,
                0xA745, 0xA745,
                0xA747, 0xA747,
                0xA749, 0xA749,
                0xA74B, 0xA74B,
                0xA74D, 0xA74D,
                0xA74F, 0xA74F,
                0xA751, 0xA751,
                0xA753, 0xA753,
                0xA755, 0xA755,
                0xA757, 0xA757,
                0xA759, 0xA759,
                0xA75B, 0xA75B,
                0xA75D, 0xA75D,
                0xA75F, 0xA75F,
                0xA761, 0xA761,
                0xA763, 0xA763,
                0xA765, 0xA765,
                0xA767, 0xA767,
                0xA769, 0xA769,
                0xA76B, 0xA76B,
                0xA76D, 0xA76D,
                0xA76F, 0xA76F,
                0xA771, 0xA778,
                0xA77A, 0xA77A,
                0xA77C, 0xA77C,
                0xA77F, 0xA77F,
                0xA781, 0xA781,
                0xA783, 0xA783,
                0xA785, 0xA785,
                0xA787, 0xA787,
                0xA78C, 0xA78C,
                0xA78E, 0xA78E,
                0xA791, 0xA791,
                0xA7A1, 0xA7A1,
                0xA7A3, 0xA7A3,
                0xA7A5, 0xA7A5,
                0xA7A7, 0xA7A7,
                0xA7A9, 0xA7A9,
                0xA7FA, 0xA7FA,
                0xFB00, 0xFB06,
                0xFB13, 0xFB17,
                0xFF41, 0xFF5A,
                0x10428, 0x1044F,
                0x1D41A, 0x1D433,
                0x1D44E, 0x1D454,
                0x1D456, 0x1D467,
                0x1D482, 0x1D49B,
                0x1D4B6, 0x1D4B9,
                0x1D4BB, 0x1D4BB,
                0x1D4BD, 0x1D4C3,
                0x1D4C5, 0x1D4CF,
                0x1D4EA, 0x1D503,
                0x1D51E, 0x1D537,
                0x1D552, 0x1D56B,
                0x1D586, 0x1D59F,
                0x1D5BA, 0x1D5D3,
                0x1D5EE, 0x1D607,
                0x1D622, 0x1D63B,
                0x1D656, 0x1D66F,
                0x1D68A, 0x1D6A5,
                0x1D6C2, 0x1D6DA,
                0x1D6DC, 0x1D6E1,
                0x1D6FC, 0x1D714,
                0x1D716, 0x1D71B,
                0x1D736, 0x1D74E,
                0x1D750, 0x1D755,
                0x1D770, 0x1D788,
                0x1D78A, 0x1D78F,
                0x1D7AA, 0x1D7C2,
                0x1D7C4, 0x1D7C9,
                0x1D7CB, 0x1D7CB
        }));


        CATEGORIES.put("Lm", pred(new int[]{
                0x2B0, 0x2C1,
                0x2C6, 0x2D1,
                0x2E0, 0x2E4,
                0x2EC, 0x2EC,
                0x2EE, 0x2EE,
                0x374, 0x374,
                0x37A, 0x37A,
                0x559, 0x559,
                0x640, 0x640,
                0x6E5, 0x6E6,
                0x7F4, 0x7F5,
                0x7FA, 0x7FA,
                0x81A, 0x81A,
                0x824, 0x824,
                0x828, 0x828,
                0x971, 0x971,
                0xE46, 0xE46,
                0xEC6, 0xEC6,
                0x10FC, 0x10FC,
                0x17D7, 0x17D7,
                0x1843, 0x1843,
                0x1AA7, 0x1AA7,
                0x1C78, 0x1C7D,
                0x1D2C, 0x1D61,
                0x1D78, 0x1D78,
                0x1D9B, 0x1DBF,
                0x2071, 0x2071,
                0x207F, 0x207F,
                0x2090, 0x209C,
                0x2C7D, 0x2C7D,
                0x2D6F, 0x2D6F,
                0x2E2F, 0x2E2F,
                0x3005, 0x3005,
                0x3031, 0x3035,
                0x303B, 0x303B,
                0x309D, 0x309E,
                0x30FC, 0x30FE,
                0xA015, 0xA015,
                0xA4F8, 0xA4FD,
                0xA60C, 0xA60C,
                0xA67F, 0xA67F,
                0xA717, 0xA71F,
                0xA770, 0xA770,
                0xA788, 0xA788,
                0xA9CF, 0xA9CF,
                0xAA70, 0xAA70,
                0xAADD, 0xAADD,
                0xFF70, 0xFF70,
                0xFF9E, 0xFF9F
        }));


        CATEGORIES.put("Lo", pred(new int[]{
                0x1BB, 0x1BB,
                0x1C0, 0x1C3,
                0x294, 0x294,
                0x5D0, 0x5EA,
                0x5F0, 0x5F2,
                0x620, 0x63F,
                0x641, 0x64A,
                0x66E, 0x66F,
                0x671, 0x6D3,
                0x6D5, 0x6D5,
                0x6EE, 0x6EF,
                0x6FA, 0x6FC,
                0x6FF, 0x6FF,
                0x710, 0x710,
                0x712, 0x72F,
                0x74D, 0x7A5,
                0x7B1, 0x7B1,
                0x7CA, 0x7EA,
                0x800, 0x815,
                0x840, 0x858,
                0x904, 0x939,
                0x93D, 0x93D,
                0x950, 0x950,
                0x958, 0x961,
                0x972, 0x977,
                0x979, 0x97F,
                0x985, 0x98C,
                0x98F, 0x990,
                0x993, 0x9A8,
                0x9AA, 0x9B0,
                0x9B2, 0x9B2,
                0x9B6, 0x9B9,
                0x9BD, 0x9BD,
                0x9CE, 0x9CE,
                0x9DC, 0x9DD,
                0x9DF, 0x9E1,
                0x9F0, 0x9F1,
                0xA05, 0xA0A,
                0xA0F, 0xA10,
                0xA13, 0xA28,
                0xA2A, 0xA30,
                0xA32, 0xA33,
                0xA35, 0xA36,
                0xA38, 0xA39,
                0xA59, 0xA5C,
                0xA5E, 0xA5E,
                0xA72, 0xA74,
                0xA85, 0xA8D,
                0xA8F, 0xA91,
                0xA93, 0xAA8,
                0xAAA, 0xAB0,
                0xAB2, 0xAB3,
                0xAB5, 0xAB9,
                0xABD, 0xABD,
                0xAD0, 0xAD0,
                0xAE0, 0xAE1,
                0xB05, 0xB0C,
                0xB0F, 0xB10,
                0xB13, 0xB28,
                0xB2A, 0xB30,
                0xB32, 0xB33,
                0xB35, 0xB39,
                0xB3D, 0xB3D,
                0xB5C, 0xB5D,
                0xB5F, 0xB61,
                0xB71, 0xB71,
                0xB83, 0xB83,
                0xB85, 0xB8A,
                0xB8E, 0xB90,
                0xB92, 0xB95,
                0xB99, 0xB9A,
                0xB9C, 0xB9C,
                0xB9E, 0xB9F,
                0xBA3, 0xBA4,
                0xBA8, 0xBAA,
                0xBAE, 0xBB9,
                0xBD0, 0xBD0,
                0xC05, 0xC0C,
                0xC0E, 0xC10,
                0xC12, 0xC28,
                0xC2A, 0xC33,
                0xC35, 0xC39,
                0xC3D, 0xC3D,
                0xC58, 0xC59,
                0xC60, 0xC61,
                0xC85, 0xC8C,
                0xC8E, 0xC90,
                0xC92, 0xCA8,
                0xCAA, 0xCB3,
                0xCB5, 0xCB9,
                0xCBD, 0xCBD,
                0xCDE, 0xCDE,
                0xCE0, 0xCE1,
                0xCF1, 0xCF2,
                0xD05, 0xD0C,
                0xD0E, 0xD10,
                0xD12, 0xD3A,
                0xD3D, 0xD3D,
                0xD4E, 0xD4E,
                0xD60, 0xD61,
                0xD7A, 0xD7F,
                0xD85, 0xD96,
                0xD9A, 0xDB1,
                0xDB3, 0xDBB,
                0xDBD, 0xDBD,
                0xDC0, 0xDC6,
                0xE01, 0xE30,
                0xE32, 0xE33,
                0xE40, 0xE45,
                0xE81, 0xE82,
                0xE84, 0xE84,
                0xE87, 0xE88,
                0xE8A, 0xE8A,
                0xE8D, 0xE8D,
                0xE94, 0xE97,
                0xE99, 0xE9F,
                0xEA1, 0xEA3,
                0xEA5, 0xEA5,
                0xEA7, 0xEA7,
                0xEAA, 0xEAB,
                0xEAD, 0xEB0,
                0xEB2, 0xEB3,
                0xEBD, 0xEBD,
                0xEC0, 0xEC4,
                0xEDC, 0xEDD,
                0xF00, 0xF00,
                0xF40, 0xF47,
                0xF49, 0xF6C,
                0xF88, 0xF8C,
                0x1000, 0x102A,
                0x103F, 0x103F,
                0x1050, 0x1055,
                0x105A, 0x105D,
                0x1061, 0x1061,
                0x1065, 0x1066,
                0x106E, 0x1070,
                0x1075, 0x1081,
                0x108E, 0x108E,
                0x10D0, 0x10FA,
                0x1100, 0x1248,
                0x124A, 0x124D,
                0x1250, 0x1256,
                0x1258, 0x1258,
                0x125A, 0x125D,
                0x1260, 0x1288,
                0x128A, 0x128D,
                0x1290, 0x12B0,
                0x12B2, 0x12B5,
                0x12B8, 0x12BE,
                0x12C0, 0x12C0,
                0x12C2, 0x12C5,
                0x12C8, 0x12D6,
                0x12D8, 0x1310,
                0x1312, 0x1315,
                0x1318, 0x135A,
                0x1380, 0x138F,
                0x13A0, 0x13F4,
                0x1401, 0x166C,
                0x166F, 0x167F,
                0x1681, 0x169A,
                0x16A0, 0x16EA,
                0x1700, 0x170C,
                0x170E, 0x1711,
                0x1720, 0x1731,
                0x1740, 0x1751,
                0x1760, 0x176C,
                0x176E, 0x1770,
                0x1780, 0x17B3,
                0x17DC, 0x17DC,
                0x1820, 0x1842,
                0x1844, 0x1877,
                0x1880, 0x18A8,
                0x18AA, 0x18AA,
                0x18B0, 0x18F5,
                0x1900, 0x191C,
                0x1950, 0x196D,
                0x1970, 0x1974,
                0x1980, 0x19AB,
                0x19C1, 0x19C7,
                0x1A00, 0x1A16,
                0x1A20, 0x1A54,
                0x1B05, 0x1B33,
                0x1B45, 0x1B4B,
                0x1B83, 0x1BA0,
                0x1BAE, 0x1BAF,
                0x1BC0, 0x1BE5,
                0x1C00, 0x1C23,
                0x1C4D, 0x1C4F,
                0x1C5A, 0x1C77,
                0x1CE9, 0x1CEC,
                0x1CEE, 0x1CF1,
                0x2135, 0x2138,
                0x2D30, 0x2D65,
                0x2D80, 0x2D96,
                0x2DA0, 0x2DA6,
                0x2DA8, 0x2DAE,
                0x2DB0, 0x2DB6,
                0x2DB8, 0x2DBE,
                0x2DC0, 0x2DC6,
                0x2DC8, 0x2DCE,
                0x2DD0, 0x2DD6,
                0x2DD8, 0x2DDE,
                0x3006, 0x3006,
                0x303C, 0x303C,
                0x3041, 0x3096,
                0x309F, 0x309F,
                0x30A1, 0x30FA,
                0x30FF, 0x30FF,
                0x3105, 0x312D,
                0x3131, 0x318E,
                0x31A0, 0x31BA,
                0x31F0, 0x31FF,
                0x3400, 0x4DB5,
                0x4E00, 0x9FCB,
                0xA000, 0xA014,
                0xA016, 0xA48C,
                0xA4D0, 0xA4F7,
                0xA500, 0xA60B,
                0xA610, 0xA61F,
                0xA62A, 0xA62B,
                0xA66E, 0xA66E,
                0xA6A0, 0xA6E5,
                0xA7FB, 0xA801,
                0xA803, 0xA805,
                0xA807, 0xA80A,
                0xA80C, 0xA822,
                0xA840, 0xA873,
                0xA882, 0xA8B3,
                0xA8F2, 0xA8F7,
                0xA8FB, 0xA8FB,
                0xA90A, 0xA925,
                0xA930, 0xA946,
                0xA960, 0xA97C,
                0xA984, 0xA9B2,
                0xAA00, 0xAA28,
                0xAA40, 0xAA42,
                0xAA44, 0xAA4B,
                0xAA60, 0xAA6F,
                0xAA71, 0xAA76,
                0xAA7A, 0xAA7A,
                0xAA80, 0xAAAF,
                0xAAB1, 0xAAB1,
                0xAAB5, 0xAAB6,
                0xAAB9, 0xAABD,
                0xAAC0, 0xAAC0,
                0xAAC2, 0xAAC2,
                0xAADB, 0xAADC,
                0xAB01, 0xAB06,
                0xAB09, 0xAB0E,
                0xAB11, 0xAB16,
                0xAB20, 0xAB26,
                0xAB28, 0xAB2E,
                0xABC0, 0xABE2,
                0xAC00, 0xD7A3,
                0xD7B0, 0xD7C6,
                0xD7CB, 0xD7FB,
                0xF900, 0xFA2D,
                0xFA30, 0xFA6D,
                0xFA70, 0xFAD9,
                0xFB1D, 0xFB1D,
                0xFB1F, 0xFB28,
                0xFB2A, 0xFB36,
                0xFB38, 0xFB3C,
                0xFB3E, 0xFB3E,
                0xFB40, 0xFB41,
                0xFB43, 0xFB44,
                0xFB46, 0xFBB1,
                0xFBD3, 0xFD3D,
                0xFD50, 0xFD8F,
                0xFD92, 0xFDC7,
                0xFDF0, 0xFDFB,
                0xFE70, 0xFE74,
                0xFE76, 0xFEFC,
                0xFF66, 0xFF6F,
                0xFF71, 0xFF9D,
                0xFFA0, 0xFFBE,
                0xFFC2, 0xFFC7,
                0xFFCA, 0xFFCF,
                0xFFD2, 0xFFD7,
                0xFFDA, 0xFFDC,
                0x10000, 0x1000B,
                0x1000D, 0x10026,
                0x10028, 0x1003A,
                0x1003C, 0x1003D,
                0x1003F, 0x1004D,
                0x10050, 0x1005D,
                0x10080, 0x100FA,
                0x10280, 0x1029C,
                0x102A0, 0x102D0,
                0x10300, 0x1031E,
                0x10330, 0x10340,
                0x10342, 0x10349,
                0x10380, 0x1039D,
                0x103A0, 0x103C3,
                0x103C8, 0x103CF,
                0x10450, 0x1049D,
                0x10800, 0x10805,
                0x10808, 0x10808,
                0x1080A, 0x10835,
                0x10837, 0x10838,
                0x1083C, 0x1083C,
                0x1083F, 0x10855,
                0x10900, 0x10915,
                0x10920, 0x10939,
                0x10A00, 0x10A00,
                0x10A10, 0x10A13,
                0x10A15, 0x10A17,
                0x10A19, 0x10A33,
                0x10A60, 0x10A7C,
                0x10B00, 0x10B35,
                0x10B40, 0x10B55,
                0x10B60, 0x10B72,
                0x10C00, 0x10C48,
                0x11003, 0x11037,
                0x11083, 0x110AF,
                0x12000, 0x1236E,
                0x13000, 0x1342E,
                0x16800, 0x16A38,
                0x1B000, 0x1B001,
                0x20000, 0x2A6D6,
                0x2A700, 0x2B734,
                0x2B740, 0x2B81D,
                0x2F800, 0x2FA1D
        }));


        CATEGORIES.put("Lt", pred(new int[]{
                0x1C5, 0x1C5,
                0x1C8, 0x1C8,
                0x1CB, 0x1CB,
                0x1F2, 0x1F2,
                0x1F88, 0x1F8F,
                0x1F98, 0x1F9F,
                0x1FA8, 0x1FAF,
                0x1FBC, 0x1FBC,
                0x1FCC, 0x1FCC,
                0x1FFC, 0x1FFC
        }));


        CATEGORIES.put("Lu", pred(new int[]{
                0x41, 0x5A,
                0xC0, 0xD6,
                0xD8, 0xDE,
                0x100, 0x100,
                0x102, 0x102,
                0x104, 0x104,
                0x106, 0x106,
                0x108, 0x108,
                0x10A, 0x10A,
                0x10C, 0x10C,
                0x10E, 0x10E,
                0x110, 0x110,
                0x112, 0x112,
                0x114, 0x114,
                0x116, 0x116,
                0x118, 0x118,
                0x11A, 0x11A,
                0x11C, 0x11C,
                0x11E, 0x11E,
                0x120, 0x120,
                0x122, 0x122,
                0x124, 0x124,
                0x126, 0x126,
                0x128, 0x128,
                0x12A, 0x12A,
                0x12C, 0x12C,
                0x12E, 0x12E,
                0x130, 0x130,
                0x132, 0x132,
                0x134, 0x134,
                0x136, 0x136,
                0x139, 0x139,
                0x13B, 0x13B,
                0x13D, 0x13D,
                0x13F, 0x13F,
                0x141, 0x141,
                0x143, 0x143,
                0x145, 0x145,
                0x147, 0x147,
                0x14A, 0x14A,
                0x14C, 0x14C,
                0x14E, 0x14E,
                0x150, 0x150,
                0x152, 0x152,
                0x154, 0x154,
                0x156, 0x156,
                0x158, 0x158,
                0x15A, 0x15A,
                0x15C, 0x15C,
                0x15E, 0x15E,
                0x160, 0x160,
                0x162, 0x162,
                0x164, 0x164,
                0x166, 0x166,
                0x168, 0x168,
                0x16A, 0x16A,
                0x16C, 0x16C,
                0x16E, 0x16E,
                0x170, 0x170,
                0x172, 0x172,
                0x174, 0x174,
                0x176, 0x176,
                0x178, 0x179,
                0x17B, 0x17B,
                0x17D, 0x17D,
                0x181, 0x182,
                0x184, 0x184,
                0x186, 0x187,
                0x189, 0x18B,
                0x18E, 0x191,
                0x193, 0x194,
                0x196, 0x198,
                0x19C, 0x19D,
                0x19F, 0x1A0,
                0x1A2, 0x1A2,
                0x1A4, 0x1A4,
                0x1A6, 0x1A7,
                0x1A9, 0x1A9,
                0x1AC, 0x1AC,
                0x1AE, 0x1AF,
                0x1B1, 0x1B3,
                0x1B5, 0x1B5,
                0x1B7, 0x1B8,
                0x1BC, 0x1BC,
                0x1C4, 0x1C4,
                0x1C7, 0x1C7,
                0x1CA, 0x1CA,
                0x1CD, 0x1CD,
                0x1CF, 0x1CF,
                0x1D1, 0x1D1,
                0x1D3, 0x1D3,
                0x1D5, 0x1D5,
                0x1D7, 0x1D7,
                0x1D9, 0x1D9,
                0x1DB, 0x1DB,
                0x1DE, 0x1DE,
                0x1E0, 0x1E0,
                0x1E2, 0x1E2,
                0x1E4, 0x1E4,
                0x1E6, 0x1E6,
                0x1E8, 0x1E8,
                0x1EA, 0x1EA,
                0x1EC, 0x1EC,
                0x1EE, 0x1EE,
                0x1F1, 0x1F1,
                0x1F4, 0x1F4,
                0x1F6, 0x1F8,
                0x1FA, 0x1FA,
                0x1FC, 0x1FC,
                0x1FE, 0x1FE,
                0x200, 0x200,
                0x202, 0x202,
                0x204, 0x204,
                0x206, 0x206,
                0x208, 0x208,
                0x20A, 0x20A,
                0x20C, 0x20C,
                0x20E, 0x20E,
                0x210, 0x210,
                0x212, 0x212,
                0x214, 0x214,
                0x216, 0x216,
                0x218, 0x218,
                0x21A, 0x21A,
                0x21C, 0x21C,
                0x21E, 0x21E,
                0x220, 0x220,
                0x222, 0x222,
                0x224, 0x224,
                0x226, 0x226,
                0x228, 0x228,
                0x22A, 0x22A,
                0x22C, 0x22C,
                0x22E, 0x22E,
                0x230, 0x230,
                0x232, 0x232,
                0x23A, 0x23B,
                0x23D, 0x23E,
                0x241, 0x241,
                0x243, 0x246,
                0x248, 0x248,
                0x24A, 0x24A,
                0x24C, 0x24C,
                0x24E, 0x24E,
                0x370, 0x370,
                0x372, 0x372,
                0x376, 0x376,
                0x386, 0x386,
                0x388, 0x38A,
                0x38C, 0x38C,
                0x38E, 0x38F,
                0x391, 0x3A1,
                0x3A3, 0x3AB,
                0x3CF, 0x3CF,
                0x3D2, 0x3D4,
                0x3D8, 0x3D8,
                0x3DA, 0x3DA,
                0x3DC, 0x3DC,
                0x3DE, 0x3DE,
                0x3E0, 0x3E0,
                0x3E2, 0x3E2,
                0x3E4, 0x3E4,
                0x3E6, 0x3E6,
                0x3E8, 0x3E8,
                0x3EA, 0x3EA,
                0x3EC, 0x3EC,
                0x3EE, 0x3EE,
                0x3F4, 0x3F4,
                0x3F7, 0x3F7,
                0x3F9, 0x3FA,
                0x3FD, 0x42F,
                0x460, 0x460,
                0x462, 0x462,
                0x464, 0x464,
                0x466, 0x466,
                0x468, 0x468,
                0x46A, 0x46A,
                0x46C, 0x46C,
                0x46E, 0x46E,
                0x470, 0x470,
                0x472, 0x472,
                0x474, 0x474,
                0x476, 0x476,
                0x478, 0x478,
                0x47A, 0x47A,
                0x47C, 0x47C,
                0x47E, 0x47E,
                0x480, 0x480,
                0x48A, 0x48A,
                0x48C, 0x48C,
                0x48E, 0x48E,
                0x490, 0x490,
                0x492, 0x492,
                0x494, 0x494,
                0x496, 0x496,
                0x498, 0x498,
                0x49A, 0x49A,
                0x49C, 0x49C,
                0x49E, 0x49E,
                0x4A0, 0x4A0,
                0x4A2, 0x4A2,
                0x4A4, 0x4A4,
                0x4A6, 0x4A6,
                0x4A8, 0x4A8,
                0x4AA, 0x4AA,
                0x4AC, 0x4AC,
                0x4AE, 0x4AE,
                0x4B0, 0x4B0,
                0x4B2, 0x4B2,
                0x4B4, 0x4B4,
                0x4B6, 0x4B6,
                0x4B8, 0x4B8,
                0x4BA, 0x4BA,
                0x4BC, 0x4BC,
                0x4BE, 0x4BE,
                0x4C0, 0x4C1,
                0x4C3, 0x4C3,
                0x4C5, 0x4C5,
                0x4C7, 0x4C7,
                0x4C9, 0x4C9,
                0x4CB, 0x4CB,
                0x4CD, 0x4CD,
                0x4D0, 0x4D0,
                0x4D2, 0x4D2,
                0x4D4, 0x4D4,
                0x4D6, 0x4D6,
                0x4D8, 0x4D8,
                0x4DA, 0x4DA,
                0x4DC, 0x4DC,
                0x4DE, 0x4DE,
                0x4E0, 0x4E0,
                0x4E2, 0x4E2,
                0x4E4, 0x4E4,
                0x4E6, 0x4E6,
                0x4E8, 0x4E8,
                0x4EA, 0x4EA,
                0x4EC, 0x4EC,
                0x4EE, 0x4EE,
                0x4F0, 0x4F0,
                0x4F2, 0x4F2,
                0x4F4, 0x4F4,
                0x4F6, 0x4F6,
                0x4F8, 0x4F8,
                0x4FA, 0x4FA,
                0x4FC, 0x4FC,
                0x4FE, 0x4FE,
                0x500, 0x500,
                0x502, 0x502,
                0x504, 0x504,
                0x506, 0x506,
                0x508, 0x508,
                0x50A, 0x50A,
                0x50C, 0x50C,
                0x50E, 0x50E,
                0x510, 0x510,
                0x512, 0x512,
                0x514, 0x514,
                0x516, 0x516,
                0x518, 0x518,
                0x51A, 0x51A,
                0x51C, 0x51C,
                0x51E, 0x51E,
                0x520, 0x520,
                0x522, 0x522,
                0x524, 0x524,
                0x526, 0x526,
                0x531, 0x556,
                0x10A0, 0x10C5,
                0x1E00, 0x1E00,
                0x1E02, 0x1E02,
                0x1E04, 0x1E04,
                0x1E06, 0x1E06,
                0x1E08, 0x1E08,
                0x1E0A, 0x1E0A,
                0x1E0C, 0x1E0C,
                0x1E0E, 0x1E0E,
                0x1E10, 0x1E10,
                0x1E12, 0x1E12,
                0x1E14, 0x1E14,
                0x1E16, 0x1E16,
                0x1E18, 0x1E18,
                0x1E1A, 0x1E1A,
                0x1E1C, 0x1E1C,
                0x1E1E, 0x1E1E,
                0x1E20, 0x1E20,
                0x1E22, 0x1E22,
                0x1E24, 0x1E24,
                0x1E26, 0x1E26,
                0x1E28, 0x1E28,
                0x1E2A, 0x1E2A,
                0x1E2C, 0x1E2C,
                0x1E2E, 0x1E2E,
                0x1E30, 0x1E30,
                0x1E32, 0x1E32,
                0x1E34, 0x1E34,
                0x1E36, 0x1E36,
                0x1E38, 0x1E38,
                0x1E3A, 0x1E3A,
                0x1E3C, 0x1E3C,
                0x1E3E, 0x1E3E,
                0x1E40, 0x1E40,
                0x1E42, 0x1E42,
                0x1E44, 0x1E44,
                0x1E46, 0x1E46,
                0x1E48, 0x1E48,
                0x1E4A, 0x1E4A,
                0x1E4C, 0x1E4C,
                0x1E4E, 0x1E4E,
                0x1E50, 0x1E50,
                0x1E52, 0x1E52,
                0x1E54, 0x1E54,
                0x1E56, 0x1E56,
                0x1E58, 0x1E58,
                0x1E5A, 0x1E5A,
                0x1E5C, 0x1E5C,
                0x1E5E, 0x1E5E,
                0x1E60, 0x1E60,
                0x1E62, 0x1E62,
                0x1E64, 0x1E64,
                0x1E66, 0x1E66,
                0x1E68, 0x1E68,
                0x1E6A, 0x1E6A,
                0x1E6C, 0x1E6C,
                0x1E6E, 0x1E6E,
                0x1E70, 0x1E70,
                0x1E72, 0x1E72,
                0x1E74, 0x1E74,
                0x1E76, 0x1E76,
                0x1E78, 0x1E78,
                0x1E7A, 0x1E7A,
                0x1E7C, 0x1E7C,
                0x1E7E, 0x1E7E,
                0x1E80, 0x1E80,
                0x1E82, 0x1E82,
                0x1E84, 0x1E84,
                0x1E86, 0x1E86,
                0x1E88, 0x1E88,
                0x1E8A, 0x1E8A,
                0x1E8C, 0x1E8C,
                0x1E8E, 0x1E8E,
                0x1E90, 0x1E90,
                0x1E92, 0x1E92,
                0x1E94, 0x1E94,
                0x1E9E, 0x1E9E,
                0x1EA0, 0x1EA0,
                0x1EA2, 0x1EA2,
                0x1EA4, 0x1EA4,
                0x1EA6, 0x1EA6,
                0x1EA8, 0x1EA8,
                0x1EAA, 0x1EAA,
                0x1EAC, 0x1EAC,
                0x1EAE, 0x1EAE,
                0x1EB0, 0x1EB0,
                0x1EB2, 0x1EB2,
                0x1EB4, 0x1EB4,
                0x1EB6, 0x1EB6,
                0x1EB8, 0x1EB8,
                0x1EBA, 0x1EBA,
                0x1EBC, 0x1EBC,
                0x1EBE, 0x1EBE,
                0x1EC0, 0x1EC0,
                0x1EC2, 0x1EC2,
                0x1EC4, 0x1EC4,
                0x1EC6, 0x1EC6,
                0x1EC8, 0x1EC8,
                0x1ECA, 0x1ECA,
                0x1ECC, 0x1ECC,
                0x1ECE, 0x1ECE,
                0x1ED0, 0x1ED0,
                0x1ED2, 0x1ED2,
                0x1ED4, 0x1ED4,
                0x1ED6, 0x1ED6,
                0x1ED8, 0x1ED8,
                0x1EDA, 0x1EDA,
                0x1EDC, 0x1EDC,
                0x1EDE, 0x1EDE,
                0x1EE0, 0x1EE0,
                0x1EE2, 0x1EE2,
                0x1EE4, 0x1EE4,
                0x1EE6, 0x1EE6,
                0x1EE8, 0x1EE8,
                0x1EEA, 0x1EEA,
                0x1EEC, 0x1EEC,
                0x1EEE, 0x1EEE,
                0x1EF0, 0x1EF0,
                0x1EF2, 0x1EF2,
                0x1EF4, 0x1EF4,
                0x1EF6, 0x1EF6,
                0x1EF8, 0x1EF8,
                0x1EFA, 0x1EFA,
                0x1EFC, 0x1EFC,
                0x1EFE, 0x1EFE,
                0x1F08, 0x1F0F,
                0x1F18, 0x1F1D,
                0x1F28, 0x1F2F,
                0x1F38, 0x1F3F,
                0x1F48, 0x1F4D,
                0x1F59, 0x1F59,
                0x1F5B, 0x1F5B,
                0x1F5D, 0x1F5D,
                0x1F5F, 0x1F5F,
                0x1F68, 0x1F6F,
                0x1FB8, 0x1FBB,
                0x1FC8, 0x1FCB,
                0x1FD8, 0x1FDB,
                0x1FE8, 0x1FEC,
                0x1FF8, 0x1FFB,
                0x2102, 0x2102,
                0x2107, 0x2107,
                0x210B, 0x210D,
                0x2110, 0x2112,
                0x2115, 0x2115,
                0x2119, 0x211D,
                0x2124, 0x2124,
                0x2126, 0x2126,
                0x2128, 0x2128,
                0x212A, 0x212D,
                0x2130, 0x2133,
                0x213E, 0x213F,
                0x2145, 0x2145,
                0x2183, 0x2183,
                0x2C00, 0x2C2E,
                0x2C60, 0x2C60,
                0x2C62, 0x2C64,
                0x2C67, 0x2C67,
                0x2C69, 0x2C69,
                0x2C6B, 0x2C6B,
                0x2C6D, 0x2C70,
                0x2C72, 0x2C72,
                0x2C75, 0x2C75,
                0x2C7E, 0x2C80,
                0x2C82, 0x2C82,
                0x2C84, 0x2C84,
                0x2C86, 0x2C86,
                0x2C88, 0x2C88,
                0x2C8A, 0x2C8A,
                0x2C8C, 0x2C8C,
                0x2C8E, 0x2C8E,
                0x2C90, 0x2C90,
                0x2C92, 0x2C92,
                0x2C94, 0x2C94,
                0x2C96, 0x2C96,
                0x2C98, 0x2C98,
                0x2C9A, 0x2C9A,
                0x2C9C, 0x2C9C,
                0x2C9E, 0x2C9E,
                0x2CA0, 0x2CA0,
                0x2CA2, 0x2CA2,
                0x2CA4, 0x2CA4,
                0x2CA6, 0x2CA6,
                0x2CA8, 0x2CA8,
                0x2CAA, 0x2CAA,
                0x2CAC, 0x2CAC,
                0x2CAE, 0x2CAE,
                0x2CB0, 0x2CB0,
                0x2CB2, 0x2CB2,
                0x2CB4, 0x2CB4,
                0x2CB6, 0x2CB6,
                0x2CB8, 0x2CB8,
                0x2CBA, 0x2CBA,
                0x2CBC, 0x2CBC,
                0x2CBE, 0x2CBE,
                0x2CC0, 0x2CC0,
                0x2CC2, 0x2CC2,
                0x2CC4, 0x2CC4,
                0x2CC6, 0x2CC6,
                0x2CC8, 0x2CC8,
                0x2CCA, 0x2CCA,
                0x2CCC, 0x2CCC,
                0x2CCE, 0x2CCE,
                0x2CD0, 0x2CD0,
                0x2CD2, 0x2CD2,
                0x2CD4, 0x2CD4,
                0x2CD6, 0x2CD6,
                0x2CD8, 0x2CD8,
                0x2CDA, 0x2CDA,
                0x2CDC, 0x2CDC,
                0x2CDE, 0x2CDE,
                0x2CE0, 0x2CE0,
                0x2CE2, 0x2CE2,
                0x2CEB, 0x2CEB,
                0x2CED, 0x2CED,
                0xA640, 0xA640,
                0xA642, 0xA642,
                0xA644, 0xA644,
                0xA646, 0xA646,
                0xA648, 0xA648,
                0xA64A, 0xA64A,
                0xA64C, 0xA64C,
                0xA64E, 0xA64E,
                0xA650, 0xA650,
                0xA652, 0xA652,
                0xA654, 0xA654,
                0xA656, 0xA656,
                0xA658, 0xA658,
                0xA65A, 0xA65A,
                0xA65C, 0xA65C,
                0xA65E, 0xA65E,
                0xA660, 0xA660,
                0xA662, 0xA662,
                0xA664, 0xA664,
                0xA666, 0xA666,
                0xA668, 0xA668,
                0xA66A, 0xA66A,
                0xA66C, 0xA66C,
                0xA680, 0xA680,
                0xA682, 0xA682,
                0xA684, 0xA684,
                0xA686, 0xA686,
                0xA688, 0xA688,
                0xA68A, 0xA68A,
                0xA68C, 0xA68C,
                0xA68E, 0xA68E,
                0xA690, 0xA690,
                0xA692, 0xA692,
                0xA694, 0xA694,
                0xA696, 0xA696,
                0xA722, 0xA722,
                0xA724, 0xA724,
                0xA726, 0xA726,
                0xA728, 0xA728,
                0xA72A, 0xA72A,
                0xA72C, 0xA72C,
                0xA72E, 0xA72E,
                0xA732, 0xA732,
                0xA734, 0xA734,
                0xA736, 0xA736,
                0xA738, 0xA738,
                0xA73A, 0xA73A,
                0xA73C, 0xA73C,
                0xA73E, 0xA73E,
                0xA740, 0xA740,
                0xA742, 0xA742,
                0xA744, 0xA744,
                0xA746, 0xA746,
                0xA748, 0xA748,
                0xA74A, 0xA74A,
                0xA74C, 0xA74C,
                0xA74E, 0xA74E,
                0xA750, 0xA750,
                0xA752, 0xA752,
                0xA754, 0xA754,
                0xA756, 0xA756,
                0xA758, 0xA758,
                0xA75A, 0xA75A,
                0xA75C, 0xA75C,
                0xA75E, 0xA75E,
                0xA760, 0xA760,
                0xA762, 0xA762,
                0xA764, 0xA764,
                0xA766, 0xA766,
                0xA768, 0xA768,
                0xA76A, 0xA76A,
                0xA76C, 0xA76C,
                0xA76E, 0xA76E,
                0xA779, 0xA779,
                0xA77B, 0xA77B,
                0xA77D, 0xA77E,
                0xA780, 0xA780,
                0xA782, 0xA782,
                0xA784, 0xA784,
                0xA786, 0xA786,
                0xA78B, 0xA78B,
                0xA78D, 0xA78D,
                0xA790, 0xA790,
                0xA7A0, 0xA7A0,
                0xA7A2, 0xA7A2,
                0xA7A4, 0xA7A4,
                0xA7A6, 0xA7A6,
                0xA7A8, 0xA7A8,
                0xFF21, 0xFF3A,
                0x10400, 0x10427,
                0x1D400, 0x1D419,
                0x1D434, 0x1D44D,
                0x1D468, 0x1D481,
                0x1D49C, 0x1D49C,
                0x1D49E, 0x1D49F,
                0x1D4A2, 0x1D4A2,
                0x1D4A5, 0x1D4A6,
                0x1D4A9, 0x1D4AC,
                0x1D4AE, 0x1D4B5,
                0x1D4D0, 0x1D4E9,
                0x1D504, 0x1D505,
                0x1D507, 0x1D50A,
                0x1D50D, 0x1D514,
                0x1D516, 0x1D51C,
                0x1D538, 0x1D539,
                0x1D53B, 0x1D53E,
                0x1D540, 0x1D544,
                0x1D546, 0x1D546,
                0x1D54A, 0x1D550,
                0x1D56C, 0x1D585,
                0x1D5A0, 0x1D5B9,
                0x1D5D4, 0x1D5ED,
                0x1D608, 0x1D621,
                0x1D63C, 0x1D655,
                0x1D670, 0x1D689,
                0x1D6A8, 0x1D6C0,
                0x1D6E2, 0x1D6FA,
                0x1D71C, 0x1D734,
                0x1D756, 0x1D76E,
                0x1D790, 0x1D7A8,
                0x1D7CA, 0x1D7CA
        }));


        CATEGORIES.put("Mc", pred(new int[]{
                0x903, 0x903,
                0x93B, 0x93B,
                0x93E, 0x940,
                0x949, 0x94C,
                0x94E, 0x94F,
                0x982, 0x983,
                0x9BE, 0x9C0,
                0x9C7, 0x9C8,
                0x9CB, 0x9CC,
                0x9D7, 0x9D7,
                0xA03, 0xA03,
                0xA3E, 0xA40,
                0xA83, 0xA83,
                0xABE, 0xAC0,
                0xAC9, 0xAC9,
                0xACB, 0xACC,
                0xB02, 0xB03,
                0xB3E, 0xB3E,
                0xB40, 0xB40,
                0xB47, 0xB48,
                0xB4B, 0xB4C,
                0xB57, 0xB57,
                0xBBE, 0xBBF,
                0xBC1, 0xBC2,
                0xBC6, 0xBC8,
                0xBCA, 0xBCC,
                0xBD7, 0xBD7,
                0xC01, 0xC03,
                0xC41, 0xC44,
                0xC82, 0xC83,
                0xCBE, 0xCBE,
                0xCC0, 0xCC4,
                0xCC7, 0xCC8,
                0xCCA, 0xCCB,
                0xCD5, 0xCD6,
                0xD02, 0xD03,
                0xD3E, 0xD40,
                0xD46, 0xD48,
                0xD4A, 0xD4C,
                0xD57, 0xD57,
                0xD82, 0xD83,
                0xDCF, 0xDD1,
                0xDD8, 0xDDF,
                0xDF2, 0xDF3,
                0xF3E, 0xF3F,
                0xF7F, 0xF7F,
                0x102B, 0x102C,
                0x1031, 0x1031,
                0x1038, 0x1038,
                0x103B, 0x103C,
                0x1056, 0x1057,
                0x1062, 0x1064,
                0x1067, 0x106D,
                0x1083, 0x1084,
                0x1087, 0x108C,
                0x108F, 0x108F,
                0x109A, 0x109C,
                0x17B6, 0x17B6,
                0x17BE, 0x17C5,
                0x17C7, 0x17C8,
                0x1923, 0x1926,
                0x1929, 0x192B,
                0x1930, 0x1931,
                0x1933, 0x1938,
                0x19B0, 0x19C0,
                0x19C8, 0x19C9,
                0x1A19, 0x1A1B,
                0x1A55, 0x1A55,
                0x1A57, 0x1A57,
                0x1A61, 0x1A61,
                0x1A63, 0x1A64,
                0x1A6D, 0x1A72,
                0x1B04, 0x1B04,
                0x1B35, 0x1B35,
                0x1B3B, 0x1B3B,
                0x1B3D, 0x1B41,
                0x1B43, 0x1B44,
                0x1B82, 0x1B82,
                0x1BA1, 0x1BA1,
                0x1BA6, 0x1BA7,
                0x1BAA, 0x1BAA,
                0x1BE7, 0x1BE7,
                0x1BEA, 0x1BEC,
                0x1BEE, 0x1BEE,
                0x1BF2, 0x1BF3,
                0x1C24, 0x1C2B,
                0x1C34, 0x1C35,
                0x1CE1, 0x1CE1,
                0x1CF2, 0x1CF2,
                0xA823, 0xA824,
                0xA827, 0xA827,
                0xA880, 0xA881,
                0xA8B4, 0xA8C3,
                0xA952, 0xA953,
                0xA983, 0xA983,
                0xA9B4, 0xA9B5,
                0xA9BA, 0xA9BB,
                0xA9BD, 0xA9C0,
                0xAA2F, 0xAA30,
                0xAA33, 0xAA34,
                0xAA4D, 0xAA4D,
                0xAA7B, 0xAA7B,
                0xABE3, 0xABE4,
                0xABE6, 0xABE7,
                0xABE9, 0xABEA,
                0xABEC, 0xABEC,
                0x11000, 0x11000,
                0x11002, 0x11002,
                0x11082, 0x11082,
                0x110B0, 0x110B2,
                0x110B7, 0x110B8,
                0x1D165, 0x1D166,
                0x1D16D, 0x1D172
        }));


        CATEGORIES.put("Me", pred(new int[]{
                0x488, 0x489,
                0x20DD, 0x20E0,
                0x20E2, 0x20E4,
                0xA670, 0xA672
        }));


        CATEGORIES.put("Mn", pred(new int[]{
                0x300, 0x36F,
                0x483, 0x487,
                0x591, 0x5BD,
                0x5BF, 0x5BF,
                0x5C1, 0x5C2,
                0x5C4, 0x5C5,
                0x5C7, 0x5C7,
                0x610, 0x61A,
                0x64B, 0x65F,
                0x670, 0x670,
                0x6D6, 0x6DC,
                0x6DF, 0x6E4,
                0x6E7, 0x6E8,
                0x6EA, 0x6ED,
                0x711, 0x711,
                0x730, 0x74A,
                0x7A6, 0x7B0,
                0x7EB, 0x7F3,
                0x816, 0x819,
                0x81B, 0x823,
                0x825, 0x827,
                0x829, 0x82D,
                0x859, 0x85B,
                0x900, 0x902,
                0x93A, 0x93A,
                0x93C, 0x93C,
                0x941, 0x948,
                0x94D, 0x94D,
                0x951, 0x957,
                0x962, 0x963,
                0x981, 0x981,
                0x9BC, 0x9BC,
                0x9C1, 0x9C4,
                0x9CD, 0x9CD,
                0x9E2, 0x9E3,
                0xA01, 0xA02,
                0xA3C, 0xA3C,
                0xA41, 0xA42,
                0xA47, 0xA48,
                0xA4B, 0xA4D,
                0xA51, 0xA51,
                0xA70, 0xA71,
                0xA75, 0xA75,
                0xA81, 0xA82,
                0xABC, 0xABC,
                0xAC1, 0xAC5,
                0xAC7, 0xAC8,
                0xACD, 0xACD,
                0xAE2, 0xAE3,
                0xB01, 0xB01,
                0xB3C, 0xB3C,
                0xB3F, 0xB3F,
                0xB41, 0xB44,
                0xB4D, 0xB4D,
                0xB56, 0xB56,
                0xB62, 0xB63,
                0xB82, 0xB82,
                0xBC0, 0xBC0,
                0xBCD, 0xBCD,
                0xC3E, 0xC40,
                0xC46, 0xC48,
                0xC4A, 0xC4D,
                0xC55, 0xC56,
                0xC62, 0xC63,
                0xCBC, 0xCBC,
                0xCBF, 0xCBF,
                0xCC6, 0xCC6,
                0xCCC, 0xCCD,
                0xCE2, 0xCE3,
                0xD41, 0xD44,
                0xD4D, 0xD4D,
                0xD62, 0xD63,
                0xDCA, 0xDCA,
                0xDD2, 0xDD4,
                0xDD6, 0xDD6,
                0xE31, 0xE31,
                0xE34, 0xE3A,
                0xE47, 0xE4E,
                0xEB1, 0xEB1,
                0xEB4, 0xEB9,
                0xEBB, 0xEBC,
                0xEC8, 0xECD,
                0xF18, 0xF19,
                0xF35, 0xF35,
                0xF37, 0xF37,
                0xF39, 0xF39,
                0xF71, 0xF7E,
                0xF80, 0xF84,
                0xF86, 0xF87,
                0xF8D, 0xF97,
                0xF99, 0xFBC,
                0xFC6, 0xFC6,
                0x102D, 0x1030,
                0x1032, 0x1037,
                0x1039, 0x103A,
                0x103D, 0x103E,
                0x1058, 0x1059,
                0x105E, 0x1060,
                0x1071, 0x1074,
                0x1082, 0x1082,
                0x1085, 0x1086,
                0x108D, 0x108D,
                0x109D, 0x109D,
                0x135D, 0x135F,
                0x1712, 0x1714,
                0x1732, 0x1734,
                0x1752, 0x1753,
                0x1772, 0x1773,
                0x17B7, 0x17BD,
                0x17C6, 0x17C6,
                0x17C9, 0x17D3,
                0x17DD, 0x17DD,
                0x180B, 0x180D,
                0x18A9, 0x18A9,
                0x1920, 0x1922,
                0x1927, 0x1928,
                0x1932, 0x1932,
                0x1939, 0x193B,
                0x1A17, 0x1A18,
                0x1A56, 0x1A56,
                0x1A58, 0x1A5E,
                0x1A60, 0x1A60,
                0x1A62, 0x1A62,
                0x1A65, 0x1A6C,
                0x1A73, 0x1A7C,
                0x1A7F, 0x1A7F,
                0x1B00, 0x1B03,
                0x1B34, 0x1B34,
                0x1B36, 0x1B3A,
                0x1B3C, 0x1B3C,
                0x1B42, 0x1B42,
                0x1B6B, 0x1B73,
                0x1B80, 0x1B81,
                0x1BA2, 0x1BA5,
                0x1BA8, 0x1BA9,
                0x1BE6, 0x1BE6,
                0x1BE8, 0x1BE9,
                0x1BED, 0x1BED,
                0x1BEF, 0x1BF1,
                0x1C2C, 0x1C33,
                0x1C36, 0x1C37,
                0x1CD0, 0x1CD2,
                0x1CD4, 0x1CE0,
                0x1CE2, 0x1CE8,
                0x1CED, 0x1CED,
                0x1DC0, 0x1DE6,
                0x1DFC, 0x1DFF,
                0x20D0, 0x20DC,
                0x20E1, 0x20E1,
                0x20E5, 0x20F0,
                0x2CEF, 0x2CF1,
                0x2D7F, 0x2D7F,
                0x2DE0, 0x2DFF,
                0x302A, 0x302F,
                0x3099, 0x309A,
                0xA66F, 0xA66F,
                0xA67C, 0xA67D,
                0xA6F0, 0xA6F1,
                0xA802, 0xA802,
                0xA806, 0xA806,
                0xA80B, 0xA80B,
                0xA825, 0xA826,
                0xA8C4, 0xA8C4,
                0xA8E0, 0xA8F1,
                0xA926, 0xA92D,
                0xA947, 0xA951,
                0xA980, 0xA982,
                0xA9B3, 0xA9B3,
                0xA9B6, 0xA9B9,
                0xA9BC, 0xA9BC,
                0xAA29, 0xAA2E,
                0xAA31, 0xAA32,
                0xAA35, 0xAA36,
                0xAA43, 0xAA43,
                0xAA4C, 0xAA4C,
                0xAAB0, 0xAAB0,
                0xAAB2, 0xAAB4,
                0xAAB7, 0xAAB8,
                0xAABE, 0xAABF,
                0xAAC1, 0xAAC1,
                0xABE5, 0xABE5,
                0xABE8, 0xABE8,
                0xABED, 0xABED,
                0xFB1E, 0xFB1E,
                0xFE00, 0xFE0F,
                0xFE20, 0xFE26,
                0x101FD, 0x101FD,
                0x10A01, 0x10A03,
                0x10A05, 0x10A06,
                0x10A0C, 0x10A0F,
                0x10A38, 0x10A3A,
                0x10A3F, 0x10A3F,
                0x11001, 0x11001,
                0x11038, 0x11046,
                0x11080, 0x11081,
                0x110B3, 0x110B6,
                0x110B9, 0x110BA,
                0x1D167, 0x1D169,
                0x1D17B, 0x1D182,
                0x1D185, 0x1D18B,
                0x1D1AA, 0x1D1AD,
                0x1D242, 0x1D244,
                0xE0100, 0xE01EF
        }));


        CATEGORIES.put("Nd", pred(new int[]{
                0x30, 0x39,
                0x660, 0x669,
                0x6F0, 0x6F9,
                0x7C0, 0x7C9,
                0x966, 0x96F,
                0x9E6, 0x9EF,
                0xA66, 0xA6F,
                0xAE6, 0xAEF,
                0xB66, 0xB6F,
                0xBE6, 0xBEF,
                0xC66, 0xC6F,
                0xCE6, 0xCEF,
                0xD66, 0xD6F,
                0xE50, 0xE59,
                0xED0, 0xED9,
                0xF20, 0xF29,
                0x1040, 0x1049,
                0x1090, 0x1099,
                0x17E0, 0x17E9,
                0x1810, 0x1819,
                0x1946, 0x194F,
                0x19D0, 0x19D9,
                0x1A80, 0x1A89,
                0x1A90, 0x1A99,
                0x1B50, 0x1B59,
                0x1BB0, 0x1BB9,
                0x1C40, 0x1C49,
                0x1C50, 0x1C59,
                0xA620, 0xA629,
                0xA8D0, 0xA8D9,
                0xA900, 0xA909,
                0xA9D0, 0xA9D9,
                0xAA50, 0xAA59,
                0xABF0, 0xABF9,
                0xFF10, 0xFF19,
                0x104A0, 0x104A9,
                0x11066, 0x1106F,
                0x1D7CE, 0x1D7FF
        }));


        CATEGORIES.put("Nl", pred(new int[]{
                0x16EE, 0x16F0,
                0x2160, 0x2182,
                0x2185, 0x2188,
                0x3007, 0x3007,
                0x3021, 0x3029,
                0x3038, 0x303A,
                0xA6E6, 0xA6EF,
                0x10140, 0x10174,
                0x10341, 0x10341,
                0x1034A, 0x1034A,
                0x103D1, 0x103D5,
                0x12400, 0x12462
        }));


        CATEGORIES.put("No", pred(new int[]{
                0xB2, 0xB3,
                0xB9, 0xB9,
                0xBC, 0xBE,
                0x9F4, 0x9F9,
                0xB72, 0xB77,
                0xBF0, 0xBF2,
                0xC78, 0xC7E,
                0xD70, 0xD75,
                0xF2A, 0xF33,
                0x1369, 0x137C,
                0x17F0, 0x17F9,
                0x19DA, 0x19DA,
                0x2070, 0x2070,
                0x2074, 0x2079,
                0x2080, 0x2089,
                0x2150, 0x215F,
                0x2189, 0x2189,
                0x2460, 0x249B,
                0x24EA, 0x24FF,
                0x2776, 0x2793,
                0x2CFD, 0x2CFD,
                0x3192, 0x3195,
                0x3220, 0x3229,
                0x3251, 0x325F,
                0x3280, 0x3289,
                0x32B1, 0x32BF,
                0xA830, 0xA835,
                0x10107, 0x10133,
                0x10175, 0x10178,
                0x1018A, 0x1018A,
                0x10320, 0x10323,
                0x10858, 0x1085F,
                0x10916, 0x1091B,
                0x10A40, 0x10A47,
                0x10A7D, 0x10A7E,
                0x10B58, 0x10B5F,
                0x10B78, 0x10B7F,
                0x10E60, 0x10E7E,
                0x11052, 0x11065,
                0x1D360, 0x1D371,
                0x1F100, 0x1F10A
        }));


        CATEGORIES.put("Pc", pred(new int[]{
                0x5F, 0x5F,
                0x203F, 0x2040,
                0x2054, 0x2054,
                0xFE33, 0xFE34,
                0xFE4D, 0xFE4F,
                0xFF3F, 0xFF3F
        }));


        CATEGORIES.put("Pd", pred(new int[]{
                0x2D, 0x2D,
                0x58A, 0x58A,
                0x5BE, 0x5BE,
                0x1400, 0x1400,
                0x1806, 0x1806,
                0x2010, 0x2015,
                0x2E17, 0x2E17,
                0x2E1A, 0x2E1A,
                0x301C, 0x301C,
                0x3030, 0x3030,
                0x30A0, 0x30A0,
                0xFE31, 0xFE32,
                0xFE58, 0xFE58,
                0xFE63, 0xFE63,
                0xFF0D, 0xFF0D
        }));


        CATEGORIES.put("Pe", pred(new int[]{
                0x29, 0x29,
                0x5D, 0x5D,
                0x7D, 0x7D,
                0xF3B, 0xF3B,
                0xF3D, 0xF3D,
                0x169C, 0x169C,
                0x2046, 0x2046,
                0x207E, 0x207E,
                0x208E, 0x208E,
                0x232A, 0x232A,
                0x2769, 0x2769,
                0x276B, 0x276B,
                0x276D, 0x276D,
                0x276F, 0x276F,
                0x2771, 0x2771,
                0x2773, 0x2773,
                0x2775, 0x2775,
                0x27C6, 0x27C6,
                0x27E7, 0x27E7,
                0x27E9, 0x27E9,
                0x27EB, 0x27EB,
                0x27ED, 0x27ED,
                0x27EF, 0x27EF,
                0x2984, 0x2984,
                0x2986, 0x2986,
                0x2988, 0x2988,
                0x298A, 0x298A,
                0x298C, 0x298C,
                0x298E, 0x298E,
                0x2990, 0x2990,
                0x2992, 0x2992,
                0x2994, 0x2994,
                0x2996, 0x2996,
                0x2998, 0x2998,
                0x29D9, 0x29D9,
                0x29DB, 0x29DB,
                0x29FD, 0x29FD,
                0x2E23, 0x2E23,
                0x2E25, 0x2E25,
                0x2E27, 0x2E27,
                0x2E29, 0x2E29,
                0x3009, 0x3009,
                0x300B, 0x300B,
                0x300D, 0x300D,
                0x300F, 0x300F,
                0x3011, 0x3011,
                0x3015, 0x3015,
                0x3017, 0x3017,
                0x3019, 0x3019,
                0x301B, 0x301B,
                0x301E, 0x301F,
                0xFD3F, 0xFD3F,
                0xFE18, 0xFE18,
                0xFE36, 0xFE36,
                0xFE38, 0xFE38,
                0xFE3A, 0xFE3A,
                0xFE3C, 0xFE3C,
                0xFE3E, 0xFE3E,
                0xFE40, 0xFE40,
                0xFE42, 0xFE42,
                0xFE44, 0xFE44,
                0xFE48, 0xFE48,
                0xFE5A, 0xFE5A,
                0xFE5C, 0xFE5C,
                0xFE5E, 0xFE5E,
                0xFF09, 0xFF09,
                0xFF3D, 0xFF3D,
                0xFF5D, 0xFF5D,
                0xFF60, 0xFF60,
                0xFF63, 0xFF63
        }));


        CATEGORIES.put("Pf", pred(new int[]{
                0xBB, 0xBB,
                0x2019, 0x2019,
                0x201D, 0x201D,
                0x203A, 0x203A,
                0x2E03, 0x2E03,
                0x2E05, 0x2E05,
                0x2E0A, 0x2E0A,
                0x2E0D, 0x2E0D,
                0x2E1D, 0x2E1D,
                0x2E21, 0x2E21
        }));


        CATEGORIES.put("Pi", pred(new int[]{
                0xAB, 0xAB,
                0x2018, 0x2018,
                0x201B, 0x201C,
                0x201F, 0x201F,
                0x2039, 0x2039,
                0x2E02, 0x2E02,
                0x2E04, 0x2E04,
                0x2E09, 0x2E09,
                0x2E0C, 0x2E0C,
                0x2E1C, 0x2E1C,
                0x2E20, 0x2E20
        }));


        CATEGORIES.put("Po", pred(new int[]{
                0x21, 0x23,
                0x25, 0x27,
                0x2A, 0x2A,
                0x2C, 0x2C,
                0x2E, 0x2F,
                0x3A, 0x3B,
                0x3F, 0x40,
                0x5C, 0x5C,
                0xA1, 0xA1,
                0xB7, 0xB7,
                0xBF, 0xBF,
                0x37E, 0x37E,
                0x387, 0x387,
                0x55A, 0x55F,
                0x589, 0x589,
                0x5C0, 0x5C0,
                0x5C3, 0x5C3,
                0x5C6, 0x5C6,
                0x5F3, 0x5F4,
                0x609, 0x60A,
                0x60C, 0x60D,
                0x61B, 0x61B,
                0x61E, 0x61F,
                0x66A, 0x66D,
                0x6D4, 0x6D4,
                0x700, 0x70D,
                0x7F7, 0x7F9,
                0x830, 0x83E,
                0x85E, 0x85E,
                0x964, 0x965,
                0x970, 0x970,
                0xDF4, 0xDF4,
                0xE4F, 0xE4F,
                0xE5A, 0xE5B,
                0xF04, 0xF12,
                0xF85, 0xF85,
                0xFD0, 0xFD4,
                0xFD9, 0xFDA,
                0x104A, 0x104F,
                0x10FB, 0x10FB,
                0x1361, 0x1368,
                0x166D, 0x166E,
                0x16EB, 0x16ED,
                0x1735, 0x1736,
                0x17D4, 0x17D6,
                0x17D8, 0x17DA,
                0x1800, 0x1805,
                0x1807, 0x180A,
                0x1944, 0x1945,
                0x1A1E, 0x1A1F,
                0x1AA0, 0x1AA6,
                0x1AA8, 0x1AAD,
                0x1B5A, 0x1B60,
                0x1BFC, 0x1BFF,
                0x1C3B, 0x1C3F,
                0x1C7E, 0x1C7F,
                0x1CD3, 0x1CD3,
                0x2016, 0x2017,
                0x2020, 0x2027,
                0x2030, 0x2038,
                0x203B, 0x203E,
                0x2041, 0x2043,
                0x2047, 0x2051,
                0x2053, 0x2053,
                0x2055, 0x205E,
                0x2CF9, 0x2CFC,
                0x2CFE, 0x2CFF,
                0x2D70, 0x2D70,
                0x2E00, 0x2E01,
                0x2E06, 0x2E08,
                0x2E0B, 0x2E0B,
                0x2E0E, 0x2E16,
                0x2E18, 0x2E19,
                0x2E1B, 0x2E1B,
                0x2E1E, 0x2E1F,
                0x2E2A, 0x2E2E,
                0x2E30, 0x2E31,
                0x3001, 0x3003,
                0x303D, 0x303D,
                0x30FB, 0x30FB,
                0xA4FE, 0xA4FF,
                0xA60D, 0xA60F,
                0xA673, 0xA673,
                0xA67E, 0xA67E,
                0xA6F2, 0xA6F7,
                0xA874, 0xA877,
                0xA8CE, 0xA8CF,
                0xA8F8, 0xA8FA,
                0xA92E, 0xA92F,
                0xA95F, 0xA95F,
                0xA9C1, 0xA9CD,
                0xA9DE, 0xA9DF,
                0xAA5C, 0xAA5F,
                0xAADE, 0xAADF,
                0xABEB, 0xABEB,
                0xFE10, 0xFE16,
                0xFE19, 0xFE19,
                0xFE30, 0xFE30,
                0xFE45, 0xFE46,
                0xFE49, 0xFE4C,
                0xFE50, 0xFE52,
                0xFE54, 0xFE57,
                0xFE5F, 0xFE61,
                0xFE68, 0xFE68,
                0xFE6A, 0xFE6B,
                0xFF01, 0xFF03,
                0xFF05, 0xFF07,
                0xFF0A, 0xFF0A,
                0xFF0C, 0xFF0C,
                0xFF0E, 0xFF0F,
                0xFF1A, 0xFF1B,
                0xFF1F, 0xFF20,
                0xFF3C, 0xFF3C,
                0xFF61, 0xFF61,
                0xFF64, 0xFF65,
                0x10100, 0x10101,
                0x1039F, 0x1039F,
                0x103D0, 0x103D0,
                0x10857, 0x10857,
                0x1091F, 0x1091F,
                0x1093F, 0x1093F,
                0x10A50, 0x10A58,
                0x10A7F, 0x10A7F,
                0x10B39, 0x10B3F,
                0x11047, 0x1104D,
                0x110BB, 0x110BC,
                0x110BE, 0x110C1,
                0x12470, 0x12473
        }));


        CATEGORIES.put("Ps", pred(new int[]{
                0x28, 0x28,
                0x5B, 0x5B,
                0x7B, 0x7B,
                0xF3A, 0xF3A,
                0xF3C, 0xF3C,
                0x169B, 0x169B,
                0x201A, 0x201A,
                0x201E, 0x201E,
                0x2045, 0x2045,
                0x207D, 0x207D,
                0x208D, 0x208D,
                0x2329, 0x2329,
                0x2768, 0x2768,
                0x276A, 0x276A,
                0x276C, 0x276C,
                0x276E, 0x276E,
                0x2770, 0x2770,
                0x2772, 0x2772,
                0x2774, 0x2774,
                0x27C5, 0x27C5,
                0x27E6, 0x27E6,
                0x27E8, 0x27E8,
                0x27EA, 0x27EA,
                0x27EC, 0x27EC,
                0x27EE, 0x27EE,
                0x2983, 0x2983,
                0x2985, 0x2985,
                0x2987, 0x2987,
                0x2989, 0x2989,
                0x298B, 0x298B,
                0x298D, 0x298D,
                0x298F, 0x298F,
                0x2991, 0x2991,
                0x2993, 0x2993,
                0x2995, 0x2995,
                0x2997, 0x2997,
                0x29D8, 0x29D8,
                0x29DA, 0x29DA,
                0x29FC, 0x29FC,
                0x2E22, 0x2E22,
                0x2E24, 0x2E24,
                0x2E26, 0x2E26,
                0x2E28, 0x2E28,
                0x3008, 0x3008,
                0x300A, 0x300A,
                0x300C, 0x300C,
                0x300E, 0x300E,
                0x3010, 0x3010,
                0x3014, 0x3014,
                0x3016, 0x3016,
                0x3018, 0x3018,
                0x301A, 0x301A,
                0x301D, 0x301D,
                0xFD3E, 0xFD3E,
                0xFE17, 0xFE17,
                0xFE35, 0xFE35,
                0xFE37, 0xFE37,
                0xFE39, 0xFE39,
                0xFE3B, 0xFE3B,
                0xFE3D, 0xFE3D,
                0xFE3F, 0xFE3F,
                0xFE41, 0xFE41,
                0xFE43, 0xFE43,
                0xFE47, 0xFE47,
                0xFE59, 0xFE59,
                0xFE5B, 0xFE5B,
                0xFE5D, 0xFE5D,
                0xFF08, 0xFF08,
                0xFF3B, 0xFF3B,
                0xFF5B, 0xFF5B,
                0xFF5F, 0xFF5F,
                0xFF62, 0xFF62
        }));


        CATEGORIES.put("Sc", pred(new int[]{
                0x24, 0x24,
                0xA2, 0xA5,
                0x60B, 0x60B,
                0x9F2, 0x9F3,
                0x9FB, 0x9FB,
                0xAF1, 0xAF1,
                0xBF9, 0xBF9,
                0xE3F, 0xE3F,
                0x17DB, 0x17DB,
                0x20A0, 0x20B9,
                0xA838, 0xA838,
                0xFDFC, 0xFDFC,
                0xFE69, 0xFE69,
                0xFF04, 0xFF04,
                0xFFE0, 0xFFE1,
                0xFFE5, 0xFFE6
        }));


        CATEGORIES.put("Sk", pred(new int[]{
                0x5E, 0x5E,
                0x60, 0x60,
                0xA8, 0xA8,
                0xAF, 0xAF,
                0xB4, 0xB4,
                0xB8, 0xB8,
                0x2C2, 0x2C5,
                0x2D2, 0x2DF,
                0x2E5, 0x2EB,
                0x2ED, 0x2ED,
                0x2EF, 0x2FF,
                0x375, 0x375,
                0x384, 0x385,
                0x1FBD, 0x1FBD,
                0x1FBF, 0x1FC1,
                0x1FCD, 0x1FCF,
                0x1FDD, 0x1FDF,
                0x1FED, 0x1FEF,
                0x1FFD, 0x1FFE,
                0x309B, 0x309C,
                0xA700, 0xA716,
                0xA720, 0xA721,
                0xA789, 0xA78A,
                0xFBB2, 0xFBC1,
                0xFF3E, 0xFF3E,
                0xFF40, 0xFF40,
                0xFFE3, 0xFFE3
        }));


        CATEGORIES.put("Sm", pred(new int[]{
                0x2B, 0x2B,
                0x3C, 0x3E,
                0x7C, 0x7C,
                0x7E, 0x7E,
                0xAC, 0xAC,
                0xB1, 0xB1,
                0xD7, 0xD7,
                0xF7, 0xF7,
                0x3F6, 0x3F6,
                0x606, 0x608,
                0x2044, 0x2044,
                0x2052, 0x2052,
                0x207A, 0x207C,
                0x208A, 0x208C,
                0x2118, 0x2118,
                0x2140, 0x2144,
                0x214B, 0x214B,
                0x2190, 0x2194,
                0x219A, 0x219B,
                0x21A0, 0x21A0,
                0x21A3, 0x21A3,
                0x21A6, 0x21A6,
                0x21AE, 0x21AE,
                0x21CE, 0x21CF,
                0x21D2, 0x21D2,
                0x21D4, 0x21D4,
                0x21F4, 0x22FF,
                0x2308, 0x230B,
                0x2320, 0x2321,
                0x237C, 0x237C,
                0x239B, 0x23B3,
                0x23DC, 0x23E1,
                0x25B7, 0x25B7,
                0x25C1, 0x25C1,
                0x25F8, 0x25FF,
                0x266F, 0x266F,
                0x27C0, 0x27C4,
                0x27C7, 0x27CA,
                0x27CC, 0x27CC,
                0x27CE, 0x27E5,
                0x27F0, 0x27FF,
                0x2900, 0x2982,
                0x2999, 0x29D7,
                0x29DC, 0x29FB,
                0x29FE, 0x2AFF,
                0x2B30, 0x2B44,
                0x2B47, 0x2B4C,
                0xFB29, 0xFB29,
                0xFE62, 0xFE62,
                0xFE64, 0xFE66,
                0xFF0B, 0xFF0B,
                0xFF1C, 0xFF1E,
                0xFF5C, 0xFF5C,
                0xFF5E, 0xFF5E,
                0xFFE2, 0xFFE2,
                0xFFE9, 0xFFEC,
                0x1D6C1, 0x1D6C1,
                0x1D6DB, 0x1D6DB,
                0x1D6FB, 0x1D6FB,
                0x1D715, 0x1D715,
                0x1D735, 0x1D735,
                0x1D74F, 0x1D74F,
                0x1D76F, 0x1D76F,
                0x1D789, 0x1D789,
                0x1D7A9, 0x1D7A9,
                0x1D7C3, 0x1D7C3
        }));


        CATEGORIES.put("So", pred(new int[]{
                0xA6, 0xA7,
                0xA9, 0xA9,
                0xAE, 0xAE,
                0xB0, 0xB0,
                0xB6, 0xB6,
                0x482, 0x482,
                0x60E, 0x60F,
                0x6DE, 0x6DE,
                0x6E9, 0x6E9,
                0x6FD, 0x6FE,
                0x7F6, 0x7F6,
                0x9FA, 0x9FA,
                0xB70, 0xB70,
                0xBF3, 0xBF8,
                0xBFA, 0xBFA,
                0xC7F, 0xC7F,
                0xD79, 0xD79,
                0xF01, 0xF03,
                0xF13, 0xF17,
                0xF1A, 0xF1F,
                0xF34, 0xF34,
                0xF36, 0xF36,
                0xF38, 0xF38,
                0xFBE, 0xFC5,
                0xFC7, 0xFCC,
                0xFCE, 0xFCF,
                0xFD5, 0xFD8,
                0x109E, 0x109F,
                0x1360, 0x1360,
                0x1390, 0x1399,
                0x1940, 0x1940,
                0x19DE, 0x19FF,
                0x1B61, 0x1B6A,
                0x1B74, 0x1B7C,
                0x2100, 0x2101,
                0x2103, 0x2106,
                0x2108, 0x2109,
                0x2114, 0x2114,
                0x2116, 0x2117,
                0x211E, 0x2123,
                0x2125, 0x2125,
                0x2127, 0x2127,
                0x2129, 0x2129,
                0x212E, 0x212E,
                0x213A, 0x213B,
                0x214A, 0x214A,
                0x214C, 0x214D,
                0x214F, 0x214F,
                0x2195, 0x2199,
                0x219C, 0x219F,
                0x21A1, 0x21A2,
                0x21A4, 0x21A5,
                0x21A7, 0x21AD,
                0x21AF, 0x21CD,
                0x21D0, 0x21D1,
                0x21D3, 0x21D3,
                0x21D5, 0x21F3,
                0x2300, 0x2307,
                0x230C, 0x231F,
                0x2322, 0x2328,
                0x232B, 0x237B,
                0x237D, 0x239A,
                0x23B4, 0x23DB,
                0x23E2, 0x23F3,
                0x2400, 0x2426,
                0x2440, 0x244A,
                0x249C, 0x24E9,
                0x2500, 0x25B6,
                0x25B8, 0x25C0,
                0x25C2, 0x25F7,
                0x2600, 0x266E,
                0x2670, 0x26FF,
                0x2701, 0x2767,
                0x2794, 0x27BF,
                0x2800, 0x28FF,
                0x2B00, 0x2B2F,
                0x2B45, 0x2B46,
                0x2B50, 0x2B59,
                0x2CE5, 0x2CEA,
                0x2E80, 0x2E99,
                0x2E9B, 0x2EF3,
                0x2F00, 0x2FD5,
                0x2FF0, 0x2FFB,
                0x3004, 0x3004,
                0x3012, 0x3013,
                0x3020, 0x3020,
                0x3036, 0x3037,
                0x303E, 0x303F,
                0x3190, 0x3191,
                0x3196, 0x319F,
                0x31C0, 0x31E3,
                0x3200, 0x321E,
                0x322A, 0x3250,
                0x3260, 0x327F,
                0x328A, 0x32B0,
                0x32C0, 0x32FE,
                0x3300, 0x33FF,
                0x4DC0, 0x4DFF,
                0xA490, 0xA4C6,
                0xA828, 0xA82B,
                0xA836, 0xA837,
                0xA839, 0xA839,
                0xAA77, 0xAA79,
                0xFDFD, 0xFDFD,
                0xFFE4, 0xFFE4,
                0xFFE8, 0xFFE8,
                0xFFED, 0xFFEE,
                0xFFFC, 0xFFFD,
                0x10102, 0x10102,
                0x10137, 0x1013F,
                0x10179, 0x10189,
                0x10190, 0x1019B,
                0x101D0, 0x101FC,
                0x1D000, 0x1D0F5,
                0x1D100, 0x1D126,
                0x1D129, 0x1D164,
                0x1D16A, 0x1D16C,
                0x1D183, 0x1D184,
                0x1D18C, 0x1D1A9,
                0x1D1AE, 0x1D1DD,
                0x1D200, 0x1D241,
                0x1D245, 0x1D245,
                0x1D300, 0x1D356,
                0x1F000, 0x1F02B,
                0x1F030, 0x1F093,
                0x1F0A0, 0x1F0AE,
                0x1F0B1, 0x1F0BE,
                0x1F0C1, 0x1F0CF,
                0x1F0D1, 0x1F0DF,
                0x1F110, 0x1F12E,
                0x1F130, 0x1F169,
                0x1F170, 0x1F19A,
                0x1F1E6, 0x1F202,
                0x1F210, 0x1F23A,
                0x1F240, 0x1F248,
                0x1F250, 0x1F251,
                0x1F300, 0x1F320,
                0x1F330, 0x1F335,
                0x1F337, 0x1F37C,
                0x1F380, 0x1F393,
                0x1F3A0, 0x1F3C4,
                0x1F3C6, 0x1F3CA,
                0x1F3E0, 0x1F3F0,
                0x1F400, 0x1F43E,
                0x1F440, 0x1F440,
                0x1F442, 0x1F4F7,
                0x1F4F9, 0x1F4FC,
                0x1F500, 0x1F53D,
                0x1F550, 0x1F567,
                0x1F5FB, 0x1F5FF,
                0x1F601, 0x1F610,
                0x1F612, 0x1F614,
                0x1F616, 0x1F616,
                0x1F618, 0x1F618,
                0x1F61A, 0x1F61A,
                0x1F61C, 0x1F61E,
                0x1F620, 0x1F625,
                0x1F628, 0x1F62B,
                0x1F62D, 0x1F62D,
                0x1F630, 0x1F633,
                0x1F635, 0x1F640,
                0x1F645, 0x1F64F,
                0x1F680, 0x1F6C5,
                0x1F700, 0x1F773
        }));


        CATEGORIES.put("Zl", pred(new int[]{
                0x2028, 0x2028
        }));


        CATEGORIES.put("Zp", pred(new int[]{
                0x2029, 0x2029
        }));


        CATEGORIES.put("Zs", pred(new int[]{
                0x20, 0x20,
                0xA0, 0xA0,
                0x1680, 0x1680,
                0x180E, 0x180E,
                0x2000, 0x200A,
                0x202F, 0x202F,
                0x205F, 0x205F,
                0x3000, 0x3000
        }));

        String c = "CLMNPSZ";
        for (int i=0; i<c.length(); i++) {
            char ch = c.charAt(i);
            IntPredicate ip = null;
            for (Map.Entry<String, IntPredicate> entry : CATEGORIES.entrySet()) {
                 if (entry.getKey().charAt(0) == ch) {
                     ip = (ip == null ? entry.getValue() : new IntUnionPredicate(ip, entry.getValue()));
                 }
            }
            CATEGORIES.put(ch+"", ip);
        }


    }

        public final static IntPredicate ESCAPE_s =
            new IntSetPredicate(IntArraySet.make(new int[]{9, 10, 13, 32}, 4));

    public final static IntPredicate ESCAPE_S = new IntComplementPredicate(ESCAPE_s);

    public final static IntPredicate ESCAPE_i = new IntPredicate() {
        public boolean matches(int value) {
            return XMLCharacterData.isNCNameStart11(value) || value==':';
        }
    };

    public final static IntPredicate ESCAPE_I = new IntPredicate() {
        public boolean matches(int value) {
            return !(XMLCharacterData.isNCNameStart11(value) || value==':');
        }
    };

    public final static IntPredicate ESCAPE_c = new IntPredicate() {
        public boolean matches(int value) {
            return XMLCharacterData.isNCName11(value) || value==':';
        }
    };

    public final static IntPredicate ESCAPE_C = new IntPredicate() {
        public boolean matches(int value) {
            return !(XMLCharacterData.isNCName11(value) || value==':');
        }
    };

    public final static IntPredicate ESCAPE_d = getCategory("Nd");

    public final static IntPredicate ESCAPE_D = new IntComplementPredicate(ESCAPE_d);

    static IntPredicate CATEGORY_P = getCategory("P");
    static IntPredicate CATEGORY_Z = getCategory("Z");
    static IntPredicate CATEGORY_C = getCategory("C");

    public final static IntPredicate ESCAPE_w = new IntPredicate () {
        public boolean matches(int value) {
            return !(CATEGORY_P.matches(value) || CATEGORY_Z.matches(value) || CATEGORY_C.matches(value));
        }
    };

    public final static IntPredicate ESCAPE_W = new IntComplementPredicate(ESCAPE_w);

    /**
     * Construct a predicate from a set of character ranges
     * @param ranges the ranges of characters
     * @return a predicate that is true for characters present in one of these ranges
     */

    private static IntPredicate pred(int[] ranges) {
        int[] startPoints = new int[ranges.length/2];
        int[] endPoints = new int[ranges.length/2];
        for (int i=0; i<ranges.length; i+=2) {
            startPoints[i/2] = ranges[i];
            endPoints[i/2] = ranges[i+1];
        }
        return new IntSetPredicate(new IntRangeSet(startPoints, endPoints));
    }

    /**
     * Get a predicate to test characters for membership of one of the Unicode
     * character categories
     * @param cat a one-character or two-character category name, for example L or Lu
     * @return a predicate that tests whether a given character belongs to the category
     */

    public static IntPredicate getCategory(String cat) {
        return CATEGORIES.get(cat);
    }


}

// The following stylesheet was used to generate this module, taking data from the XML form of the
// Unicode 6.0.0 database. The ranges not listed as individual characters (Co, Cn) were added by hand.

//<?xml version="1.0" encoding="UTF-8"?>
//<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
//    xmlns:xs="http://www.w3.org/2001/XMLSchema"
//    xmlns:f="http://local-functions/"
//    exclude-result-prefixes="xs f"
//    version="2.0">
//
//    <!-- Output Java representation of character categories -->
//
//    <xsl:output method="text"/>
//
//    <xsl:param name="v6" select="doc('ucd.all.flat.xml')"/>
//
//    <xsl:key name="cat-key" match="char" use="@gc"/>
//
//    <xsl:template name="main">
//
//    public final static HashMap&lt;String, int[]> CATEGORIES = new HashMap&lt;String, int[]>(30);
//
//    static {
//
//        <xsl:variable name="categories" select="distinct-values($v6/ucd/repertoire/char/@gc)"/>
//        <xsl:for-each select="$categories">
//            <xsl:sort select="."/>
//            <xsl:variable name="chars" select="key('cat-key', ., $v6)/@cp"/>
//            <xsl:variable name="codes" select="for $c in $chars return f:hexToInt(0,$c)"/>
//
//    CATEGORIES.put("<xsl:value-of select="."/>", new int[]{
//            <xsl:for-each-group select="$codes" group-adjacent=". - position()">
//                <xsl:if test="position() ne 1">,&#xa;</xsl:if>
//                <xsl:value-of select="'0x', f:intToHex(current-group()[1]), ', 0x', f:intToHex(current-group()[1] + count(current-group()) - 1)" separator=""/>
//            </xsl:for-each-group>
//    });
//        </xsl:for-each>
//    }
//    </xsl:template>
//
//
//    <xsl:function name="f:hexToInt" as="xs:integer">
//      <xsl:param name="acc" as="xs:integer"/>
//      <xsl:param name="in" as="xs:string"/>
//      <xsl:choose>
//        <xsl:when test="$in eq ''">
//          <xsl:sequence select="$acc"/>
//        </xsl:when>
//        <xsl:otherwise>
//          <xsl:variable name="first" select="string-length(substring-before('0123456789ABCDEF', substring($in, 1, 1)))"/>
//          <xsl:sequence select="f:hexToInt($acc * 16 + $first, substring($in, 2))"/>
//        </xsl:otherwise>
//      </xsl:choose>
//    </xsl:function>
//
//    <xsl:function name="f:intToHex" as="xs:string">
//      <xsl:param name="in" as="xs:integer"/>
//      <xsl:choose>
//        <xsl:when test="$in eq 0">
//          <xsl:sequence select="''"/>
//        </xsl:when>
//        <xsl:otherwise>
//          <xsl:variable name="last" select="substring('0123456789ABCDEF', $in mod 16 + 1, 1)"/>
//          <xsl:sequence select="concat(f:intToHex($in idiv 16), $last)"/>
//        </xsl:otherwise>
//      </xsl:choose>
//    </xsl:function>
//
//</xsl:stylesheet>
