/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.filter.testing;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.filter.Matcher;
import net.shibboleth.idp.attribute.filter.context.AttributeFilterContext;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractIdentifiedInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;

/** A simple, mock implementation of {@link Matcher}. */
public class MockMatcher extends AbstractIdentifiedInitializableComponent implements Matcher { 

    /** ID of the attribute to which this matcher applies. */
    @Nullable private String matchingAttribute;

    /** Values, of the attribute, considered to match this matcher. */
    @Nullable private Collection<?> matchingValues;
    
    /** state variable */
    private boolean initialized;
   
    /** do we fail when validate is called? do we fail when we are called?*/
    private boolean fails;

    /** Constructor. */
    public MockMatcher() {
        setId("Mock");
    }

    /**
     * Sets the ID of the attribute to which this matcher applies.
     * 
     * @param id ID of the attribute to which this matcher applies
     */
    public void setMatchingAttribute(@Nonnull final String id) {
        matchingAttribute = Constraint.isNotNull(StringSupport.trimOrNull(id), "attribute ID can not be null or empty");
        if (!initialized) {
            setId("Mock " + id);
        }
    }

    /**
     * Sets the values, of the attribute, considered to match this matcher. If null then all attribute values are
     * considered to be matching.
     * 
     * @param values values, of the attribute, considered to match this matcher
     */
    public void setMatchingValues(@Nullable final Collection<?> values) {
        matchingValues = values;
    }

    /** {@inheritDoc} */
    @Nullable public Set<IdPAttributeValue> getMatchingValues(@Nonnull final IdPAttribute attribute,
            @Nonnull final AttributeFilterContext filterContext) {
        if (fails) {
            return null;
        }
        if (!Objects.equals(attribute.getId(), matchingAttribute)) {
            return CollectionSupport.emptySet();
        }

        if (matchingValues == null) {
            return CollectionSupport.copyToSet(attribute.getValues());
        }

        final LinkedHashSet<IdPAttributeValue> values = new LinkedHashSet<>();
        for (final IdPAttributeValue value : attribute.getValues()) {
            if (matchingValues != null && matchingValues.contains(value)) {
                values.add(value);
            }
        }

        return values;
    }


    /** {@inheritDoc} */
    @Override
    public boolean isInitialized() {
        return initialized;
    }

    /** {@inheritDoc} */
    @Override
    public void doInitialize()  {
        initialized = true;
    }


    /** set whether we are going to fail {{@link #getMatchingValues(IdPAttribute, AttributeFilterContext)}.
     * @param doFail whether we are going to fail.
     */
    public void setFailValidate(final boolean doFail) {
        fails = doFail;
    }
    
}