/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.testing;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.ResolutionException;
import net.shibboleth.idp.attribute.resolver.ResolverAttributeDefinitionDependency;
import net.shibboleth.idp.attribute.resolver.ResolverDataConnectorDependency;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolutionContext;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.ComponentInitializationException;

/** A data connector that just returns a static collection of attributes. */
@ThreadSafe
public class MockDataConnector extends AbstractDataConnector {

    /** Number of times {@link #destroy()} was called. */
    private int destroyCount;

    /** Number of times {@link #initialize()} was called. */
    private int initializeCount;

    /** Static collection of values returned by this connector. */
    @Nullable private final Map<String, IdPAttribute> values;

    /** Exception thrown by resolution. */
    @Nullable private ResolutionException resolutionException;

    /**
     * Constructor.
     * 
     * @param id unique ID for this data connector
     * @param connectorValues static collection of values returned by this connector
     * 
     * @throws ComponentInitializationException ...
     */
    public MockDataConnector(@Nonnull @NotEmpty String id, @Nullable Map<String, IdPAttribute> connectorValues)
                    throws ComponentInitializationException {
        setId(id);
        values = connectorValues;
    }

    /**
     * Constructor.
     *
     * @param id unique ID for this data connector
     * @param connectorValues static collection of values returned by this connector
     * @param attrDepends The AD dependencies
     * @param dcDepends The DC dependencies
     * @throws ComponentInitializationException if initialization failed
     */
    public MockDataConnector(@Nonnull @NotEmpty String id, @Nullable Map<String, IdPAttribute> connectorValues, 
            @Nullable Set<ResolverAttributeDefinitionDependency> attrDepends,
            @Nullable Set<ResolverDataConnectorDependency> dcDepends) throws ComponentInitializationException {
        setAttributeDependencies(attrDepends == null? CollectionSupport.emptySet() : attrDepends);
        setDataConnectorDependencies(dcDepends == null? CollectionSupport.emptySet() : dcDepends);
        setId(id);
        values = connectorValues;
        initialize();
    }

    /**
     * Fail resolutions
     * 
     * @param fail <code>true</code> to return a {@link ResolutionException}, otherwise <code>null</code>
     */
    public void setFailure(final boolean fail) {
        if (fail) {
            resolutionException = new ResolutionException();
        } else {
            resolutionException = null;
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected Map<String,IdPAttribute> doDataConnectorResolve(
            @Nonnull final AttributeResolutionContext resolutionContext,
            @Nonnull final AttributeResolverWorkContext workContext) throws ResolutionException {
        if (resolutionException != null) {
            throw resolutionException;
        }

        return values;
    }

    /** {@inheritDoc} */
    @Override
    public void doDestroy() {
        super.doDestroy();
        destroyCount += 1;
    }


    /** {@inheritDoc} */
    @Override
    public boolean isInitialized() {
        return initializeCount > 0;
    }

    /** {@inheritDoc} */
    @Override
    public void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        initializeCount += 1;
    }

    /**
     * Gets the number of times {@link #destroy()} was called.
     * 
     * @return number of times {@link #destroy()} was called
     */
    public int getDestroyCount() {
        return destroyCount;
    }

    /**
     * Gets the number of times {@link #initialize()} was called.
     * 
     * @return number of times {@link #initialize()} was called
     */
    public int getInitializeCount() {
        return initializeCount;
    }

}