/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.testing;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.resolver.AbstractAttributeDefinition;
import net.shibboleth.idp.attribute.resolver.ResolutionException;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolutionContext;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;

/** An attribute definition that simply returns a static value.   Used for testing only.  This is 
 * a cut and paste job from StaticAttributeDefinition in idp-attribute-resolver-impl */
@ThreadSafe
public class MockStaticAttributeDefinition extends AbstractAttributeDefinition {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(MockStaticAttributeDefinition.class);

    /** Static value returned by this definition. */
    @Nullable private IdPAttribute value;

    /**
     * Set the attribute value we are returning.
     * 
     * @param newAttribute what to set.
     */
    public void setValue(@Nullable final IdPAttribute newAttribute) {
        checkSetterPreconditions();
        
        value = newAttribute;
    }

    /**
     * Return the static attribute we are returning.
     * 
     * @return the attribute.
     */
    @Nullable public IdPAttribute getValue() {
        return value;
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (null == value) {
            throw new ComponentInitializationException("Static Attribute definition " + getId()
                    + " does not have an attribute set up.");
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected IdPAttribute doAttributeDefinitionResolve(
            @Nonnull final AttributeResolutionContext resolutionContext,
            @Nonnull final AttributeResolverWorkContext workContext) throws ResolutionException {
        checkComponentActive();
        log.debug("Attribute definition '{}': Resolving static attribute {}", getId(), value);
        return value;
    }
    
}