/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.testing;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.ResolutionException;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolutionContext;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.primitive.LoggerFactory;

/** An attribute definition that simply returns a static value.   Used for testing only.  This is 
 * a cut and paste job from StaticDataConnector in idp-attribute-resolver-impl */
@ThreadSafe
public class MockStaticDataConnector extends AbstractDataConnector {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(MockStaticDataConnector.class);

    /** Static collection of values returned by this connector. */
    @Nullable private Map<String, IdPAttribute> attributes;

    /**
     * Get the static values returned by this connector.
     * 
     * @return static values returned by this connector
     */
    @Nullable public Map<String, IdPAttribute> getAttributes() {
        return attributes;
    }

    /**
     * Set static values returned by this connector.
     * 
     * @param newValues static values returned by this connector
     */
    public void setValues(@Nullable Collection<IdPAttribute> newValues) {
        checkSetterPreconditions();

        if (null == newValues) {
            attributes = null;
            return;
        } 
        
        final Map<String, IdPAttribute> map = new HashMap<>(newValues.size());
        for (final IdPAttribute attr : newValues) {
            if (null == attr) {
                continue;
            }
            map.put(attr.getId(), attr);
        }
        
        attributes = Map.copyOf(map);
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (null == attributes) {
            throw new ComponentInitializationException("Static Data connector " + getId()
                    + " does not have values set up.");
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected Map<String, IdPAttribute> doDataConnectorResolve(
            @Nonnull final AttributeResolutionContext resolutionContext,
            @Nonnull final AttributeResolverWorkContext workContext) throws ResolutionException {
        checkComponentActive();
        log.debug("Data connector '{}': Resolving static attribute {}", getId(), attributes);
        return attributes;
    }
    
}