/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.testing;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.StringAttributeValue;
import net.shibboleth.idp.attribute.resolver.AttributeDefinition;
import net.shibboleth.idp.attribute.resolver.DataConnector;
import net.shibboleth.idp.attribute.resolver.ResolutionException;
import net.shibboleth.idp.attribute.resolver.ResolverPlugin;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolutionContext;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.ComponentInitializationException;

/** Support class for attribute resolvers testing. */
public class ResolverTestSupport {

    /** eduPersonAfiliation. */
    @Nonnull @NotEmpty public static final String EPA_ATTRIB_ID = "urn:oid:1.3.6.1.4.1.5923.1.1.1.1";

    /** Two affiliations. */
    @Nonnull @NotEmpty public static final String[] EPA1_VALUES = new String[] {"student", "member"};

    /** Another two affiliations. */
    @Nonnull @NotEmpty public static final String[] EPA2_VALUES = new String[] {"staff", "member"};

    /** Three affiliations. */
    @Nonnull @NotEmpty public static final String[] EPA3_VALUES = new String[] {"student", "part-time-student", "library-walk-in"};

    /** eduPersonEntitlement. */
    @Nonnull @NotEmpty public static final String EPE_ATTRIB_ID = "urn:oid:1.3.6.1.4.1.5923.1.1.1.7";

    /** Two entitlement URIs. */
    @Nonnull @NotEmpty public static final String[] EPE1_VALUES = new String[] {"http://example.org/entitlement1",
            "http://example.org/entitlement2"};

    /** An entitlement URN. */
    @Nonnull @NotEmpty public static final String[] EPE2_VALUES = new String[] {"urn:example.org:entitlement1"};

    /** A different entitlement URN. */
    @Nonnull @NotEmpty public static final String[] EPE3_VALUES = new String[] {"urn:example.org:entitlement2"};

    /** work method to build a {@link AttributeResolutionContext}
     * @param plugins what to pupulate 
     * @return the context
     */
    @Nonnull public static AttributeResolutionContext buildResolutionContext(final ResolverPlugin<?>... plugins) {
        final AttributeResolutionContext resolutionContext = new AttributeResolutionContext();
        final AttributeResolverWorkContext workContext =
                resolutionContext.ensureSubcontext(AttributeResolverWorkContext.class);

        assert workContext != null;
        
        MockStaticAttributeDefinition definition;
        MockStaticDataConnector connector;
        try {
            for (ResolverPlugin<?> plugin : plugins) {
                if (plugin instanceof MockStaticAttributeDefinition) {
                    definition = (MockStaticAttributeDefinition) plugin;
                    workContext.recordAttributeDefinitionResolution(definition,
                            definition.resolve(resolutionContext));
                }

                if (plugin instanceof MockStaticDataConnector) {
                    connector = (MockStaticDataConnector) plugin;
                    workContext.recordDataConnectorResolution(connector, connector.resolve(resolutionContext));
                }
            }
        } catch (final ResolutionException e) {
            // this can't happen here
            e.printStackTrace();
        }

        return resolutionContext;
    }

    /** Construct a test {@link IdPAttribute}
     * @param attributeId its name
     * @param values its values (as strings)
     * @return the Attribute
     */
    @Nonnull public static IdPAttribute buildAttribute(@Nonnull @NotEmpty final String attributeId, @Nonnull final String... values) {
        final IdPAttribute attribute = new IdPAttribute(attributeId);
        
        List<IdPAttributeValue> valueList = new ArrayList<>();
        for (final String value : values) {
            valueList.add(StringAttributeValue.valueOf(value));
        }
        attribute.setValues(valueList);

        return attribute;
    }

    /** Construct a test {@link AttributeDefinition}
     * @param attributeId the name
     * @param values the values
     * @return the definition
     */
    @Nullable public static MockStaticAttributeDefinition buildAttributeDefinition(@Nonnull @NotEmpty final String attributeId,
            @Nonnull final String... values) {
        final IdPAttribute attribute = buildAttribute(attributeId, values);

        try {
            final MockStaticAttributeDefinition definition = new MockStaticAttributeDefinition();
            definition.setId(attributeId);
            definition.setValue(attribute);
            definition.initialize();
            return definition;
        } catch (final ComponentInitializationException e) {
            // this can't happen here
            e.printStackTrace();
            return null;
        }
    }

    /** Construct a test {@link DataConnector}
     * @param connectorId the name
     * @param attributes what to return
     * @return the definition
     */
    @Nullable public static MockStaticDataConnector buildDataConnector(@Nonnull @NotEmpty final String connectorId,
            @Nonnull final IdPAttribute... attributes) {

        try {
            final MockStaticDataConnector connector = new MockStaticDataConnector();
            connector.setId(connectorId);
            connector.setValues(Arrays.asList(attributes));
            connector.initialize();

            return connector;
        } catch (final ComponentInitializationException e) {
            // this can't happen here
            e.printStackTrace();
            return null;
        }
    }
    
}