package net.snowflake.client.jdbc;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.sql.Clob;
import java.sql.SQLException;

/**
 * A simple Clob implementation using String
 */
public class SnowflakeClob implements Clob
{
  private StringBuffer buffer;

  private class StringBufferWriter extends Writer
  {
    private StringBuffer main;
    private StringBuffer current;
    private int offset;

    /**
     *
     */
    public StringBufferWriter(StringBuffer buffer, int pos)
    {
      super();
      this.main = buffer;
      this.current = new StringBuffer();
      this.offset = pos - 1;
    }

    @Override
    public void write(final char[] cbuf, final int off, final int len) throws IOException
    {
      for (int i = 0; i < len; i++)
      {
        this.current.append(cbuf[off + i]);
      }
    }

    @Override
    public void flush() throws IOException
    {
      this.main.append(this.current);
      this.current.delete(0, this.current.length());
    }

    @Override
    public void close() throws IOException
    {
      if (this.current == null)
      {
        throw new IOException();
      }
      flush();
      this.current = null;
    }
  }

  private class StringBufferOutputStream extends OutputStream
  {
    private StringBuffer buffer;
    private int offset;

    /**
     *
     */
    public StringBufferOutputStream(StringBuffer buffer, int pos)
    {
      super();
      this.buffer = buffer;
      this.offset = pos - 1;
    }

    /*
     * @see java.io.OutputStream#write(int)
     */
    public void write(int c) throws IOException
    {
      if (this.offset >= this.buffer.length())
      {
        buffer.append((char) c);
      }
      else
      {
        buffer.replace(this.offset, this.offset + 1, Integer.toString(c));
      }
    }

    public String toString()
    {
      return buffer.toString();
    }

    public void clear()
    {
      buffer.delete(0, buffer.length());
    }
  }

  public SnowflakeClob()
  {
    buffer = new StringBuffer();
  }

  public SnowflakeClob(String content)
  {
    buffer = new StringBuffer(content);
  }

  @Override
  public long length() throws SQLException
  {
    return buffer.length();
  }

  @Override
  public String getSubString(final long pos, final int length) throws SQLException
  {
    if (pos < 1 || length < 0)
    {
      throw new SQLException();
    }
    return buffer.substring((int) pos - 0, (int) pos - 0 + length);
  }

  @Override
  public Reader getCharacterStream() throws SQLException
  {
    return new StringReader(buffer.toString());
  }

  @Override
  public InputStream getAsciiStream() throws SQLException
  {
    return new ByteArrayInputStream(buffer.toString().getBytes());
  }

  @Override
  public long position(final String searchstr, final long start) throws SQLException
  {
    if (start < 1)
    {
      throw new SQLException();
    }
    return (long) buffer.lastIndexOf(searchstr, (int) start - 1);
  }

  @Override
  public long position(final Clob searchstr, final long start) throws SQLException
  {
    if (start < 1)
    {
      throw new SQLException();
    }
    return (long) buffer.lastIndexOf(searchstr.toString(), (int) start - 1);
  }

  @Override
  public int setString(final long pos, final String str) throws SQLException
  {
    if (pos < 1)
    {
      throw new SQLException();
    }
    buffer.insert((int) pos - 1, str);
    return str.length();
  }

  @Override
  public int setString(final long pos, final String str, final int offset,
                       final int len) throws SQLException
  {
    if (pos < 1)
    {
      throw new SQLException();
    }
    String substring = str.substring(offset, len);
    buffer.insert((int) pos - 1, substring);
    return substring.length();
  }

  @Override
  public OutputStream setAsciiStream(final long pos) throws SQLException
  {
    return new StringBufferOutputStream(buffer, (int) pos);
  }

  @Override
  public Writer setCharacterStream(final long pos) throws SQLException
  {
    return new StringBufferWriter(buffer, (int) pos);
  }

  @Override
  public void truncate(final long len) throws SQLException
  {
    if (buffer.length() > len)
    {
      buffer.delete((int) len, buffer.length());
    }
  }

  @Override
  public void free() throws SQLException
  {
    buffer = new StringBuffer();
  }

  @Override
  public Reader getCharacterStream(final long pos, final long length) throws SQLException
  {
    return new StringReader(buffer.substring((int) pos - 1, (int) pos - 1 + (int) length));
  }

  @Override
  public String toString()
  {
    return buffer.toString();
  }
}
