/*
 * This file is part of the OWL API.
 *
 * The contents of this file are subject to the LGPL License, Version 3.0.
 *
 * Copyright (C) 2011, The University of Manchester
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 *
 * Alternatively, the contents of this file may be used under the terms of the Apache License, Version 2.0
 * in which case, the provisions of the Apache License Version 2.0 are applicable instead of those above.
 *
 * Copyright 2011, University of Manchester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.semanticweb.owlapi.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;

import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLRuntimeException;

/** An ontology document source which reads an ontology from a reader.
 * 
 * @author Matthew Horridge, The University Of Manchester, Bio-Health
 *         Informatics Group, Date: 17-Nov-2007 */
public class ReaderDocumentSource implements OWLOntologyDocumentSource {
    private static int counter = 0;
    private final IRI documentIRI;
    private String buffer;

    /** Constructs and ontology input source which will read an ontology from a
     * reader.
     * 
     * @param reader
     *            The reader that will be used to read an ontology. */
    public ReaderDocumentSource(Reader reader) {
        this(reader, getNextDocumentIRI());
    }

    /** @return a fresh IRI */
    public static synchronized IRI getNextDocumentIRI() {
        counter = counter + 1;
        return IRI.create("reader:ontology" + counter);
    }

    /** Constructs and ontology input source which will read an ontology from a
     * reader.
     * 
     * @param reader
     *            The reader that will be used to read an ontology.
     * @param documentIRI
     *            The ontology document IRI which will be used as the base of
     *            the document if needed. */
    public ReaderDocumentSource(Reader reader, IRI documentIRI) {
        this.documentIRI = documentIRI;
        fillBuffer(reader);
    }

    private void fillBuffer(Reader reader) {
        try {
            StringBuilder builder = new StringBuilder();
            final int length = 100000;
            char[] tempBuffer = new char[length];
            int read = 0;
            do {
                read = reader.read(tempBuffer, 0, length);
                if (read > 0) {
                    builder.append(tempBuffer, 0, read);
                }
            } while (read > 0);
            buffer = builder.toString();
        } catch (IOException e) {
            throw new OWLRuntimeException(e);
        }
    }

    @Override
    public IRI getDocumentIRI() {
        return documentIRI;
    }

    @Override
    public Reader getReader() {
        return new StringReader(buffer);
    }

    @Override
    public boolean isReaderAvailable() {
        return true;
    }

    @Override
    public boolean isInputStreamAvailable() {
        return false;
    }

    @Override
    public InputStream getInputStream() {
        throw new OWLRuntimeException(
                "InputStream not available.  Check with ReaderDocumentSource.isReaderAvailable() first!");
    }
}
