/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.utils.collections;

import io.netty.util.collection.LongObjectHashMap;
import java.lang.reflect.Array;
import java.util.Comparator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.function.ToLongFunction;
import org.apache.activemq.artemis.utils.collections.LinkedList;
import org.apache.activemq.artemis.utils.collections.LinkedListIterator;

public class LinkedListImpl<E>
implements LinkedList<E> {
    private static final int INITIAL_ITERATOR_ARRAY_SIZE = 10;
    private final Node<E> head = new NodeHolder<E>(null);
    private final Comparator<E> comparator;
    LongObjectHashMap<Node<E>> nodeMap;
    private Node<E> tail = null;
    private int size;
    private volatile Iterator[] iters = this.createIteratorArray(10);
    private int numIters;
    private int nextIndex;
    private ToLongFunction<E> idSupplier;

    public LinkedListImpl() {
        this(null, null);
    }

    public LinkedListImpl(Comparator<E> comparator) {
        this(comparator, null);
    }

    public LinkedListImpl(Comparator<E> comparator, ToLongFunction<E> supplier) {
        this.comparator = comparator;
        this.idSupplier = supplier;
        this.nodeMap = this.idSupplier != null ? this.newLongHashMap() : null;
    }

    @Override
    public void clearID() {
        this.idSupplier = null;
        if (this.nodeMap != null) {
            this.nodeMap.clear();
            this.nodeMap = null;
        }
    }

    @Override
    public void setIDSupplier(ToLongFunction<E> supplier) {
        this.idSupplier = supplier;
        this.nodeMap = this.newLongHashMap();
        try (Iterator iterator = (Iterator)this.iterator();){
            while (iterator.hasNext()) {
                Object value = iterator.next();
                Node position = iterator.last;
                this.putID(value, position);
            }
        }
    }

    private LongObjectHashMap<Node<E>> newLongHashMap() {
        return new LongObjectHashMap<Node<E>>(Math.max(8, this.size));
    }

    private void putID(E value, Node<E> position) {
        long id = this.idSupplier.applyAsLong(value);
        if (id >= 0L) {
            this.nodeMap.put(id, position);
        }
    }

    @Override
    public void addHead(E e) {
        Node node = Node.with(e);
        node.next = ((Node)this.head).next;
        node.prev = (Node)this.head;
        ((Node)this.head).next = node;
        if (this.size == 0) {
            this.tail = node;
        } else {
            node.next.prev = node;
        }
        this.itemAdded(node, e);
        ++this.size;
    }

    @Override
    public E removeWithID(long id) {
        if (this.nodeMap == null) {
            return null;
        }
        Node<E> node = this.nodeMap.get(id);
        if (node == null) {
            return null;
        }
        this.removeAfter(((Node)node).prev);
        return node.val();
    }

    private void itemAdded(Node<E> node, E item) {
        if (this.nodeMap != null) {
            this.putID(item, node);
        }
    }

    private void itemRemoved(Node<E> node) {
        long id;
        if (this.nodeMap != null && (id = this.idSupplier.applyAsLong(node.val())) >= 0L) {
            this.nodeMap.remove(id);
        }
    }

    @Override
    public void addTail(E e) {
        if (this.size == 0) {
            this.addHead(e);
        } else {
            Node node = Node.with(e);
            node.prev = (Node)this.tail;
            ((Node)this.tail).next = node;
            this.tail = node;
            this.itemAdded(node, e);
            ++this.size;
        }
    }

    public void addSorted(E e) {
        if (this.comparator == null) {
            throw new NullPointerException("comparator=null");
        }
        if (this.size != 0) {
            if (this.comparator.compare(((Node)this.head).next.val(), e) < 0) {
                this.addHead(e);
                return;
            }
            if (this.comparator.compare(this.tail.val(), e) >= 0) {
                this.addTail(e);
                return;
            }
            Node fetching = ((Node)this.head).next;
            while (fetching.next != null) {
                int compareNext = this.comparator.compare(fetching.next.val(), e);
                if (compareNext <= 0) {
                    this.addAfter(fetching, e);
                    return;
                }
                fetching = fetching.next;
            }
            throw new IllegalStateException("Cannot find a suitable place for your element, There's a mismatch in the comparator or there was concurrent adccess on the queue");
        }
        this.addHead(e);
    }

    private void addAfter(Node<E> node, E e) {
        Node newNode = Node.with(e);
        Node nextNode = ((Node)node).next;
        ((Node)node).next = newNode;
        newNode.prev = (Node)node;
        newNode.next = nextNode;
        nextNode.prev = newNode;
        this.itemAdded(node, e);
        ++this.size;
    }

    @Override
    public E poll() {
        Node ret = ((Node)this.head).next;
        if (ret != null) {
            this.removeAfter(this.head);
            return (E)ret.val();
        }
        return null;
    }

    @Override
    public void clear() {
        while (this.poll() != null) {
        }
    }

    @Override
    public int size() {
        return this.size;
    }

    public int getSizeOfSuppliedIDs() {
        return this.nodeMap == null ? 0 : this.nodeMap.size();
    }

    @Override
    public LinkedListIterator<E> iterator() {
        return new Iterator();
    }

    public String toString() {
        StringBuilder str = new StringBuilder("LinkedListImpl [ ");
        Node node = this.head;
        while (node != null) {
            str.append(node.toString());
            if (node.next != null) {
                str.append(", ");
            }
            node = node.next;
        }
        return str.toString();
    }

    public int numIters() {
        return this.numIters;
    }

    private Iterator[] createIteratorArray(int size) {
        return (Iterator[])Array.newInstance(Iterator.class, size);
    }

    private void removeAfter(Node<E> node) {
        Node toRemove = ((Node)node).next;
        ((Node)node).next = toRemove.next;
        if (toRemove.next != null) {
            toRemove.next.prev = (Node)node;
        }
        this.itemRemoved(toRemove);
        if (toRemove == this.tail) {
            this.tail = node;
        }
        --this.size;
        if (toRemove.iterCount != 0) {
            this.nudgeIterators(toRemove);
        }
        toRemove.next = (toRemove.prev = null);
    }

    private synchronized void nudgeIterators(Node<E> node) {
        for (int i = 0; i < this.numIters; ++i) {
            Iterator iter = this.iters[i];
            if (iter == null) continue;
            iter.nudged(node);
        }
    }

    private synchronized void addIter(Iterator iter) {
        if (this.numIters == this.iters.length) {
            this.resize(2 * this.numIters);
        }
        this.iters[this.nextIndex++] = iter;
        ++this.numIters;
    }

    private synchronized void resize(int newSize) {
        Iterator[] newIters = this.createIteratorArray(newSize);
        System.arraycopy(this.iters, 0, newIters, 0, this.numIters);
        this.iters = newIters;
    }

    private synchronized void removeIter(Iterator iter) {
        for (int i = 0; i < this.numIters; ++i) {
            if (iter != this.iters[i]) continue;
            this.iters[i] = null;
            if (i != this.numIters - 1) {
                System.arraycopy(this.iters, i + 1, this.iters, i, this.numIters - i - 1);
            }
            --this.numIters;
            if (this.numIters >= 10 && this.numIters == this.iters.length / 2) {
                this.resize(this.numIters);
            }
            --this.nextIndex;
            return;
        }
        throw new IllegalStateException("Cannot find iter to remove");
    }

    private class Iterator
    implements LinkedListIterator<E> {
        Node<E> last;
        Node<E> current;
        boolean repeat;

        Iterator() {
            this.current = LinkedListImpl.this.head.next;
            if (this.current != null) {
                this.current.iterCount++;
            }
            LinkedListImpl.this.addIter(this);
        }

        @Override
        public void repeat() {
            this.repeat = true;
        }

        @Override
        public boolean hasNext() {
            Node e = this.getNode();
            if (e != null && (e != this.last || this.repeat)) {
                return true;
            }
            return this.canAdvance();
        }

        @Override
        public E next() {
            Node e = this.getNode();
            if (this.repeat) {
                this.repeat = false;
                if (e != null) {
                    return e.val();
                }
                if (this.canAdvance()) {
                    this.advance();
                    e = this.getNode();
                    return e.val();
                }
                throw new NoSuchElementException();
            }
            if (e == null || e == this.last) {
                if (this.canAdvance()) {
                    this.advance();
                    e = this.getNode();
                } else {
                    throw new NoSuchElementException();
                }
            }
            this.last = e;
            this.repeat = false;
            return e.val();
        }

        @Override
        public void remove() {
            if (this.last == null) {
                throw new NoSuchElementException();
            }
            if (this.current == null) {
                throw new NoSuchElementException();
            }
            LinkedListImpl.this.removeAfter(this.current.prev);
            this.last = null;
        }

        @Override
        public void close() {
            LinkedListImpl.this.removeIter(this);
        }

        public void nudged(Node<E> node) {
            if (this.current == node) {
                if (this.canAdvance()) {
                    this.advance();
                } else if (this.current.prev != LinkedListImpl.this.head) {
                    this.current.iterCount--;
                    this.current = this.current.prev;
                    this.current.iterCount++;
                } else {
                    this.current = null;
                }
            }
        }

        private Node<E> getNode() {
            if (this.current == null) {
                this.current = LinkedListImpl.this.head.next;
                if (this.current != null) {
                    this.current.iterCount++;
                }
            }
            if (this.current != null) {
                return this.current;
            }
            return null;
        }

        private boolean canAdvance() {
            if (this.current == null) {
                this.current = LinkedListImpl.this.head.next;
                if (this.current != null) {
                    this.current.iterCount++;
                }
            }
            return this.current != null && this.current.next != null;
        }

        private void advance() {
            if (this.current == null || this.current.next == null) {
                throw new NoSuchElementException();
            }
            this.current.iterCount--;
            this.current = this.current.next;
            this.current.iterCount++;
        }
    }

    public static class Node<T> {
        private Node<T> next;
        private Node<T> prev;
        private int iterCount;

        private static <T> Node<T> with(T o) {
            Objects.requireNonNull(o, "Only HEAD nodes are allowed to hold null values");
            if (o instanceof Node) {
                Node node = (Node)o;
                if (node.prev == null && node.next == null) {
                    node.iterCount = 0;
                    return node;
                }
            }
            return new NodeHolder(o);
        }

        protected T val() {
            return (T)this;
        }

        protected final Node<T> next() {
            return this.next;
        }

        protected final Node<T> prev() {
            return this.prev;
        }

        public String toString() {
            return this.val() == this ? "Intrusive Node" : "Node, value = " + this.val();
        }
    }

    private static final class NodeHolder<T>
    extends Node<T> {
        private final T val;

        private NodeHolder(T e) {
            this.val = e;
        }

        @Override
        protected T val() {
            return this.val;
        }
    }
}

