/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.hadoop2.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.flink.hadoop2.shaded.com.google.common.base.Preconditions;
import org.apache.flink.hadoop2.shaded.com.google.common.collect.Lists;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.protocol.CacheDirectiveInfo;
import org.apache.hadoop.hdfs.protocol.CachePoolInfo;
import org.apache.hadoop.hdfs.security.token.delegation.DelegationTokenIdentifier;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfo;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.StorageInfo;
import org.apache.hadoop.hdfs.server.namenode.AclFeature;
import org.apache.hadoop.hdfs.server.namenode.AclStorage;
import org.apache.hadoop.hdfs.server.namenode.BackupJournalManager;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOp;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogOpCodes;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeSymlink;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.JournalSet;
import org.apache.hadoop.hdfs.server.namenode.LogsPurgeable;
import org.apache.hadoop.hdfs.server.namenode.MetaRecoveryContext;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeLayoutVersion;
import org.apache.hadoop.hdfs.server.namenode.metrics.NameNodeMetrics;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLogManifest;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.security.token.delegation.DelegationKey;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class FSEditLog
implements LogsPurgeable {
    static final Log LOG = LogFactory.getLog(FSEditLog.class);
    private State state = State.UNINITIALIZED;
    private JournalSet journalSet = null;
    private EditLogOutputStream editLogStream = null;
    private long txid = 0L;
    private long synctxid = 0L;
    private long curSegmentTxId = -12345L;
    private long lastPrintTime;
    private volatile boolean isSyncRunning = false;
    private volatile boolean isAutoSyncScheduled = false;
    private long numTransactions;
    private long numTransactionsBatchedInSync;
    private long totalTimeTransactions;
    private NameNodeMetrics metrics;
    private final NNStorage storage;
    private final Configuration conf;
    private final List<URI> editsDirs;
    private final ThreadLocal<FSEditLogOp.OpInstanceCache> cache = new ThreadLocal<FSEditLogOp.OpInstanceCache>(){

        @Override
        protected FSEditLogOp.OpInstanceCache initialValue() {
            return new FSEditLogOp.OpInstanceCache();
        }
    };
    private final List<URI> sharedEditsDirs;
    private static final ThreadLocal<TransactionId> myTransactionId = new ThreadLocal<TransactionId>(){

        @Override
        protected synchronized TransactionId initialValue() {
            return new TransactionId(Long.MAX_VALUE);
        }
    };

    FSEditLog(Configuration conf, NNStorage storage, List<URI> editsDirs) {
        this.conf = conf;
        this.storage = storage;
        this.metrics = NameNode.getNameNodeMetrics();
        this.lastPrintTime = Time.now();
        this.editsDirs = Lists.newArrayList(editsDirs);
        this.sharedEditsDirs = FSNamesystem.getSharedEditsDirs(conf);
    }

    public synchronized void initJournalsForWrite() {
        Preconditions.checkState(this.state == State.UNINITIALIZED || this.state == State.CLOSED, "Unexpected state: %s", new Object[]{this.state});
        this.initJournals(this.editsDirs);
        this.state = State.BETWEEN_LOG_SEGMENTS;
    }

    public synchronized void initSharedJournalsForRead() {
        if (this.state == State.OPEN_FOR_READING) {
            LOG.warn((Object)"Initializing shared journals for READ, already open for READ", (Throwable)new Exception());
            return;
        }
        Preconditions.checkState(this.state == State.UNINITIALIZED || this.state == State.CLOSED);
        this.initJournals(this.sharedEditsDirs);
        this.state = State.OPEN_FOR_READING;
    }

    private synchronized void initJournals(List<URI> dirs) {
        int minimumRedundantJournals = this.conf.getInt("dfs.namenode.edits.dir.minimum", 1);
        this.journalSet = new JournalSet(minimumRedundantJournals);
        for (URI u : dirs) {
            boolean required = FSNamesystem.getRequiredNamespaceEditsDirs(this.conf).contains(u);
            if (u.getScheme().equals("file")) {
                Storage.StorageDirectory sd = this.storage.getStorageDirectory(u);
                if (sd == null) continue;
                this.journalSet.add(new FileJournalManager(this.conf, sd, this.storage), required, this.sharedEditsDirs.contains(u));
                continue;
            }
            this.journalSet.add(this.createJournal(u), required, this.sharedEditsDirs.contains(u));
        }
        if (this.journalSet.isEmpty()) {
            LOG.error((Object)"No edits directories configured!");
        }
    }

    Collection<URI> getEditURIs() {
        return this.editsDirs;
    }

    synchronized void openForWrite() throws IOException {
        Preconditions.checkState(this.state == State.BETWEEN_LOG_SEGMENTS, "Bad state: %s", new Object[]{this.state});
        long segmentTxId = this.getLastWrittenTxId() + 1L;
        ArrayList<EditLogInputStream> streams = new ArrayList<EditLogInputStream>();
        this.journalSet.selectInputStreams(streams, segmentTxId, true);
        if (!streams.isEmpty()) {
            String error = String.format("Cannot start writing at txid %s when there is a stream available for read: %s", segmentTxId, streams.get(0));
            IOUtils.cleanup(LOG, streams.toArray(new EditLogInputStream[0]));
            throw new IllegalStateException(error);
        }
        this.startLogSegment(segmentTxId, true);
        assert (this.state == State.IN_SEGMENT) : "Bad state: " + (Object)((Object)this.state);
    }

    synchronized boolean isOpenForWrite() {
        return this.state == State.IN_SEGMENT || this.state == State.BETWEEN_LOG_SEGMENTS;
    }

    synchronized boolean isSegmentOpen() {
        return this.state == State.IN_SEGMENT;
    }

    public synchronized boolean isOpenForRead() {
        return this.state == State.OPEN_FOR_READING;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void close() {
        if (this.state == State.CLOSED) {
            LOG.debug((Object)"Closing log when already closed");
            return;
        }
        try {
            if (this.state == State.IN_SEGMENT) {
                assert (this.editLogStream != null);
                this.waitForSyncToFinish();
                this.endCurrentLogSegment(true);
            }
        }
        finally {
            if (this.journalSet != null && !this.journalSet.isEmpty()) {
                try {
                    this.journalSet.close();
                }
                catch (IOException ioe) {
                    LOG.warn((Object)"Error closing journalSet", (Throwable)ioe);
                }
            }
            this.state = State.CLOSED;
        }
    }

    synchronized void formatNonFileJournals(NamespaceInfo nsInfo) throws IOException {
        Preconditions.checkState(this.state == State.BETWEEN_LOG_SEGMENTS, "Bad state: %s", new Object[]{this.state});
        for (JournalManager jm : this.journalSet.getJournalManagers()) {
            if (jm instanceof FileJournalManager) continue;
            jm.format(nsInfo);
        }
    }

    synchronized List<Storage.FormatConfirmable> getFormatConfirmables() {
        Preconditions.checkState(this.state == State.BETWEEN_LOG_SEGMENTS, "Bad state: %s", new Object[]{this.state});
        ArrayList<Storage.FormatConfirmable> ret = Lists.newArrayList();
        for (JournalManager jm : this.journalSet.getJournalManagers()) {
            if (jm instanceof FileJournalManager) continue;
            ret.add(jm);
        }
        return ret;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void logEdit(FSEditLogOp op) {
        FSEditLog fSEditLog = this;
        synchronized (fSEditLog) {
            assert (this.isOpenForWrite()) : "bad state: " + (Object)((Object)this.state);
            this.waitIfAutoSyncScheduled();
            long start = this.beginTransaction();
            op.setTransactionId(this.txid);
            try {
                this.editLogStream.write(op);
            }
            catch (IOException ex) {
                // empty catch block
            }
            this.endTransaction(start);
            if (!this.shouldForceSync()) {
                return;
            }
            this.isAutoSyncScheduled = true;
        }
        this.logSync();
    }

    synchronized void waitIfAutoSyncScheduled() {
        try {
            while (this.isAutoSyncScheduled) {
                this.wait(1000L);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    synchronized void doneWithAutoSyncScheduling() {
        if (this.isAutoSyncScheduled) {
            this.isAutoSyncScheduled = false;
            this.notifyAll();
        }
    }

    private boolean shouldForceSync() {
        return this.editLogStream.shouldForceSync();
    }

    private long beginTransaction() {
        assert (Thread.holdsLock(this));
        ++this.txid;
        TransactionId id = myTransactionId.get();
        id.txid = this.txid;
        return Time.now();
    }

    private void endTransaction(long start) {
        assert (Thread.holdsLock(this));
        long end = Time.now();
        ++this.numTransactions;
        this.totalTimeTransactions += end - start;
        if (this.metrics != null) {
            this.metrics.addTransaction(end - start);
        }
    }

    public synchronized long getLastWrittenTxId() {
        return this.txid;
    }

    synchronized long getCurSegmentTxId() {
        Preconditions.checkState(this.isSegmentOpen(), "Bad state: %s", new Object[]{this.state});
        return this.curSegmentTxId;
    }

    synchronized void setNextTxId(long nextTxId) {
        Preconditions.checkArgument(this.synctxid <= this.txid && nextTxId >= this.txid, "May not decrease txid. synctxid=%s txid=%s nextTxId=%s", this.synctxid, this.txid, nextTxId);
        this.txid = nextTxId - 1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void logSyncAll() {
        FSEditLog fSEditLog = this;
        synchronized (fSEditLog) {
            TransactionId id = myTransactionId.get();
            id.txid = this.txid;
        }
        this.logSync();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void logSync() {
        long syncStart = 0L;
        long mytxid = FSEditLog.myTransactionId.get().txid;
        boolean sync = false;
        try {
            EditLogOutputStream logStream = null;
            FSEditLog fSEditLog = this;
            synchronized (fSEditLog) {
                try {
                    this.printStatistics(false);
                    while (mytxid > this.synctxid && this.isSyncRunning) {
                        try {
                            this.wait(1000L);
                        }
                        catch (InterruptedException ie) {}
                    }
                    if (mytxid <= this.synctxid) {
                        ++this.numTransactionsBatchedInSync;
                        if (this.metrics == null) return;
                        this.metrics.incrTransactionsBatchedInSync();
                        return;
                    }
                    syncStart = this.txid;
                    this.isSyncRunning = true;
                    sync = true;
                    try {
                        if (this.journalSet.isEmpty()) {
                            throw new IOException("No journals available to flush");
                        }
                        this.editLogStream.setReadyToFlush();
                    }
                    catch (IOException e) {
                        String msg = "Could not sync enough journals to persistent storage due to " + e.getMessage() + ". " + "Unsynced transactions: " + (this.txid - this.synctxid);
                        LOG.fatal((Object)msg, (Throwable)new Exception());
                        IOUtils.cleanup(LOG, this.journalSet);
                        ExitUtil.terminate(1, msg);
                    }
                }
                finally {
                    this.doneWithAutoSyncScheduling();
                }
                logStream = this.editLogStream;
            }
            long start = Time.now();
            try {
                if (logStream != null) {
                    logStream.flush();
                }
            }
            catch (IOException ex) {
                FSEditLog throwable = this;
                synchronized (throwable) {
                    String msg = "Could not sync enough journals to persistent storage. Unsynced transactions: " + (this.txid - this.synctxid);
                    LOG.fatal((Object)msg, (Throwable)new Exception());
                    IOUtils.cleanup(LOG, this.journalSet);
                    ExitUtil.terminate(1, msg);
                }
            }
            long elapsed = Time.now() - start;
            if (this.metrics == null) return;
            this.metrics.addSync(elapsed);
            return;
        }
        finally {
            FSEditLog ie = this;
            synchronized (ie) {
                if (sync) {
                    this.synctxid = syncStart;
                    this.isSyncRunning = false;
                }
                this.notifyAll();
            }
        }
    }

    private void printStatistics(boolean force) {
        long now = Time.now();
        if (this.lastPrintTime + 60000L > now && !force) {
            return;
        }
        this.lastPrintTime = now;
        StringBuilder buf = new StringBuilder();
        buf.append("Number of transactions: ");
        buf.append(this.numTransactions);
        buf.append(" Total time for transactions(ms): ");
        buf.append(this.totalTimeTransactions);
        buf.append(" Number of transactions batched in Syncs: ");
        buf.append(this.numTransactionsBatchedInSync);
        buf.append(" Number of syncs: ");
        buf.append(this.editLogStream.getNumSync());
        buf.append(" SyncTimes(ms): ");
        buf.append(this.journalSet.getSyncTimes());
        LOG.info((Object)buf);
    }

    private void logRpcIds(FSEditLogOp op, boolean toLogRpcIds) {
        if (toLogRpcIds) {
            op.setRpcClientId(Server.getClientId());
            op.setRpcCallId(Server.getCallId());
        }
    }

    public void logOpenFile(String path, INodeFile newNode, boolean toLogRpcIds) {
        Preconditions.checkArgument(newNode.isUnderConstruction());
        PermissionStatus permissions = newNode.getPermissionStatus();
        FSEditLogOp.AddOp op = (FSEditLogOp.AddOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)FSEditLogOp.AddOp.getInstance(this.cache.get()).setInodeId(newNode.getId())).setPath(path)).setReplication(newNode.getFileReplication())).setModificationTime(newNode.getModificationTime())).setAccessTime(newNode.getAccessTime())).setBlockSize(newNode.getPreferredBlockSize())).setBlocks(newNode.getBlocks())).setPermissionStatus(permissions)).setClientName(newNode.getFileUnderConstructionFeature().getClientName())).setClientMachine(newNode.getFileUnderConstructionFeature().getClientMachine());
        AclFeature f = newNode.getAclFeature();
        if (f != null) {
            op.setAclEntries(AclStorage.readINodeLogicalAcl(newNode));
        }
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    public void logCloseFile(String path, INodeFile newNode) {
        FSEditLogOp.CloseOp op = (FSEditLogOp.CloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)((FSEditLogOp.AddCloseOp)FSEditLogOp.CloseOp.getInstance(this.cache.get()).setPath(path)).setReplication(newNode.getFileReplication())).setModificationTime(newNode.getModificationTime())).setAccessTime(newNode.getAccessTime())).setBlockSize(newNode.getPreferredBlockSize())).setBlocks(newNode.getBlocks())).setPermissionStatus(newNode.getPermissionStatus());
        this.logEdit(op);
    }

    public void logAddBlock(String path, INodeFile file) {
        Preconditions.checkArgument(file.isUnderConstruction());
        BlockInfo[] blocks = file.getBlocks();
        Preconditions.checkState(blocks != null && blocks.length > 0);
        BlockInfo pBlock = blocks.length > 1 ? blocks[blocks.length - 2] : null;
        BlockInfo lastBlock = blocks[blocks.length - 1];
        FSEditLogOp.AddBlockOp op = FSEditLogOp.AddBlockOp.getInstance(this.cache.get()).setPath(path).setPenultimateBlock(pBlock).setLastBlock(lastBlock);
        this.logEdit(op);
    }

    public void logUpdateBlocks(String path, INodeFile file, boolean toLogRpcIds) {
        Preconditions.checkArgument(file.isUnderConstruction());
        FSEditLogOp.UpdateBlocksOp op = FSEditLogOp.UpdateBlocksOp.getInstance(this.cache.get()).setPath(path).setBlocks(file.getBlocks());
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    public void logMkDir(String path, INode newNode) {
        PermissionStatus permissions = newNode.getPermissionStatus();
        FSEditLogOp.MkdirOp op = FSEditLogOp.MkdirOp.getInstance(this.cache.get()).setInodeId(newNode.getId()).setPath(path).setTimestamp(newNode.getModificationTime()).setPermissionStatus(permissions);
        AclFeature f = newNode.getAclFeature();
        if (f != null) {
            op.setAclEntries(AclStorage.readINodeLogicalAcl(newNode));
        }
        this.logEdit(op);
    }

    void logRename(String src, String dst, long timestamp, boolean toLogRpcIds) {
        FSEditLogOp.RenameOldOp op = FSEditLogOp.RenameOldOp.getInstance(this.cache.get()).setSource(src).setDestination(dst).setTimestamp(timestamp);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logRename(String src, String dst, long timestamp, boolean toLogRpcIds, Options.Rename ... options) {
        FSEditLogOp.RenameOp op = FSEditLogOp.RenameOp.getInstance(this.cache.get()).setSource(src).setDestination(dst).setTimestamp(timestamp).setOptions(options);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logSetReplication(String src, short replication) {
        FSEditLogOp.SetReplicationOp op = FSEditLogOp.SetReplicationOp.getInstance(this.cache.get()).setPath(src).setReplication(replication);
        this.logEdit(op);
    }

    void logSetQuota(String src, long nsQuota, long dsQuota) {
        FSEditLogOp.SetQuotaOp op = FSEditLogOp.SetQuotaOp.getInstance(this.cache.get()).setSource(src).setNSQuota(nsQuota).setDSQuota(dsQuota);
        this.logEdit(op);
    }

    void logSetPermissions(String src, FsPermission permissions) {
        FSEditLogOp.SetPermissionsOp op = FSEditLogOp.SetPermissionsOp.getInstance(this.cache.get()).setSource(src).setPermissions(permissions);
        this.logEdit(op);
    }

    void logSetOwner(String src, String username, String groupname) {
        FSEditLogOp.SetOwnerOp op = FSEditLogOp.SetOwnerOp.getInstance(this.cache.get()).setSource(src).setUser(username).setGroup(groupname);
        this.logEdit(op);
    }

    void logConcat(String trg, String[] srcs, long timestamp, boolean toLogRpcIds) {
        FSEditLogOp.ConcatDeleteOp op = FSEditLogOp.ConcatDeleteOp.getInstance(this.cache.get()).setTarget(trg).setSources(srcs).setTimestamp(timestamp);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logDelete(String src, long timestamp, boolean toLogRpcIds) {
        FSEditLogOp.DeleteOp op = FSEditLogOp.DeleteOp.getInstance(this.cache.get()).setPath(src).setTimestamp(timestamp);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logGenerationStampV1(long genstamp) {
        FSEditLogOp.SetGenstampV1Op op = FSEditLogOp.SetGenstampV1Op.getInstance(this.cache.get()).setGenerationStamp(genstamp);
        this.logEdit(op);
    }

    void logGenerationStampV2(long genstamp) {
        FSEditLogOp.SetGenstampV2Op op = FSEditLogOp.SetGenstampV2Op.getInstance(this.cache.get()).setGenerationStamp(genstamp);
        this.logEdit(op);
    }

    void logAllocateBlockId(long blockId) {
        FSEditLogOp.AllocateBlockIdOp op = FSEditLogOp.AllocateBlockIdOp.getInstance(this.cache.get()).setBlockId(blockId);
        this.logEdit(op);
    }

    void logTimes(String src, long mtime, long atime) {
        FSEditLogOp.TimesOp op = FSEditLogOp.TimesOp.getInstance(this.cache.get()).setPath(src).setModificationTime(mtime).setAccessTime(atime);
        this.logEdit(op);
    }

    void logSymlink(String path, String value, long mtime, long atime, INodeSymlink node, boolean toLogRpcIds) {
        FSEditLogOp.SymlinkOp op = FSEditLogOp.SymlinkOp.getInstance(this.cache.get()).setId(node.getId()).setPath(path).setValue(value).setModificationTime(mtime).setAccessTime(atime).setPermissionStatus(node.getPermissionStatus());
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logGetDelegationToken(DelegationTokenIdentifier id, long expiryTime) {
        FSEditLogOp.GetDelegationTokenOp op = FSEditLogOp.GetDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id).setExpiryTime(expiryTime);
        this.logEdit(op);
    }

    void logRenewDelegationToken(DelegationTokenIdentifier id, long expiryTime) {
        FSEditLogOp.RenewDelegationTokenOp op = FSEditLogOp.RenewDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id).setExpiryTime(expiryTime);
        this.logEdit(op);
    }

    void logCancelDelegationToken(DelegationTokenIdentifier id) {
        FSEditLogOp.CancelDelegationTokenOp op = FSEditLogOp.CancelDelegationTokenOp.getInstance(this.cache.get()).setDelegationTokenIdentifier(id);
        this.logEdit(op);
    }

    void logUpdateMasterKey(DelegationKey key) {
        FSEditLogOp.UpdateMasterKeyOp op = FSEditLogOp.UpdateMasterKeyOp.getInstance(this.cache.get()).setDelegationKey(key);
        this.logEdit(op);
    }

    void logReassignLease(String leaseHolder, String src, String newHolder) {
        FSEditLogOp.ReassignLeaseOp op = FSEditLogOp.ReassignLeaseOp.getInstance(this.cache.get()).setLeaseHolder(leaseHolder).setPath(src).setNewHolder(newHolder);
        this.logEdit(op);
    }

    void logCreateSnapshot(String snapRoot, String snapName, boolean toLogRpcIds) {
        FSEditLogOp.CreateSnapshotOp op = FSEditLogOp.CreateSnapshotOp.getInstance(this.cache.get()).setSnapshotRoot(snapRoot).setSnapshotName(snapName);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logDeleteSnapshot(String snapRoot, String snapName, boolean toLogRpcIds) {
        FSEditLogOp.DeleteSnapshotOp op = FSEditLogOp.DeleteSnapshotOp.getInstance(this.cache.get()).setSnapshotRoot(snapRoot).setSnapshotName(snapName);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logRenameSnapshot(String path, String snapOldName, String snapNewName, boolean toLogRpcIds) {
        FSEditLogOp.RenameSnapshotOp op = FSEditLogOp.RenameSnapshotOp.getInstance(this.cache.get()).setSnapshotRoot(path).setSnapshotOldName(snapOldName).setSnapshotNewName(snapNewName);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logAllowSnapshot(String path) {
        FSEditLogOp.AllowSnapshotOp op = FSEditLogOp.AllowSnapshotOp.getInstance(this.cache.get()).setSnapshotRoot(path);
        this.logEdit(op);
    }

    void logDisallowSnapshot(String path) {
        FSEditLogOp.DisallowSnapshotOp op = FSEditLogOp.DisallowSnapshotOp.getInstance(this.cache.get()).setSnapshotRoot(path);
        this.logEdit(op);
    }

    void logAddCacheDirectiveInfo(CacheDirectiveInfo directive, boolean toLogRpcIds) {
        FSEditLogOp.AddCacheDirectiveInfoOp op = FSEditLogOp.AddCacheDirectiveInfoOp.getInstance(this.cache.get()).setDirective(directive);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logModifyCacheDirectiveInfo(CacheDirectiveInfo directive, boolean toLogRpcIds) {
        FSEditLogOp.ModifyCacheDirectiveInfoOp op = FSEditLogOp.ModifyCacheDirectiveInfoOp.getInstance(this.cache.get()).setDirective(directive);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logRemoveCacheDirectiveInfo(Long id, boolean toLogRpcIds) {
        FSEditLogOp.RemoveCacheDirectiveInfoOp op = FSEditLogOp.RemoveCacheDirectiveInfoOp.getInstance(this.cache.get()).setId(id);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logAddCachePool(CachePoolInfo pool, boolean toLogRpcIds) {
        FSEditLogOp.AddCachePoolOp op = FSEditLogOp.AddCachePoolOp.getInstance(this.cache.get()).setPool(pool);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logModifyCachePool(CachePoolInfo info, boolean toLogRpcIds) {
        FSEditLogOp.ModifyCachePoolOp op = FSEditLogOp.ModifyCachePoolOp.getInstance(this.cache.get()).setInfo(info);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logRemoveCachePool(String poolName, boolean toLogRpcIds) {
        FSEditLogOp.RemoveCachePoolOp op = FSEditLogOp.RemoveCachePoolOp.getInstance(this.cache.get()).setPoolName(poolName);
        this.logRpcIds(op, toLogRpcIds);
        this.logEdit(op);
    }

    void logSetAcl(String src, List<AclEntry> entries) {
        FSEditLogOp.SetAclOp op = FSEditLogOp.SetAclOp.getInstance();
        op.src = src;
        op.aclEntries = entries;
        this.logEdit(op);
    }

    void logStartRollingUpgrade(long startTime) {
        FSEditLogOp.RollingUpgradeOp op = FSEditLogOp.RollingUpgradeOp.getStartInstance(this.cache.get());
        op.setTime(startTime);
        this.logEdit(op);
    }

    void logFinalizeRollingUpgrade(long finalizeTime) {
        FSEditLogOp.RollingUpgradeOp op = FSEditLogOp.RollingUpgradeOp.getFinalizeInstance(this.cache.get());
        op.setTime(finalizeTime);
        this.logEdit(op);
    }

    synchronized List<JournalSet.JournalAndStream> getJournals() {
        return this.journalSet.getAllJournalStreams();
    }

    @VisibleForTesting
    public synchronized JournalSet getJournalSet() {
        return this.journalSet;
    }

    @VisibleForTesting
    synchronized void setJournalSetForTesting(JournalSet js) {
        this.journalSet = js;
    }

    @VisibleForTesting
    void setMetricsForTests(NameNodeMetrics metrics) {
        this.metrics = metrics;
    }

    public synchronized RemoteEditLogManifest getEditLogManifest(long fromTxId) throws IOException {
        return this.journalSet.getEditLogManifest(fromTxId);
    }

    synchronized long rollEditLog() throws IOException {
        LOG.info((Object)"Rolling edit logs");
        this.endCurrentLogSegment(true);
        long nextTxId = this.getLastWrittenTxId() + 1L;
        this.startLogSegment(nextTxId, true);
        assert (this.curSegmentTxId == nextTxId);
        return nextTxId;
    }

    synchronized void startLogSegment(long segmentTxId, boolean writeHeaderTxn) throws IOException {
        LOG.info((Object)("Starting log segment at " + segmentTxId));
        Preconditions.checkArgument(segmentTxId > 0L, "Bad txid: %s", segmentTxId);
        Preconditions.checkState(this.state == State.BETWEEN_LOG_SEGMENTS, "Bad state: %s", new Object[]{this.state});
        Preconditions.checkState(segmentTxId > this.curSegmentTxId, "Cannot start writing to log segment " + segmentTxId + " when previous log segment started at " + this.curSegmentTxId);
        Preconditions.checkArgument(segmentTxId == this.txid + 1L, "Cannot start log segment at txid %s when next expected txid is %s", segmentTxId, this.txid + 1L);
        this.numTransactionsBatchedInSync = 0L;
        this.totalTimeTransactions = 0L;
        this.numTransactions = 0L;
        this.storage.attemptRestoreRemovedStorage();
        try {
            this.editLogStream = this.journalSet.startLogSegment(segmentTxId, NameNodeLayoutVersion.CURRENT_LAYOUT_VERSION);
        }
        catch (IOException ex) {
            throw new IOException("Unable to start log segment " + segmentTxId + ": too few journals successfully started.", ex);
        }
        this.curSegmentTxId = segmentTxId;
        this.state = State.IN_SEGMENT;
        if (writeHeaderTxn) {
            this.logEdit(FSEditLogOp.LogSegmentOp.getInstance(this.cache.get(), FSEditLogOpCodes.OP_START_LOG_SEGMENT));
            this.logSync();
        }
    }

    synchronized void endCurrentLogSegment(boolean writeEndTxn) {
        LOG.info((Object)("Ending log segment " + this.curSegmentTxId));
        Preconditions.checkState(this.isSegmentOpen(), "Bad state: %s", new Object[]{this.state});
        if (writeEndTxn) {
            this.logEdit(FSEditLogOp.LogSegmentOp.getInstance(this.cache.get(), FSEditLogOpCodes.OP_END_LOG_SEGMENT));
            this.logSync();
        }
        this.printStatistics(true);
        long lastTxId = this.getLastWrittenTxId();
        try {
            this.journalSet.finalizeLogSegment(this.curSegmentTxId, lastTxId);
            this.editLogStream = null;
        }
        catch (IOException e) {
            // empty catch block
        }
        this.state = State.BETWEEN_LOG_SEGMENTS;
    }

    synchronized void abortCurrentLogSegment() {
        try {
            if (this.editLogStream != null) {
                this.editLogStream.abort();
                this.editLogStream = null;
                this.state = State.BETWEEN_LOG_SEGMENTS;
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"All journals failed to abort", (Throwable)e);
        }
    }

    @Override
    public synchronized void purgeLogsOlderThan(long minTxIdToKeep) {
        if (!this.isOpenForWrite()) {
            return;
        }
        assert (this.curSegmentTxId == -12345L || minTxIdToKeep <= this.curSegmentTxId) : "cannot purge logs older than txid " + minTxIdToKeep + " when current segment starts at " + this.curSegmentTxId;
        if (minTxIdToKeep == 0L) {
            return;
        }
        try {
            this.journalSet.purgeLogsOlderThan(minTxIdToKeep);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    synchronized void waitForSyncToFinish() {
        while (this.isSyncRunning) {
            try {
                this.wait(1000L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    synchronized long getSyncTxId() {
        return this.synctxid;
    }

    synchronized void setOutputBufferCapacity(int size) {
        this.journalSet.setOutputBufferCapacity(size);
    }

    synchronized void registerBackupNode(NamenodeRegistration bnReg, NamenodeRegistration nnReg) throws IOException {
        if (bnReg.isRole(HdfsServerConstants.NamenodeRole.CHECKPOINT)) {
            return;
        }
        BackupJournalManager jas = this.findBackupJournal(bnReg);
        if (jas != null) {
            LOG.info((Object)("Backup node " + bnReg + " re-registers"));
            return;
        }
        LOG.info((Object)("Registering new backup node: " + bnReg));
        BackupJournalManager bjm = new BackupJournalManager(bnReg, nnReg);
        this.journalSet.add(bjm, true);
    }

    synchronized void releaseBackupStream(NamenodeRegistration registration) throws IOException {
        BackupJournalManager bjm = this.findBackupJournal(registration);
        if (bjm != null) {
            LOG.info((Object)("Removing backup journal " + bjm));
            this.journalSet.remove(bjm);
        }
    }

    private synchronized BackupJournalManager findBackupJournal(NamenodeRegistration bnReg) {
        for (JournalManager bjm : this.journalSet.getJournalManagers()) {
            if (!(bjm instanceof BackupJournalManager) || !((BackupJournalManager)bjm).matchesRegistration(bnReg)) continue;
            return (BackupJournalManager)bjm;
        }
        return null;
    }

    synchronized void logEdit(int length, byte[] data) {
        long start = this.beginTransaction();
        try {
            this.editLogStream.writeRaw(data, 0, length);
        }
        catch (IOException ex) {
            // empty catch block
        }
        this.endTransaction(start);
    }

    synchronized void recoverUnclosedStreams() {
        Preconditions.checkState(this.state == State.BETWEEN_LOG_SEGMENTS, "May not recover segments - wrong state: %s", new Object[]{this.state});
        try {
            this.journalSet.recoverUnfinalizedSegments();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public long getSharedLogCTime() throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            return jas.getManager().getJournalCTime();
        }
        throw new IOException("No shared log found.");
    }

    public synchronized void doPreUpgradeOfSharedLog() throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            jas.getManager().doPreUpgrade();
        }
    }

    public synchronized void doUpgradeOfSharedLog() throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            jas.getManager().doUpgrade(this.storage);
        }
    }

    public synchronized void doFinalizeOfSharedLog() throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            jas.getManager().doFinalize();
        }
    }

    public synchronized boolean canRollBackSharedLog(StorageInfo prevStorage, int targetLayoutVersion) throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            return jas.getManager().canRollBack(this.storage, prevStorage, targetLayoutVersion);
        }
        throw new IOException("No shared log found.");
    }

    public synchronized void doRollback() throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            if (!jas.isShared()) continue;
            jas.getManager().doRollback();
        }
    }

    public synchronized void discardSegments(long markerTxid) throws IOException {
        for (JournalSet.JournalAndStream jas : this.journalSet.getAllJournalStreams()) {
            jas.getManager().discardSegments(markerTxid);
        }
    }

    @Override
    public void selectInputStreams(Collection<EditLogInputStream> streams, long fromTxId, boolean inProgressOk) throws IOException {
        this.journalSet.selectInputStreams(streams, fromTxId, inProgressOk);
    }

    public Collection<EditLogInputStream> selectInputStreams(long fromTxId, long toAtLeastTxId) throws IOException {
        return this.selectInputStreams(fromTxId, toAtLeastTxId, null, true);
    }

    public synchronized Collection<EditLogInputStream> selectInputStreams(long fromTxId, long toAtLeastTxId, MetaRecoveryContext recovery, boolean inProgressOk) throws IOException {
        ArrayList<EditLogInputStream> streams = new ArrayList<EditLogInputStream>();
        this.selectInputStreams(streams, fromTxId, inProgressOk);
        try {
            this.checkForGaps(streams, fromTxId, toAtLeastTxId, inProgressOk);
        }
        catch (IOException e) {
            if (recovery != null) {
                LOG.error((Object)e);
            }
            FSEditLog.closeAllStreams(streams);
            throw e;
        }
        return streams;
    }

    private void checkForGaps(List<EditLogInputStream> streams, long fromTxId, long toAtLeastTxId, boolean inProgressOk) throws IOException {
        Iterator<EditLogInputStream> iter = streams.iterator();
        long txId = fromTxId;
        while (true) {
            EditLogInputStream elis;
            if (txId > toAtLeastTxId) {
                return;
            }
            if (!iter.hasNext() || (elis = iter.next()).getFirstTxId() > txId) break;
            long next = elis.getLastTxId();
            if (next == -12345L) {
                if (!inProgressOk) {
                    throw new RuntimeException("inProgressOk = false, but selectInputStreams returned an in-progress edit log input stream (" + elis + ")");
                }
                return;
            }
            txId = next + 1L;
        }
        throw new IOException(String.format("Gap in transactions. Expected to be able to read up until at least txid %d but unable to find any edit logs containing txid %d", toAtLeastTxId, txId));
    }

    static void closeAllStreams(Iterable<EditLogInputStream> streams) {
        for (EditLogInputStream s : streams) {
            IOUtils.closeStream(s);
        }
    }

    static Class<? extends JournalManager> getJournalClass(Configuration conf, String uriScheme) {
        String key = "dfs.namenode.edits.journal-plugin." + uriScheme;
        Class<JournalManager> clazz = null;
        try {
            clazz = conf.getClass(key, null, JournalManager.class);
        }
        catch (RuntimeException re) {
            throw new IllegalArgumentException("Invalid class specified for " + uriScheme, re);
        }
        if (clazz == null) {
            LOG.warn((Object)("No class configured for " + uriScheme + ", " + key + " is empty"));
            throw new IllegalArgumentException("No class configured for " + uriScheme);
        }
        return clazz;
    }

    private JournalManager createJournal(URI uri) {
        Class<? extends JournalManager> clazz = FSEditLog.getJournalClass(this.conf, uri.getScheme());
        try {
            Constructor<? extends JournalManager> cons = clazz.getConstructor(Configuration.class, URI.class, NamespaceInfo.class);
            return cons.newInstance(this.conf, uri, this.storage.getNamespaceInfo());
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Unable to construct journal, " + uri, e);
        }
    }

    private static class TransactionId {
        public long txid;

        TransactionId(long value) {
            this.txid = value;
        }
    }

    private static enum State {
        UNINITIALIZED,
        BETWEEN_LOG_SEGMENTS,
        IN_SEGMENT,
        OPEN_FOR_READING,
        CLOSED;

    }
}

