/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.CatalogNotExistException;
import org.apache.flink.table.api.TableSchema;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.Catalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.ExternalCatalog;
import org.apache.flink.table.catalog.ExternalCatalogTable;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.descriptors.Descriptor;
import org.apache.flink.table.factories.TableFactoryUtil;
import org.apache.flink.table.sinks.TableSink;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class CatalogManager {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogManager.class);
    private Map<String, Catalog> catalogs;
    private Map<String, ExternalCatalog> externalCatalogs;
    private String currentCatalogName;
    private String currentDatabaseName;
    private final String builtInCatalogName;

    public CatalogManager(String defaultCatalogName, Catalog defaultCatalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)defaultCatalogName) ? 1 : 0) != 0, (Object)"Default catalog name cannot be null or empty");
        Preconditions.checkNotNull((Object)defaultCatalog, (String)"Default catalog cannot be null");
        this.catalogs = new LinkedHashMap<String, Catalog>();
        this.externalCatalogs = new LinkedHashMap<String, ExternalCatalog>();
        this.catalogs.put(defaultCatalogName, defaultCatalog);
        this.currentCatalogName = defaultCatalogName;
        this.currentDatabaseName = defaultCatalog.getDefaultDatabase();
        this.builtInCatalogName = defaultCatalogName;
    }

    public void registerCatalog(String catalogName, Catalog catalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalog, (String)"Catalog cannot be null");
        if (this.catalogs.containsKey(catalogName) || this.externalCatalogs.containsKey(catalogName)) {
            throw new CatalogException(String.format("Catalog %s already exists.", catalogName));
        }
        this.catalogs.put(catalogName, catalog);
        catalog.open();
    }

    public Optional<Catalog> getCatalog(String catalogName) {
        return Optional.ofNullable(this.catalogs.get(catalogName));
    }

    @Deprecated
    public void registerExternalCatalog(String catalogName, ExternalCatalog catalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"The catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalog, (String)"The catalog cannot be null.");
        if (this.externalCatalogs.containsKey(catalogName) || this.catalogs.containsKey(catalogName)) {
            throw new CatalogException(String.format("An external catalog named [%s] already exists.", catalogName));
        }
        this.externalCatalogs.put(catalogName, catalog);
    }

    @Deprecated
    public Optional<ExternalCatalog> getExternalCatalog(String externalCatalogName) {
        return Optional.ofNullable(this.externalCatalogs.get(externalCatalogName));
    }

    public Set<String> getCatalogs() {
        return this.catalogs.keySet();
    }

    @Deprecated
    public Set<String> getExternalCatalogs() {
        return this.externalCatalogs.keySet();
    }

    public String getCurrentCatalog() {
        return this.currentCatalogName;
    }

    public void setCurrentCatalog(String catalogName) throws CatalogNotExistException {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        if (this.externalCatalogs.containsKey(catalogName)) {
            throw new CatalogException("An external catalog cannot be set as the default one.");
        }
        Catalog potentialCurrentCatalog = this.catalogs.get(catalogName);
        if (potentialCurrentCatalog == null) {
            throw new CatalogException(String.format("A catalog with name [%s] does not exist.", catalogName));
        }
        if (!this.currentCatalogName.equals(catalogName)) {
            this.currentCatalogName = catalogName;
            this.currentDatabaseName = potentialCurrentCatalog.getDefaultDatabase();
            LOG.info("Set the current default catalog as [{}] and the current default database as [{}].", (Object)this.currentCatalogName, (Object)this.currentDatabaseName);
        }
    }

    public String getCurrentDatabase() {
        return this.currentDatabaseName;
    }

    public void setCurrentDatabase(String databaseName) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)databaseName) ? 1 : 0) != 0, (Object)"The database name cannot be null or empty.");
        if (!this.catalogs.get(this.currentCatalogName).databaseExists(databaseName)) {
            throw new CatalogException(String.format("A database with name [%s] does not exist in the catalog: [%s].", databaseName, this.currentCatalogName));
        }
        if (!this.currentDatabaseName.equals(databaseName)) {
            this.currentDatabaseName = databaseName;
            LOG.info("Set the current default database as [{}] in the current default catalog [{}].", (Object)this.currentDatabaseName, (Object)this.currentCatalogName);
        }
    }

    public String getBuiltInCatalogName() {
        return this.builtInCatalogName;
    }

    public String getBuiltInDatabaseName() {
        return this.catalogs.get(this.getBuiltInCatalogName()).getDefaultDatabase();
    }

    public Optional<ResolvedTable> resolveTable(String ... tablePath) {
        Preconditions.checkArgument((tablePath != null && tablePath.length != 0 ? 1 : 0) != 0, (Object)"Table path must not be null or empty.");
        List<String> userPath = Arrays.asList(tablePath);
        List<List> prefixes = Arrays.asList(Arrays.asList(this.currentCatalogName, this.currentDatabaseName), Collections.singletonList(this.currentCatalogName), Collections.emptyList());
        for (List prefix : prefixes) {
            Optional<ResolvedTable> potentialTable = this.lookupPath(prefix, userPath);
            if (!potentialTable.isPresent()) continue;
            return potentialTable;
        }
        return Optional.empty();
    }

    private Optional<ResolvedTable> lookupPath(List<String> prefix, List<String> userPath) {
        try {
            ArrayList<String> path = new ArrayList<String>(prefix);
            path.addAll(userPath);
            Optional<ResolvedTable> potentialTable = this.lookupCatalogTable(path);
            if (!potentialTable.isPresent()) {
                potentialTable = this.lookupExternalTable(path);
            }
            return potentialTable;
        }
        catch (TableNotExistException e) {
            return Optional.empty();
        }
    }

    private Optional<ResolvedTable> lookupCatalogTable(List<String> path) throws TableNotExistException {
        if (path.size() == 3) {
            Catalog currentCatalog = this.catalogs.get(path.get(0));
            String currentDatabaseName = path.get(1);
            String tableName = String.join((CharSequence)".", path.subList(2, path.size()));
            ObjectPath objectPath = new ObjectPath(currentDatabaseName, tableName);
            if (currentCatalog != null && currentCatalog.tableExists(objectPath)) {
                CatalogBaseTable table = currentCatalog.getTable(objectPath);
                return Optional.of(ResolvedTable.catalogTable(Arrays.asList(path.get(0), currentDatabaseName, tableName), table));
            }
        }
        return Optional.empty();
    }

    private Optional<ResolvedTable> lookupExternalTable(List<String> path) {
        ExternalCatalog currentCatalog = this.externalCatalogs.get(path.get(0));
        return Optional.ofNullable(currentCatalog).flatMap(externalCatalog -> this.extractPath((ExternalCatalog)externalCatalog, path.subList(1, path.size() - 1))).map(finalCatalog -> finalCatalog.getTable((String)path.get(path.size() - 1))).map(table -> ResolvedTable.externalTable(path, table, CatalogManager.getTableSchema(table)));
    }

    private Optional<ExternalCatalog> extractPath(ExternalCatalog rootExternalCatalog, List<String> path) {
        ExternalCatalog schema = rootExternalCatalog;
        for (String pathPart : path) {
            if ((schema = schema.getSubCatalog(pathPart)) != null) continue;
            return Optional.empty();
        }
        return Optional.of(schema);
    }

    private static TableSchema getTableSchema(ExternalCatalogTable externalTable) {
        if (externalTable.isTableSource()) {
            return TableFactoryUtil.findAndCreateTableSource((Descriptor)externalTable).getTableSchema();
        }
        TableSink tableSink = TableFactoryUtil.findAndCreateTableSink((Descriptor)externalTable);
        return tableSink.getTableSchema();
    }

    public String[] getFullTablePath(List<String> paths) {
        String tableName;
        String dbName;
        String catalogName;
        if (paths == null) {
            throw new ValidationException("Table paths can not be null!");
        }
        if (paths.size() < 1 || paths.size() > 3) {
            throw new ValidationException("Table paths length must be between 1(inclusive) and 3(inclusive)");
        }
        if (paths.stream().anyMatch(StringUtils::isNullOrWhitespaceOnly)) {
            throw new ValidationException("Table paths contain null or while-space-only string");
        }
        if (paths.size() == 3) {
            return new String[]{paths.get(0), paths.get(1), paths.get(2)};
        }
        if (paths.size() == 1) {
            catalogName = this.getCurrentCatalog();
            dbName = this.getCurrentDatabase();
            tableName = paths.get(0);
        } else {
            catalogName = this.getCurrentCatalog();
            dbName = paths.get(0);
            tableName = paths.get(1);
        }
        return new String[]{catalogName, dbName, tableName};
    }

    public static class ResolvedTable {
        private final ExternalCatalogTable externalCatalogTable;
        private final CatalogBaseTable catalogTable;
        private final TableSchema tableSchema;
        private final List<String> tablePath;

        static ResolvedTable externalTable(List<String> tablePath, ExternalCatalogTable table, TableSchema tableSchema) {
            return new ResolvedTable(table, null, tableSchema, tablePath);
        }

        static ResolvedTable catalogTable(List<String> tablePath, CatalogBaseTable table) {
            return new ResolvedTable(null, table, table.getSchema(), tablePath);
        }

        private ResolvedTable(ExternalCatalogTable externalCatalogTable, CatalogBaseTable catalogTable, TableSchema tableSchema, List<String> tablePath) {
            this.externalCatalogTable = externalCatalogTable;
            this.catalogTable = catalogTable;
            this.tableSchema = tableSchema;
            this.tablePath = tablePath;
        }

        public Optional<ExternalCatalogTable> getExternalCatalogTable() {
            return Optional.ofNullable(this.externalCatalogTable);
        }

        public Optional<CatalogBaseTable> getCatalogTable() {
            return Optional.ofNullable(this.catalogTable);
        }

        public TableSchema getTableSchema() {
            return this.tableSchema;
        }

        public List<String> getTablePath() {
            return this.tablePath;
        }
    }
}

