/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.impl.Log4JLogger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSImage;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.log4j.Level;
import org.junit.Assert;
import org.junit.Test;

public class TestEditLogToleration {
    private static final Log LOG = LogFactory.getLog(TestEditLogToleration.class);
    private static final int TOLERATION_LENGTH = 1024;
    private static final byte[] PADS = new byte[]{0, -1};
    private static final int LOOP = 8;
    private static final Random RANDOM = new Random();

    public TestEditLogToleration() {
        ((Log4JLogger)FSEditLog.LOG).getLogger().setLevel(Level.ALL);
        ((Log4JLogger)LogFactory.getLog(MBeans.class)).getLogger().setLevel(Level.OFF);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void runTest(EditFileModifier modifier) throws IOException {
        Configuration conf = new Configuration();
        conf.setInt("dfs.namenode.edits.toleration.length", 1024);
        MiniDFSCluster cluster = new MiniDFSCluster(conf, 0, true, null);
        try {
            cluster.waitActive();
            FileSystem fs = cluster.getFileSystem();
            fs.mkdirs(new Path("/user/foo"));
            fs.mkdirs(new Path("/user/bar"));
            cluster.shutdownNameNode();
            for (File dir : FSNamesystem.getNamespaceEditsDirs((Configuration)conf)) {
                File editFile = new File(new File(dir, "current"), FSImage.NameNodeFile.EDITS.getName());
                Assert.assertTrue((String)("Should exist: " + editFile), (boolean)editFile.exists());
                modifier.modify(editFile);
            }
            try {
                cluster.restartNameNode();
                Assert.assertTrue((boolean)modifier.isTolerable());
            }
            catch (IOException e) {
                LOG.info((Object)"Got an exception", (Throwable)e);
                Assert.assertFalse((boolean)modifier.isTolerable());
            }
        }
        finally {
            cluster.shutdown();
        }
    }

    @Test
    public void testNoModification() throws IOException {
        this.runTest(new NullModifier());
    }

    @Test
    public void testTruncatedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int truncation = RANDOM.nextInt(100);
            this.runTest(new Truncating(truncation));
        }
    }

    @Test
    public void testNormalPaddedEditLog() throws IOException {
        for (int i = 0; i < 4; ++i) {
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(0, padding, pad);
            Assert.assertTrue((boolean)p.isTolerable());
            this.runTest(p);
        }
    }

    @Test
    public void testTolerableErrorPaddedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int corruption = RANDOM.nextInt(1024) + 1;
            Assert.assertTrue((corruption > 0 ? 1 : 0) != 0);
            Assert.assertTrue((corruption <= 1024 ? 1 : 0) != 0);
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(corruption, padding, pad);
            Assert.assertTrue((boolean)p.isTolerable());
            this.runTest(p);
        }
    }

    @Test
    public void testIntolerableErrorPaddedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int corruption = RANDOM.nextInt(1024) + 1024 + 1;
            Assert.assertTrue((corruption > 1024 ? 1 : 0) != 0);
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(corruption, padding, pad);
            Assert.assertFalse((boolean)p.isTolerable());
            this.runTest(p);
        }
    }

    @Test
    public void testTruncateAndNormalPaddedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int truncation = RANDOM.nextInt(100);
            Truncating t = new Truncating(truncation);
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(0, padding, pad);
            ChainModifier chain = new ChainModifier(t, p);
            Assert.assertTrue((boolean)chain.isTolerable());
            this.runTest(chain);
        }
    }

    @Test
    public void testTruncateAndTolerableErrorPaddedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int truncation = RANDOM.nextInt(100);
            Truncating t = new Truncating(truncation);
            int corruption = RANDOM.nextInt(512) + 1;
            Assert.assertTrue((corruption > 0 ? 1 : 0) != 0);
            Assert.assertTrue((corruption <= 512 ? 1 : 0) != 0);
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(corruption, padding, pad);
            ChainModifier chain = new ChainModifier(t, p);
            Assert.assertTrue((boolean)chain.isTolerable());
            this.runTest(chain);
        }
    }

    @Test
    public void testTruncateAndIntolerableErrorPaddedEditLog() throws IOException {
        for (int i = 0; i < 8; ++i) {
            int truncation = RANDOM.nextInt(100);
            Truncating t = new Truncating(truncation);
            int corruption = RANDOM.nextInt(1024) + 1024 + 1;
            Assert.assertTrue((corruption > 1024 ? 1 : 0) != 0);
            int padding = RANDOM.nextInt(2048);
            byte pad = PADS[RANDOM.nextInt(PADS.length)];
            Padding p = new Padding(corruption, padding, pad);
            ChainModifier chain = new ChainModifier(t, p);
            Assert.assertFalse((boolean)chain.isTolerable());
            this.runTest(chain);
        }
    }

    static class ChainModifier
    extends EditFileModifier {
        final List<EditFileModifier> modifers;

        ChainModifier(EditFileModifier ... modifiers) {
            this.modifers = Arrays.asList(modifiers);
        }

        @Override
        void modify(File editFile) throws IOException {
            for (EditFileModifier m : this.modifers) {
                m.modify(editFile);
            }
        }

        @Override
        boolean isTolerable() {
            for (EditFileModifier m : this.modifers) {
                if (m.isTolerable()) continue;
                return false;
            }
            return true;
        }
    }

    static class Padding
    extends EditFileModifier {
        static final byte[] PAD_BUFFER = new byte[4096];
        final int corruptionLength;
        final int paddingLength;
        final byte pad;

        Padding(int corruptionLength, int paddingLength, byte pad) {
            this.corruptionLength = corruptionLength;
            this.paddingLength = paddingLength;
            this.pad = pad;
        }

        @Override
        void modify(File f) throws IOException {
            this.log(f);
            RandomAccessFile raf = new RandomAccessFile(f, "rw");
            raf.seek(f.length());
            if (this.corruptionLength > 0) {
                Padding.pad(raf, this.pad, this.corruptionLength - 1);
                raf.write(171);
            }
            Padding.pad(raf, this.pad, this.paddingLength);
            raf.close();
            LOG.info((Object)(this.getClass().getSimpleName() + ": new length=" + f.length() + ", corruptionLength=" + this.corruptionLength + ", paddingLength=" + this.paddingLength));
        }

        @Override
        boolean isTolerable() {
            return this.corruptionLength <= 1024;
        }

        private static void pad(DataOutput out, byte pad, int length) throws IOException {
            Arrays.fill(PAD_BUFFER, pad);
            while (length > 0) {
                int n = length < PAD_BUFFER.length ? length : PAD_BUFFER.length;
                out.write(PAD_BUFFER, 0, n);
                length -= n;
            }
        }
    }

    static class Truncating
    extends EditFileModifier {
        final int truncationLength;

        Truncating(int truncationLength) {
            this.truncationLength = truncationLength;
        }

        @Override
        void modify(File f) throws IOException {
            this.log(f);
            RandomAccessFile raf = new RandomAccessFile(f, "rw");
            raf.setLength(f.length() - (long)this.truncationLength);
            raf.close();
            LOG.info((Object)(this.getClass().getSimpleName() + ": new length=" + f.length() + ", truncationLength=" + this.truncationLength));
        }

        @Override
        boolean isTolerable() {
            return true;
        }
    }

    static class NullModifier
    extends EditFileModifier {
        NullModifier() {
        }

        @Override
        void modify(File f) throws IOException {
            this.log(f);
        }

        @Override
        boolean isTolerable() {
            return true;
        }
    }

    static abstract class EditFileModifier {
        EditFileModifier() {
        }

        void log(File f) {
            LOG.info((Object)(this.getClass().getSimpleName() + ": length=" + f.length() + ", f=" + f));
        }

        abstract void modify(File var1) throws IOException;

        abstract boolean isTolerable();
    }
}

