/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.client.RegionReplicaUtil;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.exceptions.MergeRegionException;
import org.apache.hadoop.hbase.master.CatalogJanitor;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.assignment.TransitRegionStateProcedure;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.ServerRegionReplicaUtil;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class MetaFixer {
    private static final Logger LOG = LoggerFactory.getLogger(MetaFixer.class);
    private static final String MAX_MERGE_COUNT_KEY = "hbase.master.metafixer.max.merge.count";
    private static final int MAX_MERGE_COUNT_DEFAULT = 64;
    private final MasterServices masterServices;
    private final int maxMergeCount;

    MetaFixer(MasterServices masterServices) {
        this.masterServices = masterServices;
        this.maxMergeCount = this.masterServices.getConfiguration().getInt(MAX_MERGE_COUNT_KEY, 64);
    }

    void fix() throws IOException {
        CatalogJanitor.Report report = this.masterServices.getCatalogJanitor().getLastReport();
        if (report == null) {
            LOG.info("CatalogJanitor has not generated a report yet; run 'catalogjanitor_run' in shell or wait until CatalogJanitor chore runs.");
            return;
        }
        this.fixHoles(report);
        this.fixOverlaps(report);
        this.masterServices.runReplicationBarrierCleaner();
    }

    void fixHoles(CatalogJanitor.Report report) {
        List<Pair<RegionInfo, RegionInfo>> holes = report.getHoles();
        if (holes.isEmpty()) {
            LOG.info("CatalogJanitor Report contains no holes to fix. Skipping.");
            return;
        }
        LOG.info("Identified {} region holes to fix. Detailed fixup progress logged at DEBUG.", (Object)holes.size());
        List<RegionInfo> newRegionInfos = MetaFixer.createRegionInfosForHoles(holes);
        List<RegionInfo> newMetaEntries = MetaFixer.createMetaEntries(this.masterServices, newRegionInfos);
        TransitRegionStateProcedure[] assignProcedures = this.masterServices.getAssignmentManager().createRoundRobinAssignProcedures(newMetaEntries);
        this.masterServices.getMasterProcedureExecutor().submitProcedures((Procedure[])assignProcedures);
        LOG.info("Scheduled {}/{} new regions for assignment.", (Object)assignProcedures.length, (Object)holes.size());
    }

    private static List<RegionInfo> createRegionInfosForHoles(List<Pair<RegionInfo, RegionInfo>> holes) {
        List<RegionInfo> newRegionInfos = holes.stream().map(MetaFixer::getHoleCover).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        LOG.debug("Constructed {}/{} RegionInfo descriptors corresponding to identified holes.", (Object)newRegionInfos.size(), (Object)holes.size());
        return newRegionInfos;
    }

    private static Optional<RegionInfo> getHoleCover(Pair<RegionInfo, RegionInfo> hole) {
        RegionInfo left = (RegionInfo)hole.getFirst();
        RegionInfo right = (RegionInfo)hole.getSecond();
        if (left.getTable().equals((Object)right.getTable())) {
            if (Bytes.compareTo((byte[])left.getEndKey(), (byte[])right.getStartKey()) >= 0) {
                LOG.warn("Skipping hole fix; left-side endKey is not less than right-side startKey; left=<{}>, right=<{}>", (Object)left, (Object)right);
                return Optional.empty();
            }
            return Optional.of(MetaFixer.buildRegionInfo(left.getTable(), left.getEndKey(), right.getStartKey()));
        }
        boolean leftUndefined = left.equals(RegionInfo.UNDEFINED);
        boolean rightUndefined = right.equals(RegionInfo.UNDEFINED);
        boolean last = left.isLast();
        boolean first = right.isFirst();
        if (leftUndefined && rightUndefined) {
            LOG.warn("Skipping hole fix; both the hole left-side and right-side RegionInfos are UNDEFINED; left=<{}>, right=<{}>", (Object)left, (Object)right);
            return Optional.empty();
        }
        if (leftUndefined || last) {
            return Optional.of(MetaFixer.buildRegionInfo(right.getTable(), HConstants.EMPTY_START_ROW, right.getStartKey()));
        }
        if (rightUndefined || first) {
            return Optional.of(MetaFixer.buildRegionInfo(left.getTable(), left.getEndKey(), HConstants.EMPTY_END_ROW));
        }
        LOG.warn("Skipping hole fix; don't know what to do with left=<{}>, right=<{}>", (Object)left, (Object)right);
        return Optional.empty();
    }

    private static RegionInfo buildRegionInfo(TableName tn, byte[] start, byte[] end) {
        return RegionInfoBuilder.newBuilder((TableName)tn).setStartKey(start).setEndKey(end).build();
    }

    private static List<RegionInfo> createMetaEntries(MasterServices masterServices, List<RegionInfo> newRegionInfos) {
        List addMetaEntriesResults = newRegionInfos.stream().map(regionInfo -> {
            try {
                TableDescriptor td = masterServices.getTableDescriptors().get(regionInfo.getTable());
                List newRegions = RegionReplicaUtil.addReplicas(Collections.singletonList(regionInfo), (int)1, (int)td.getRegionReplication());
                MetaTableAccessor.addRegionsToMeta((Connection)masterServices.getConnection(), (List)newRegions, (int)td.getRegionReplication());
                if (td.getRegionReplication() > 1) {
                    ServerRegionReplicaUtil.setupRegionReplicaReplication(masterServices.getConfiguration());
                }
                return Either.ofLeft(newRegions);
            }
            catch (IOException e) {
                return Either.ofRight(e);
            }
        }).collect(Collectors.toList());
        List<RegionInfo> createMetaEntriesSuccesses = addMetaEntriesResults.stream().filter(Either::hasLeft).map(Either::getLeft).flatMap(Collection::stream).collect(Collectors.toList());
        List<IOException> createMetaEntriesFailures = addMetaEntriesResults.stream().filter(Either::hasRight).map(Either::getRight).collect(Collectors.toList());
        LOG.debug("Added {}/{} entries to hbase:meta", (Object)createMetaEntriesSuccesses.size(), (Object)newRegionInfos.size());
        if (!createMetaEntriesFailures.isEmpty()) {
            LOG.warn("Failed to create entries in hbase:meta for {}/{} RegionInfo descriptors. First failure message included; full list of failures with accompanying stack traces is available at log level DEBUG. message={}", new Object[]{createMetaEntriesFailures.size(), addMetaEntriesResults.size(), ((IOException)createMetaEntriesFailures.get(0)).getMessage()});
            if (LOG.isDebugEnabled()) {
                createMetaEntriesFailures.forEach(ioe -> LOG.debug("Attempt to fix region hole in hbase:meta failed.", (Throwable)ioe));
            }
        }
        return createMetaEntriesSuccesses;
    }

    void fixOverlaps(CatalogJanitor.Report report) throws IOException {
        for (Set set : MetaFixer.calculateMerges(this.maxMergeCount, report.getOverlaps())) {
            RegionInfo[] regionsArray = set.toArray(new RegionInfo[0]);
            try {
                this.masterServices.mergeRegions(regionsArray, true, 0L, 0L);
            }
            catch (MergeRegionException mre) {
                LOG.warn("Failed overlap fix of {}", (Object)regionsArray, (Object)mre);
            }
        }
    }

    @VisibleForTesting
    static List<SortedSet<RegionInfo>> calculateMerges(int maxMergeCount, List<Pair<RegionInfo, RegionInfo>> overlaps) {
        if (overlaps.isEmpty()) {
            LOG.debug("No overlaps.");
            return Collections.emptyList();
        }
        ArrayList<SortedSet<RegionInfo>> merges = new ArrayList<SortedSet<RegionInfo>>();
        TreeSet<Object> currentMergeSet = new TreeSet<Object>();
        HashSet<Object> regionsInMergeSet = new HashSet<Object>();
        RegionInfo regionInfoWithlargestEndKey = null;
        for (Pair<RegionInfo, RegionInfo> pair : overlaps) {
            if (!(regionInfoWithlargestEndKey == null || MetaFixer.isOverlap(regionInfoWithlargestEndKey, pair) && currentMergeSet.size() < maxMergeCount)) {
                if (currentMergeSet.size() >= maxMergeCount) {
                    LOG.warn("Ran into maximum-at-a-time merges limit={}", (Object)maxMergeCount);
                }
                if (currentMergeSet.size() <= 1) {
                    for (RegionInfo regionInfo : currentMergeSet) {
                        regionsInMergeSet.remove(regionInfo);
                    }
                    currentMergeSet.clear();
                } else {
                    merges.add(currentMergeSet);
                    currentMergeSet = new TreeSet();
                }
            }
            if (!regionsInMergeSet.contains(pair.getFirst())) {
                currentMergeSet.add(pair.getFirst());
                regionsInMergeSet.add(pair.getFirst());
            }
            if (!regionsInMergeSet.contains(pair.getSecond())) {
                currentMergeSet.add(pair.getSecond());
                regionsInMergeSet.add(pair.getSecond());
            }
            regionInfoWithlargestEndKey = MetaFixer.getRegionInfoWithLargestEndKey(MetaFixer.getRegionInfoWithLargestEndKey((RegionInfo)pair.getFirst(), (RegionInfo)pair.getSecond()), regionInfoWithlargestEndKey);
        }
        merges.add(currentMergeSet);
        return merges;
    }

    @VisibleForTesting
    static RegionInfo getRegionInfoWithLargestEndKey(RegionInfo a, RegionInfo b) {
        if (a == null) {
            return b;
        }
        if (b == null) {
            return a;
        }
        if (!a.getTable().equals((Object)b.getTable())) {
            return b;
        }
        if (a.isLast()) {
            return a;
        }
        if (b.isLast()) {
            return b;
        }
        int compare = Bytes.compareTo((byte[])a.getEndKey(), (byte[])b.getEndKey());
        return compare == 0 || compare > 0 ? a : b;
    }

    @VisibleForTesting
    static boolean isOverlap(RegionInfo ri, Pair<RegionInfo, RegionInfo> pair) {
        if (ri == null || pair == null) {
            return false;
        }
        return ri.isOverlap((RegionInfo)pair.getFirst()) || ri.isOverlap((RegionInfo)pair.getSecond());
    }

    private static class Either<L, R> {
        private final L left;
        private final R right;

        public static <L, R> Either<L, R> ofLeft(L left) {
            return new Either<L, Object>(left, null);
        }

        public static <L, R> Either<L, R> ofRight(R right) {
            return new Either<Object, R>(null, right);
        }

        Either(L left, R right) {
            this.left = left;
            this.right = right;
        }

        public boolean hasLeft() {
            return this.left != null;
        }

        public L getLeft() {
            if (!this.hasLeft()) {
                throw new IllegalStateException("Either contains no left.");
            }
            return this.left;
        }

        public boolean hasRight() {
            return this.right != null;
        }

        public R getRight() {
            if (!this.hasRight()) {
                throw new IllegalStateException("Either contains no right.");
            }
            return this.right;
        }
    }
}

