/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.ssl;

import io.netty.util.ResourceLeakDetector;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.Principal;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.network.ConnectionMode;
import org.apache.kafka.common.security.TestSecurityConfig;
import org.apache.kafka.common.security.auth.SslEngineFactory;
import org.apache.kafka.common.security.ssl.DefaultSslEngineFactory;
import org.apache.kafka.common.security.ssl.SslFactory;
import org.apache.kafka.common.security.ssl.mock.TestProviderCreator;
import org.apache.kafka.test.TestSslUtils;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public abstract class SslFactoryTest {
    protected final String tlsProtocol;
    protected final Class<? extends SslEngineFactory> engineFactoryClass;

    public SslFactoryTest(String tlsProtocol, Class<? extends SslEngineFactory> engineFactoryClass) {
        this.tlsProtocol = tlsProtocol;
        this.engineFactoryClass = engineFactoryClass;
        ResourceLeakDetector.setLevel((ResourceLeakDetector.Level)ResourceLeakDetector.Level.PARANOID);
    }

    protected void configureSslBuilderClass(Map<String, Object> conf) {
        conf.put("ssl.engine.factory.class", this.engineFactoryClass);
    }

    @Test
    public void testSslFactoryConfiguration() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(serverSslConfig);
        try (SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);){
            sslFactory.configure(serverSslConfig);
            SSLEngine engine = sslFactory.createSslEngine("localhost", 0);
            Assertions.assertNotNull((Object)engine);
            Assertions.assertEquals(Set.of(this.tlsProtocol), Set.of(engine.getEnabledProtocols()));
            Assertions.assertFalse((boolean)engine.getUseClientMode());
        }
    }

    @Test
    public void testSslFactoryConfigurationWithBrokerSSLListenerDynamicReloadOn() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(serverSslConfig);
        serverSslConfig.put("confluent.broker.listeners.ssl.file.store.reload.enable", "true");
        try (SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);){
            sslFactory.configure(serverSslConfig);
            Assertions.assertTrue((boolean)Boolean.parseBoolean((String)serverSslConfig.get("confluent.ssl.enable.dynamic.store.update")));
            SSLEngine engine = sslFactory.createSslEngine("localhost", 0);
            Assertions.assertNotNull((Object)engine);
            Assertions.assertFalse((boolean)engine.getUseClientMode());
        }
    }

    @Test
    public void testSslFactoryConfigurationACCPOn() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(serverSslConfig);
        serverSslConfig.put("confluent.accp.enabled", "true");
        try (SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);){
            sslFactory.configure(serverSslConfig);
            String sslEngineFactoryClassName = sslFactory.sslEngineFactory().getClass().getSimpleName();
            Assertions.assertEquals((Object)"DefaultSslEngineFactory", (Object)sslEngineFactoryClassName);
            SSLEngine engine = sslFactory.createSslEngine("localhost", 0);
            Assertions.assertNotNull((Object)engine);
            Assertions.assertFalse((boolean)engine.getUseClientMode());
        }
    }

    @Test
    public void testSslFactoryConfigWithManyKeyStoreEntries() throws Exception {
        Map<String, Object> serverSslConfig = TestSslUtils.generateConfigsWithCertificateChains(this.tlsProtocol);
        this.configureSslBuilderClass(serverSslConfig);
        try (SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);){
            sslFactory.configure(serverSslConfig);
            SSLEngine engine = sslFactory.createSslEngine("localhost", 0);
            Assertions.assertNotNull((Object)engine);
            Assertions.assertEquals(Set.of(this.tlsProtocol), Set.of(engine.getEnabledProtocols()));
            Assertions.assertFalse((boolean)engine.getUseClientMode());
        }
    }

    @Test
    public void testSslFactoryWithCustomKeyManagerConfiguration() {
        TestProviderCreator testProviderCreator = new TestProviderCreator();
        Map<String, Object> serverSslConfig = TestSslUtils.createSslConfig("TestAlgorithm", "TestAlgorithm", this.tlsProtocol);
        this.configureSslBuilderClass(serverSslConfig);
        serverSslConfig.put("security.providers", testProviderCreator.getClass().getName());
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(serverSslConfig);
        Assertions.assertNotNull((Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not created");
        Security.removeProvider(testProviderCreator.getProvider().getName());
    }

    @Test
    public void testSslFactoryWithoutProviderClassConfiguration() {
        Map<String, Object> serverSslConfig = TestSslUtils.createSslConfig("TestAlgorithm", "TestAlgorithm", this.tlsProtocol);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        Assertions.assertThrows(KafkaException.class, () -> sslFactory.configure(serverSslConfig));
    }

    @Test
    public void testSslFactoryWithIncorrectProviderClassConfiguration() {
        Map<String, Object> serverSslConfig = TestSslUtils.createSslConfig("TestAlgorithm", "TestAlgorithm", this.tlsProtocol);
        serverSslConfig.put("security.providers", "com.fake.ProviderClass1,com.fake.ProviderClass2");
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        Assertions.assertThrows(KafkaException.class, () -> sslFactory.configure(serverSslConfig));
    }

    @Test
    public void testSslFactoryWithoutPasswordConfiguration() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(serverSslConfig);
        serverSslConfig.remove("ssl.truststore.password");
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        try {
            sslFactory.configure(serverSslConfig);
        }
        catch (Exception e) {
            Assertions.fail((String)("An exception was thrown when configuring the truststore without a password: " + String.valueOf(e)));
        }
    }

    @Test
    public void testClientMode() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> clientSslConfig = this.sslConfigsBuilder(ConnectionMode.CLIENT).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        this.configureSslBuilderClass(clientSslConfig);
        SslFactory sslFactory = new SslFactory(ConnectionMode.CLIENT);
        sslFactory.configure(clientSslConfig);
        SSLEngine engine = sslFactory.createSslEngine("localhost", 0);
        Assertions.assertTrue((boolean)engine.getUseClientMode());
    }

    @Test
    public void staleSslEngineFactoryShouldBeClosed() throws IOException, GeneralSecurityException {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> clientSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        clientSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(clientSslConfig);
        TestSslUtils.TestSslEngineFactory sslEngineFactory = (TestSslUtils.TestSslEngineFactory)sslFactory.sslEngineFactory();
        Assertions.assertNotNull((Object)sslEngineFactory);
        Assertions.assertFalse((boolean)sslEngineFactory.closed);
        trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        clientSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        clientSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        sslFactory.reconfigure(clientSslConfig);
        TestSslUtils.TestSslEngineFactory newSslEngineFactory = (TestSslUtils.TestSslEngineFactory)sslFactory.sslEngineFactory();
        Assertions.assertNotEquals((Object)sslEngineFactory, (Object)newSslEngineFactory);
        Assertions.assertTrue((boolean)sslEngineFactory.closed);
    }

    private Principal getPrincipal(SslEngineFactory engineFactory) throws Exception {
        List allEntries = SslFactory.CertificateEntries.create((KeyStore)engineFactory.keystore());
        Assertions.assertEquals((int)1, (int)allEntries.size());
        SslFactory.CertificateEntries entries = (SslFactory.CertificateEntries)allEntries.get(0);
        return entries.subjectPrincipal;
    }

    @Test
    public void testIncompatibleKeystoreReconfiguration() throws Exception {
        File trustStoreFile = File.createTempFile("truststore", ".jks");
        Map<String, Object> initialSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).cn("Initial CN").build();
        this.configureSslBuilderClass(initialSslConfig);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(initialSslConfig);
        SslEngineFactory sslEngineFactory = sslFactory.sslEngineFactory();
        Assertions.assertNotNull((Object)sslEngineFactory, (String)"SslEngineFactory not created");
        Assertions.assertEquals((Object)"O=A server,CN=Initial CN", (Object)this.getPrincipal(sslEngineFactory).getName());
        Map<String, Object> updatedSslConfigs = this.sslConfigsBuilder(ConnectionMode.SERVER).useExistingTrustStore(trustStoreFile).cn("Different CN").build();
        Assertions.assertThrows(ConfigException.class, () -> sslFactory.validateReconfiguration(updatedSslConfigs));
        updatedSslConfigs.put("confluent.require.compatible.keystore.updates", true);
        Assertions.assertThrows(ConfigException.class, () -> sslFactory.validateReconfiguration(updatedSslConfigs));
        updatedSslConfigs.put("confluent.require.compatible.keystore.updates", false);
        sslFactory.reconfigure(updatedSslConfigs);
        Assertions.assertEquals((Object)"O=A server,CN=Different CN", (Object)this.getPrincipal(sslFactory.sslEngineFactory()).getName());
    }

    @Test
    public void testReconfiguration() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(sslConfig);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        Map<String, Object> finalSslConfig = sslConfig;
        Exception e = (Exception)Assertions.assertThrows(ConfigException.class, () -> sslFactory.validateReconfiguration(finalSslConfig));
        Assertions.assertEquals((Object)"SSL reconfiguration failed due to java.lang.IllegalStateException: SslFactory has not been configured.", (Object)e.getMessage());
        sslFactory.configure(sslConfig);
        SslEngineFactory sslEngineFactory = sslFactory.sslEngineFactory();
        Assertions.assertNotNull((Object)sslEngineFactory, (String)"SslEngineFactory not created");
        sslFactory.reconfigure(sslConfig);
        Assertions.assertSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory recreated unnecessarily");
        trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> newSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(newSslConfig);
        sslFactory.reconfigure(newSslConfig);
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        trustStoreFile = File.createTempFile("truststore", ".jks");
        sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        sslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        Assertions.assertEquals((Object)sslFactory.sslEngineFactory().getClass(), this.engineFactoryClass);
        sslFactory.reconfigure(sslConfig);
        Assertions.assertTrue((boolean)(sslFactory.sslEngineFactory() instanceof TestSslUtils.TestSslEngineFactory));
        sslEngineFactory = sslFactory.sslEngineFactory();
        trustStoreFile.setLastModified(System.currentTimeMillis() + 10000L);
        sslFactory.reconfigure(newSslConfig);
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        File keyStoreFile = new File((String)newSslConfig.get("ssl.keystore.location"));
        keyStoreFile.setLastModified(System.currentTimeMillis() + 10000L);
        sslFactory.reconfigure(newSslConfig);
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        keyStoreFile.setLastModified(System.currentTimeMillis() + 15000L);
        sslFactory.validateReconfiguration(newSslConfig);
        sslFactory.reconfigure(newSslConfig);
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        keyStoreFile.setLastModified(System.currentTimeMillis() + 20000L);
        Files.delete(keyStoreFile.toPath());
        sslFactory.reconfigure(newSslConfig);
        Assertions.assertSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory recreated unnecessarily");
    }

    @Test
    public void testReconfigurationWithoutTruststore() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(sslConfig);
        sslConfig.remove("ssl.truststore.location");
        sslConfig.remove("ssl.truststore.password");
        sslConfig.remove("ssl.truststore.type");
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(sslConfig);
        SSLContext sslContext = ((DefaultSslEngineFactory)sslFactory.sslEngineFactory()).sslContext();
        Assertions.assertNotNull((Object)sslContext, (String)"SSL context not created");
        Assertions.assertSame((Object)sslContext, (Object)((DefaultSslEngineFactory)sslFactory.sslEngineFactory()).sslContext(), (String)"SSL context recreated unnecessarily");
        Assertions.assertFalse((boolean)sslFactory.createSslEngine("localhost", 0).getUseClientMode());
        Map<String, Object> sslConfig2 = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(sslConfig2);
        try {
            sslFactory.validateReconfiguration(sslConfig2);
            Assertions.fail((String)"Truststore configured dynamically for listener without previous truststore");
        }
        catch (ConfigException configException) {
            // empty catch block
        }
    }

    @Test
    public void testReconfigurationWithoutKeystore() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(sslConfig);
        sslConfig.remove("ssl.keystore.location");
        sslConfig.remove("ssl.keystore.password");
        sslConfig.remove("ssl.keystore.type");
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(sslConfig);
        SSLContext sslContext = ((DefaultSslEngineFactory)sslFactory.sslEngineFactory()).sslContext();
        Assertions.assertNotNull((Object)sslContext, (String)"SSL context not created");
        Assertions.assertSame((Object)sslContext, (Object)((DefaultSslEngineFactory)sslFactory.sslEngineFactory()).sslContext(), (String)"SSL context recreated unnecessarily");
        Assertions.assertFalse((boolean)sslFactory.createSslEngine("localhost", 0).getUseClientMode());
        File newTrustStoreFile = TestUtils.tempFile("truststore", ".jks");
        sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(newTrustStoreFile).build();
        this.configureSslBuilderClass(sslConfig);
        sslConfig.remove("ssl.keystore.location");
        sslConfig.remove("ssl.keystore.password");
        sslConfig.remove("ssl.keystore.type");
        sslFactory.reconfigure(sslConfig);
        Assertions.assertNotSame((Object)sslContext, (Object)((DefaultSslEngineFactory)sslFactory.sslEngineFactory()).sslContext(), (String)"SSL context not recreated");
        sslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(newTrustStoreFile).build();
        this.configureSslBuilderClass(sslConfig);
        try {
            sslFactory.validateReconfiguration(sslConfig);
            Assertions.fail((String)"Keystore configured dynamically for listener without previous keystore");
        }
        catch (ConfigException configException) {
            // empty catch block
        }
    }

    @Test
    public void testPemReconfiguration() throws Exception {
        Properties props = new Properties();
        props.putAll(this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(null).usePem(true).build());
        TestSecurityConfig sslConfig = new TestSecurityConfig(props);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(sslConfig.values());
        SslEngineFactory sslEngineFactory = sslFactory.sslEngineFactory();
        Assertions.assertNotNull((Object)sslEngineFactory, (String)"SslEngineFactory not created");
        props.put("some.config", "some.value");
        sslConfig = new TestSecurityConfig(props);
        sslFactory.reconfigure(sslConfig.values());
        Assertions.assertSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory recreated unnecessarily");
        props.put("ssl.keystore.key", new Password(((Password)props.get("ssl.keystore.key")).value() + " "));
        sslConfig = new TestSecurityConfig(props);
        sslFactory.reconfigure(sslConfig.values());
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        props.put("ssl.keystore.certificate.chain", new Password(((Password)props.get("ssl.keystore.certificate.chain")).value() + " "));
        sslConfig = new TestSecurityConfig(props);
        sslFactory.reconfigure(sslConfig.values());
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
        props.put("ssl.truststore.certificates", new Password(((Password)props.get("ssl.truststore.certificates")).value() + " "));
        sslConfig = new TestSecurityConfig(props);
        sslFactory.reconfigure(sslConfig.values());
        Assertions.assertNotSame((Object)sslEngineFactory, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not recreated");
        sslEngineFactory = sslFactory.sslEngineFactory();
    }

    @Test
    public void testKeyStoreTrustStoreValidation() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).build();
        this.configureSslBuilderClass(serverSslConfig);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(serverSslConfig);
        Assertions.assertNotNull((Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory not created");
    }

    @Test
    public void testUntrustedKeyStoreValidationFails() throws Exception {
        File trustStoreFile1 = TestUtils.tempFile("truststore1", ".jks");
        File trustStoreFile2 = TestUtils.tempFile("truststore2", ".jks");
        Map<String, Object> sslConfig1 = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile1).build();
        this.configureSslBuilderClass(sslConfig1);
        Map<String, Object> sslConfig2 = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile2).build();
        this.configureSslBuilderClass(sslConfig2);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);
        for (String key : Arrays.asList("ssl.truststore.location", "ssl.truststore.password", "ssl.truststore.type", "ssl.trustmanager.algorithm")) {
            sslConfig1.put(key, sslConfig2.get(key));
        }
        try {
            sslFactory.configure(sslConfig1);
            Assertions.fail((String)"Validation did not fail with untrusted truststore");
        }
        catch (ConfigException configException) {
            // empty catch block
        }
    }

    @Test
    public void testKeystoreVerifiableUsingTruststore() throws Exception {
        this.verifyKeystoreVerifiableUsingTruststore(false, this.tlsProtocol);
    }

    @Test
    public void testPemKeystoreVerifiableUsingTruststore() throws Exception {
        this.verifyKeystoreVerifiableUsingTruststore(true, this.tlsProtocol);
    }

    private void verifyKeystoreVerifiableUsingTruststore(boolean usePem, String tlsProtocol) throws Exception {
        File trustStoreFile1 = usePem ? null : TestUtils.tempFile("truststore1", ".jks");
        Map<String, Object> sslConfig1 = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile1).usePem(usePem).build();
        this.configureSslBuilderClass(sslConfig1);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER, null, true);
        sslFactory.configure(sslConfig1);
        File trustStoreFile2 = usePem ? null : TestUtils.tempFile("truststore2", ".jks");
        Map<String, Object> sslConfig2 = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile2).usePem(usePem).build();
        this.configureSslBuilderClass(sslConfig2);
        try {
            sslFactory.validateReconfiguration(sslConfig2);
            Assertions.fail((String)"ValidateReconfiguration did not fail as expected");
        }
        catch (ConfigException configException) {
            // empty catch block
        }
    }

    @Test
    public void testCertificateEntriesValidation() throws Exception {
        this.verifyCertificateEntriesValidation(false, this.tlsProtocol);
    }

    @Test
    public void testPemCertificateEntriesValidation() throws Exception {
        this.verifyCertificateEntriesValidation(true, this.tlsProtocol);
    }

    private void verifyCertificateEntriesValidation(boolean usePem, String tlsProtocol) throws Exception {
        File trustStoreFile = usePem ? null : TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).usePem(usePem).build();
        this.configureSslBuilderClass(serverSslConfig);
        File newTrustStoreFile = usePem ? null : TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> newCnConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(newTrustStoreFile).cn("Another CN").usePem(usePem).build();
        this.configureSslBuilderClass(newCnConfig);
        KeyStore ks1 = this.sslKeyStore(serverSslConfig);
        KeyStore ks2 = this.sslKeyStore(serverSslConfig);
        Assertions.assertEquals((Object)SslFactory.CertificateEntries.create((KeyStore)ks1), (Object)SslFactory.CertificateEntries.create((KeyStore)ks2));
        ks2.setCertificateEntry("another", ks1.getCertificate("localhost"));
        Assertions.assertEquals((Object)SslFactory.CertificateEntries.create((KeyStore)ks1), (Object)SslFactory.CertificateEntries.create((KeyStore)ks2));
        KeyStore ks3 = this.sslKeyStore(newCnConfig);
        Assertions.assertNotEquals((Object)SslFactory.CertificateEntries.create((KeyStore)ks1), (Object)SslFactory.CertificateEntries.create((KeyStore)ks3));
    }

    @Test
    public void testClientSpecifiedSslEngineFactoryUsed() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> clientSslConfig = this.sslConfigsBuilder(ConnectionMode.CLIENT).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        clientSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        SslFactory sslFactory = new SslFactory(ConnectionMode.CLIENT);
        sslFactory.configure(clientSslConfig);
        Assertions.assertInstanceOf(TestSslUtils.TestSslEngineFactory.class, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory must be of expected type");
    }

    @Test
    public void testEngineFactoryClosed() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> clientSslConfig = this.sslConfigsBuilder(ConnectionMode.CLIENT).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        clientSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        SslFactory sslFactory = new SslFactory(ConnectionMode.CLIENT);
        sslFactory.configure(clientSslConfig);
        TestSslUtils.TestSslEngineFactory engine = (TestSslUtils.TestSslEngineFactory)sslFactory.sslEngineFactory();
        Assertions.assertFalse((boolean)engine.closed);
        sslFactory.close();
        Assertions.assertTrue((boolean)engine.closed);
    }

    @Test
    public void testServerSpecifiedSslEngineFactoryUsed() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        serverSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(serverSslConfig);
        Assertions.assertInstanceOf(TestSslUtils.TestSslEngineFactory.class, (Object)sslFactory.sslEngineFactory(), (String)"SslEngineFactory must be of expected type");
    }

    @Test
    public void testInvalidSslEngineFactory() throws Exception {
        File trustStoreFile = TestUtils.tempFile("truststore", ".jks");
        Map<String, Object> clientSslConfig = this.sslConfigsBuilder(ConnectionMode.CLIENT).createNewTrustStore(trustStoreFile).useClientCert(false).build();
        clientSslConfig.put("ssl.engine.factory.class", String.class);
        SslFactory sslFactory = new SslFactory(ConnectionMode.CLIENT);
        Assertions.assertThrows(ClassCastException.class, () -> sslFactory.configure(clientSslConfig));
    }

    @Test
    public void testUsedConfigs() throws IOException, GeneralSecurityException {
        Map<String, Object> serverSslConfig = this.sslConfigsBuilder(ConnectionMode.SERVER).createNewTrustStore(TestUtils.tempFile("truststore", ".jks")).useClientCert(false).build();
        serverSslConfig.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        TestSecurityConfig securityConfig = new TestSecurityConfig(serverSslConfig);
        SslFactory sslFactory = new SslFactory(ConnectionMode.SERVER);
        sslFactory.configure(securityConfig.values());
        Assertions.assertFalse((boolean)securityConfig.unused().contains("ssl.engine.factory.class"));
    }

    @Test
    public void testDynamicUpdateCompatibility() throws Exception {
        KeyPair keyPair = TestSslUtils.generateKeyPair("RSA");
        KeyStore ks = this.createKeyStore(keyPair, "*.example.com", "Kafka", true, "localhost", "*.example.com");
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)ks, (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.com", "Kafka", true, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, " *.example.com", " Kafka ", true, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.COM", "Kafka", true, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "KAFKA", true, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "Kafka", true, "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "Kafka", true, "localhost"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.com", "Kafka", false, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.COM", "Kafka", false, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "KAFKA", false, "localhost", "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "Kafka", false, "*.example.com"), (boolean)false, (boolean)false);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "Kafka", false, "localhost"), (boolean)false, (boolean)false);
        Assertions.assertThrows(ConfigException.class, () -> SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, " *.example.com", " Kafka ", false, "localhost", "*.example.com"), (boolean)false, (boolean)false));
        Assertions.assertThrows(ConfigException.class, () -> SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.another.example.com", "Kafka", true, "*.example.com"), (boolean)false, (boolean)false));
        Assertions.assertThrows(ConfigException.class, () -> SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.EXAMPLE.COM", "Kafka", true, "*.another.example.com"), (boolean)false, (boolean)false));
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, " *.another.example.com", "Kafka ", true, "localhost", "*.another.example.com"), (boolean)true, (boolean)true);
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.com", "Kafka", true, "localhost", "*.another.example.com"), (boolean)false, (boolean)true);
        Assertions.assertThrows(ConfigException.class, () -> SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.example.com", "Kafka", true, "localhost", "*.another.example.com"), (boolean)true, (boolean)false));
        SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.another.example.com", "Kafka", true, "localhost", "*.example.com"), (boolean)true, (boolean)false);
        Assertions.assertThrows(ConfigException.class, () -> SslFactory.CertificateEntries.ensureCompatible((KeyStore)ks, (KeyStore)this.createKeyStore(keyPair, "*.another.example.com", "Kafka", true, "localhost", "*.example.com"), (boolean)false, (boolean)true));
    }

    private KeyStore createKeyStore(KeyPair keyPair, String commonName, String org, boolean utf8, String ... dnsNames) throws Exception {
        X509Certificate cert = new TestSslUtils.CertificateBuilder().sanDnsNames(dnsNames).generate(commonName, org, utf8, keyPair);
        KeyStore ks = KeyStore.getInstance("PKCS12");
        ks.load(null, null);
        ks.setKeyEntry("kafka", keyPair.getPrivate(), null, new X509Certificate[]{cert});
        return ks;
    }

    private KeyStore sslKeyStore(Map<String, Object> sslConfig) {
        Object store = sslConfig.get("ssl.keystore.location") != null ? new DefaultSslEngineFactory.FileBasedStore((String)sslConfig.get("ssl.keystore.type"), (String)sslConfig.get("ssl.keystore.location"), (Password)sslConfig.get("ssl.keystore.password"), (Password)sslConfig.get("ssl.key.password"), true, false) : new DefaultSslEngineFactory.PemStore((Password)sslConfig.get("ssl.keystore.certificate.chain"), (Password)sslConfig.get("ssl.keystore.key"), (Password)sslConfig.get("ssl.key.password"), false);
        return store.get();
    }

    TestSslUtils.SslConfigsBuilder sslConfigsBuilder(ConnectionMode connectionMode) {
        return new TestSslUtils.SslConfigsBuilder(connectionMode).tlsProtocol(this.tlsProtocol).sslEnabledProtocols(Collections.singletonList(this.tlsProtocol));
    }
}

