/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.utils;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.Watchable;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.kafka.common.utils.FileWatchService;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class FileWatchServiceTest {
    private final List<FileWatchListener> listeners = new ArrayList<FileWatchListener>();
    private FileWatchService watchService;

    @BeforeEach
    public void setUp() {
        FileWatchService.useHighSensitivity();
        this.watchService = new FileWatchService();
    }

    @AfterEach
    public void tearDown() {
        if (this.watchService != null) {
            this.watchService.close();
        }
        FileWatchService.resetSensitivity();
    }

    @Test
    public void testFileWatchService() throws Exception {
        File watchDir = TestUtils.tempDirectory();
        File file = new File(watchDir, "test");
        this.verifyFileWatchListener(file, file, file.getParentFile(), s -> {
            try {
                TestUtils.writeToFile(file, s);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    @Test
    public void testWatchSymLink() throws Exception {
        File watchDir = TestUtils.tempDirectory();
        File dataDir = new File(watchDir, "..data");
        File file = new File(watchDir, "test");
        Files.createSymbolicLink(file.toPath(), Paths.get(dataDir.getAbsolutePath(), "test"), new FileAttribute[0]);
        this.verifyFileWatchListener(dataDir, file, watchDir, s -> {
            try {
                File tmpDir = new File(watchDir, (String)s);
                Files.createDirectories(tmpDir.toPath(), new FileAttribute[0]);
                TestUtils.writeToFile(new File(tmpDir, "test"), s);
                Utils.delete((File)dataDir);
                Files.createSymbolicLink(dataDir.toPath(), tmpDir.toPath(), new FileAttribute[0]);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void verifyFileWatchListener(File watchFile, File file, File watchDir, Consumer<String> writeToFile) throws Exception {
        Assertions.assertNull((Object)this.executorService());
        writeToFile.accept("1");
        FileWatchListener listener = new FileWatchListener(watchFile, file);
        WatchKey key = this.watchService.add((FileWatchService.Listener)listener);
        Assertions.assertTrue((boolean)key.isValid());
        Assertions.assertEquals(Collections.emptyList(), key.pollEvents());
        Assertions.assertEquals((Object)watchDir.getAbsoluteFile().toPath(), (Object)key.watchable());
        Assertions.assertEquals((int)1, (int)listener.currentValue);
        Assertions.assertNotNull((Object)this.executorService());
        Thread.sleep(FileWatchService.MIN_WATCH_INTERVAL.toMillis());
        int beforeUpdateCount = listener.updateCount.get();
        writeToFile.accept("2");
        TestUtils.waitForCondition(() -> listener.currentValue == 2, "Update event not processed");
        int updateCount = listener.updateCount.get();
        Assertions.assertTrue((updateCount >= beforeUpdateCount + 1 ? 1 : 0) != 0, (String)String.format("Listener update count not incremented, updateCount=%d, beforeUpdateCount=%d", updateCount, beforeUpdateCount));
        Assertions.assertTrue((updateCount <= beforeUpdateCount + 2 ? 1 : 0) != 0, (String)String.format("Listener updated too many times, updateCount=%d, beforeUpdateCount=%d", updateCount, beforeUpdateCount));
        this.watchService.remove((FileWatchService.Listener)listener);
        Assertions.assertFalse((boolean)key.isValid());
        Assertions.assertNull((Object)this.executorService());
        this.watchService.add((FileWatchService.Listener)listener);
        Assertions.assertNotNull((Object)this.executorService());
        this.watchService.close();
        Assertions.assertNull((Object)this.executorService());
    }

    @Test
    public void testMultipleListeners() throws Exception {
        File file1 = TestUtils.tempFile("1");
        File file2 = TestUtils.tempFile("2");
        File file3 = TestUtils.tempFile("3");
        File file4 = new File(TestUtils.tempDirectory(), "file.props");
        TestUtils.writeToFile(file4, "4");
        File file5 = new File(TestUtils.tempDirectory(), "file.props");
        TestUtils.writeToFile(file5, "5");
        this.addListener(file1);
        this.addListener(file1);
        this.addListener(file2);
        this.addListener(file3);
        this.addListener(file4);
        this.addListener(file5);
        this.verifyValues(1, 1, 2, 3, 4, 5);
        Thread.sleep(FileWatchService.MIN_WATCH_INTERVAL.toMillis());
        TestUtils.writeToFile(file1, "10");
        TestUtils.writeToFile(file2, "20");
        TestUtils.writeToFile(file4, "40");
        this.waitForUpdate(10, 10, 20, 3, 40, 5);
        for (int i = 0; i < this.listeners.size(); ++i) {
            this.watchService.remove((FileWatchService.Listener)this.listeners.remove(0));
            Assertions.assertEquals((Object)this.listeners.isEmpty(), (Object)(this.executorService() == null ? 1 : 0));
        }
    }

    @Test
    public void testEventTypes() throws Exception {
        ArrayBlockingQueue<MockWatchKey> watchKeys = new ArrayBlockingQueue<MockWatchKey>(10);
        final WatchService mockWatchService = (WatchService)Mockito.mock(WatchService.class);
        final File dir1 = TestUtils.tempDirectory();
        File dir2 = TestUtils.tempDirectory();
        File file1 = new File(dir1, "file.props");
        File file2 = new File(dir2, "file.props");
        File file3 = new File(dir1, "file3.props");
        TestUtils.writeToFile(file1, "1");
        TestUtils.writeToFile(file2, "2");
        TestUtils.writeToFile(file3, "3");
        final MockWatchKey key1 = new MockWatchKey(dir1.getAbsoluteFile().toPath());
        final MockWatchKey key2 = new MockWatchKey(dir2.getAbsoluteFile().toPath());
        final AtomicInteger registrations = new AtomicInteger();
        this.watchService.close();
        this.watchService = new FileWatchService(this){
            final /* synthetic */ FileWatchServiceTest this$0;
            {
                this.this$0 = this$0;
            }

            protected WatchService newWatchService() {
                return mockWatchService;
            }

            protected WatchKey register(Path watchDir) {
                registrations.incrementAndGet();
                return watchDir.getFileName().toString().equals(dir1.getName()) ? key1 : key2;
            }
        };
        Mockito.when((Object)mockWatchService.take()).then(unused -> watchKeys.take());
        Assertions.assertSame((Object)key1, (Object)this.addListener(file1));
        Assertions.assertSame((Object)key1, (Object)this.addListener(file1));
        Assertions.assertSame((Object)key2, (Object)this.addListener(file2));
        Assertions.assertSame((Object)key1, (Object)this.addListener(file3));
        Assertions.assertEquals((int)2, (int)registrations.get());
        TestUtils.writeToFile(file1, "10");
        TestUtils.writeToFile(file3, "30");
        TestUtils.writeToFile(file2, "20");
        key2.events.add(this.watchEvent(file2.toPath(), StandardWatchEventKinds.ENTRY_CREATE));
        watchKeys.add(key2);
        this.waitForUpdate(1, 1, 20, 3);
        this.verifyUpdateCounts(0, 0, 1, 0);
        key1.events.add(this.watchEvent(file1.toPath(), StandardWatchEventKinds.ENTRY_MODIFY));
        watchKeys.add(key1);
        this.waitForUpdate(10, 10, 20, 3);
        this.verifyUpdateCounts(1, 1, 1, 0);
        TestUtils.writeToFile(file1, "11");
        TestUtils.writeToFile(file2, "22");
        TestUtils.writeToFile(file3, "33");
        key1.events.add(this.watchEvent(null, StandardWatchEventKinds.OVERFLOW));
        watchKeys.add(key1);
        this.waitForUpdate(11, 11, 20, 33);
        this.verifyUpdateCounts(2, 2, 1, 1);
        TestUtils.writeToFile(file3, "34");
        file2.delete();
        key2.events.add(this.watchEvent(file2.toPath(), StandardWatchEventKinds.ENTRY_DELETE));
        key1.events.add(this.watchEvent(file3.toPath(), StandardWatchEventKinds.ENTRY_MODIFY));
        watchKeys.add(key2);
        watchKeys.add(key1);
        this.waitForUpdate(11, 11, 20, 34);
        this.verifyUpdateCounts(2, 2, 1, 2);
    }

    private <T> WatchEvent<T> watchEvent(T context, WatchEvent.Kind<T> eventType) {
        WatchEvent event = (WatchEvent)Mockito.mock(WatchEvent.class);
        Mockito.when(event.kind()).thenReturn(eventType);
        Mockito.when(event.context()).thenReturn(context);
        return event;
    }

    private ExecutorService executorService() {
        return (ExecutorService)TestUtils.fieldValue(this.watchService, FileWatchService.class, "executorService");
    }

    private WatchKey addListener(File file) {
        FileWatchListener listener = new FileWatchListener(file);
        this.listeners.add(listener);
        return this.watchService.add((FileWatchService.Listener)listener);
    }

    private void verifyValues(Integer ... values) {
        Assertions.assertEquals(Arrays.asList(values), this.listeners.stream().map(l -> l.currentValue).collect(Collectors.toList()));
    }

    private void verifyUpdateCounts(Integer ... counts) {
        Assertions.assertEquals(Arrays.asList(counts), this.listeners.stream().map(l -> l.updateCount.get()).collect(Collectors.toList()));
    }

    private void waitForUpdate(Integer ... newValues) throws Exception {
        TestUtils.waitForCondition(() -> Arrays.equals((Object[])newValues, this.listeners.stream().map(l -> l.currentValue).toArray()), "Update event not processed");
    }

    private static class FileWatchListener
    implements FileWatchService.Listener {
        private final File file;
        private final File configFile;
        volatile int currentValue;
        AtomicInteger updateCount;

        FileWatchListener(File file) {
            this(file, file);
        }

        FileWatchListener(File file, File configFile) {
            this.file = file;
            this.configFile = configFile;
            this.updateCount = new AtomicInteger();
        }

        public File file() {
            return this.file;
        }

        public void onInit() {
            this.update();
        }

        public void onUpdate() {
            this.updateCount.incrementAndGet();
            this.update();
        }

        private void update() {
            try {
                this.currentValue = Integer.parseInt(Utils.readFileAsString((String)this.configFile.getAbsolutePath()).trim());
            }
            catch (Exception e) {
                this.currentValue = -1;
            }
        }
    }

    private static class MockWatchKey
    implements WatchKey {
        private final Path path;
        private List<WatchEvent<?>> events = new ArrayList();

        MockWatchKey(Path path) {
            this.path = path;
        }

        @Override
        public List<WatchEvent<?>> pollEvents() {
            List<WatchEvent<?>> currentEvents = this.events;
            this.events = new ArrayList();
            return currentEvents;
        }

        @Override
        public boolean reset() {
            return false;
        }

        @Override
        public void cancel() {
        }

        @Override
        public boolean isValid() {
            return true;
        }

        @Override
        public Watchable watchable() {
            return this.path;
        }
    }
}

