/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.metadata.BrokerRegistrationChangeRecord;
import org.apache.kafka.common.metadata.FenceBrokerRecord;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.metadata.PartitionChangeRecord;
import org.apache.kafka.common.metadata.PartitionRecord;
import org.apache.kafka.common.metadata.RegisterBrokerRecord;
import org.apache.kafka.common.metadata.RemoveTopicRecord;
import org.apache.kafka.common.metadata.TopicRecord;
import org.apache.kafka.common.metadata.UnfenceBrokerRecord;
import org.apache.kafka.common.metadata.UnregisterBrokerRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.controller.ControllerMetrics;
import org.apache.kafka.controller.TopicIdPartition;
import org.apache.kafka.metadata.BrokerRegistrationFencingChange;
import org.apache.kafka.server.common.ApiMessageAndVersion;

final class ControllerMetricsManager {
    private final Set<Integer> registeredBrokers = new HashSet<Integer>();
    private final Set<Integer> fencedBrokers = new HashSet<Integer>();
    private int topicCount = 0;
    private final Map<TopicIdPartition, PartitionState> topicPartitions = new HashMap<TopicIdPartition, PartitionState>();
    private final Set<TopicIdPartition> offlineTopicPartitions = new HashSet<TopicIdPartition>();
    private final Set<TopicIdPartition> imbalancedTopicPartitions = new HashSet<TopicIdPartition>();
    private final ControllerMetrics controllerMetrics;

    ControllerMetricsManager(ControllerMetrics controllerMetrics) {
        this.controllerMetrics = controllerMetrics;
    }

    void replayBatch(long baseOffset, List<ApiMessageAndVersion> messages) {
        int i = 1;
        for (ApiMessageAndVersion message : messages) {
            try {
                this.replay(message.message());
            }
            catch (Exception e) {
                String failureMessage = String.format("Unable to update controller metrics for %s record, it was %d of %d record(s) in the batch with baseOffset %d.", message.message().getClass().getSimpleName(), i, messages.size(), baseOffset);
                throw new IllegalArgumentException(failureMessage, e);
            }
            ++i;
        }
    }

    void replay(ApiMessage message) {
        MetadataRecordType type = MetadataRecordType.fromId(message.apiKey());
        switch (type) {
            case REGISTER_BROKER_RECORD: {
                this.replay((RegisterBrokerRecord)message);
                break;
            }
            case UNREGISTER_BROKER_RECORD: {
                this.replay((UnregisterBrokerRecord)message);
                break;
            }
            case FENCE_BROKER_RECORD: {
                this.replay((FenceBrokerRecord)message);
                break;
            }
            case UNFENCE_BROKER_RECORD: {
                this.replay((UnfenceBrokerRecord)message);
                break;
            }
            case BROKER_REGISTRATION_CHANGE_RECORD: {
                this.replay((BrokerRegistrationChangeRecord)message);
                break;
            }
            case TOPIC_RECORD: {
                this.replay((TopicRecord)message);
                break;
            }
            case PARTITION_RECORD: {
                this.replay((PartitionRecord)message);
                break;
            }
            case PARTITION_CHANGE_RECORD: {
                this.replay((PartitionChangeRecord)message);
                break;
            }
            case REMOVE_TOPIC_RECORD: {
                this.replay((RemoveTopicRecord)message);
                break;
            }
            case CONFIG_RECORD: 
            case FEATURE_LEVEL_RECORD: 
            case CLIENT_QUOTA_RECORD: 
            case PRODUCER_IDS_RECORD: 
            case ACCESS_CONTROL_ENTRY_RECORD: 
            case REMOVE_ACCESS_CONTROL_ENTRY_RECORD: 
            case NO_OP_RECORD: {
                break;
            }
            case BROKER_REPLICA_EXCLUSION_RECORD: 
            case CLUSTER_LINK_RECORD: 
            case ENCRYPTED_ENVELOPE_RECORD: 
            case INSTALL_METADATA_ENCRYPTOR_RECORD: 
            case MIRROR_TOPIC_CHANGE_RECORD: 
            case MIRROR_TOPIC_RECORD: 
            case REMOVE_CLUSTER_LINK_RECORD: {
                break;
            }
            default: {
                throw new RuntimeException("Unhandled record type " + (Object)((Object)type));
            }
        }
    }

    private void replay(RegisterBrokerRecord record) {
        Integer brokerId = record.brokerId();
        this.registeredBrokers.add(brokerId);
        if (record.fenced()) {
            this.fencedBrokers.add(brokerId);
        } else {
            this.fencedBrokers.remove(brokerId);
        }
        this.updateBrokerStateMetrics();
    }

    private void replay(UnregisterBrokerRecord record) {
        Integer brokerId = record.brokerId();
        this.registeredBrokers.remove(brokerId);
        this.fencedBrokers.remove(brokerId);
        this.updateBrokerStateMetrics();
    }

    private void replay(FenceBrokerRecord record) {
        this.handleFencingChange(record.id(), BrokerRegistrationFencingChange.FENCE);
    }

    private void replay(UnfenceBrokerRecord record) {
        this.handleFencingChange(record.id(), BrokerRegistrationFencingChange.UNFENCE);
    }

    private void replay(BrokerRegistrationChangeRecord record) {
        BrokerRegistrationFencingChange fencingChange = BrokerRegistrationFencingChange.fromValue(record.fenced()).orElseThrow(() -> new IllegalArgumentException(String.format("Registration change record for %d has unknown value for fenced field: %x", record.brokerId(), record.fenced())));
        this.handleFencingChange(record.brokerId(), fencingChange);
    }

    private void handleFencingChange(Integer brokerId, BrokerRegistrationFencingChange fencingChange) {
        if (!this.registeredBrokers.contains(brokerId)) {
            throw new IllegalArgumentException(String.format("Broker with id %s is not registered", brokerId));
        }
        if (fencingChange == BrokerRegistrationFencingChange.FENCE) {
            this.fencedBrokers.add(brokerId);
            this.updateBrokerStateMetrics();
        } else if (fencingChange == BrokerRegistrationFencingChange.UNFENCE) {
            this.fencedBrokers.remove(brokerId);
            this.updateBrokerStateMetrics();
        }
    }

    private void updateBrokerStateMetrics() {
        this.controllerMetrics.setFencedBrokerCount(this.fencedBrokers.size());
        HashSet<Integer> activeBrokers = new HashSet<Integer>(this.registeredBrokers);
        activeBrokers.removeAll(this.fencedBrokers);
        this.controllerMetrics.setActiveBrokerCount(activeBrokers.size());
    }

    private void replay(TopicRecord record) {
        ++this.topicCount;
        this.controllerMetrics.setGlobalTopicCount(this.topicCount);
    }

    private void replay(PartitionRecord record) {
        TopicIdPartition tp = new TopicIdPartition(record.topicId(), record.partitionId());
        PartitionState partitionState = new PartitionState(record.leader(), record.replicas().get(0));
        this.topicPartitions.put(tp, partitionState);
        this.updateBasedOnPartitionState(tp, partitionState);
        this.updateTopicAndPartitionMetrics();
    }

    private void replay(PartitionChangeRecord record) {
        TopicIdPartition tp = new TopicIdPartition(record.topicId(), record.partitionId());
        if (!this.topicPartitions.containsKey(tp)) {
            throw new IllegalArgumentException(String.format("Unknown topic partitions %s", tp));
        }
        PartitionState partitionState = this.topicPartitions.computeIfPresent(tp, (key, oldValue) -> {
            PartitionState newValue = oldValue;
            if (record.replicas() != null) {
                newValue = new PartitionState(newValue.leader(), record.replicas().get(0));
            }
            if (record.leader() != -2) {
                newValue = new PartitionState(record.leader(), newValue.preferredReplica());
            }
            return newValue;
        });
        this.updateBasedOnPartitionState(tp, partitionState);
        this.updateTopicAndPartitionMetrics();
    }

    private void replay(RemoveTopicRecord record) {
        Uuid topicId = record.topicId();
        Predicate<TopicIdPartition> matchesTopic = tp -> tp.topicId() == topicId;
        --this.topicCount;
        this.topicPartitions.keySet().removeIf(matchesTopic);
        this.offlineTopicPartitions.removeIf(matchesTopic);
        this.imbalancedTopicPartitions.removeIf(matchesTopic);
        this.updateTopicAndPartitionMetrics();
    }

    private void updateBasedOnPartitionState(TopicIdPartition tp, PartitionState partitionState) {
        if (partitionState.leader() == -1) {
            this.offlineTopicPartitions.add(tp);
        } else {
            this.offlineTopicPartitions.remove(tp);
        }
        if (partitionState.leader() == partitionState.preferredReplica()) {
            this.imbalancedTopicPartitions.remove(tp);
        } else {
            this.imbalancedTopicPartitions.add(tp);
        }
    }

    private void updateTopicAndPartitionMetrics() {
        this.controllerMetrics.setGlobalTopicCount(this.topicCount);
        this.controllerMetrics.setGlobalPartitionCount(this.topicPartitions.size());
        this.controllerMetrics.setOfflinePartitionCount(this.offlineTopicPartitions.size());
        this.controllerMetrics.setPreferredReplicaImbalanceCount(this.imbalancedTopicPartitions.size());
    }

    void reset() {
        this.registeredBrokers.clear();
        this.fencedBrokers.clear();
        this.topicCount = 0;
        this.topicPartitions.clear();
        this.offlineTopicPartitions.clear();
        this.imbalancedTopicPartitions.clear();
        this.updateBrokerStateMetrics();
        this.updateTopicAndPartitionMetrics();
    }

    private static final class PartitionState {
        final int leader;
        final int preferredReplica;

        PartitionState(int leader, int preferredReplica) {
            this.leader = leader;
            this.preferredReplica = preferredReplica;
        }

        int leader() {
            return this.leader;
        }

        int preferredReplica() {
            return this.preferredReplica;
        }
    }
}

