/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.AbstractMap;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.common.message.BrokerHeartbeatRequestData;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.BrokerControlState;
import org.apache.kafka.controller.BrokerControlStates;
import org.apache.kafka.controller.BrokerHeartbeatTracker;
import org.apache.kafka.controller.BrokerIdAndEpoch;
import org.apache.kafka.metadata.placement.UsableBroker;
import org.slf4j.Logger;

public class BrokerHeartbeatManager {
    private final Logger log;
    private final BrokerHeartbeatTracker tracker;
    private final HashMap<Integer, BrokerHeartbeatState> brokers;
    private final TreeSet<BrokerHeartbeatState> active;
    private final Map<Integer, Long> shutdowns;

    BrokerHeartbeatManager(LogContext logContext, Time time, long sessionTimeoutNs) {
        this.log = logContext.logger(BrokerHeartbeatManager.class);
        this.tracker = new BrokerHeartbeatTracker(time, sessionTimeoutNs);
        this.brokers = new HashMap();
        this.active = new TreeSet<BrokerHeartbeatState>(MetadataOffsetComparator.INSTANCE);
        this.shutdowns = new HashMap<Integer, Long>();
    }

    Set<BrokerHeartbeatState> active() {
        return new TreeSet<BrokerHeartbeatState>((SortedSet<BrokerHeartbeatState>)this.active);
    }

    BrokerHeartbeatTracker tracker() {
        return this.tracker;
    }

    Time time() {
        return this.tracker.time();
    }

    Collection<BrokerHeartbeatState> brokers() {
        return this.brokers.values();
    }

    OptionalLong controlledShutdownOffset(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null || !broker.shuttingDown()) {
            return OptionalLong.empty();
        }
        return OptionalLong.of(this.shutdowns.getOrDefault(broker.id, 0L));
    }

    NavigableMap<Integer, Long> shutdowns() {
        return new TreeMap<Integer, Long>(this.shutdowns);
    }

    void fence(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker != null) {
            broker.fenced = true;
            this.active.remove(broker);
            broker.inControlledShutdown = false;
            this.shutdowns.remove(brokerId);
        }
    }

    void remove(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.remove(brokerId);
        if (broker != null) {
            this.active.remove(broker);
            this.shutdowns.remove(brokerId);
        }
    }

    boolean hasValidSession(int brokerId, long brokerEpoch) {
        return this.tracker.hasValidSession(new BrokerIdAndEpoch(brokerId, brokerEpoch));
    }

    void register(int brokerId, boolean fenced) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        long metadataOffset = -1L;
        if (broker == null) {
            broker = new BrokerHeartbeatState(brokerId, fenced, -1L, false);
            this.brokers.put(brokerId, broker);
            this.shutdowns.remove(brokerId);
        } else if (broker.fenced() != fenced) {
            metadataOffset = broker.metadataOffset;
        }
        this.touch(brokerId, fenced, metadataOffset);
    }

    void touch(int brokerId, boolean fenced, long metadataOffset) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        this.active.remove(broker);
        broker.fenced = fenced;
        broker.metadataOffset = metadataOffset;
        boolean isActive = false;
        if (fenced) {
            broker.inControlledShutdown = false;
            this.shutdowns.remove(brokerId);
        } else if (!broker.shuttingDown()) {
            isActive = true;
        }
        if (isActive) {
            this.active.add(broker);
        }
    }

    Map.Entry<Long, Integer> lowestActiveOffsetAndBroker() {
        Iterator<BrokerHeartbeatState> iterator = this.active.iterator();
        if (!iterator.hasNext()) {
            return new AbstractMap.SimpleImmutableEntry<Long, Integer>(Long.MAX_VALUE, -1);
        }
        BrokerHeartbeatState first = iterator.next();
        return new AbstractMap.SimpleImmutableEntry<Long, Integer>(first.metadataOffset, first.id);
    }

    void updateControlledShutdownOffset(int brokerId, long controlledShutDownOffset) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        if (broker.fenced()) {
            throw new RuntimeException("Cannot update the controlled shutdown state of broker " + brokerId + ", because it is fenced.");
        }
        this.active.remove(broker);
        broker.inControlledShutdown = true;
        this.shutdowns.put(brokerId, controlledShutDownOffset);
        this.log.debug("Updated the controlled shutdown offset for broker {} to {}.", (Object)brokerId, (Object)controlledShutDownOffset);
    }

    void maybeUpdateControlledShutdownOffset(int brokerId, long controlledShutDownOffset) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        if (broker.fenced()) {
            throw new RuntimeException("Fenced brokers cannot enter controlled shutdown.");
        }
        this.active.remove(broker);
        if (!broker.inControlledShutdown) {
            broker.inControlledShutdown = true;
            this.shutdowns.put(brokerId, controlledShutDownOffset);
            this.log.debug("Updated the controlled shutdown offset for broker {} to {}.", (Object)brokerId, (Object)controlledShutDownOffset);
        }
    }

    Iterator<UsableBroker> usableBrokers(Function<Integer, Optional<String>> idToRack, Set<Integer> excludedBrokerIds, Function<Integer, Integer> idToCell) {
        return new UsableBrokerIterator(this.brokers.values().iterator(), idToRack, excludedBrokerIds, idToCell);
    }

    BrokerControlState currentBrokerState(BrokerHeartbeatState broker) {
        if (broker.shuttingDown()) {
            return BrokerControlState.CONTROLLED_SHUTDOWN;
        }
        if (broker.fenced()) {
            return BrokerControlState.FENCED;
        }
        return BrokerControlState.UNFENCED;
    }

    BrokerControlStates calculateNextBrokerState(int brokerId, BrokerHeartbeatRequestData request, long registerBrokerRecordOffset, Supplier<Boolean> hasLeaderships) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        BrokerControlState currentState = this.currentBrokerState(broker);
        switch (currentState) {
            case FENCED: {
                if (request.wantShutDown()) {
                    this.log.info("Fenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                if (!request.wantFence()) {
                    if (request.currentMetadataOffset() >= registerBrokerRecordOffset) {
                        this.log.info("The request from broker {} to unfence has been granted because it has caught up with the offset of its register broker record {}.", (Object)brokerId, (Object)registerBrokerRecordOffset);
                        return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
                    }
                    if (this.log.isDebugEnabled()) {
                        this.log.debug("The request from broker {} to unfence cannot yet be granted because it has not caught up with the offset of its register broker record {}. It is still at offset {}.", new Object[]{brokerId, registerBrokerRecordOffset, request.currentMetadataOffset()});
                    }
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                return new BrokerControlStates(currentState, BrokerControlState.FENCED);
            }
            case UNFENCED: {
                if (request.wantFence()) {
                    if (request.wantShutDown()) {
                        this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted fencing", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                if (request.wantShutDown()) {
                    if (hasLeaderships.get().booleanValue()) {
                        this.log.info("Unfenced broker {} has requested and been granted a controlled shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
            }
            case CONTROLLED_SHUTDOWN: {
                if (hasLeaderships.get().booleanValue()) {
                    this.log.debug("Broker {} is in controlled shutdown state, but can not shut down because more leaders still need to be moved.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                }
                Map.Entry<Long, Integer> lowestActiveOffsetAndBroker = this.lowestActiveOffsetAndBroker();
                long controlledShutdownOffset = this.shutdowns.getOrDefault(brokerId, 0L);
                if (controlledShutdownOffset <= lowestActiveOffsetAndBroker.getKey()) {
                    this.log.info("The request from broker {} to shut down has been granted since the lowest active offset {} is now greater than the broker's controlled shutdown offset {}.", new Object[]{brokerId, lowestActiveOffsetAndBroker.getKey(), controlledShutdownOffset});
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                this.log.info("The request from broker {} to shut down can not yet be granted because the lowest active offset {} from broker {} is not greater than the broker's shutdown offset {}.", new Object[]{brokerId, lowestActiveOffsetAndBroker.getKey(), lowestActiveOffsetAndBroker.getValue(), controlledShutdownOffset});
                return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
            }
        }
        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
    }

    private BrokerHeartbeatState heartbeatStateOrThrow(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            throw new IllegalStateException("Broker " + brokerId + " is not registered.");
        }
        return broker;
    }

    static class MetadataOffsetComparator
    implements Comparator<BrokerHeartbeatState> {
        static final MetadataOffsetComparator INSTANCE = new MetadataOffsetComparator();

        MetadataOffsetComparator() {
        }

        @Override
        public int compare(BrokerHeartbeatState a, BrokerHeartbeatState b) {
            if (a.metadataOffset < b.metadataOffset) {
                return -1;
            }
            if (a.metadataOffset > b.metadataOffset) {
                return 1;
            }
            if (a.id < b.id) {
                return -1;
            }
            if (a.id > b.id) {
                return 1;
            }
            return 0;
        }
    }

    static class BrokerHeartbeatState {
        private final int id;
        private boolean fenced;
        private long metadataOffset;
        private boolean inControlledShutdown;

        BrokerHeartbeatState(int id, boolean fenced, long metadataOffset, boolean inControlledShutdown) {
            this.id = id;
            this.fenced = fenced;
            this.metadataOffset = metadataOffset;
            this.inControlledShutdown = inControlledShutdown;
        }

        int id() {
            return this.id;
        }

        boolean fenced() {
            return this.fenced;
        }

        long metadataOffset() {
            return this.metadataOffset;
        }

        void setMetadataOffset(long metadataOffset) {
            this.metadataOffset = metadataOffset;
        }

        boolean shuttingDown() {
            return this.inControlledShutdown;
        }
    }

    static class UsableBrokerIterator
    implements Iterator<UsableBroker> {
        private final Iterator<BrokerHeartbeatState> iterator;
        private final Function<Integer, Optional<String>> idToRack;
        private final Set<Integer> excludedBrokerIds;
        private final Function<Integer, Integer> idToCell;
        private UsableBroker next;

        UsableBrokerIterator(Iterator<BrokerHeartbeatState> iterator, Function<Integer, Optional<String>> idToRack, Set<Integer> excludedBrokerIds, Function<Integer, Integer> idToCell) {
            this.iterator = iterator;
            this.idToRack = idToRack;
            this.excludedBrokerIds = excludedBrokerIds;
            this.idToCell = idToCell;
            this.next = null;
        }

        @Override
        public boolean hasNext() {
            BrokerHeartbeatState result;
            if (this.next != null) {
                return true;
            }
            do {
                if (this.iterator.hasNext()) continue;
                return false;
            } while (this.excludedBrokerIds.contains((result = this.iterator.next()).id()));
            Optional<String> rack = this.idToRack.apply(result.id());
            int cellId = this.idToCell.apply(result.id());
            boolean treatAsFenced = result.fenced() || result.shuttingDown();
            this.next = new UsableBroker(result.id(), rack, treatAsFenced, cellId);
            return true;
        }

        @Override
        public UsableBroker next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            UsableBroker result = this.next;
            this.next = null;
            return result;
        }
    }
}

