/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import io.confluent.kafka.multitenant.TenantUtils;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cell;
import org.apache.kafka.common.PartitionPlacementStrategy;
import org.apache.kafka.common.Tenant;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.errors.ResourceNotFoundException;
import org.apache.kafka.common.message.AssignTenantsToCellRequestData;
import org.apache.kafka.common.message.AssignTenantsToCellResponseData;
import org.apache.kafka.common.message.DeleteTenantsResponseData;
import org.apache.kafka.common.message.DescribeTenantsResponseData;
import org.apache.kafka.common.metadata.MetadataRecordType;
import org.apache.kafka.common.metadata.RemoveTenantRecord;
import org.apache.kafka.common.metadata.TenantRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ApiError;
import org.apache.kafka.common.requests.AssignTenantsToCellRequest;
import org.apache.kafka.common.requests.DeleteTenantsRequest;
import org.apache.kafka.common.requests.DescribeTenantsRequest;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.controller.CellControlManager;
import org.apache.kafka.controller.ControllerResult;
import org.apache.kafka.controller.FeatureControlManager;
import org.apache.kafka.metadata.placement.CellAssignor;
import org.apache.kafka.metadata.placement.CellDescriber;
import org.apache.kafka.metadata.placement.TenantDescriber;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.mutable.BoundedList;
import org.slf4j.Logger;

public class TenantControlManager
implements TenantDescriber {
    private final Logger log;
    private final FeatureControlManager featureControl;
    private final CellControlManager cellControl;
    private final PartitionPlacementStrategy defaultPartitionPlacementStrategy;
    private final short replicationFactor;
    private final short tenantStripeFactor;

    public TenantControlManager(LogContext logContext, FeatureControlManager featureControl, CellControlManager cellControlManager, PartitionPlacementStrategy partitionPlacementStrategy, short replicationFactor, short tenantStripeFactor) {
        this.log = logContext.logger(TenantControlManager.class);
        this.featureControl = featureControl;
        this.cellControl = cellControlManager;
        this.defaultPartitionPlacementStrategy = partitionPlacementStrategy;
        this.replicationFactor = replicationFactor;
        this.tenantStripeFactor = tenantStripeFactor;
    }

    public ControllerResult<AssignTenantsToCellResponseData> assignTenantsToCells(AssignTenantsToCellRequest request, Set<Integer> usableBrokers, short requestVersion) {
        this.cellControl.confirmCellsSupported();
        AssignTenantsToCellResponseData data = new AssignTenantsToCellResponseData();
        ArrayList<AssignTenantsToCellResponseData.TenantAssignmentErrors> failedTenants = new ArrayList<AssignTenantsToCellResponseData.TenantAssignmentErrors>();
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (AssignTenantsToCellRequestData.TenantToCellAssignment assignment : request.tenantsToAssign()) {
            String tenantId = assignment.tenantId();
            List<Integer> cellIds = requestVersion == 0 ? List.of(Integer.valueOf(assignment.cellId())) : assignment.cellIds();
            ApiError err = this.assignTenantToCells(tenantId, cellIds, assignment.force(), usableBrokers, ((List)records)::add);
            if (ApiError.NONE.equals((Object)err)) continue;
            AssignTenantsToCellResponseData.TenantAssignmentErrors tenantAssignmentErrors = new AssignTenantsToCellResponseData.TenantAssignmentErrors();
            tenantAssignmentErrors.setTenantId(tenantId);
            tenantAssignmentErrors.setError(err.error().code());
            tenantAssignmentErrors.setErrorMessage(err.message());
            if (requestVersion == 0) {
                tenantAssignmentErrors.setCellId(assignment.cellId());
            } else {
                tenantAssignmentErrors.setCellIds(cellIds);
            }
            failedTenants.add(tenantAssignmentErrors);
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data.setFailedTenants(failedTenants));
    }

    public ControllerResult<DescribeTenantsResponseData> describeTenants(DescribeTenantsRequest request, short requestVersion) {
        this.cellControl.confirmCellsSupported();
        DescribeTenantsResponseData data = new DescribeTenantsResponseData();
        List<String> tenantIds = request.tenants().isEmpty() ? this.cellControl.tenantIds() : request.tenants();
        Collections.sort(tenantIds);
        ArrayList<DescribeTenantsResponseData.TenantDescription> tenantDescriptions = new ArrayList<DescribeTenantsResponseData.TenantDescription>();
        for (String tenantId : tenantIds) {
            Optional<List<Cell>> tenantCells = this.getTenantCells(tenantId);
            if (!TenantControlManager.isTenantExisting(tenantCells)) {
                return ControllerResult.atomicOf(List.of(), data.setErrorCode(Errors.TENANT_NOT_FOUND.code()).setErrorMessage(String.format("Tenant %s does not exist", tenantId)));
            }
            if (!TenantControlManager.isTenantMappedToCells(tenantCells)) {
                return ControllerResult.atomicOf(List.of(), data.setErrorCode(Errors.TENANT_NOT_FOUND.code()).setErrorMessage(String.format("Tenant's: %s metadata exist but no cell is assigned to the tenant", tenantId)));
            }
            List<Integer> cellIds = tenantCells.get().stream().map(Cell::cellId).collect(Collectors.toList());
            tenantDescriptions.add(TenantControlManager.createTenantDescription(requestVersion, tenantId, cellIds, this.defaultPartitionPlacementStrategy));
        }
        return ControllerResult.atomicOf(List.of(), data.setTenantDescriptions(tenantDescriptions));
    }

    void updateUnassignedTenantDescription(List<DescribeTenantsResponseData.TenantDescription> tenantDescriptions, Map<String, Uuid> clusterTopicIds, short requestVersion) {
        HashSet unassignedTenants = new HashSet();
        HashSet<String> cellAssignedTenants = new HashSet<String>();
        for (DescribeTenantsResponseData.TenantDescription tenantDescription : tenantDescriptions) {
            cellAssignedTenants.add(tenantDescription.tenantId());
        }
        for (Map.Entry entry : clusterTopicIds.entrySet()) {
            String topic = (String)entry.getKey();
            Optional tenantId = TenantUtils.extractTenantId((String)topic);
            tenantId.ifPresent(unassignedTenants::add);
        }
        unassignedTenants.removeAll(cellAssignedTenants);
        for (String string : unassignedTenants) {
            tenantDescriptions.add(TenantControlManager.createTenantDescription(requestVersion, string, null, PartitionPlacementStrategy.TENANT_IN_CELL));
        }
    }

    private static DescribeTenantsResponseData.TenantDescription createTenantDescription(int requestVersion, String tenantId, List<Integer> tenantCellIds, PartitionPlacementStrategy strategy) {
        DescribeTenantsResponseData.TenantDescription tenantDescription = new DescribeTenantsResponseData.TenantDescription().setTenantId(tenantId).setPartitionPlacementStrategy(strategy.code().intValue());
        if (requestVersion >= 1) {
            return tenantDescription.setCellIds(tenantCellIds != null ? tenantCellIds : CellDescriber.NO_CELLS);
        }
        return tenantDescription.setCellId(tenantCellIds != null && !tenantCellIds.isEmpty() ? tenantCellIds.get(0) : -1);
    }

    public ControllerResult<DeleteTenantsResponseData> deleteTenants(DeleteTenantsRequest request) {
        this.cellControl.confirmCellsSupported();
        DeleteTenantsResponseData data = new DeleteTenantsResponseData();
        ArrayList tenantIds = new ArrayList(request.data().tenants());
        Collections.sort(tenantIds);
        BoundedList records = BoundedList.newArrayBacked((int)10000);
        for (String tenantId : tenantIds) {
            Optional<List<Cell>> tenantCellOpt = this.getTenantCells(tenantId);
            if (!TenantControlManager.isTenantExisting(tenantCellOpt)) continue;
            this.removeTenant(tenantId, ((List)records)::add);
        }
        return ControllerResult.atomicOf((List<ApiMessageAndVersion>)records, data.setFailedTenants(new ArrayList()));
    }

    void replay(TenantRecord tenantRecord) {
        this.cellControl.replay(tenantRecord);
    }

    void replay(RemoveTenantRecord removeTenantRecord) {
        this.cellControl.replay(removeTenantRecord);
    }

    List<Integer> createTenantToCellsAssignmentIfNotExists(String tenantId, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<List<Cell>> tenantCellOpt = this.getTenantCells(tenantId);
        if (TenantControlManager.isTenantExisting(tenantCellOpt) && TenantControlManager.isTenantMappedToCells(tenantCellOpt)) {
            return tenantCellOpt.get().stream().map(Cell::cellId).collect(Collectors.toList());
        }
        List<Integer> tenantAssignableCells = this.cellControl.computeUsableCells(usableBrokers, this.replicationFactor, this.tenantStripeFactor).stream().map(Cell::cellId).sorted().toList();
        if (tenantAssignableCells.isEmpty()) {
            if (this.cellControl.isCellMigrationEnabled()) {
                return CellDescriber.NO_CELLS;
            }
            String errorMessage = "Cluster is unable to create partitions due to it not having any usable cells.";
            this.log.error(errorMessage);
            throw new ResourceNotFoundException(errorMessage);
        }
        if (tenantAssignableCells.size() < this.tenantStripeFactor) {
            this.log.warn("Tenant is being assigned to cells less than the required stripe factor: {}", (Object)this.tenantStripeFactor);
        }
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)this.getTenantRecord(tenantId, tenantAssignableCells), this.featureControl.metadataVersionOrThrow().tenantRecordVersion()));
        return tenantAssignableCells;
    }

    TenantRecord getTenantRecord(String tenantId, Cell cell) {
        return this.getTenantRecord(tenantId, CellDescriber.getCells(cell.cellId()));
    }

    TenantRecord getTenantRecord(String tenantId, List<Integer> cellIds) {
        if (this.featureControl.metadataVersionOrThrow().tenantRecordVersion() >= 1) {
            return new TenantRecord().setTenantId(tenantId).setCellIds(cellIds);
        }
        if (cellIds.size() > 1) {
            throw new IllegalArgumentException("Can't create tenant" + tenantId + " with multiple cells: " + cellIds.stream().map(String::valueOf).collect(Collectors.joining(", ")) + " when TenantRecord does not support multiple cells");
        }
        return new TenantRecord().setTenantId(tenantId).setCellId(cellIds.get(0));
    }

    ApiError createTenant(String tenantId, int cellId, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<Cell> cellOpt = this.cellControl.getCell(cellId);
        if (!cellOpt.isPresent()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Cell %s does not exist", cellId));
        }
        if (this.cellControl.containsTenant(tenantId)) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s already exists", tenantId));
        }
        Cell cell = cellOpt.get();
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)this.getTenantRecord(tenantId, CellDescriber.getCells(cell.cellId())), MetadataRecordType.TENANT_RECORD.highestSupportedVersion()));
        return ApiError.NONE;
    }

    ApiError assignTenantToCells(String tenantId, List<Integer> cellIds, boolean force, Set<Integer> usableBrokers, Consumer<ApiMessageAndVersion> recordConsumer) {
        Optional<List<Cell>> sourceCellsOpt = this.getTenantCells(tenantId);
        List nonExistentCellIds = cellIds.stream().filter(cellId -> !this.cellControl.getCell((int)cellId).isPresent()).collect(Collectors.toList());
        if (!nonExistentCellIds.isEmpty()) {
            return new ApiError(Errors.CELL_NOT_FOUND, String.format("Cells %s does not exist", nonExistentCellIds.stream().map(String::valueOf).collect(Collectors.joining(", "))));
        }
        List targetCells = cellIds.stream().map(this.cellControl::getCell).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        if (!TenantControlManager.isTenantExisting(sourceCellsOpt) || !TenantControlManager.isTenantMappedToCells(sourceCellsOpt)) {
            if (this.cellControl.isCellMigrationEnabled()) {
                this.log.info("Assigning cells: {} to tenant: {} with no pre-existing cell assignment", (Object)cellIds.stream().map(String::valueOf).collect(Collectors.joining(", ")), (Object)tenantId);
            } else {
                return new ApiError(Errors.TENANT_NOT_FOUND, String.format("Tenant %s does not exist", tenantId));
            }
        }
        if (sourceCellsOpt.isPresent() && sourceCellsOpt.get().size() == targetCells.size() && new HashSet(sourceCellsOpt.get()).containsAll(targetCells)) {
            return ApiError.NONE;
        }
        List cellsNotOpenForAssignment = targetCells.stream().filter(targetCell -> !CellAssignor.isCellOpenForAssignment(targetCell, usableBrokers, this.replicationFactor)).collect(Collectors.toList());
        if (!cellsNotOpenForAssignment.isEmpty()) {
            return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved to cells %s since the cells either does not have enough brokers to meet its minSize or does not have at least %s alive brokers", tenantId, cellsNotOpenForAssignment.stream().map(Cell::cellId).map(String::valueOf).collect(Collectors.joining(", ")), this.replicationFactor));
        }
        if (!force) {
            List targetCellsInProhibitedState = targetCells.stream().filter(targetCell -> CellControlManager.PROHIBITED_TARGET_STATES.contains(targetCell.state())).collect(Collectors.toList());
            if (!targetCellsInProhibitedState.isEmpty()) {
                return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved to cells %s since it is prohibited", tenantId, targetCellsInProhibitedState.stream().map(Cell::cellId).map(String::valueOf).collect(Collectors.joining(", "))));
            }
            List sourceCellsInProhibitedSourceState = sourceCellsOpt.map(cells -> cells.stream().filter(cell -> CellControlManager.PROHIBITED_SOURCE_STATES.contains(cell.state())).collect(Collectors.toList())).orElse(List.of());
            if (!sourceCellsInProhibitedSourceState.isEmpty()) {
                return new ApiError(Errors.INVALID_REQUEST, String.format("Tenant %s cannot be moved from cells %s since it is prohibited", tenantId, sourceCellsInProhibitedSourceState.stream().map(Cell::cellId).map(String::valueOf).collect(Collectors.joining(", "))));
            }
        }
        TenantRecord record = this.getTenantRecord(tenantId, cellIds);
        recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, this.featureControl.metadataVersionOrThrow().tenantRecordVersion()));
        this.log.info("Tenant {} is manually assigned cells {}", (Object)tenantId, (Object)cellIds.stream().map(String::valueOf).collect(Collectors.joining(", ")));
        return ApiError.NONE;
    }

    boolean removeTenant(String tenantId, Consumer<ApiMessageAndVersion> recordConsumer) {
        boolean tenantExists = this.cellControl.containsTenant(tenantId);
        if (tenantExists) {
            RemoveTenantRecord record = new RemoveTenantRecord().setTenantId(tenantId);
            recordConsumer.accept(new ApiMessageAndVersion((ApiMessage)record, MetadataRecordType.REMOVE_TENANT_RECORD.highestSupportedVersion()));
            this.log.info("Deleted tenant {} information", (Object)tenantId);
        } else {
            this.log.warn("Tenant {} information was already deleted", (Object)tenantId);
        }
        return tenantExists;
    }

    @Override
    public List<Integer> getTenantCellIds(String tenantId) {
        if (this.defaultPartitionPlacementStrategy != PartitionPlacementStrategy.TENANT_IN_CELL || !this.featureControl.metadataVersionOrThrow().isCellsSupported()) {
            return CellDescriber.NO_CELLS;
        }
        Optional<List<Cell>> tenantCells = this.getTenantCells(tenantId);
        if (TenantControlManager.isTenantExisting(tenantCells) && TenantControlManager.isTenantMappedToCells(tenantCells)) {
            return tenantCells.get().stream().map(Cell::cellId).collect(Collectors.toList());
        }
        return CellDescriber.NO_CELLS;
    }

    PartitionPlacementStrategy calculatePartitionPlacementStrategy(Optional<KafkaPrincipal> principalOpt) {
        if (!this.featureControl.metadataVersionOrThrow().isCellsSupported()) {
            return PartitionPlacementStrategy.CLUSTER_WIDE;
        }
        return CellAssignor.calculatePartitionPlacementStrategy(principalOpt, this.defaultPartitionPlacementStrategy);
    }

    boolean isTenantCellPlacementEnabled(Optional<KafkaPrincipal> principalOpt) {
        if (!this.featureControl.metadataVersionOrThrow().isCellsSupported()) {
            return false;
        }
        return CellAssignor.isTenantCellPlacementEnabled(principalOpt, this.defaultPartitionPlacementStrategy);
    }

    boolean isTenantCellPlacementEnabled(String tenantId) {
        if (!this.cellControl.isCellsEnabled()) {
            return false;
        }
        return !tenantId.isEmpty();
    }

    private Optional<List<Cell>> getTenantCells(String tenantId) {
        Tenant tenant = this.cellControl.getTenant(tenantId);
        if (tenant == null) {
            return Optional.empty();
        }
        Optional<List<Cell>> cellList = Optional.of(new ArrayList());
        for (Integer cellId : tenant.cellIds()) {
            Optional<Cell> cellOpt = this.cellControl.getCell(cellId);
            if (!cellOpt.isPresent()) {
                this.log.error("Tenant {} is assigned to cell {}, however the cell does not exist", (Object)tenantId, (Object)cellId);
                continue;
            }
            cellList.get().add(cellOpt.get());
        }
        return cellList;
    }

    static boolean isTenantExisting(Optional<List<Cell>> maybeCellIds) {
        return maybeCellIds.isPresent();
    }

    static boolean isTenantMappedToCells(Optional<List<Cell>> maybeCellIds) {
        return !maybeCellIds.get().isEmpty();
    }
}

