/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Random;
import org.apache.kafka.common.Node;

public class RequestManager {
    private final Map<String, ConnectionState> connections = new HashMap<String, ConnectionState>();
    private final ArrayList<Node> bootstrapServers;
    private final int retryBackoffMs;
    private final int requestTimeoutMs;
    private final Random random;

    public RequestManager(Collection<Node> bootstrapServers, int retryBackoffMs, int requestTimeoutMs, Random random) {
        this.bootstrapServers = new ArrayList<Node>(bootstrapServers);
        this.retryBackoffMs = retryBackoffMs;
        this.requestTimeoutMs = requestTimeoutMs;
        this.random = random;
    }

    public boolean hasAnyInflightRequest(long currentTimeMs) {
        boolean result = false;
        Iterator<ConnectionState> iterator = this.connections.values().iterator();
        while (iterator.hasNext()) {
            ConnectionState connection = iterator.next();
            if (connection.hasRequestTimedOut(currentTimeMs)) {
                iterator.remove();
                continue;
            }
            if (connection.isBackoffComplete(currentTimeMs)) {
                iterator.remove();
                continue;
            }
            if (!connection.hasInflightRequest(currentTimeMs)) continue;
            result = true;
            break;
        }
        return result;
    }

    public Optional<Node> findReadyBootstrapServer(long currentTimeMs) {
        if (this.hasAnyInflightRequest(currentTimeMs)) {
            return Optional.empty();
        }
        int startIndex = this.random.nextInt(this.bootstrapServers.size());
        Optional<Node> result = Optional.empty();
        for (int i = 0; i < this.bootstrapServers.size(); ++i) {
            int index = (startIndex + i) % this.bootstrapServers.size();
            Node node = this.bootstrapServers.get(index);
            if (!this.isReady(node, currentTimeMs)) continue;
            result = Optional.of(node);
            break;
        }
        return result;
    }

    public long backoffBeforeAvailableBootstrapServer(long currentTimeMs) {
        long minBackoffMs = this.retryBackoffMs;
        Iterator<ConnectionState> iterator = this.connections.values().iterator();
        while (iterator.hasNext()) {
            ConnectionState connection = iterator.next();
            if (connection.hasRequestTimedOut(currentTimeMs)) {
                iterator.remove();
                continue;
            }
            if (connection.isBackoffComplete(currentTimeMs)) {
                iterator.remove();
                continue;
            }
            if (connection.hasInflightRequest(currentTimeMs)) {
                return connection.remainingRequestTimeMs(currentTimeMs);
            }
            if (!connection.isBackingOff(currentTimeMs)) continue;
            minBackoffMs = Math.min(minBackoffMs, connection.remainingBackoffMs(currentTimeMs));
        }
        for (Node node : this.bootstrapServers) {
            if (!this.isReady(node, currentTimeMs)) continue;
            return 0L;
        }
        return minBackoffMs;
    }

    public boolean hasRequestTimedOut(Node node, long timeMs) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return false;
        }
        return state.hasRequestTimedOut(timeMs);
    }

    public boolean isReady(Node node, long timeMs) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return true;
        }
        boolean ready = state.isReady(timeMs);
        if (ready) {
            this.reset(node);
        }
        return ready;
    }

    public boolean isBackingOff(Node node, long timeMs) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return false;
        }
        return state.isBackingOff(timeMs);
    }

    public long remainingRequestTimeMs(Node node, long timeMs) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return 0L;
        }
        return state.remainingRequestTimeMs(timeMs);
    }

    public long remainingBackoffMs(Node node, long timeMs) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return 0L;
        }
        return state.remainingBackoffMs(timeMs);
    }

    public boolean isResponseExpected(Node node, long correlationId) {
        ConnectionState state = this.connections.get(node.idString());
        if (state == null) {
            return false;
        }
        return state.isResponseExpected(correlationId);
    }

    public void onResponseResult(Node node, long correlationId, boolean success, long timeMs) {
        if (this.isResponseExpected(node, correlationId)) {
            if (success) {
                this.reset(node);
            } else {
                this.connections.get(node.idString()).onResponseError(correlationId, timeMs);
            }
        }
    }

    public void onRequestSent(Node node, long correlationId, long timeMs) {
        ConnectionState state = this.connections.computeIfAbsent(node.idString(), key -> new ConnectionState(node, this.retryBackoffMs, this.requestTimeoutMs));
        state.onRequestSent(correlationId, timeMs);
    }

    public void reset(Node node) {
        this.connections.remove(node.idString());
    }

    public void resetAll() {
        this.connections.clear();
    }

    private static final class ConnectionState {
        private final Node node;
        private final int retryBackoffMs;
        private final int requestTimeoutMs;
        private State state = State.READY;
        private long lastSendTimeMs = 0L;
        private long lastFailTimeMs = 0L;
        private OptionalLong inFlightCorrelationId = OptionalLong.empty();

        private ConnectionState(Node node, int retryBackoffMs, int requestTimeoutMs) {
            this.node = node;
            this.retryBackoffMs = retryBackoffMs;
            this.requestTimeoutMs = requestTimeoutMs;
        }

        private boolean isBackoffComplete(long timeMs) {
            return this.state == State.BACKING_OFF && timeMs >= this.lastFailTimeMs + (long)this.retryBackoffMs;
        }

        boolean hasRequestTimedOut(long timeMs) {
            return this.state == State.AWAITING_RESPONSE && timeMs >= this.lastSendTimeMs + (long)this.requestTimeoutMs;
        }

        boolean isReady(long timeMs) {
            if (this.isBackoffComplete(timeMs) || this.hasRequestTimedOut(timeMs)) {
                this.state = State.READY;
            }
            return this.state == State.READY;
        }

        boolean isBackingOff(long timeMs) {
            if (this.state != State.BACKING_OFF) {
                return false;
            }
            return !this.isBackoffComplete(timeMs);
        }

        private boolean hasInflightRequest(long timeMs) {
            if (this.state != State.AWAITING_RESPONSE) {
                return false;
            }
            return !this.hasRequestTimedOut(timeMs);
        }

        long remainingRequestTimeMs(long timeMs) {
            if (this.hasInflightRequest(timeMs)) {
                return this.lastSendTimeMs + (long)this.requestTimeoutMs - timeMs;
            }
            return 0L;
        }

        long remainingBackoffMs(long timeMs) {
            if (this.isBackingOff(timeMs)) {
                return this.lastFailTimeMs + (long)this.retryBackoffMs - timeMs;
            }
            return 0L;
        }

        boolean isResponseExpected(long correlationId) {
            return this.inFlightCorrelationId.isPresent() && this.inFlightCorrelationId.getAsLong() == correlationId;
        }

        void onResponseError(long correlationId, long timeMs) {
            this.inFlightCorrelationId.ifPresent(inflightRequestId -> {
                if (inflightRequestId == correlationId) {
                    this.lastFailTimeMs = timeMs;
                    this.state = State.BACKING_OFF;
                    this.inFlightCorrelationId = OptionalLong.empty();
                }
            });
        }

        void onRequestSent(long correlationId, long timeMs) {
            this.lastSendTimeMs = timeMs;
            this.inFlightCorrelationId = OptionalLong.of(correlationId);
            this.state = State.AWAITING_RESPONSE;
        }

        public String toString() {
            return String.format("ConnectionState(node=%s, state=%s, lastSendTimeMs=%d, lastFailTimeMs=%d, inFlightCorrelationId=%s)", new Object[]{this.node, this.state, this.lastSendTimeMs, this.lastFailTimeMs, this.inFlightCorrelationId.isPresent() ? Long.valueOf(this.inFlightCorrelationId.getAsLong()) : "undefined"});
        }
    }

    private static enum State {
        AWAITING_RESPONSE,
        BACKING_OFF,
        READY;

    }
}

