/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.replication.push;

import com.yammer.metrics.core.Gauge;
import io.confluent.kafka.replication.push.PushManager;
import io.confluent.kafka.replication.push.PushSession;
import io.confluent.kafka.replication.push.PushSessionEndReason;
import io.confluent.kafka.replication.push.Pusher;
import io.confluent.kafka.replication.push.PusherThread;
import io.confluent.kafka.replication.push.ReplicationConfig;
import io.confluent.kafka.replication.push.buffer.RefCountingMemoryTracker;
import io.confluent.kafka.replication.push.metrics.PushReplicationManagerMetrics;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.utils.ThreadUtils;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.common.TopicIdPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PushManagerImpl
implements PushManager {
    private static final Logger log = LoggerFactory.getLogger(PushManagerImpl.class);
    private final PushReplicationManagerMetrics pushReplicationManagerMetrics;
    private final List<Pusher> pushers;
    private final List<ExecutorService> callbackExecutors;
    private final RefCountingMemoryTracker<MemoryRecords> tracker;
    private final AtomicReference<State> state;
    private final ConcurrentHashMap<Integer, Integer> brokerPusherMap;
    private final AtomicInteger numBrokersAssigned;
    private final int numPushersPerBroker;
    private final boolean pushReplicationModeEnabled;
    private final boolean enablePushReplicationForInternalTopics;

    public PushManagerImpl(ReplicationConfig config, Time time, PushReplicationManagerMetrics pushReplicationManagerMetrics, Function<Integer, KafkaClient> networkClientResolver) {
        final RefCountingMemoryTracker<MemoryRecords> tracker = new RefCountingMemoryTracker<MemoryRecords>(MemoryRecords::sizeInBytes, config.maxMemoryBufferBytes(), ignored -> {});
        ArrayList<ExecutorService> callbackExecutors = new ArrayList<ExecutorService>();
        this.pushers = Collections.unmodifiableList(PushManagerImpl.initPushers(config, time, pushReplicationManagerMetrics, networkClientResolver, tracker, callbackExecutors));
        this.callbackExecutors = Collections.unmodifiableList(callbackExecutors);
        this.tracker = tracker;
        this.state = new AtomicReference<State>(State.NOT_STARTED);
        this.brokerPusherMap = new ConcurrentHashMap();
        this.numBrokersAssigned = new AtomicInteger(0);
        this.numPushersPerBroker = config.numPushersPerBroker();
        this.pushReplicationManagerMetrics = pushReplicationManagerMetrics;
        this.pushReplicationManagerMetrics.registerMemoryBytesUsedGauge(new Gauge<Long>(){

            public Long value() {
                return tracker.totalBytes();
            }
        });
        this.pushReplicationModeEnabled = ReplicationConfig.pushReplicationModeEnabled(config.replicationMode());
        this.enablePushReplicationForInternalTopics = config.enablePushForInternalTopics();
    }

    @Override
    public void onLeaderAppend(TopicIdPartition topicIdPartition, Set<Integer> replicaIds, long appendOffset, AbstractRecords records) {
        int numEnqueued = replicaIds.size();
        if (numEnqueued == 0) {
            log.trace("No replicas to push to for partition {}", (Object)topicIdPartition);
            return;
        }
        if (!(records instanceof MemoryRecords)) {
            String errorMessage = String.format("Only MemoryRecords supported currently, %s given", records);
            log.error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
        if (!this.tracker.initCount((MemoryRecords)records, numEnqueued)) {
            log.debug("Tracker memory limit reached while trying to buffer records {} for partition {} and replicas {}. Stopping all active push sessions for that partition.", new Object[]{records, topicIdPartition, replicaIds});
            this.stopPush(topicIdPartition, replicaIds, PushSessionEndReason.MEMORY_BUFFER_EXHAUSTED);
            return;
        }
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicIdPartition, replicaId);
            pusher.onLeaderAppend(topicIdPartition, replicaId, appendOffset, records);
        }
    }

    @Override
    public void onHighWatermarkUpdate(TopicIdPartition topicIdPartition, Set<Integer> replicaIds, long updatedHighWatermark) {
        log.trace("onHighWatermarkUpdate called with value {} for partition {} and replicas {}", new Object[]{updatedHighWatermark, topicIdPartition, replicaIds});
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicIdPartition, replicaId);
            pusher.onHighWatermarkUpdate(topicIdPartition, replicaId, updatedHighWatermark);
        }
    }

    @Override
    public void onLogStartOffsetUpdate(TopicIdPartition topicIdPartition, Set<Integer> replicaIds, long updatedLogStartOffset) {
        log.trace("onLogStartOffsetUpdate called with offset {} for partition {} and replicas {}", new Object[]{updatedLogStartOffset, topicIdPartition, replicaIds});
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicIdPartition, replicaId);
            pusher.onLogStartOffsetUpdate(topicIdPartition, replicaId, updatedLogStartOffset);
        }
    }

    @Override
    public void startPush(TopicIdPartition topicIdPartition, PushSession pushSession) {
        int replicaId = pushSession.replicaNode().id();
        log.trace("startPush called with session {} for partition {} and replica {}", new Object[]{pushSession, topicIdPartition, replicaId});
        Pusher pusher = this.getPusher(topicIdPartition, replicaId);
        pusher.startPush(topicIdPartition, pushSession);
    }

    @Override
    public void stopPush(TopicIdPartition topicIdPartition, Set<Integer> replicaIds, PushSessionEndReason pushSessionEndReason) {
        log.trace("stopPush called for partition {} and replicas {} with pushSessionEndReason={}", new Object[]{topicIdPartition, replicaIds, pushSessionEndReason});
        for (int replicaId : replicaIds) {
            Pusher pusher = this.getPusher(topicIdPartition, replicaId);
            pusher.stopPush(topicIdPartition, replicaId, pushSessionEndReason);
        }
    }

    @Override
    public boolean isPushReplicationSupported(boolean isInternalTopic, boolean isClusterLinkDestination) {
        return this.pushReplicationModeEnabled && !isClusterLinkDestination && (!isInternalTopic || this.enablePushReplicationForInternalTopics);
    }

    @Override
    public void recordFollowerNotCaughtUp(TopicIdPartition topicIdPartition, Integer replicaId) {
        this.pushReplicationManagerMetrics.incrementFollowersNotCatchingUpCount();
    }

    @Override
    public boolean startup() {
        if (this.state.compareAndSet(State.NOT_STARTED, State.RUNNING)) {
            log.info("Starting up PushManager...");
            this.pushers.forEach(Pusher::start);
            return true;
        }
        log.info("PushManager is already started.");
        return false;
    }

    @Override
    public boolean shutdown() {
        if (this.state.compareAndSet(State.RUNNING, State.SHUTDOWN)) {
            log.info("Shutting down PushManager...");
            this.tracker.close();
            this.shutdownPushers();
            this.shutdownCallbackExecutors();
            this.pushReplicationManagerMetrics.close();
            return true;
        }
        log.info("PushManager is not running.");
        return false;
    }

    @Override
    public boolean isActive() {
        return this.state.get() == State.RUNNING;
    }

    public Pusher getPusher(TopicIdPartition topicIdPartition, int destinationBrokerId) {
        if (!this.isActive()) {
            String errorMessage = "PushManager has already been shut down!";
            log.error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
        Integer pusherId = this.brokerPusherMap.get(destinationBrokerId);
        if (pusherId == null) {
            pusherId = this.brokerPusherMap.computeIfAbsent(destinationBrokerId, k -> Math.abs(this.numBrokersAssigned.getAndAccumulate(this.numPushersPerBroker, Integer::sum) % this.pushers.size()));
        }
        int idModifier = this.numPushersPerBroker == 1 ? 0 : Math.abs(topicIdPartition.hashCode() % this.numPushersPerBroker);
        return this.pushers.get((pusherId + idModifier) % this.pushers.size());
    }

    private static List<Pusher> initPushers(ReplicationConfig config, Time time, PushReplicationManagerMetrics pushReplicationManagerMetrics, Function<Integer, KafkaClient> networkClientResolver, RefCountingMemoryTracker<MemoryRecords> tracker, List<ExecutorService> callbackExecutors) {
        ArrayList<Pusher> newPushers = new ArrayList<Pusher>();
        for (int pusherId = 0; pusherId < config.maxPushers(); ++pusherId) {
            ExecutorService pusherExecutor = Executors.newSingleThreadExecutor();
            PusherThread newPusher = PusherThread.newPusher(pusherId, config, pushReplicationManagerMetrics, networkClientResolver.apply(pusherId), tracker, time, pusherExecutor);
            newPushers.add(newPusher);
            callbackExecutors.add(pusherExecutor);
        }
        return newPushers;
    }

    private void shutdownPushers() {
        this.pushers.forEach(Pusher::shutdown);
    }

    private void shutdownCallbackExecutors() {
        this.callbackExecutors.forEach(executorService -> ThreadUtils.shutdownExecutorServiceQuietly((ExecutorService)executorService, (long)30L, (TimeUnit)TimeUnit.SECONDS));
    }

    private static enum State {
        NOT_STARTED,
        RUNNING,
        SHUTDOWN;

    }
}

