/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.epoch;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.server.common.CheckpointFileConfig;
import org.apache.kafka.server.util.Scheduler;
import org.apache.kafka.storage.internals.checkpoint.LeaderEpochCheckpointFile;
import org.apache.kafka.storage.internals.log.EpochEntry;
import org.slf4j.Logger;

public final class LeaderEpochFileCache {
    private final Logger log;
    private final TopicPartition topicPartition;
    private final LeaderEpochCheckpointFile checkpoint;
    private final Scheduler scheduler;
    private final ReentrantReadWriteLock updateLock = new ReentrantReadWriteLock();
    private final Object flushLock = new Object();
    private final TreeMap<Integer, EpochEntry> epochs = new TreeMap();
    private int epochsVersion;
    private int flushedEpochsVersion;

    public LeaderEpochFileCache(TopicPartition topicPartition, LeaderEpochCheckpointFile checkpoint, Scheduler scheduler) {
        this.checkpoint = checkpoint;
        this.topicPartition = topicPartition;
        this.scheduler = scheduler;
        LogContext logContext = new LogContext("[LeaderEpochCache " + String.valueOf(topicPartition) + "] ");
        this.log = logContext.logger(LeaderEpochFileCache.class);
        checkpoint.read().forEach(this::assign);
    }

    public void handleCheckpointFileConfigChange(CheckpointFileConfig config) {
        this.checkpoint.updateChecksumProtection(config.checksumProtected());
        if (!config.checksumProtected()) {
            this.forceFlush();
        }
    }

    private boolean shouldFlushEpochCache(int currEpochsVersion) {
        return currEpochsVersion > this.flushedEpochsVersion;
    }

    public boolean isDirty() {
        return this.shouldFlushEpochCache(this.epochsVersion);
    }

    private LeaderEpochFileCache(List<EpochEntry> epochEntries, TopicPartition topicPartition, LeaderEpochCheckpointFile checkpoint, Scheduler scheduler) {
        this.checkpoint = checkpoint;
        this.topicPartition = topicPartition;
        this.scheduler = scheduler;
        LogContext logContext = new LogContext("[LeaderEpochCache " + String.valueOf(topicPartition) + "] ");
        this.log = logContext.logger(LeaderEpochFileCache.class);
        for (EpochEntry entry : epochEntries) {
            this.epochs.put(entry.epoch, entry);
        }
    }

    public void assign(int epoch, long startOffset) {
        EpochEntry entry = new EpochEntry(epoch, startOffset);
        if (this.assign(entry)) {
            this.log.debug("Appended new epoch entry {}. Cache now contains {} entries.", (Object)entry, (Object)this.epochs.size());
        }
    }

    private boolean isUpdateNeeded(EpochEntry entry) {
        return this.latestEntry().map(lastEntry -> entry.epoch != lastEntry.epoch || entry.startOffset < lastEntry.startOffset).orElse(true);
    }

    private boolean assign(EpochEntry entry) {
        if (entry.epoch < 0 || entry.startOffset < 0L) {
            throw new IllegalArgumentException("Received invalid partition leader epoch entry " + String.valueOf(entry));
        }
        if (!this.isUpdateNeeded(entry)) {
            return false;
        }
        this.updateLock.writeLock().lock();
        try {
            if (this.isUpdateNeeded(entry)) {
                this.maybeTruncateNonMonotonicEntries(entry);
                this.epochs.put(entry.epoch, entry);
                ++this.epochsVersion;
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void restore(List<EpochEntry> entries) {
        EpochSnapshot epochsSnapshotToFlush;
        this.updateLock.writeLock().lock();
        try {
            this.epochs.clear();
            for (EpochEntry entry : entries) {
                this.maybeTruncateNonMonotonicEntries(entry);
                this.epochs.put(entry.epoch, entry);
            }
            ++this.epochsVersion;
            epochsSnapshotToFlush = this.createEpochSnapshot();
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
        this.writeToFile(epochsSnapshotToFlush);
    }

    private void maybeTruncateNonMonotonicEntries(EpochEntry newEntry) {
        List<EpochEntry> removedEpochs = LeaderEpochFileCache.removeWhileMatching(this.epochs.descendingMap().entrySet().iterator(), entry -> entry.epoch >= newEntry.epoch || entry.startOffset >= newEntry.startOffset);
        if (removedEpochs.size() > 1 || !removedEpochs.isEmpty() && removedEpochs.get((int)0).startOffset != newEntry.startOffset) {
            this.log.warn("New epoch entry {} caused truncation of conflicting entries {}. Cache now contains {} entries.", new Object[]{newEntry, removedEpochs, this.epochs.size()});
        }
    }

    private static List<EpochEntry> removeWhileMatching(Iterator<Map.Entry<Integer, EpochEntry>> iterator, Predicate<EpochEntry> predicate) {
        ArrayList<EpochEntry> removedEpochs = new ArrayList<EpochEntry>();
        while (iterator.hasNext()) {
            EpochEntry entry = iterator.next().getValue();
            if (predicate.test(entry)) {
                removedEpochs.add(entry);
                iterator.remove();
                continue;
            }
            return removedEpochs;
        }
        return removedEpochs;
    }

    public boolean nonEmpty() {
        this.updateLock.readLock().lock();
        try {
            boolean bl = !this.epochs.isEmpty();
            return bl;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public Optional<EpochEntry> latestEntry() {
        this.updateLock.readLock().lock();
        try {
            Optional<EpochEntry> optional = Optional.ofNullable(this.epochs.lastEntry()).map(Map.Entry::getValue);
            return optional;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public Optional<Integer> latestEpoch() {
        return this.latestEntry().map(epochEntry -> epochEntry.epoch);
    }

    public OptionalInt previousEpoch() {
        this.updateLock.readLock().lock();
        try {
            OptionalInt optionalInt = this.latestEntry().flatMap(entry -> Optional.ofNullable(this.epochs.lowerEntry(entry.epoch))).map(integerEpochEntryEntry -> OptionalInt.of((Integer)integerEpochEntryEntry.getKey())).orElseGet(OptionalInt::empty);
            return optionalInt;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public Optional<EpochEntry> earliestEntry() {
        this.updateLock.readLock().lock();
        try {
            Optional<EpochEntry> optional = Optional.ofNullable(this.epochs.firstEntry()).map(Map.Entry::getValue);
            return optional;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public OptionalInt previousEpoch(int epoch) {
        this.updateLock.readLock().lock();
        try {
            OptionalInt optionalInt = LeaderEpochFileCache.toOptionalInt(this.epochs.lowerKey(epoch));
            return optionalInt;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public Optional<EpochEntry> previousEntry(int epoch) {
        this.updateLock.readLock().lock();
        try {
            Optional<EpochEntry> optional = Optional.ofNullable(this.epochs.lowerEntry(epoch)).map(Map.Entry::getValue);
            return optional;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public OptionalInt nextEpoch(int epoch) {
        this.updateLock.readLock().lock();
        try {
            OptionalInt optionalInt = LeaderEpochFileCache.toOptionalInt(this.epochs.higherKey(epoch));
            return optionalInt;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    private static OptionalInt toOptionalInt(Integer value) {
        return value != null ? OptionalInt.of(value) : OptionalInt.empty();
    }

    public Optional<EpochEntry> epochEntry(int epoch) {
        this.updateLock.readLock().lock();
        try {
            Optional<EpochEntry> optional = Optional.ofNullable(this.epochs.get(epoch));
            return optional;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map.Entry<Integer, Long> endOffsetFor(int requestedEpoch, long logEndOffset) {
        this.updateLock.readLock().lock();
        try {
            Map.Entry<Integer, EpochEntry> floorEntry;
            Map.Entry<Integer, EpochEntry> higherEntry;
            AbstractMap.SimpleImmutableEntry<Integer, Long> epochAndOffset = requestedEpoch == -1 ? new AbstractMap.SimpleImmutableEntry<Integer, Long>(-1, -1L) : (this.latestEpoch().isPresent() && this.latestEpoch().get() == requestedEpoch ? new AbstractMap.SimpleImmutableEntry<Integer, Long>(requestedEpoch, logEndOffset) : ((higherEntry = this.epochs.higherEntry(requestedEpoch)) == null ? new AbstractMap.SimpleImmutableEntry<Integer, Long>(-1, -1L) : ((floorEntry = this.epochs.floorEntry(requestedEpoch)) == null ? new AbstractMap.SimpleImmutableEntry<Integer, Long>(requestedEpoch, higherEntry.getValue().startOffset) : new AbstractMap.SimpleImmutableEntry<Integer, Long>(floorEntry.getValue().epoch, higherEntry.getValue().startOffset))));
            if (this.log.isTraceEnabled()) {
                this.log.trace("Processed end offset request for epoch {} and returning epoch {} with end offset {} from epoch cache of size {}}", new Object[]{requestedEpoch, epochAndOffset.getKey(), epochAndOffset.getValue(), this.epochs.size()});
            }
            AbstractMap.SimpleImmutableEntry<Integer, Long> simpleImmutableEntry = epochAndOffset;
            return simpleImmutableEntry;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long offsetForEpoch(int requestedEpoch) {
        this.updateLock.readLock().lock();
        try {
            long requestedStartOffset = requestedEpoch == -1 || !this.epochs.containsKey(requestedEpoch) ? -1L : this.epochs.get((Object)Integer.valueOf((int)requestedEpoch)).startOffset;
            this.log.debug("Processed start offset request for epoch {} and returning start offset {}", (Object)requestedEpoch, (Object)requestedStartOffset);
            long l = requestedStartOffset;
            return l;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long findDivergenceInEpochCache(List<EpochEntry> tieredEpochState, long firstTieredOffset, long lastTieredOffset, long firstLocalOffset, long lastLocalOffset) {
        this.updateLock.readLock().lock();
        try {
            long divergenceOffset = -1L;
            if (this.epochs.isEmpty() || tieredEpochState.isEmpty()) {
                this.log.info("Local epoch cache or the tiered state is empty. Hence, no divergence.");
                long l = divergenceOffset;
                return l;
            }
            this.log.info("Find divergence between local leader epoch cache {} [startOffset: {} lastOffset: {}] and tiered leader epoch state {} [startOffset: {} lastOffset: {}]", new Object[]{this.epochs, firstLocalOffset, lastLocalOffset, tieredEpochState, firstTieredOffset, lastTieredOffset});
            class EpochAndOffsetRange {
                final int epoch;
                final long startOffset;
                final long numMessages;

                EpochAndOffsetRange(EpochEntry epochEntry, long numMessages) {
                    this.epoch = epochEntry.epoch;
                    this.startOffset = epochEntry.startOffset;
                    this.numMessages = numMessages;
                }
            }
            ArrayList<EpochAndOffsetRange> tieredEpochAndOffsetRanges = new ArrayList<EpochAndOffsetRange>();
            for (int i = 0; i < tieredEpochState.size(); ++i) {
                EpochEntry epochEntry = tieredEpochState.get(i);
                long numMessages = i + 1 == tieredEpochState.size() ? lastTieredOffset - epochEntry.startOffset + 1L : tieredEpochState.get((int)(i + 1)).startOffset - epochEntry.startOffset;
                tieredEpochAndOffsetRanges.add(new EpochAndOffsetRange(epochEntry, numMessages));
            }
            HashMap<Integer, EpochAndOffsetRange> localEpochAndOffsetRanges = new HashMap<Integer, EpochAndOffsetRange>();
            for (Map.Entry<Integer, EpochEntry> entry : this.epochs.entrySet()) {
                Map.Entry<Integer, EpochEntry> nextEntry = this.epochs.higherEntry(entry.getKey());
                long numMessages = nextEntry == null ? lastLocalOffset - entry.getValue().startOffset + 1L : nextEntry.getValue().startOffset - entry.getValue().startOffset;
                localEpochAndOffsetRanges.put(entry.getValue().epoch, new EpochAndOffsetRange(entry.getValue(), numMessages));
            }
            Iterator it = tieredEpochAndOffsetRanges.iterator();
            while (divergenceOffset == -1L && it.hasNext()) {
                EpochAndOffsetRange tieredRange = (EpochAndOffsetRange)it.next();
                EpochAndOffsetRange localRange = (EpochAndOffsetRange)localEpochAndOffsetRanges.get(tieredRange.epoch);
                if (localRange != null) {
                    if (tieredRange.startOffset != localRange.startOffset && (tieredRange.epoch != this.epochs.firstEntry().getValue().epoch || localRange.startOffset < tieredRange.startOffset)) {
                        divergenceOffset = Math.min(tieredRange.startOffset, localRange.startOffset);
                    }
                    if (divergenceOffset != -1L) continue;
                    if (tieredRange.startOffset + tieredRange.numMessages < localRange.startOffset + localRange.numMessages && tieredRange.startOffset + tieredRange.numMessages - 1L != lastTieredOffset) {
                        divergenceOffset = tieredRange.startOffset + tieredRange.numMessages;
                        continue;
                    }
                    if (tieredRange.startOffset + tieredRange.numMessages <= localRange.startOffset + localRange.numMessages || localRange.startOffset + localRange.numMessages - 1L == lastLocalOffset) continue;
                    divergenceOffset = localRange.startOffset + localRange.numMessages;
                    continue;
                }
                if (tieredRange.startOffset + tieredRange.numMessages - 1L < firstLocalOffset || tieredRange.startOffset > lastLocalOffset || tieredRange.numMessages == 0L) continue;
                divergenceOffset = tieredRange.startOffset;
            }
            this.log.info("Divergence reported: {}", (Object)divergenceOffset);
            long l = divergenceOffset;
            return l;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void truncateFromEndAsyncFlush(long endOffset) {
        this.updateLock.writeLock().lock();
        try {
            List<EpochEntry> removedEntries = this.truncateFromEnd(this.epochs, endOffset);
            if (!removedEntries.isEmpty()) {
                this.scheduler.scheduleOnce("leader-epoch-cache-flush-" + String.valueOf(this.topicPartition), this::maybeFlushIfDirExists);
                this.log.debug("Cleared entries {} from epoch cache after truncating to end offset {}, leaving {} entries in the cache.", new Object[]{removedEntries, endOffset, this.epochs.size()});
            }
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void truncateFromStartAsyncFlush(long startOffset) {
        this.updateLock.writeLock().lock();
        try {
            List<EpochEntry> removedEntries = this.truncateFromStart(this.epochs, startOffset);
            if (!removedEntries.isEmpty()) {
                this.scheduler.scheduleOnce("leader-epoch-cache-flush-" + String.valueOf(this.topicPartition), this::maybeFlushIfDirExists);
                EpochEntry updatedFirstEntry = removedEntries.get(removedEntries.size() - 1);
                this.log.debug("Cleared entries {} and rewrote first entry {} after truncating to start offset {}, leaving {} in the cache.", new Object[]{removedEntries, updatedFirstEntry, startOffset, this.epochs.size()});
            }
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
    }

    public void maybeFlushIfDirExists() {
        Optional<EpochSnapshot> epochsSnapshotToFlush = Optional.empty();
        this.updateLock.readLock().lock();
        try {
            if (this.isDirty()) {
                epochsSnapshotToFlush = Optional.of(this.createEpochSnapshot());
            }
            epochsSnapshotToFlush.ifPresent(this::writeToFileIfDirExists);
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void forceFlush() {
        this.updateLock.readLock().lock();
        try {
            Object object = this.flushLock;
            synchronized (object) {
                this.log.info("Force flushing leader epoch cache due to config change for partition {}, isDirty is {}", (Object)this.topicPartition, (Object)this.isDirty());
                this.checkpoint.write(this.createEpochSnapshot().epochEntries);
            }
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    private List<EpochEntry> truncateFromStart(TreeMap<Integer, EpochEntry> epochs, long startOffset) {
        List<EpochEntry> removedEntries = LeaderEpochFileCache.removeWhileMatching(epochs.entrySet().iterator(), entry -> entry.startOffset <= startOffset);
        if (!removedEntries.isEmpty()) {
            EpochEntry firstBeforeStartOffset = removedEntries.get(removedEntries.size() - 1);
            EpochEntry updatedFirstEntry = new EpochEntry(firstBeforeStartOffset.epoch, startOffset);
            epochs.put(updatedFirstEntry.epoch, updatedFirstEntry);
            ++this.epochsVersion;
        }
        return removedEntries;
    }

    private List<EpochEntry> truncateFromEnd(TreeMap<Integer, EpochEntry> epochs, long endOffset) {
        Optional<EpochEntry> epochEntry = this.latestEntry();
        if (endOffset >= 0L && epochEntry.isPresent() && epochEntry.get().startOffset >= endOffset) {
            List<EpochEntry> removedEntries = LeaderEpochFileCache.removeWhileMatching(epochs.descendingMap().entrySet().iterator(), x -> x.startOffset >= endOffset);
            if (!removedEntries.isEmpty()) {
                ++this.epochsVersion;
            }
            return removedEntries;
        }
        return List.of();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public OptionalInt epochForOffset(long offset) {
        this.updateLock.readLock().lock();
        try {
            OptionalInt previousEpoch = OptionalInt.empty();
            for (EpochEntry epochEntry : this.epochs.values()) {
                int epoch = epochEntry.epoch;
                long startOffset = epochEntry.startOffset;
                if (startOffset == offset) {
                    OptionalInt optionalInt = OptionalInt.of(epoch);
                    return optionalInt;
                }
                if (startOffset > offset) {
                    OptionalInt optionalInt = previousEpoch;
                    return optionalInt;
                }
                previousEpoch = OptionalInt.of(epoch);
            }
            OptionalInt optionalInt = previousEpoch;
            return optionalInt;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public LeaderEpochFileCache withCheckpoint(LeaderEpochCheckpointFile leaderEpochCheckpoint) {
        this.updateLock.readLock().lock();
        try {
            LeaderEpochFileCache leaderEpochFileCache = new LeaderEpochFileCache(this.epochEntries(), this.topicPartition, leaderEpochCheckpoint, this.scheduler);
            return leaderEpochFileCache;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<EpochEntry> epochEntriesInRange(long startOffset, long endOffset) {
        this.updateLock.readLock().lock();
        try {
            TreeMap<Integer, EpochEntry> epochsCopy = new TreeMap<Integer, EpochEntry>((SortedMap<Integer, EpochEntry>)this.epochs);
            if (startOffset >= 0L) {
                this.truncateFromStart(epochsCopy, startOffset);
            }
            this.truncateFromEnd(epochsCopy, endOffset);
            ArrayList<EpochEntry> arrayList = new ArrayList<EpochEntry>(epochsCopy.values());
            return arrayList;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public void clearAndFlush() {
        EpochSnapshot epochsSnapshotToFlush;
        this.updateLock.writeLock().lock();
        try {
            this.epochs.clear();
            ++this.epochsVersion;
            epochsSnapshotToFlush = this.createEpochSnapshot();
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
        this.writeToFile(epochsSnapshotToFlush);
    }

    public void clear() {
        this.updateLock.writeLock().lock();
        try {
            this.epochs.clear();
            ++this.epochsVersion;
        }
        finally {
            this.updateLock.writeLock().unlock();
        }
    }

    public List<EpochEntry> epochEntries() {
        this.updateLock.readLock().lock();
        try {
            ArrayList<EpochEntry> arrayList = new ArrayList<EpochEntry>(this.epochs.values());
            return arrayList;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    public Map<Integer, Long> copyEpochEntries() {
        this.updateLock.readLock().lock();
        try {
            Map<Integer, Long> map = this.epochs.values().stream().collect(Collectors.toMap(e -> e.epoch, e -> e.startOffset));
            return map;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] snapshotForSegment(long endOffsetInclusive) {
        this.updateLock.readLock().lock();
        try {
            byte[] byArray = this.checkpoint.toByteArray(this.epochs.values().stream().filter(e -> e.startOffset <= endOffsetInclusive).collect(Collectors.toList()));
            return byArray;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    private EpochSnapshot createEpochSnapshot() {
        return new EpochSnapshot(new ArrayList<EpochEntry>(this.epochs.values()), this.epochsVersion);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NavigableMap<Integer, Long> epochWithOffsets() {
        this.updateLock.readLock().lock();
        try {
            TreeMap<Integer, Long> epochWithOffsets = new TreeMap<Integer, Long>();
            for (EpochEntry epochEntry : this.epochs.values()) {
                epochWithOffsets.put(epochEntry.epoch, epochEntry.startOffset);
            }
            TreeMap<Integer, Long> treeMap = epochWithOffsets;
            return treeMap;
        }
        finally {
            this.updateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeToFile(EpochSnapshot epochSnapshot) {
        Object object = this.flushLock;
        synchronized (object) {
            if (this.shouldFlushEpochCache(epochSnapshot.version)) {
                this.log.debug("Flushing leader epoch cache for partition {}, isDirty is {}", (Object)this.topicPartition, (Object)this.isDirty());
                this.checkpoint.write(epochSnapshot.epochEntries);
                this.flushedEpochsVersion = epochSnapshot.version;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeToFileIfDirExists(EpochSnapshot epochSnapshot) {
        Object object = this.flushLock;
        synchronized (object) {
            if (this.shouldFlushEpochCache(epochSnapshot.version)) {
                this.log.debug("Flushing leader epoch cache for partition {}, isDirty is {}", (Object)this.topicPartition, (Object)this.isDirty());
                this.checkpoint.writeIfDirExists(epochSnapshot.epochEntries);
                this.flushedEpochsVersion = epochSnapshot.version;
            }
        }
    }

    private static class EpochSnapshot {
        final Collection<EpochEntry> epochEntries;
        final int version;

        EpochSnapshot(Collection<EpochEntry> epochEntries, int version) {
            this.epochEntries = epochEntries;
            this.version = version;
        }
    }
}

